// -*- mode: js; js-indent-level: 4; indent-tabs-mode: nil -*-
var Cu = Components.utils;
var Ci = Components.interfaces;
var Cc = Components.classes;
var Cr = Components.results;

Cu.import("resource://gre/modules/XPCOMUtils.jsm");
Cu.import("resource://gre/modules/Services.jsm");
Cu.import("resource://gre/modules/ctypes.jsm");

Cu.import("resource://unity/libnotify.js");
Cu.import("resource://unity/unity-api.js");
Cu.import("resource://unity/unity-callback-manager.js");
Cu.import("resource://unity/unity-context-manager.js");
Cu.import("resource://unity/unity-favicon-utils.js");
Cu.import("resource://unity/unity-misc-utils.js");
Cu.import("resource://unity/unity-popup-manager.js");
Cu.import("resource://unity/unity-preview-utils.js");
Cu.import("resource://unity/unity-view-manager.js");
Cu.import("resource://unity/unity-webapps.js");
Cu.import("resource://unity/unity-window-helper.js");
Cu.import("resource://unity/unity-xid-helper.js");

var XIDHelper = new UnityXIDHelper();

var consoleService = Cc["@mozilla.org/consoleservice;1"].getService(Components.interfaces.nsIConsoleService);
var eTLDService = Cc["@mozilla.org/network/effective-tld-service;1"].getService(Components.interfaces.nsIEffectiveTLDService);
var prefs = Cc["@mozilla.org/preferences-service;1"].getService(Components.interfaces.nsIPrefService);

prefs = prefs.getBranch("extensions.unity.");

var allowed_domains = [];

function log(str) {
    /*extensions.unity.logging*/
    var enabled = false;
    try {
	enabled = prefs.getBoolPref("logging");
    } catch (x) {}
    if (enabled) {
        consoleService.logStringMessage(str);
    }
}

function setTimeout(callback, time) {
    var timer = Components.classes["@mozilla.org/timer;1"]
        .createInstance(Components.interfaces.nsITimer);
    timer.initWithCallback(callback, time, Components.interfaces.nsITimer.TYPE_ONE_SHOT);
    return timer;
}

function toDataURL(uri, callback, sx, sy, sw, sh) {
    if (uri.match(/^\s{0,}file:/i))
	return;
    if (uri.match(/^\s{0,}https:/i))
	return;

    var mediator = Cc["@mozilla.org/appshell/window-mediator;1"]
	.getService(Ci.nsIWindowMediator);
    var win = mediator.getMostRecentWindow("navigator:browser");
    var canvas = win.document.createElementNS("http://www.w3.org/1999/xhtml",
                                              "canvas");
    var img = new win.Image();

    img.onload = function() {
	var width = img.width, height = img.height;

	var hasExtraArgs = sx || sx === 0;
	if (hasExtraArgs) {
            width = sw;
            height = sh;
	}

	var dx  = 0, dy = 0;

	if (width > height)
            dy = width - height;
	else
            dx = height - width;

	canvas.width = width + dx;
	canvas.height = height + dy;

	var ctx = canvas.getContext('2d');
	if (hasExtraArgs)
            ctx.drawImage(img, sx, sy, sw, sh, dx / 2, dy / 2, width, height);
	else
            ctx.drawImage(img, dx / 2, dy / 2);
	callback(true, canvas.toDataURL());
    };
    img.src = uri;
}

function requestIntegration(uwa, aWindow, domain, name, iconUrl, realInit) {
    function integrateActionCallback() {
	uwa.permissions_allow_domain(domain);

	realInit(name, iconUrl, true);
    }
    function dontAskActionCallback() {
	uwa.permissions_dontask_domain(domain);
    }

    if (uwa.permissions_get_domain_dontask(domain)) {
	return;
    }

    if (domain == "" || uwa.permissions_get_domain_allowed(domain)) {
	try {
	    realInit(name, iconUrl);
	}
	catch (e) {
	    consoleService.logStringMessage("Unity Extension: Error calling realInit: " + e.toString() + e.line + e);
	}
	return;
    }
    var mainWindow = aWindow.QueryInterface(Components.interfaces.nsIInterfaceRequestor)
	.getInterface(Components.interfaces.nsIWebNavigation)
	.QueryInterface(Components.interfaces.nsIDocShellTreeItem)
	.rootTreeItem
	.QueryInterface(Components.interfaces.nsIInterfaceRequestor)
        .getInterface(Components.interfaces.nsIDOMWindow);
    var PopupManager = new UnityPopupManager(mainWindow);

    PopupManager.requestIntegration(aWindow.document,
				    name, domain,
				    integrateActionCallback,
				    dontAskActionCallback);
}

function BrandingRemover(mainWindow) {
    this._init(mainWindow);
}

BrandingRemover.prototype = {
    _value: '',

    _tabs: [],

    _init: function (mainWindow) {
	this._mainWindow = mainWindow;
    },

    _func: function (id, oldValue, newValue) {
	this._value = newValue;
	if (this._mainWindow.gBrowser.contentWindow.document.title)
	    return this._mainWindow.gBrowser.contentWindow.document.title;
	return newValue;
    },

    activate: function(aWindow) {
	if (this._tabs.indexOf(aWindow) == -1)
	    this._tabs.push(aWindow);
	if (this._tabs.length == 1) {
	    this._mainWindow.document.watch('title', this._func.bind(this));
	    this._mainWindow.document.title = this._mainWindow.document.title;
	}
    },

    deactivate: function(aWindow) {
	this._tabs = this._tabs.filter(function (element, index, array) {
            return element != aWindow;
        });
	if (!this._tabs.length) {
	    this._mainWindow.document.unwatch('title');
	    if (this._value)
		this._mainWindow.document.title = this._value;
	    this._value = undefined;
	}
    }
};

var _windows = new WeakMap();

function getBrandingRemover(mainWindow) {
    var res = _windows.get(mainWindow);

    if (!res) {
	res = new BrandingRemover(mainWindow);
	_windows.set(mainWindow, res);
    }

    return res;
}

function UnityObject (uwa, service, ContextManager, ViewManager, aWindow) {
    this._init(uwa, service, ContextManager, ViewManager, aWindow);
}

UnityObject.prototype = {
    _unity: null,
    uwa: null,
    service: null,
    _contextManager: null,
    _viewManager: null,

    _aWindow: null,
    _brandingRemover: null,
    _mainWindow: null,
    _windowHelper: null,
    _appRaiseCb: null,
    _appCloseCb: null,

    _callbackManager: null,

    _init: function (uwa, service, ContextManager, ViewManager, aWindow) {
        this._events = [];
        this._unity = { context: null, onInitCallbacks: [], self: this };
        this.uwa = uwa;
        this.service = service;
        this._contextManager = ContextManager;
        this._viewManager = ViewManager;

	this._callbackManager = new UnityCallbackManager();

        this._setWindow(aWindow);
    },

    _listenEvent: function(target, name, cb) {
        var descr = { target: target, name: name, cb: cb.bind(this) };

        target.addEventListener(name, descr.cb, false);

        this._events.push(descr);
    },

    _removeAllListeners: function() {
        for (var i = 0; i < this._events.length; i++) {
            let descr = this._events[i];
            descr.target.removeEventListener(descr.name, descr.cb, false);
        }
        this._events = [];
    },

    getAPI: function() {
        return makeAPI(setTimeout, this._contextAddApplicationActionsAPI.bind(this), this._initAPI.bind(this), toDataURL,
                       log, this.uwa, this._unity, this._callbackManager, this._acceptDataAPICb.bind(this));
    },

    _initAPI: function(props) {
	var name = props.name, iconUrl = props.iconUrl, oninit = props.onInit, domain = props.domain, homepage = props.homepage, login = props.login, mimeTypes = props.mimeTypes;
	var toplevelWindow = this._windowHelper.FindToplevelContentWindow (this._aWindow);

	if (!mimeTypes) {
	    mimeTypes = null;
	}
	if (!iconUrl) {
	    iconUrl = null;
	}

	if (domain === undefined) {
	    domain = this._aWindow.document.domain;
	} else {
	    var currentBaseDomain = eTLDService.getBaseDomainFromHost(this._aWindow.document.domain);

	    if ((domain.indexOf(currentBaseDomain,
				domain.length - currentBaseDomain.length) == -1)) {
		throw Error("invalid supplied domain " + domain + " does not have base suffix (" + currentBaseDomain + ")");
	    }
	}

        if (login) {
            Cc["@mozilla.org/observer-service;1"]
                .getService(Components.interfaces.nsIObserverService)
                .notifyObservers(null, "webapps-login", JSON.stringify({ login: login, domain: domain }));
        }

        this._props = {
            name: name,
            iconUrl: iconUrl,
            oninit: oninit,
            domain: domain,
            homepage: homepage,
            login: login,
            mimeTypes: mimeTypes
        };

        requestIntegration(this.uwa, this._aWindow, domain, name, iconUrl, this._realInit.bind(this));
    },

    _contextPrepared: function (preparedContext) {
        if (this._unity.context == null) {return;}

        allowed_domains.push(this._props.domain);

	var previewRequested = this._callbackManager.makeCallback (this.uwa.ContextPreviewCallbackType,
                                                                   this._previewRequested.bind(this), "preview");

        this._unity.contextReady = true;

	this.uwa.context_add_icon(this._unity.context, UnityFaviconUtils.getFaviconForDocument (this._aWindow.document), 16);

	this.uwa.context_on_raise_callback (this._unity.context, this._appRaiseCb, null);
	this.uwa.context_on_close_callback (this._unity.context, this._appCloseCb, null);
	this.uwa.context_set_preview_requested_callback (this._unity.context, previewRequested, null);

	if (this._props.homepage)
	    this.uwa.context_set_homepage (this._unity.context, this._props.homepage);

	for (var c in this._unity.onInitCallbacks) {
	    var callback = this._unity.onInitCallbacks[c];
	    if (callback != null) {
		try {
		    callback();
		} catch (e) {
		    consoleService.logStringMessage("Unity Extension, error calling onInit callback: " + e.toString());
		}
	    }
	}

	var toplevelWindow = this._windowHelper.FindToplevelContentWindow (this._aWindow);
	if (this._viewManager.windowIsSelected (this._aWindow, toplevelWindow)) {
	    this._viewManager.setActiveView (this._unity.context, toplevelWindow);
	} else {
	    this.uwa.context_set_view_is_active (this._unity.context, 0);
	}

	this.uwa.context_set_view_location (this._unity.context, this._aWindow.document.location.toString());
	XIDHelper.setWindowXID(this._aWindow, this.uwa, this._unity.context);

	this._listenEvent(this._mainWindow.gBrowser.tabContainer, "TabSelect", this._onTabSelect);

	this._onTabSelect();
    },

    _realInit: function(name, iconUrl, firstTime) {
	var toplevelWindow = this._windowHelper.FindToplevelContentWindow(this._aWindow);

	// If we are adding the site for the first time and there is no homepage, store the point of integration.
	if (firstTime && this._props.homepage == undefined) {
	    this._props.homepage = toplevelWindow.document.location.toString();
	}

	this._unity.context = this._contextManager.findContextByWindow(toplevelWindow);

	if (this._unity.context != null) {
	    if (this._unity.contextReady && this._props.oninit != null) {
		try {
		    this._props.oninit();
		} catch (e) {
		    consoleService.logStringMessage("Unity Extension, exception calling onInit callback: " + e.toString());
		}
	    } else if (this._unity.contextReady == false) {
		this._unity.onInitCallbacks.push(this._props.oninit);
	    }
            this._props.oninit = null;
	    return;
	}

	this._unity.context = this.uwa.context_new_lazy(this.service, this._props.name, this._props.domain, this._props.iconUrl, this._props.mimeTypes);

	this._unity.context.window = toplevelWindow;
	this._unity.context.parent = this._aWindow.parent;
	this._unity.context.mainWindow = this._mainWindow;
	this._unity.context.isActive = false;
	this._unity.contextReady = false;

	this._unity.onInitCallbacks.push(this._props.oninit);
        this._props.oninit = null;

	this._contextManager.addContext(this._unity.context);

	this.uwa.context_prepare(this._unity.context,
				 this._callbackManager.makeCallback (this.uwa.ContextReadyCallbackType, this._contextPrepared.bind(this), "prepare"),
				 null);

    },

    _contextAddApplicationActionsAPI: function(actions) {
        var array = this.uwa.ApplicationActionDesc.array()(actions.length);
        for (var i = 0; i < actions.length; i++) {
            array[i].path = ctypes.char.array()(actions[i].name);
            array[i].callback = actions[i].callback;
            array[i].user_data = null;
        }
        this.uwa.context_add_application_actions(this._unity.context, array, actions.length);
        actions = [];
    },

    _acceptDataAPICb: function(context, mimes, func) {
        var array = this.uwa.StrWrapperDesc.array()(mimes.length);
        for (var i = 0; i < mimes.length; i++) {
            array[i].str = ctypes.char.array()(mimes[i]);
        }
        this.uwa.context_set_application_accept_data(this.unity.context, array, mimes.length);
        this._unity.acceptDataCallback = func;
    },

    _setWindow: function(aWindow) {
        this._aWindow = aWindow;
        this._windowHelper = new UnityWindowHelper(aWindow);
        this._mainWindow = aWindow.QueryInterface(Components.interfaces.nsIInterfaceRequestor)
	    .getInterface(Components.interfaces.nsIWebNavigation)
	    .QueryInterface(Components.interfaces.nsIDocShellTreeItem)
	    .rootTreeItem
	    .QueryInterface(Components.interfaces.nsIInterfaceRequestor)
            .getInterface(Components.interfaces.nsIDOMWindow);
        this._brandingRemover = getBrandingRemover(this._mainWindow);

        this._listenEvent(this._mainWindow.gBrowser.tabContainer, "TabClose", function(event) {
            if (this._mainWindow.gBrowser.getBrowserForTab(event.target) == this._mainWindow.gBrowser.getBrowserForDocument(
                this._windowHelper.FindToplevelContentWindow(aWindow).document)) {

            	this._loseInterest(true);
            }
        });
        this._listenEvent(aWindow, "pagehide", function (event) {
            this._loseInterest(false);
        });

	this._appRaiseCb = this._callbackManager.makeCallback (this.uwa.ContextRaiseCallbackType,
							       this._windowHelper.makeRaiseCallback(this._unity), "raise");
	this._appCloseCb = this._callbackManager.makeCallback (this.uwa.ContextCloseCallbackType,
							       this._windowHelper.makeCloseCallback(), "close");

    },

    _previewRequested: function (context, user_data) {
	var previewData = null;
	try {
	    previewData = UnityPreviewUtils.getPreviewForWindow(this._aWindow.document.defaultView);
	} catch (e) {
	    consoleService.logStringMessage("Unity Preview Error: " + e.toString());
	}
	return ctypes.char.array(previewData.length+1)(previewData+'\0');
    },

    _onTabSelect: function(event) {
	let window = this._mainWindow.gBrowser.contentWindow;
	let context = this._contextManager.findContextByWindow(window);

	this._viewManager.setActiveView (context, window);

	if (window != this._aWindow) {
	    this._brandingRemover.deactivate(this._aWindow);
	} else {
	    this._brandingRemover.activate(this._aWindow);
	}
    },

    _loseInterest: function (abandoned) {
        if (this._contextManager.removeContext(this._unity.context)) {
            this.uwa.context_destroy(this._unity.context, abandoned);
        }

        this._unity.context = null;
        this._brandingRemover.deactivate(this._aWindow);
        this._removeAllListeners();
        this._callbackManager.releaseAll();
    }
}

var nope = Function();

function Notification() {
}

Notification.prototype = {
    __exposedProps__: { dispatchEvent: 'r', removeEventListener: 'r', addEventListener: 'r', cancel: 'r',
                        show: 'r', ondisplay: 'r', onerror: 'r', onclick: 'r', onclose: 'r', onshow: 'r',
                        replaceId: 'r', dir: '' },
    dispatchEvent: nope,
    removeEventListener: nope,
    addEventListener: nope,
    cancel: nope,
    show: nope,
    ondisplay: null,
    onerror: null,
    onclick: null,
    onclose: null,
    onshow: null,
    replaceId: '',
    dir: ''
};

function WebkitNotifications(uwa, libnotify, aWindow) {
    this._init(uwa, libnotify, aWindow);
}

WebkitNotifications.prototype = {
    __exposedProps__: { requestPermission: 'r', checkPermission: 'r',
                        createNotification: 'r', createHTMLNotification: 'r' },

    PERMISSION_ALLOWED: 0,
    PERMISSION_NOT_ALLOWED: 1,
    PERMISSION_DENIED: 2,

    _init: function(uwa, libnotify, aWindow) {
        this._uwa = uwa;
        this._libnotify = libnotify;
        this._aWindow = aWindow;
    },

    requestPermission: function (callback) {
	var domain = this._aWindow.document.domain;

        requestIntegration(this._uwa, this._aWindow, domain, domain, null, function () {
            if (callback)
                callback();
        });
    },

    checkPermission: function () {
	var domain = this._aWindow.document.domain;

        if (this._uwa.permissions_get_domain_dontask(domain))
            return this.PERMISSION_DENIED;
        if (domain === "" || this._uwa.permissions_get_domain_allowed(domain))
            return this.PERMISSION_ALLOWED;
        return this.PERMISSION_NOT_ALLOWED;
    },

    createNotification: function(iconUrl, title, body) {
        if (this.checkPermission() !== this.PERMISSION_ALLOWED) {
            throw new Error("not allowed");
        }
        var notification = new Notification();

        notification.show = function () {
            var notification = this._libnotify.notification_new(title, body, null/*iconUrl*/);
            this._libnotify.notification_show(notification, null);
            this._libnotify.notification_unref(notification);
        }.bind(this);

        return notification;
    },

    createHTMLNotification: function() {
        if (this.checkPermission() !== this.PERMISSION_ALLOWED) {
            throw new Error("not allowed");
        }

        return new Notification();
    }
};

function UnityObjectFactory() {
    this._init();
}

FindToplevelContentWindow = function (aWindow) {
    var toplevelContentWindow = aWindow;

    while (toplevelContentWindow.parent != toplevelContentWindow){
	toplevelContentWindow = toplevelContentWindow.parent;
    }

    return toplevelContentWindow;
}

UnityObjectFactory.prototype = {
    _init: function() {
        this.uwa = new UnityWebapps();
        this.libnotify = new Libnotify();
        this.service = this.uwa.service_new();
        this.contextManager = new UnityContextManager();
        this.viewManager = new UnityViewManager(this.contextManager, this.uwa);

        this.libnotify.init("firefox");

        this._windows = new WeakMap();
    },

    _get: function(aWindow) {
        aWindow = FindToplevelContentWindow(aWindow);

        if (!this._windows.has(aWindow)) {
            this._windows.set(aWindow, { unity: new UnityObject(this.uwa, this.service, this.contextManager, this.viewManager, aWindow).getAPI(),
                                         webkitNotifications: new WebkitNotifications(this.uwa, this.libnotify, aWindow) });
        }
        return this._windows.get(aWindow);
    },

    getForWindow: function(aWindow) {
        return this._get(aWindow).unity;
    },

    getWebkitNotificationsForWindow: function(aWindow) {
        return this._get(aWindow).webkitNotifications;
    }
};

var objectFactory = new UnityObjectFactory();

UnityGlobalPropertyInitializer = {
    handle: function (aWindow) {
        if (!objectFactory.uwa.permissions_is_integration_allowed())
            return;

        var api = objectFactory.getForWindow(aWindow);

        function unity() {
        }
        unity.prototype = {
            __proto__: aWindow.wrappedJSObject.external,
            getUnityObject: function (version) {
                if (version === 1)
                    return api;
                throw new Error("incorrect version");
            },
            __exposedProps__: { 'getUnityObject': "r" }
        };
        aWindow.wrappedJSObject.external = new unity();
        aWindow.wrappedJSObject.webkitNotifications = objectFactory.getWebkitNotificationsForWindow(aWindow);
    }
};

var EXPORTED_SYMBOLS = [ "UnityGlobalPropertyInitializer", "allowed_domains" ];
