# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for ubuntuone.couch.sign."""

import ubuntu_sso

from twisted.trial.unittest import TestCase
from mocker import Mocker, ANY
from ubuntuone.couch import auth
from oauth import oauth

from ubuntuone.couch.auth import (
    APP_NAME, HMAC_SHA1, get_oauth_request_header, get_oauth_credentials,
    get_oauth_token, request)

CONSUMER_KEY = u'this_consumer_key'
CONSUMER_SECRET = u'sssssh!'
TOKEN_KEY = u'tokentokentoken'
TOKEN_SECRET = u'ssssssshhhhhh!'

CONSUMER = oauth.OAuthConsumer(CONSUMER_KEY, CONSUMER_SECRET)
TOKEN = oauth.OAuthToken(TOKEN_KEY, TOKEN_SECRET)

URL = 'https://example.com'


class QueryTestCase(TestCase):
    """Test case for u1couch.query."""

    def setUp(self):
        self.mocker = Mocker()

    def tearDown(self):
        self.mocker.restore()
        self.mocker.verify()

    def test_get_oauth_request_header(self):
        """Test get_oauth_request_header returns correct headers."""
        fake_headers = {'Authorization':
             'OAuth realm="", oauth_nonce="39941541", '
             'oauth_timestamp="1297958903", '
             'oauth_consumer_key="this_consumer_key", '
             'oauth_signature_method="HMAC-SHA1", oauth_version="1.0", '
             'oauth_token="tokentokentoken", '
             'oauth_signature="TNIfersCweWluuuJW%2FT%2FbW9IHD0%3D"'}
        mock_oauth = self.mocker.replace("oauth.oauth")
        mock_oauth.OAuthRequest                # pylint: disable=W0104
        MockOAuthRequest = self.mocker.mock()  # pylint: disable=C0103
        self.mocker.result(MockOAuthRequest)
        MockOAuthRequest.from_consumer_and_token(
            http_url=URL, http_method='GET', oauth_consumer=CONSUMER,
            token=TOKEN, parameters=None)
        oauth_request = self.mocker.mock()
        self.mocker.result(oauth_request)
        oauth_request.sign_request(HMAC_SHA1, CONSUMER, TOKEN)
        oauth_request.to_header()
        self.mocker.result(fake_headers)
        self.mocker.replay()
        headers = get_oauth_request_header(CONSUMER, TOKEN, URL)
        self.assertEquals(fake_headers, headers)

    def test_get_oauth_request_header_http(self):
        """Test get_oauth_request_header fails on http urls."""
        self.assertRaises(
            AssertionError, get_oauth_request_header, CONSUMER,
            TOKEN, 'http://example.com')

    def test_get_oauth_credentials(self):
        """Test get_oauth_credentials returns proper oauth data."""
        self.patch(auth, "_undbusify", lambda x: x)
        dbus = self.mocker.replace("dbus")
        self.mocker.replace("dbus.mainloop.glib.DBusGMainLoop")
        bus = dbus.SessionBus()
        bus.get_object(
            ubuntu_sso.DBUS_BUS_NAME, ubuntu_sso.DBUS_CRED_PATH,
            follow_name_owner_changes=True)
        mock_proxy = self.mocker.mock()
        self.mocker.result(mock_proxy)
        mock_proxy.find_credentials(APP_NAME)
        self.mocker.result({
            u'token': TOKEN_KEY,
            u'token_secret': TOKEN_SECRET,
            u'consumer_secret': CONSUMER_SECRET,
            u'consumer_key': CONSUMER_KEY})
        self.mocker.replay()
        oauth_data = get_oauth_credentials()
        self.assertEquals(TOKEN_KEY, oauth_data['token'])
        self.assertEquals(TOKEN_SECRET, oauth_data['token_secret'])
        self.assertEquals(CONSUMER_KEY, oauth_data['consumer_key'])
        self.assertEquals(CONSUMER_SECRET, oauth_data['consumer_secret'])

    def test_get_oauth_token(self):
        """Test get_oauth_token turns strings into proper OAuth."""
        token = get_oauth_token(TOKEN_KEY, TOKEN_SECRET)
        self.assertEquals(TOKEN_KEY, token.key)
        self.assertEquals(TOKEN_SECRET, token.secret)

    def test_request(self):
        """Test a full request."""
        mock_get_oauth_credentials = self.mocker.replace(
            'ubuntuone.couch.auth.get_oauth_credentials')
        mock_get_oauth_credentials()
        self.mocker.result({
            u'token': TOKEN_KEY,
            u'token_secret': TOKEN_SECRET,
            u'consumer_key': CONSUMER_KEY,
            u'consumer_secret': CONSUMER_SECRET})
        Http = self.mocker.replace("httplib2.Http")  # pylint: disable=C0103
        http = Http()
        http.request(
            'https://something.one.ubuntu.com/', method='GET', headers=ANY,
            body=None)
        fake_result = ({'status': '200'}, "Totally A-OK!")
        self.mocker.result(fake_result)
        self.mocker.replay()
        result = request('https://something.one.ubuntu.com/')
        self.assertEquals(fake_result, result)

    def test_request_plaintext(self):
        """Test a full request."""
        mock_get_oauth_credentials = self.mocker.replace(
            'ubuntuone.couch.auth.get_oauth_credentials')
        mock_get_oauth_credentials()
        self.mocker.result({
            u'token': TOKEN_KEY,
            u'token_secret': TOKEN_SECRET,
            u'consumer_key': CONSUMER_KEY,
            u'consumer_secret': CONSUMER_SECRET})
        Http = self.mocker.replace("httplib2.Http")  # pylint: disable=C0103
        http = Http()
        http.request(
            'https://something.one.ubuntu.com/', method='GET', headers=ANY,
            body=None)
        fake_result = ({'status': '200'}, "Totally A-OK!")
        self.mocker.result(fake_result)
        self.mocker.replay()
        result = request(
            'https://something.one.ubuntu.com/', sigmeth='PLAINTEXT')
        self.assertEquals(fake_result, result)
