# -*- coding: utf-8 -*-
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The test suite for the UI for the control panel for Ubuntu One."""

import os

from twisted.internet import defer

from ubuntuone.controlpanel import gui
from ubuntuone.controlpanel.backend import (
    ControlBackend,
    DEVICE_TYPE_COMPUTER,
    DEVICE_TYPE_PHONE,
)
from ubuntuone.controlpanel.tests import USER_HOME, ROOT_PATH

# Attribute 'yyy' defined outside __init__, access to a protected member
# pylint: disable=W0201, W0212


FAKE_ACCOUNT_INFO = {'type': 'Payed', 'name': 'Test me',
    'email': 'test.com', 'quota_total': '12345', 'quota_used': '9999'}

ROOT = {
    u'volume_id': '', u'path': ROOT_PATH,
    u'subscribed': 'True', u'type': ControlBackend.ROOT_TYPE,
    u'display_name': u'My Ubuntu',
}

MUSIC_PATH = os.path.join(USER_HOME, u'.ubuntuone',
                          u'Purchased from Ubuntu One')
MUSIC_FOLDER = {
    u'volume_id': u'58236', u'subscribed': u'True',
    u'type': ControlBackend.FOLDER_TYPE,
    u'path': MUSIC_PATH,
    u'suggested_path': u'~/.ubuntuone/Purchased from Ubuntu One',
    u'display_name': u'.ubuntuone/Purchased from Ubuntu One',
}

FAKE_FOLDERS_INFO = [
    # backend send this ordered by path
    {u'volume_id': u'1', u'path': os.path.join(USER_HOME, 'bar'),
     u'suggested_path': u'~/bar', u'subscribed': u'True',
     u'type': ControlBackend.FOLDER_TYPE, u'display_name': u'bar'},
    {u'volume_id': u'2', u'path': os.path.join(USER_HOME, 'baz'),
     u'suggested_path': u'~/baz', u'subscribed': u'True',
     u'type': ControlBackend.FOLDER_TYPE, u'display_name': u'baz'},
    {u'volume_id': u'0', u'path': os.path.join(USER_HOME, 'foo'),
     u'suggested_path': u'~/foo', u'subscribed': u'',
     u'type': ControlBackend.FOLDER_TYPE, u'display_name': u'foo'},
]

FAKE_SHARES_INFO = [
    # backend send this ordered by path
    {u'volume_id': u'1234', u'name': u'do',
     u'realpath': os.path.join(USER_HOME, '.local', 'share', 'ubuntuone',
                           'shares', 'do from Other User'),
     u'path': os.path.join(ROOT_PATH, 'Shared With Me', 'do from Other User'),
     u'subscribed': u'', u'type': ControlBackend.SHARE_TYPE,
     u'display_name': u'do'},
    {u'volume_id': u'5678', u'name': u're',
     u'realpath': os.path.join(USER_HOME, '.local', 'share', 'ubuntuone',
                           'shares', 're from Other User'),
     u'path': os.path.join(ROOT_PATH, 'Shared With Me', 're from Other User'),
     u'subscribed': u'True', u'type': ControlBackend.SHARE_TYPE,
     u'display_name': u're'},
]

FAKE_VOLUMES_INFO = [
    (u'', u'147852369', [ROOT] + FAKE_FOLDERS_INFO),
    (u'Other User', gui.SHARES_MIN_SIZE_FULL, FAKE_SHARES_INFO),
]

FAKE_VOLUMES_NO_FREE_SPACE_INFO = [
    (u'', u'500', [ROOT]),
    (u'No free space', u'0',
     [{u'volume_id': u'0', u'name': u'full', u'path': u'full-share',
       u'subscribed': u'', u'type': ControlBackend.SHARE_TYPE,
       u'display_name': u'something',
      }]),
    (u'Almost no free space', gui.SHARES_MIN_SIZE_FULL - 1,
     [{u'volume_id': u'1', u'name': u'almostfull', u'path': u'almost-full',
       u'subscribed': u'', u'type': ControlBackend.SHARE_TYPE,
       u'display_name': u'yadda',
      }]),
]

FAKE_VOLUMES_ONLY_ROOT_INFO = [(u'', u'5368709120', [ROOT])]
FAKE_VOLUMES_MINIMAL_INFO = [(u'', u'147852369', [ROOT, MUSIC_FOLDER])]

FAKE_DEVICE_INFO = {
    'device_id': '1258-6854', 'device_name': 'Baz',
    'device_type': DEVICE_TYPE_COMPUTER,
    'is_local': 'True', 'configurable': 'True', 'limit_bandwidth': 'True',
    'max_upload_speed': '1000', 'max_download_speed': '72548',
    'show_all_notifications': 'True',
}

FAKE_DEVICES_INFO = [
    {'device_id': '0', 'name': 'Ubuntu One @ Foo',
     'type': DEVICE_TYPE_COMPUTER,
     'is_local': '', 'configurable': ''},
    {'device_id': '1', 'name': 'Ubuntu One @ Bar',
     'type': DEVICE_TYPE_PHONE,
     'is_local': '', 'configurable': ''},
    {'device_id': '2', 'name': 'Ubuntu One @ Z',
     'type': DEVICE_TYPE_COMPUTER,
     'is_local': '', 'configurable': 'True', 'limit_bandwidth': '',
     'max_upload_speed': '0', 'max_download_speed': '0',
     'show_all_notifications': ''},
    {'device_id': '1258-6854', 'name': 'Ubuntu One @ Baz',
     'type': DEVICE_TYPE_COMPUTER,
     'is_local': 'True', 'configurable': 'True', 'limit_bandwidth': 'True',
     'max_upload_speed': '1000', 'max_download_speed': '72548',
     'show_all_notifications': 'True'},  # local
]


class FakedObject(object):
    """Fake an object, record every call."""

    next_result = None
    exposed_methods = []
    exposed_results = {}
    raise_attr_error = True

    def __init__(self, *args, **kwargs):
        self._args = args
        self._kwargs = kwargs
        self._called = {}
        for i in self.exposed_methods:
            setattr(self, i, self._record_call(i))

    def __call__(self, *args, **kwargs):
        """Skip."""

    def __getattribute__(self, attr_name):
        super_getattr = super(FakedObject, self).__getattribute__

        try:
            result = super_getattr(attr_name)
        except AttributeError:
            if super_getattr('raise_attr_error'):
                raise
            else:
                result = FakedObject()
                result.raise_attr_error = super_getattr('raise_attr_error')

        return result

    def _record_call(self, func_name):
        """Store values when calling 'func_name'."""

        def faked_call_in_faked_object(*args, **kwargs):
            """Fake 'func_name'."""
            self._called[func_name] = (args, kwargs)
            no_func_result = object()
            result = self.exposed_results.get(func_name, no_func_result)
            if result is no_func_result:
                result = self.next_result
            else:
                result = defer.succeed(result)
            return result

        return faked_call_in_faked_object


class FakeSignal(object):

    """A fake PyQt signal."""

    def __init__(self, *args, **kwargs):
        """Initialize."""
        self.target = None

    def connect(self, target):
        """Fake connect."""
        self.target = target

    def disconnect(self, *args):
        """Fake disconnect."""
        self.target = None

    def emit(self, *args):
        """Fake emit."""
        if self.target:
            self.target(*args)
