# -*- coding: utf-8 -*-
#
# Copyright 2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the First-time wizard."""

from twisted.internet import defer

from ubuntuone.controlpanel.gui.qt import wizard as gui
from ubuntuone.controlpanel.gui.qt.tests import BaseTestCase, TOKEN


BUTTONS = [
    'BackButton', 'CancelButton', 'CommitButton',
    'CustomButton1', 'CustomButton2', 'CustomButton3',
    'FinishButton', 'HelpButton', 'NextButton',
]


class AreYouSureTestCase(BaseTestCase):
    """Test suite for the "Are you sure?" dialog."""

    class_ui = gui.AreYouSure

    def test_title(self):
        """Check the window title."""
        self.assertEqual(self.ui.windowTitle(), gui.APP_NAME)
        self.assertEqual(self.ui.ui.title_label.text(), gui.ARE_YOU_SURE_TITLE)

    def test_message_label(self):
        """Check the message label text."""
        link = gui.LINK_STYLE.format(link_url=gui.UBUNTUONE_LINK,
                                     link_text=gui.UBUNTUONE_LINK)
        msg = u'%s<p>%s' % (gui.ARE_YOU_SURE_SUBTITLE,
                            gui.ARE_YOU_SURE_HELP.format(support_url=link))
        self.assertEqual(unicode(self.ui.ui.message_label.text()), msg)

    def test_buttons(self):
        """The buttons have the correct text."""
        self.assertEqual(self.ui.ui.yes_button.text(), gui.ARE_YOU_SURE_YES)
        self.assertEqual(self.ui.ui.no_button.text(), gui.ARE_YOU_SURE_NO)


class UbuntuOnePageTestCase(BaseTestCase):
    """Test the UbuntuOnePage widget."""

    class_ui = gui.UbuntuOnePage
    main_title = ''
    panel_class = gui.QtGui.QFrame
    sub_title = ''

    def test_panel_class(self):
        """The panel_class is correct."""
        self.assertIsInstance(self.ui.panel, self.panel_class)

    def test_panel_is_created(self):
        """The panel is properly added to the layout."""
        self.assertEqual(self.ui.layout().itemAt(2).widget(), self.ui.panel)

    def test_title(self):
        """The title is correct."""
        self.assertIn(self.main_title, self.ui.title())  # avoid markup

    def test_subtitle(self):
        """The subtitle is correct."""
        self.assertEqual(self.ui.subTitle(), self.sub_title)

    def test_error_label(self):
        """The error label is hidden."""
        self.assertFalse(self.ui.form_errors_label.isVisible())

    def test_is_final(self):
        """The page is not final."""
        # from the doc:

        # After calling setFinalPage(true), isFinalPage() returns true and the
        # Finish button is visible (and enabled if isComplete() returns true).

        # After calling setFinalPage(false), isFinalPage() returns true if
        # nextId() returns -1; otherwise, it returns false.

        self.patch(self.ui, 'nextId', lambda *a: 0)
        self.assertFalse(self.ui.isFinalPage())


class LicensePageTestCase(UbuntuOnePageTestCase):
    """Test the LicensePage wizard page."""

    class_ui = gui.LicensePage
    panel_class = gui.QtGui.QTextBrowser

    def test_content(self):
        """The page content is correct."""
        expected = gui.QtGui.QTextBrowser()
        expected.setHtml(gui.LICENSE_CONTENT)
        self.assertEqual(self.ui.panel.toHtml(), expected.toHtml())


class SignInPageTestCase(UbuntuOnePageTestCase):
    """Test the SignInPage wizard page."""

    class_ui = gui.SignInPage
    panel_class = gui.SignInPanel


class CloudToComputerPageTestCase(UbuntuOnePageTestCase):
    """Test the CloudToComputerPage wizard page."""

    class_ui = gui.CloudToComputerPage
    main_title = gui.CLOUD_TO_COMPUTER_TITLE
    panel_class = gui.RemoteFoldersPanel
    sub_title = gui.CLOUD_TO_COMPUTER_SUBTITLE

    def test_folder_panel_shows_remote_folders_only(self):
        """The FolderPanel shows only remote folders."""
        self.assertTrue(self.ui.panel.remote_folders)


class SettingsPageTestCase(UbuntuOnePageTestCase):
    """Test the SettingsPage wizard page."""

    class_ui = gui.SettingsPage
    panel_class = gui.PreferencesPanel


class ComputerToCloudPageTestCase(UbuntuOnePageTestCase):
    """Test the ComputerToCloudPage wizard page."""

    class_ui = gui.ComputerToCloudPage
    main_title = gui.COMPUTER_TO_CLOUD_TITLE
    panel_class = gui.LocalFoldersPanel
    sub_title = gui.COMPUTER_TO_CLOUD_SUBTITLE

    def test_is_final(self):
        """The page is not final."""
        self.assertTrue(self.ui.isFinalPage())


class UbuntuOneWizardTestCase(BaseTestCase):
    """Test the UbuntuOneWizard widget."""

    class_ui = gui.UbuntuOneWizard
    confirm_response = gui.QtGui.QDialog.Accepted

    @defer.inlineCallbacks
    def setUp(self):
        yield super(UbuntuOneWizardTestCase, self).setUp()
        self.patch(gui.utils, 'uninstall_application', self.fail)
        self.patch(self.ui.confirm_dialog, 'exec_',
                   lambda: self.confirm_response)

    def test_options(self):
        """Tne wizard options are correct."""
        options = [
            (self.ui.NoBackButtonOnStartPage, True),
            (self.ui.HaveFinishButtonOnEarlyPages, False),
        ]
        for option, expected in options:
            self.assertEqual(expected, self.ui.testOption(option))

    def test_style(self):
        """Tne wizard style is Modern."""
        self.assertEqual(self.ui.wizardStyle(), self.ui.ModernStyle)

    def test_side_widget(self):
        """The side widget is correct."""
        self.assertIsInstance(self.ui.side_widget, gui.SideWidget)
        self.assertIs(self.ui.sideWidget(), self.ui.side_widget)

    def test_confirm_dialog(self):
        """The confirm dialog is correct."""
        self.assertIsInstance(self.ui.confirm_dialog, gui.AreYouSure)

    def test_first_page(self):
        """The first page is the correct one."""
        expected = self.ui.pages[self.ui.signin_page]
        self.assertEqual(self.ui.startId(), expected)

    def test_done_accepted(self):
        """When the wizard reached the end, emit finished."""
        self.ui.finished.connect(self._set_called)

        self.ui.done(gui.QtGui.QDialog.Accepted)

        self.assertEqual(self._called, ((gui.QtGui.QDialog.Accepted,), {}))


class UbuntuOneWizardSignInTestCase(UbuntuOneWizardTestCase):
    """Test the SignInPage wizard page."""

    buttons = {'CancelButton': (gui.CLOSE_AND_SETUP_LATER, 'rejected', ())}
    page_name = 'signin'
    stage_name = page_name

    @defer.inlineCallbacks
    def setUp(self):
        yield super(UbuntuOneWizardSignInTestCase, self).setUp()
        self.page = getattr(self.ui, '%s_page' % self.page_name)
        self._move_to_this_page()

    def _move_to_this_page(self):
        """Fake the wizard is moved to this page."""
        page_id = self.ui.pages[self.page]
        if self.ui.currentId() != page_id:
            self.ui._next_id = page_id
            self.ui.next()

    def test_was_added(self):
        """The SignInPage is added correctly."""
        page_id = self.ui.pages[self.page]
        self.assertIs(self.ui.page(page_id), self.page)

    def test_enabled_at_startup(self):
        """The page is enabled at startup."""
        self.assertTrue(self.page.isEnabled())

    def test_focus_order(self):
        """The focus order is correct."""
        # can not be tested due to Qt API limitations

    def test_button_layout(self):
        """The button layout is correct."""
        msg = '%s should not be visible.'
        for button_name in BUTTONS:
            button = self.ui.button(getattr(self.ui, button_name))
            if button_name in self.buttons:
                self.assertTrue(button.isVisible(),
                                '%s should be visible.' % button_name)
            else:
                self.assertFalse(button.isVisible(), msg % button_name)

    def test_side_widget_stage(self):
        """When in this page, the side_widget' stage is correct."""
        self.assertEqual(self.ui.side_widget.stage,
            getattr(self.ui.side_widget, '%s_stage' % self.stage_name))

    def test_buttons_behavior(self):
        """Send the rejected signal when the cancel button is clicked."""
        msg = '%r should emit %r with args %r when clicked.'

        for name, (text, signal, signal_args) in self.buttons.iteritems():
            button = self.ui.button(getattr(self.ui, name))

            if text is not None:
                self.assertEqual(unicode(button.text()), text)

            if signal:
                getattr(self.ui, signal).connect(self._set_called)
                button.click()

                self.assertEqual(self._called, (signal_args, {}),
                                 msg % (name, signal, signal_args))
                self._called = False

    def test_done_rejected(self):
        """When the wizard was cancelled and user confirmed, finish."""
        self.patch(self, 'confirm_response', gui.QtGui.QDialog.Accepted)
        self.ui.rejected.connect(self._set_called)

        assert not self.ui.page(self.ui.currentId()).isFinalPage()
        self.ui.done(gui.QtGui.QDialog.Rejected)

        self.assertEqual(self._called, ((), {}))

    def test_done_rejected_confirmation_rejected(self):
        """When the wizard was cancelled but user unconfimed, do not finish."""
        self.patch(self, 'confirm_response', gui.QtGui.QDialog.Rejected)
        self.ui.accepted.connect(self._set_called)
        self.ui.rejected.connect(self._set_called)
        self.ui.finished.connect(self._set_called)

        assert not self.ui.page(self.ui.currentId()).isFinalPage()
        self.ui.done(gui.QtGui.QDialog.Rejected)

        self.assertEqual(self._called, False)


class UbuntuOneWizardCloudToComputerTestCase(UbuntuOneWizardSignInTestCase):
    """Test the CloudToComputerPage wizard page."""

    buttons = {'NextButton': (None, 'currentIdChanged', (4,))}
    page_name = 'cloud_folders'
    stage_name = 'folders'


class UbuntuOneWizardSettingsTestCase(UbuntuOneWizardSignInTestCase):
    """Test the CloudToComputerPage wizard page."""

    buttons = {'BackButton': (None, 'currentIdChanged', (1,))}
    page_name = 'settings'
    stage_name = 'folders'


class UbuntuOneWizardComputerToCloudTestCase(UbuntuOneWizardSignInTestCase):
    """Test the CloudToComputerPage wizard page."""

    buttons = {
        'FinishButton': (None, 'finished', (gui.QtGui.QDialog.Accepted,)),
        'BackButton': (None, 'currentIdChanged', (1,)),
    }
    page_name = 'local_folders'
    stage_name = 'sync'

    def test_done_rejected(self):
        """When the wizard is closed on the final page, emit rejected."""
        self.ui.finished.connect(self._set_called)

        assert self.ui.page(self.ui.currentId()).isFinalPage()
        self.ui.done(gui.QtGui.QDialog.Rejected)

        self.assertEqual(self._called, ((gui.QtGui.QDialog.Rejected,), {}))

    def test_done_rejected_confirmation_rejected(self):
        """When the wizard was cancelled but user unconfimed, do not finish."""
        # does not apply to this page


class UbuntuOneWizardLicensePage(UbuntuOneWizardSignInTestCase):
    """Test the LicensePage wizard page."""

    buttons = {
        'CommitButton': (gui.LICENSE_AGREE, None, ()),
        'CancelButton': (gui.LICENSE_DISAGREE, 'rejected', ()),
    }
    page_name = 'license'
    stage_name = 'install'

    @defer.inlineCallbacks
    def setUp(self):
        yield super(UbuntuOneWizardLicensePage, self).setUp()
        self.patch(gui.utils, 'uninstall_application', self._set_called)

    def test_done_rejected(self):
        """When the wizard is closed on the final page, emit rejected."""
        super(UbuntuOneWizardLicensePage, self).test_done_rejected()
        self.assertEqual(self._called, ((), {}))


class UbuntuOneWizardLoginTestCase(UbuntuOneWizardTestCase):
    """Test the login through the wizard."""

    method = 'login'

    @defer.inlineCallbacks
    def test_with_credentials(self):
        """Wizard is done when credentials were retrieved."""
        self.ui.currentIdChanged.connect(self._set_called)
        d = defer.succeed(TOKEN)

        def check():
            """Confirm the ui is disabled while processing."""
            self.assertFalse(self.ui.signin_page.isEnabled())
            return d

        self.patch(self.ui.backend, self.method, check)
        button = getattr(self.ui.signin_page.panel.ui,
                         '%s_button' % self.method)
        button.click()

        yield d

        self.assertTrue(self.ui.signin_page.isEnabled())
        expected_next_id = self.ui.pages[self.ui.cloud_folders_page]
        self.assertEqual(self._called, ((expected_next_id,), {}))

    @defer.inlineCallbacks
    def test_without_credentials(self):
        """Wizard is done when credentials were retrieved."""
        self.ui.currentIdChanged.connect(self._set_called)
        d = defer.succeed(None)

        def check():
            """Confirm the ui is disabled while processing."""
            self.assertFalse(self.ui.signin_page.isEnabled())
            return d

        self.patch(self.ui.backend, self.method, check)
        button = getattr(self.ui.signin_page.panel.ui,
                         '%s_button' % self.method)
        button.click()

        yield d

        self.assertTrue(self.ui.signin_page.isEnabled())
        self.assertFalse(self._called)

    @defer.inlineCallbacks
    def test_with_error(self):
        """Wizard is enabled after an error was handled."""
        self.ui.currentIdChanged.connect(self._set_called)
        d = defer.fail(ValueError())

        def check():
            """Confirm the ui is disabled while processing."""
            self.assertFalse(self.ui.signin_page.isEnabled())
            return d

        self.patch(self.ui.backend, self.method, check)
        button = getattr(self.ui.signin_page.panel.ui,
                         '%s_button' % self.method)
        button.click()

        yield d

        self.assertTrue(self.ui.signin_page.isEnabled())
        self.assertFalse(self._called)  # the wizard page was not changed


class UbuntuOneWizardRegisterTestCase(UbuntuOneWizardLoginTestCase):
    """Test the register through the wizard."""

    method = 'register'
