# -*- coding: utf-8 *-*

# Copyright 2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The user interface for the control panel for Ubuntu One."""

import os

from PyQt4 import QtGui, QtCore
from twisted.internet.defer import inlineCallbacks

from ubuntuone.platform import expand_user

from ubuntuone.controlpanel.logger import setup_logging
from ubuntuone.controlpanel.gui import (
    COPY_LINK,
    OPEN,
    SEARCH_FILES,
    SHARED_FILES,
)

# Unused import images_rc, pylint: disable=W0611
from ubuntuone.controlpanel.gui.qt.ui import (
    images_rc,
    share_links_ui,
)
# pylint: enable=W0611
from ubuntuone.controlpanel.gui.qt.share_file import ShareFileWidget
from ubuntuone.controlpanel.gui.qt.share_links_search import (
    get_system_icon_for_filename,
)
from ubuntuone.controlpanel.gui.qt.ubuntuonebin import UbuntuOneBin


logger = setup_logging('qt.share_links')


FILE_NAME_COL = 0
PUBLIC_LINK_COL = 1
ACTIONS_COL = 2


class ShareLinksPanel(UbuntuOneBin):
    """The Share Links Tab Panel widget"""

    ui_class = share_links_ui
    logger = logger
    _enhanced_line = None
    home_dir = ''
    _shared_files = {}

    def _setup(self):
        """Do some extra setupping for the UI."""
        super(ShareLinksPanel, self)._setup()
        self.home_dir = ''
        self.ui.search_files_lbl.setText(SEARCH_FILES)
        self.ui.shared_group.setTitle(SHARED_FILES)

        # Set enhanced line edits
        self._enhanced_line = EnhancedLineEdit(self.ui.line_search,
            self._line_close_btn, icon=":/delete_search.png",
            style='enhanced_borderless')
        self._enhanced_line.btn_operation.hide()
        self.ui.line_search.popup.popupHidden.connect(
            self._hide_line_btn_close_hide)
        self.ui.line_search.popup.popupShown.connect(
            self._hide_line_btn_close_show)
        EnhancedLineEdit(self.ui.line_copy_link, self._copy_link_from_line,
            text=COPY_LINK)

        self.ui.line_search.itemSelected.connect(self.share_file)
        self.ui.back_to_file_list.clicked.connect(self._move_to_main_list)
        self.ui.open_in_browser.clicked.connect(self._open_in_browser)

        # Connect backend signals
        self.backend.set_public_files_list_handler(self._load_public_files)
        self.backend.set_public_access_changed_handler(self._file_shared)
        self.backend.set_public_access_change_error_handler(
            lambda: self._set_is_processing(False))
        self.get_public_files()

    @inlineCallbacks
    def share_file(self, file_path):
        """Clean the previous file share details and publish file_path."""
        file_path = unicode(file_path)
        if self.ui.hbox_share_file.count() > 0:
            widget = self.ui.hbox_share_file.takeAt(0).widget()
            widget.close()
        full_path = expand_user(file_path.encode('utf-8')).decode('utf-8')
        if full_path not in self._shared_files:
            self.is_processing = True
            share_file_widget = ShareFileWidget(file_path)
            self.ui.hbox_share_file.addWidget(share_file_widget)
            share_file_widget.linkDisabled.connect(
                lambda: self.ui.line_copy_link.setText(''))
            yield self.backend.change_public_access(
                full_path, True)
        else:
            share_file_widget = ShareFileWidget(file_path)
            self.ui.hbox_share_file.addWidget(share_file_widget)
            share_file_widget.linkDisabled.connect(
                lambda: self.ui.line_copy_link.setText(''))
            self.ui.line_copy_link.setText(self._shared_files[full_path])
            self.ui.stacked_widget.setCurrentIndex(1)

    def _file_shared(self, info):
        """Receive the notification that the file has been published."""
        url = info.get("public_url")
        self.ui.line_copy_link.setText(url)
        self.ui.stacked_widget.setCurrentIndex(1)
        self.is_processing = False

    def _open_in_browser(self):
        """Open the link in line_copy_link in the browser."""
        url = self.ui.line_copy_link.text()
        QtGui.QDesktopServices.openUrl(QtCore.QUrl(url))

    def _copy_link_from_line(self):
        """Copy link into clipboard from line edit."""
        app = QtGui.QApplication.instance()
        app.clipboard().setText(self.ui.line_copy_link.text())

    def _move_to_main_list(self):
        """Set the share files list as current widget."""
        self.ui.stacked_widget.setCurrentIndex(0)
        self.get_public_files()

    @inlineCallbacks
    def get_public_files(self):
        """Request the list of public files."""
        self.is_processing = True
        self.home_dir = yield self.backend.get_home_dir()
        yield self.backend.get_public_files()

    def _load_public_files(self, publicfiles):
        """Load the list of public files."""
        self.ui.tree_shared_files.clear()
        self._shared_files = {}
        for pfile in publicfiles:
            item = QtGui.QTreeWidgetItem()
            path = pfile['path']
            public_url = pfile['public_url']
            self._shared_files[path] = public_url
            name = os.path.basename(path)
            item.setText(FILE_NAME_COL, name)
            tooltip = path
            if tooltip.startswith(self.home_dir):
                tooltip = tooltip.replace(self.home_dir, '~', 1)
            item.setToolTip(FILE_NAME_COL, tooltip)
            icon = get_system_icon_for_filename(path.encode('utf-8'))
            item.setIcon(FILE_NAME_COL, icon)

            self.ui.tree_shared_files.setColumnWidth(PUBLIC_LINK_COL, 300)
            item.setSizeHint(FILE_NAME_COL, QtCore.QSize(-1, 35))
            self.ui.tree_shared_files.addTopLevelItem(item)

            link = ('<a href="%s"><span style="font-size: 12px;'
                    'color: #dd4814";>%s</span></a>'
                    % (public_url, public_url))
            label = QtGui.QLabel(link, self.ui.tree_shared_files)
            label.setOpenExternalLinks(True)
            self.ui.tree_shared_files.setItemWidget(item, PUBLIC_LINK_COL,
                label)

            actions = ActionsButtons(path, public_url,
                self.ui.tree_shared_files)
            self.ui.tree_shared_files.setItemWidget(item, ACTIONS_COL, actions)
        self.is_processing = False

    def _line_close_btn(self):
        """Close button in the line edit was pressed, hide the popup."""
        self.ui.line_search.popup.hide()
        self.ui.line_search.setFocus()

    def _hide_line_btn_close_hide(self):
        """Hide the button inside the search line edit-"""
        self._enhanced_line.btn_operation.hide()

    def _hide_line_btn_close_show(self):
        """Show the button inside the search line edit-"""
        self._enhanced_line.btn_operation.show()


class ActionsButtons(QtGui.QWidget):
    """Widget that contains the open and copy link actions on the list."""

    def __init__(self, path, link, parent=None):
        super(ActionsButtons, self).__init__(parent)
        self.path = path
        self.link = link

        hbox = QtGui.QHBoxLayout(self)
        btn_open = QtGui.QPushButton(OPEN)
        btn_copy = QtGui.QPushButton(COPY_LINK)
        btn_open.setObjectName('action_button')
        btn_copy.setObjectName('action_button')
        hbox.addSpacerItem(QtGui.QSpacerItem(1, 0,
            QtGui.QSizePolicy.Expanding))
        hbox.addWidget(btn_open)
        hbox.addWidget(btn_copy)

        btn_open.clicked.connect(self.open)
        btn_copy.clicked.connect(self.copy)

    def open(self):
        """Open the file."""
        file_path = u'file://%s' % self.path
        QtGui.QDesktopServices.openUrl(QtCore.QUrl(file_path))

    def copy(self):
        """Copy the public link of the file in the clipboard."""
        app = QtGui.QApplication.instance()
        app.clipboard().setText(self.link)


class EnhancedLineEdit(object):
    """Add a button inside the QLineEdit received."""

    def __init__(self, line_edit, operation, text=None, icon=None, style=None):
        hbox = QtGui.QHBoxLayout(line_edit)
        hbox.setMargin(0)
        line_edit.setLayout(hbox)
        hbox.addStretch()
        self.btn_operation = QtGui.QPushButton(line_edit)
        if text:
            self.btn_operation.setText(text)
        if icon:
            self.btn_operation.setIcon(QtGui.QIcon(icon))
        if style:
            self.btn_operation.setObjectName(style)
        hbox.addWidget(self.btn_operation)
        self.btn_operation.clicked.connect(operation)
