# -*- coding: utf-8 -*-
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Test the windows utils functions."""

import os
import sys

from twisted.internet import defer

from ubuntuone.controlpanel import utils
from ubuntuone.controlpanel.tests import TestCase
from ubuntuone.devtools.testcases import skipIfJenkins

# let me use protected methods
# pylint: disable=W0212

SOME_EXE_NAME = 'foo.exe'


class FrozenTestCase(TestCase):
    """A base test case for handling frozen/not frozen systems."""

    frozen = True
    executable = 'path/to/current/exe/ubuntuone/dist/executable.exe'

    @defer.inlineCallbacks
    def setUp(self):
        """Set the different tests."""
        yield super(FrozenTestCase, self).setUp()

        for attr_name in ('frozen', 'executable'):
            value_not_set = object()
            value = getattr(utils.windows.sys, attr_name, value_not_set)

            if self.frozen is not None:
                setattr(utils.windows.sys, attr_name, getattr(self, attr_name))
            elif self.frozen is None and value is not value_not_set:
                delattr(utils.windows.sys, attr_name)

            if self.frozen is not None and value is value_not_set:
                self.addCleanup(delattr, utils.windows.sys, attr_name)
            elif value is not value_not_set:
                self.addCleanup(setattr, utils.windows.sys, attr_name, value)


class AutoupdaterTestCase(TestCase):
    """Test the code that is used for the auto update process."""

    @defer.inlineCallbacks
    def setUp(self):
        """Prepare for the diff tests."""
        yield super(AutoupdaterTestCase, self).setUp()
        self._base_path = r'path\to\exe'
        self.auto_update_path = os.path.join(self._base_path,
                                    utils.windows.AUTOUPDATE_EXE_NAME)
        self.return_from_call = 0
        self.command = None
        self.args = []

        def fake_execute_process(command, args=None, path=None):
            """Fake async process execution."""
            self.command = command
            self.args = args
            return self.return_from_call

        self.patch(utils.windows, 'getProcessValue', fake_execute_process)
        self.patch(utils.windows, 'get_exe_path',
                   lambda exe_name: os.path.join(self._base_path, exe_name))

    @defer.inlineCallbacks
    def test_are_updates_present_true(self):
        """Test when updates are present."""
        # the idea is simple, set the value to be returned from
        # the fake call, assert that we get true and also that
        # we did use the correct parameters.
        self.return_from_call = 0
        are_present = yield utils.are_updates_present()
        self.assertTrue(are_present, 'Updates should be present.')
        # lets assert that we did use the correct args
        expected_args = ('--mode', 'unattended')
        self.assertEqual(expected_args, self.args)
        self.assertEqual(self.command, self.auto_update_path)

    @defer.inlineCallbacks
    def test_are_updates_present_false(self):
        """Test when updates are not present."""
        # similar to test_are_updates_present_true but with diff retcode
        self.return_from_call = 40
        are_present = yield utils.are_updates_present()
        self.assertFalse(are_present, 'Updates should NOT be present.')
        # lets assert that we did use the correct args
        expected_args = ('--mode', 'unattended')
        self.assertEqual(expected_args, self.args)
        self.assertEqual(self.command, self.auto_update_path)

    def test_perform_update(self):
        """Test the method that performs the update."""
        self.patch(utils.windows.win32api, 'ShellExecute', self._set_called)
        utils.perform_update()
        args = (None, 'runas', self.auto_update_path,
                '--unattendedmodeui none', '', 0)
        self.assertEqual(self._called, (args, {}))


class FakeOpenKey(object):

    """A Fake OpenKey class.

    This class becomes a method-like callable on FakeRegistry, allowing
    FakeRegistry.OpenKey to operate as a context manager."""

    def __init__(self):
        self.openkey_args = None
        super(FakeOpenKey, self).__init__()

    def __call__(self, *args, **kwargs):
        self.openkey_args = (args, kwargs)
        return self

    def __enter__(self, *args, **kwargs):
        #self.openkey_args = (args, kwargs)
        return self

    def __exit__(self, *args):
        pass


class FakeRegistry(object):

    """A fake registry."""

    # pylint: disable=C0103
    HKEY_CURRENT_USER = 2
    KEY_ALL_ACCESS = 4
    REG_SZ = 1

    def __init__(self):
        self.HKEY_CURRENT_USER = 2
        self.KEY_ALL_ACCESS = 4
        self.openkey_args = None
        self.query_args = None
        self.set_args = []

        self.OpenKey = FakeOpenKey()

    def QueryValueEx(self, *args, **kwargs):
        """Fake QueryValueEx."""
        self.query_args = (args, kwargs)

    def SetValueEx(self, *args, **kwargs):
        """Fake SetValueEx."""
        self.set_args.append((args, kwargs))


class AutostartTestCase(TestCase):

    """Test add_syncdaemon_to_autostart."""

    @defer.inlineCallbacks
    def setUp(self):
        """Initialize this test instance."""
        self.registry = FakeRegistry()
        self.patch(utils.windows, "_winreg", self.registry)
        yield super(AutostartTestCase, self).setUp()

    def test_add_syncdaemon_to_autostart(self):
        """Check that the registry is updated correctly."""
        # I can't patch sys because frozen is not there by default
        sys.frozen = True
        self.addCleanup(delattr, sys, 'frozen')
        utils.windows.add_to_autostart()
        self.assertEqual(self.registry.OpenKey.openkey_args,
            ((self.registry.HKEY_CURRENT_USER, utils.windows.AUTORUN_KEY, 0,
              self.registry.KEY_ALL_ACCESS), {}))
        self.assertEqual(self.registry.query_args, None)
        path = os.path.dirname(os.path.abspath(sys.executable))
        self.assertEqual(self.registry.set_args,
             [((self.registry.OpenKey, 'Ubuntu One', 0, 1,
             '"%s\\ubuntuone-syncdaemon.exe"' % path), {}),
             ((self.registry.OpenKey, 'Ubuntu One Icon', 0, 1,
             '"%s\\ubuntuone-control-panel-qt.exe" '
             '--minimized --with-icon' % path),
             {})])

    def test_not_added_if_not_frozen(self):
        """Not frozen binaries are not added to the registry."""
        utils.windows.add_to_autostart()
        self.assertEqual(self.registry.openkey_args, None)
        self.assertEqual(self.registry.query_args, None)
        self.assertEqual(self.registry.set_args, [])


class DefaultFoldersTestCase(TestCase):
    """Test the default_folders method."""

    @skipIfJenkins("win32", "Fails due to Jenkins setup on Windows")
    def test_special_folders(self, names=None):
        """Test that default_folders does the right thing."""
        folders = utils.default_folders()

        if names is None:
            names = ('PERSONAL', 'MYMUSIC', 'MYPICTURES', 'MYVIDEO')

        self.assertEqual(len(folders), len(names))
        self.assertEqual(folders, [os.path.abspath(f) for f in folders])

        expected = []
        for name in names:
            name = getattr(utils.windows.shellcon, 'CSIDL_%s' % name)
            folder = utils.windows.shell.SHGetFolderPath(0, name, None, 0)
            expected.append(folder)

        self.assertEqual(sorted(folders), sorted(expected))

    @skipIfJenkins("win32", "Fails due to Jenkins setup on Windows")
    def test_special_folders_with_error(self):
        """Test that default_folders handles errors."""
        failing_name = 'MYVIDEO'
        original_func = utils.windows.shell.SHGetFolderPath

        class SomeError(Exception):
            """For test only."""

        def fail_for_some(code, name, parent, flag):
            """Custom failer."""
            fail = getattr(utils.windows.shellcon, 'CSIDL_%s' % failing_name)
            if name == fail:
                raise SomeError('The parameter is incorrect.')
            else:
                return original_func(code, name, parent, flag)

        self.patch(utils.windows.shell, 'SHGetFolderPath', fail_for_some)
        self.test_special_folders(names=('PERSONAL', 'MYMUSIC', 'MYPICTURES'))


class GetExePathTestCase(FrozenTestCase):
    """Test the path calculator when sys is frozen."""

    @defer.inlineCallbacks
    def setUp(self):
        """Set the different tests."""
        yield super(GetExePathTestCase, self).setUp()
        self.called = []
        self.exists = True

        def fake_abspath(path):
            """Fake os.path.abspath."""
            self.called.append('os.path.abspath')
            return path

        def fake_dirname(path):
            """Fake os.path.dirname."""
            self.called.append('os.path.dirname')
            return path

        def fake_exists(path):
            """Fake os.path.exists."""
            self.called.append('os.path.exists')
            return self.exists

        # patch the os.path functions used
        self.patch(utils.windows.os.path, 'abspath', fake_abspath)
        self.patch(utils.windows.os.path, 'dirname', fake_dirname)
        self.patch(utils.windows.os.path, 'exists', fake_exists)

    def test_get_exe_path(self):
        """Test the method used to get the autoupdate."""
        path = utils.windows.get_exe_path(exe_name=SOME_EXE_NAME)

        self.assertEqual(os.path.join(self.executable, SOME_EXE_NAME), path)

        expected = ['os.path.abspath', 'os.path.dirname', 'os.path.dirname',
                    'os.path.exists']
        self.assertEqual(expected, self.called)

    def test_get_exe_path_not_present(self):
        """Test the method used to get the autoupdate."""
        self.exists = False

        # called method and assert that we have the correct result
        path = utils.windows.get_exe_path(exe_name=SOME_EXE_NAME)
        self.assertTrue(path is None)


class GetExePathNotFrozenTestCase(GetExePathTestCase):
    """Test the path calculator when sys is not frozen."""

    frozen = None

    def test_get_exe_path(self):
        """Test the method used to get the autoupdate."""
        self.patch(utils.windows, '__file__', self.executable)

        path = utils.windows.get_exe_path(exe_name=SOME_EXE_NAME)
        self.assertEqual(os.path.join(self.executable, SOME_EXE_NAME), path)

        expected = ['os.path.dirname', 'os.path.dirname', 'os.path.dirname',
                    'os.path.exists']
        self.assertEqual(expected, self.called)


class UninstallApplicationTestCase(FrozenTestCase):
    """Test the uninstall_application helper when sys is frozen."""

    @defer.inlineCallbacks
    def setUp(self):
        yield super(UninstallApplicationTestCase, self).setUp()
        self.patch(utils.windows.win32api, "ShellExecute", self._set_called)
        self.patch(os.path, "exists", lambda path: True)

    def test_uninstall(self):
        """The uninstaller is run."""
        utils.uninstall_application()

        exe_name = utils.windows.UNINSTALL_EXE_NAME
        uninstall_path = utils.windows.get_exe_path(exe_name=exe_name)
        self.assertEqual(self._called,
            ((None, '', uninstall_path, '--mode win32', '', 0), {}))

    def test_uninstall_exe_not_present(self):
        """The uninstaller is not run if not available."""
        self.patch(os.path, "exists", lambda path: False)

        utils.uninstall_application()

        self.assertFalse(self._called)


class UninstallApplicationNotFrozenTestCase(UninstallApplicationTestCase):
    """Test the uninstall_application helper when sys is not frozen."""

    frozen = None
