# -*- coding: utf-8 -*-
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The user interface for the control panel for Ubuntu One."""

from PyQt4 import QtGui, QtCore

from twisted.internet import defer

from ubuntuone.controlpanel import utils
from ubuntuone.controlpanel.logger import setup_logging
from ubuntuone.controlpanel.gui import UPDATE_TITLE, UPDATE_SOFTWARE
from ubuntuone.controlpanel.gui.qt.systray import TrayIcon
from ubuntuone.controlpanel.gui.qt.ui import mainwindow_ui

# pylint: disable=E0611
try:
    from gi.repository import Unity
    USE_LIBUNITY = True
except ImportError:
    USE_LIBUNITY = False
# pylint: enable=E0611

logger = setup_logging('qt.gui')

U1_DOTDESKTOP = "ubuntuone-installer.desktop"


class MainWindow(QtGui.QMainWindow):
    """The Main Window of the Control Panel."""

    def __init__(self, close_callback=None, installer=False):
        """Initialize this instance with the UI layout."""
        super(MainWindow, self).__init__()
        self.ui = mainwindow_ui.Ui_MainWindow()
        self.ui.setupUi(self)
        self.close_callback = close_callback
        self._setup()
        self.quit_action = QtGui.QAction(self, triggered=self.close)
        self.quit_action.setShortcuts(["Ctrl+q", "Ctrl+w"])
        self.addAction(self.quit_action)
        self.installer = installer
        if installer:
            self.ui.control_panel.start_from_license()
        utils.add_to_autostart()
        if USE_LIBUNITY:
            self.entry = Unity.LauncherEntry.get_for_desktop_id(U1_DOTDESKTOP)
        else:
            self.entry = None

    def _setup(self):
        """Do some extra setupping for the UI."""
        self.ui.control_panel.finished.connect(self.close)

    def switch_to(self, tabname="folders"):
        """Switch control panel to the required tab."""
        tabnames = ["folders", "share_links", "devices", "settings", "account"]
        if tabname in tabnames:
            self.ui.control_panel.ui.tab_widget.setCurrentIndex(
                tabnames.index(tabname))

    # Invalid names "closeEvent" "focusInEvent"
    # pylint: disable=C0103

    def closeEvent(self, event):
        """The main window is being closed, call any custom callback."""
        if self.close_callback is not None:
            self.close_callback()
        event.accept()

    def set_urgent(self, value):
        """Set the urgent attribute in the launcher to value."""
        if self.entry:
            self.entry.set_property('urgent', value)

    def focusInEvent(self, event):
        """The main window got focus, remove urgent bit."""
        self.set_urgent(False)
        return super(MainWindow, self).focusInEvent(event)

    # pylint: enable=C0103

    @defer.inlineCallbacks
    def check_updates(self):
        """Offer the user to update if there are updates available."""
        logger.debug('Checking for updates.')
        are_present = yield utils.are_updates_present()
        logger.info('Updates available? %r', are_present)
        if are_present:
            buttons = QtGui.QMessageBox.Yes | QtGui.QMessageBox.No
            result = QtGui.QMessageBox.question(self, UPDATE_TITLE,
                                                UPDATE_SOFTWARE,
                                                buttons, QtGui.QMessageBox.Yes)
            if result == QtGui.QMessageBox.Yes:
                logger.info('Performing auto-update.')
                utils.perform_update()
            else:
                logger.info('User do not want to update.')


def start(close_callback, minimized=False, with_icon=False, installer=False):
    """Show the UI elements."""
    # pylint: disable=W0404, F0401
    if not minimized:
        if with_icon or minimized:
            window = MainWindow(installer=installer)
        else:
            window = MainWindow(close_callback=close_callback,
                                installer=installer)
        app = QtGui.QApplication.instance()
        style = QtGui.QStyle.alignedRect(
            QtCore.Qt.LeftToRight, QtCore.Qt.AlignCenter,
            window.size(), app.desktop().availableGeometry())
        window.setGeometry(style)
        window.check_updates()
        window.show()
        window.raise_()
    else:
        window = None
    if with_icon or minimized:
        QtGui.QApplication.instance().setQuitOnLastWindowClosed(False)
        icon = TrayIcon(window=window, close_callback=close_callback)
    else:
        icon = None
    return icon, window
