# -*- coding: utf-8 -*-
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the Control Panel."""

import copy
import logging
import operator
import os
import Queue
import shutil
import sys

from PyQt4 import QtGui
from twisted.internet import defer
from ubuntuone.devtools.handlers import MementoHandler
from ubuntuone.devtools.testcases import skipIfOS

from ubuntuone.controlpanel.tests import helper_fail
from ubuntuone.controlpanel.gui.tests import (
    FAKE_VOLUMES_INFO,
    FAKE_VOLUMES_MINIMAL_INFO,
    FAKE_VOLUMES_NO_FREE_SPACE_INFO,
    MUSIC_FOLDER, ROOT, USER_HOME,
)
from ubuntuone.controlpanel.gui.qt import folders as gui
from ubuntuone.controlpanel.gui.qt.tests import (
    BaseTestCase,
    FakedDialog,
)
from ubuntuone.controlpanel.gui.qt.tests.test_ubuntuonebin import (
    UbuntuOneBinTestCase,
)

# Access to a protected member
# Instance of 'ControlBackend' has no '_called' member
# pylint: disable=W0212, E1103


def volumes_with_music_unsubscribed():
    """Return a copy of FAKE_VOLUMES_MINIMAL_INFO with music unsubscribed."""
    volumes = copy.deepcopy(FAKE_VOLUMES_MINIMAL_INFO)
    volumes[0][2][1][u'subscribed'] = u''
    return volumes


def _build_name(name):
    """Helper to build the name expected when showing folder info."""
    if name:
        name = gui.FOLDER_SHARED_BY % {'other_user_display_name': name}
    else:
        name = gui.FOLDER_OWNED_BY
    return name


class ExploreFolderButtonTestCase(BaseTestCase):
    """Test the ExploreFolderButton widget."""

    class_ui = gui.ExploreFolderButton
    kwargs = dict(folder_path=u'foo')

    def test_text(self):
        """The button text is correct."""
        self.assertEqual(unicode(self.ui.text()),
            gui.FOLDERS_COLUMN_EXPLORE)

    def test_clicked(self):
        """Clicking the button opens the folder in the default file manager."""
        self.patch(gui, 'uri_hook', self._set_called)
        self.ui.click()

        url = gui.QtCore.QUrl.fromLocalFile(self.kwargs['folder_path'])
        expected = unicode(url.toString())
        self.assertEqual(self._called, ((expected,), {}))


class FoldersPanelTestCase(UbuntuOneBinTestCase):
    """Test the qt cloud folders tab."""

    innerclass_ui = gui.folders_ui
    innerclass_name = "Ui_Form"
    class_ui = gui.FoldersPanel

    @defer.inlineCallbacks
    def setUp(self):
        yield super(FoldersPanelTestCase, self).setUp()
        self.patch(self.ui.backend, 'volumes_info',
                   lambda *a, **kw: defer.succeed(FAKE_VOLUMES_MINIMAL_INFO))

        self.memento = MementoHandler()
        self.memento.setLevel(logging.DEBUG)
        gui.logger.addHandler(self.memento)

    def set_item_checked(self, item, checked=True):
        """Make item to be checked."""
        checkbox = self.ui.ui.folders.itemWidget(item, gui.SUBSCRIPTION_COL)
        checkbox.setCheckState(gui.CHECKED if checked else gui.UNCHECKED)

    def get_item_checked(self, item):
        """Get if item is checked."""
        checkbox = self.ui.ui.folders.itemWidget(item, gui.SUBSCRIPTION_COL)
        return (checkbox.checkState() == gui.CHECKED)


class FoldersPanelVolumesInfoTestCase(FoldersPanelTestCase):
    """Test the qt cloud folders tab."""

    @defer.inlineCallbacks
    def setUp(self):
        yield super(FoldersPanelVolumesInfoTestCase, self).setUp()
        yield self.ui.load()

    def assert_folder_group_header_correct(self, item, name):
        """Check that the folder group header is correct."""
        self.assertEqual(unicode(item.text(gui.FOLDER_NAME_COL)), name)
        self.assertEqual(unicode(item.text(gui.SUBSCRIPTION_COL)),
                         gui.FOLDERS_COLUMN_SYNC_LOCALLY)
        if not self.ui.remote_folders:
            self.assertEqual(unicode(item.text(gui.EXPLORE_COL)), '')

    def assert_folder_row_correct(self, item, label, icon_name, volume,
                                  tweaked_path=None):
        """Check that the folder row 'item' is correct."""
        folders = self.ui.ui.folders

        actual_label = unicode(item.text(gui.FOLDER_NAME_COL))
        self.assertEqual(label, actual_label)

        if volume['type'] == self.ui.backend.ROOT_TYPE:
            # no check box but the ALWAYS_SUBSCRIBED legend
            self.assertEqual(unicode(item.text(gui.SUBSCRIPTION_COL)),
                             gui.ALWAYS_SUBSCRIBED)
        else:
            subscribed = self.get_item_checked(item)
            self.assertEqual(subscribed, bool(volume['subscribed']))

        actual_icon_name = item.icon_obj.icon_name
        self.assertEqual(icon_name, actual_icon_name)

        self.assertEqual(item.volume_id, volume['volume_id'])

        expected_path = volume['path']
        if tweaked_path is not None:
            expected_path = tweaked_path
        self.assertEqual(item.volume_path, expected_path)

        # tooltips are correct
        self.assertEqual(item.toolTip(gui.FOLDER_NAME_COL), label)
        self.assertEqual(item.toolTip(gui.EXPLORE_COL),
                         gui.FOLDERS_COLUMN_EXPLORE)

        if not self.ui.remote_folders:
            # explore button is in place
            model_index = folders.indexFromItem(item, gui.EXPLORE_COL)
            button = folders.indexWidget(model_index)
            self.assertEqual(button.isEnabled(), bool(volume['subscribed']))

    @defer.inlineCallbacks
    def test_is_processing_while_asking_info(self):
        """The ui is processing while the contents are loaded."""
        def check(with_storage_info):
            """The ui must be is_processing."""
            self.assertTrue(self.ui.is_processing, 'ui must be processing')
            return FAKE_VOLUMES_MINIMAL_INFO

        self.patch(self.ui.backend, 'volumes_info', check)

        yield self.ui.load()  # trigger the info request

    def test_is_not_processing_after_info_ready(self):
        """The ui is not processing when contents are load."""
        self.ui.process_info(FAKE_VOLUMES_INFO)

        self.assertFalse(self.ui.is_processing)

    def test_info_is_requested_on_load(self):
        """The volumes info is requested to the backend."""
        self.assert_backend_called('volumes_info', with_storage_info=False,
                                   refresh=self.ui.remote_folders)

    def test_process_info(self):
        """The volumes info is processed when ready."""
        self.ui.process_info(FAKE_VOLUMES_INFO)
        folders = self.ui.ui.folders

        root = folders.invisibleRootItem()
        self.assertEqual(len(FAKE_VOLUMES_INFO), root.childCount())

        treeiter = gui.QtGui.QTreeWidgetItemIterator(folders)
        for name, _, volumes in FAKE_VOLUMES_INFO:
            name = _build_name(name)

            # get first child, matching "My Folders"
            item = treeiter.value()
            self.assertFalse(item.is_empty)

            self.assert_folder_group_header_correct(item, name)

            self.assertEqual(len(volumes), item.childCount())
            sorted_vols = sorted(volumes, key=operator.itemgetter('path'))
            for volume in sorted_vols:
                treeiter += 1
                item = treeiter.value()  # get child folder
                self.assertTrue(item is not None)

                name = volume['path'].replace(USER_HOME + os.path.sep, '')
                expected_path = volume['path']
                if volume['type'] == self.ui.backend.SHARE_TYPE:
                    name = volume['name']
                    expected_path = volume['realpath']

                if volume['type'] != self.ui.backend.SHARE_TYPE:
                    icon_name = gui.FOLDER_ICON_NAME
                else:
                    icon_name = gui.SHARE_ICON_NAME

                self.assert_folder_row_correct(item, name, icon_name, volume,
                                               tweaked_path=expected_path)

            treeiter += 1
            item = treeiter.value()

    def test_process_info_clears_the_list(self):
        """The old volumes info is cleared before updated."""
        self.ui.process_info(FAKE_VOLUMES_INFO)
        self.ui.process_info(FAKE_VOLUMES_INFO)

        root = self.ui.ui.folders.invisibleRootItem()
        self.assertEqual(len(FAKE_VOLUMES_INFO), root.childCount())

    def test_process_info_with_no_volumes(self):
        """When there are no volumes, a notification is shown."""
        self.ui.process_info([])

        root = self.ui.ui.folders.invisibleRootItem()
        self.assertEqual(root.childCount(), 0)

    def test_process_info_highlights_little_free_space(self):
        """The free space is red if is zero (or close to 0)."""
        self.ui.process_info(FAKE_VOLUMES_NO_FREE_SPACE_INFO)

        child_index = 0
        root = self.ui.ui.folders.invisibleRootItem()
        for name, _, _ in FAKE_VOLUMES_NO_FREE_SPACE_INFO:
            name = _build_name(name)

            item = root.child(child_index)
            self.assertFalse(item.is_empty)
            self.assert_folder_group_header_correct(item, name)

            child_index += 1
            item = root.child(child_index)

    def test_process_info_handles_no_quota_info(self):
        """The lack of free space is handled."""
        info = [
            (u'', self.ui.backend.FREE_BYTES_NOT_AVAILABLE, [ROOT]),
            (u'No free space available',
             self.ui.backend.FREE_BYTES_NOT_AVAILABLE,
             [{u'volume_id': u'0', u'name': u'full', u'display_name': u'test',
               u'path': u'full-share', u'subscribed': u'',
               u'type': self.ui.backend.SHARE_TYPE}]),
        ]
        self.ui.process_info(info)

        child_index = 0
        root = self.ui.ui.folders.invisibleRootItem()
        for name, _, _ in info:
            name = _build_name(name)

            item = root.child(child_index)
            self.assertFalse(item.is_empty)
            self.assert_folder_group_header_correct(item, name)

            child_index += 1
            item = root.child(child_index)

    def test_process_info_preserves_position(self):
        """process_info should not scroll the list to the top."""
        self.ui.process_info(FAKE_VOLUMES_INFO)
        scrollbar = self.ui.ui.folders.verticalScrollBar()
        maximum = scrollbar.maximum()
        scrollbar.setValue(maximum)
        self.ui.process_info(FAKE_VOLUMES_INFO)
        self.assertEqual(self.ui.ui.folders.verticalScrollBar().value(),
            maximum)

    def test_clicking_on_row_opens_folder(self):
        """The folder activated is opened."""
        self.patch(gui.os.path, 'exists', lambda *a: True)
        self.patch(gui, 'uri_hook', self._set_called)

        path = ROOT['path']
        item = gui.QtGui.QTreeWidgetItem()
        item.volume_path = path
        self.ui.on_folders_itemActivated(item)

        self.assertEqual(self._called,
            ((unicode(gui.QtCore.QUrl.fromLocalFile(path).toString()),), {}))
        self.assertTrue(gui.QtCore.QUrl(self._called[0][0]).isValid())

    def test_clicking_on_row_handles_path_none(self):
        """None paths are properly handled."""
        self.patch(gui, 'uri_hook', self._set_called)

        item = gui.QtGui.QTreeWidgetItem()
        item.volume_path = None
        self.ui.on_folders_itemActivated(item)

        self.assertTrue(self.memento.check_warning(str(item),
                                                   'volume_path', 'is None'))
        self.assertEqual(self._called, False)

    def test_clicking_on_row_handles_path_non_existent(self):
        """Not-existent paths are properly handled."""
        self.patch(gui.os.path, 'exists', lambda *a: False)
        self.patch(gui, 'uri_hook', self._set_called)

        path = 'not-in-disk'
        item = gui.QtGui.QTreeWidgetItem()
        item.volume_path = path
        self.ui.on_folders_itemActivated(item)

        self.assertTrue(self.memento.check_warning(path, 'does not exist'))
        self.assertEqual(self._called, False)

    def test_process_info_with_music_folder(self, volumes=None):
        """The volumes info is processed when ready."""
        if volumes is None:
            volumes = FAKE_VOLUMES_MINIMAL_INFO

        self.ui.process_info(volumes)
        folders = self.ui.ui.folders

        treeiter = gui.QtGui.QTreeWidgetItemIterator(folders)

        # walk to 'My folders' children
        treeiter += 1

        # grab next row since first one is root
        treeiter += 1
        item = treeiter.value()

        volume = volumes[0][2][1]

        self.assert_folder_row_correct(item, gui.MUSIC_DISPLAY_NAME,
                                       gui.MUSIC_ICON_NAME, volume)

    def test_focus_order(self):
        """Ensure that the inner widgets are in the correct tab order."""
        self.ui.process_info(FAKE_VOLUMES_INFO)
        folders = self.ui.ui.folders

        widget = self.ui.ui.folders.nextInFocusChain()
        treeiter = gui.QtGui.QTreeWidgetItemIterator(folders)
        for name, _, volumes in FAKE_VOLUMES_INFO:
            item = treeiter.value()
            sorted_vols = sorted(volumes, key=operator.itemgetter('path'))
            for volume in sorted_vols:
                treeiter += 1
                item = treeiter.value()  # get child folder

                name = volume['path'].replace(USER_HOME + os.path.sep, '')
                if volume['type'] == self.ui.backend.SHARE_TYPE:
                    name = volume['name']
                self.assertEqual(unicode(item.text(gui.FOLDER_NAME_COL)), name)

                if volume['type'] != self.ui.backend.ROOT_TYPE:
                    self.assertIsInstance(widget, QtGui.QCheckBox)
                    self.assertEqual(unicode(
                        self.ui.widget_items[widget].text(0)), name)
                    widget = widget.nextInFocusChain()

                if not self.ui.remote_folders:
                    self.assertIsInstance(widget, QtGui.QPushButton)
                    self.assertEqual(unicode(
                        self.ui.widget_items[widget].text(0)), name)
                    widget = widget.nextInFocusChain()

            treeiter += 1
            item = treeiter.value()

    def test_widget_dict(self):
        """Ensure the widget_items dictionary is full."""
        self.ui.process_info(FAKE_VOLUMES_INFO)
        it = QtGui.QTreeWidgetItemIterator(self.ui.ui.folders)
        while it.value():
            item = it.value()
            checkbox = self.ui.ui.folders.itemWidget(item,
                gui.SUBSCRIPTION_COL)
            button = self.ui.ui.folders.itemWidget(item,
                gui.EXPLORE_COL)
            if checkbox:
                self.assertEqual(self.ui.widget_items[checkbox],
                item)
            if button:
                self.assertEqual(self.ui.widget_items[button],
                item)
            it += 1

    def test_tree_view_header(self):
        """The tree view header is hidden but the text is correct."""
        self.assertFalse(self.ui.ui.folders.header().isVisible())

        name = self.ui.ui.folders.headerItem().text(gui.FOLDER_NAME_COL)
        self.assertEqual(name, gui.FOLDERS_COLUMN_NAME)
        name = self.ui.ui.folders.headerItem().text(gui.SUBSCRIPTION_COL)
        self.assertEqual(name, gui.ALWAYS_SUBSCRIBED)
        name = self.ui.ui.folders.headerItem().text(gui.EXPLORE_COL)
        self.assertEqual(name, gui.FOLDERS_COLUMN_EXPLORE)

    def test_share_publish_button(self):
        """When clicking the share/publish button, the proper url is opened."""
        self.assertTrue(self.ui.ui.share_publish_button.isVisible())
        self.assert_uri_hook_called(self.ui.ui.share_publish_button,
                                    gui.MANAGE_FILES_LINK)

    def test_add_folder_button(self):
        """The 'add_folder_button' is visible by default."""
        self.assertEqual(self.ui.ui.add_folder_button.add_folder_func,
                         self.ui.backend.create_folder)
        self.assertTrue(self.ui.ui.add_folder_button.isVisible())

    def test_check_settings_button(self):
        """The 'check_settings_button' is not visible by default."""
        self.assertFalse(self.ui.ui.check_settings_button.isVisible())


class FoldersPanelAddFolderTestCase(FoldersPanelTestCase):
    """The test suite for the folder creation from a local dir."""

    timeout = 3

    @defer.inlineCallbacks
    def setUp(self):
        yield super(FoldersPanelAddFolderTestCase, self).setUp()

        # simulate volumes info properly processed
        self.ui.process_info([])
        # reset backend state
        self.ui.backend._called.clear()

    def test_not_is_processing(self):
        """Before clicking the add folder button, the UI is not processing."""
        self.assertFalse(self.ui.is_processing, 'ui must not be processing')

    @defer.inlineCallbacks
    def test_reload_volumes_info_on_success(self):
        """Once the folder is created, the volumes info is reloaded."""
        d = defer.Deferred()
        self.patch(self.ui, 'load', lambda *a, **kw: d.callback((a, kw)))

        self.ui.ui.add_folder_button.folderCreated.emit(USER_HOME)

        result = yield d  # when d is fired, self.ui.load() was called
        self.assertEqual(result, ((), {}))


class FoldersPanelSubscriptionTestCase(FoldersPanelTestCase):
    """The test suite for the folder subscription."""

    faked_volumes = FAKE_VOLUMES_MINIMAL_INFO

    @defer.inlineCallbacks
    def setUp(self):
        yield super(FoldersPanelSubscriptionTestCase, self).setUp()
        self.patch(gui.os.path, 'exists', lambda path: True)
        FakedDialog.response = gui.YES

        self.ui.process_info(self.faked_volumes)
        # the music folder
        self.item = self.ui.ui.folders.topLevelItem(0).child(1)

    def set_item_checked(self, item=None, checked=True):
        """Make item to be checked."""
        if item is None:
            item = self.item
        test = super(FoldersPanelSubscriptionTestCase, self).set_item_checked
        test(item=item, checked=checked)

    @defer.inlineCallbacks
    def test_on_folders_item_changed(self):
        """Clicking on 'subscribed' updates the folder subscription."""
        self.patch(self.ui, 'load', self._set_called)
        volume = MUSIC_FOLDER
        fid = volume['volume_id']
        subscribed = not bool(volume['subscribed'])

        self.ui.is_processing = True
        self.set_item_checked(self.item, subscribed)
        self.ui.is_processing = False

        yield self.ui.on_folders_itemChanged(self.item)

        # backend was called
        self.assert_backend_called('change_volume_settings',
                                   fid, {'subscribed': subscribed})

        value = self.get_item_checked(self.item)
        self.assertEqual(value, bool(subscribed))

        # folder list was reloaded
        self.assertEqual(self._called, ((), {}))

    @defer.inlineCallbacks
    def test_on_folders_item_changed_is_processing(self):
        """Clicking on 'subscribed' sets is_processing flag until done."""
        def check(volume_id, settings):
            """The ui must be is_processing when a change was requested."""
            self.assertTrue(self.ui.is_processing, 'ui must be processing')

        self.patch(self.ui.backend, 'change_volume_settings', check)

        yield self.ui.on_folders_itemChanged(self.item)

        # the ui is no loner processing
        self.assertFalse(self.ui.is_processing, 'ui must not be processing')

    @defer.inlineCallbacks
    def test_on_folders_item_changed_volume_id_is_none(self):
        """If item.volume_id is None, log warning and return."""
        self.patch(self.ui.backend, 'change_volume_settings', self._set_called)

        self.item.volume_id = None
        yield self.ui.on_folders_itemChanged(self.item)

        self.assertTrue(self.memento.check_warning(str(self.item),
                                                   'volume_id', 'is None'))
        self.assertEqual(self._called, False)

    @defer.inlineCallbacks
    def test_on_folders_item_changed_does_nothing_if_processing(self):
        """If UI is already processing, do nothing and return."""
        self.patch(self.ui.backend, 'change_volume_settings', self._set_called)

        self.ui.is_processing = True

        yield self.ui.on_folders_itemChanged(self.item)

        self.assertEqual(self._called, False)

    @defer.inlineCallbacks
    def test_confirm_dialog_if_path_exists(self):
        """The confirmation dialog is correct."""
        self.patch(gui.os.path, 'exists', lambda path: True)

        # make sure the item is subscribed
        self.ui.is_processing = True
        self.set_item_checked()
        self.ui.is_processing = False

        yield self.ui.on_folders_itemChanged(self.item)

        volume_path = self.item.volume_path
        msg = gui.FOLDERS_CONFIRM_MERGE % {'folder_path': volume_path}
        buttons = gui.YES | gui.NO | gui.CANCEL
        self.assertEqual(FakedDialog.args,
                         (self.ui, '', msg, buttons, gui.YES))
        self.assertEqual(FakedDialog.kwargs, {})

    @defer.inlineCallbacks
    def test_confirm_dialog_if_path_does_not_exist(self):
        """The confirmation dialog is not created if not needed."""
        self.patch(gui.os.path, 'exists', lambda path: False)

        # make sure the item is unsubscribed
        self.ui.is_processing = True
        self.set_item_checked()
        self.ui.is_processing = False

        yield self.ui.on_folders_itemChanged(self.item)

        self.assertEqual(FakedDialog.args, None)
        self.assertEqual(FakedDialog.kwargs, None)

    @defer.inlineCallbacks
    def test_subscribe_does_not_call_backend_if_dialog_closed(self):
        """Backend is not called if users closes the confirmation dialog."""
        FakedDialog.response = gui.CANCEL

        # make sure the item is subscribed
        self.ui.is_processing = True
        self.set_item_checked()
        self.ui.is_processing = False

        yield self.ui.on_folders_itemChanged(self.item)

        self.assertFalse(FakedDialog.args is None, 'warning was called')
        self.assertNotIn('change_volume_settings', self.ui.backend._called)
        self.assertFalse(self.ui.is_processing)

        self.assertFalse(self.get_item_checked(self.item))

    @defer.inlineCallbacks
    def test_subscribe_does_not_call_backend_if_answer_is_no(self):
        """Backend is not called if users clicks on 'No'."""
        FakedDialog.response = gui.NO

        # make sure the item is subscribed
        self.ui.is_processing = True
        self.set_item_checked()
        self.ui.is_processing = False

        yield self.ui.on_folders_itemChanged(self.item)

        self.assertFalse(FakedDialog.args is None, 'warning was called')
        self.assertNotIn('change_volume_settings', self.ui.backend._called)
        self.assertFalse(self.ui.is_processing)

        self.assertFalse(self.get_item_checked(self.item))

    @defer.inlineCallbacks
    def test_no_confirmation_if_unsubscribing(self):
        """The confirmation dialog is not shown if unsubscribing."""
        # make sure the item is unsubscribed
        self.ui.is_processing = True
        self.set_item_checked(checked=False)
        self.ui.is_processing = False

        # the confirm dialog was not called so far
        assert FakedDialog.args is None
        assert FakedDialog.kwargs is None

        yield self.ui.on_folders_itemChanged(self.item)

        self.assertTrue(FakedDialog.args is None, 'dialog was not run')
        self.assertTrue(FakedDialog.kwargs is None, 'dialog was hid')


class RemoteFoldersPanelTestCase(FoldersPanelVolumesInfoTestCase):
    """The test case for the RemoteFoldersPanel widget."""

    class_ui = gui.RemoteFoldersPanel

    def test_process_info_with_music_folder(self, volumes=None):
        """The volumes info is processed when ready."""
        volumes = volumes_with_music_unsubscribed()
        parent = super(RemoteFoldersPanelTestCase, self)
        parent.test_process_info_with_music_folder(volumes=volumes)

    def test_share_publish_button(self):
        """When clicking the share/publish button, the proper url is opened."""
        self.assertFalse(self.ui.ui.share_publish_button.isVisible())

    def test_add_folder_button(self):
        """The 'add_folder_button' is not visible by default."""
        self.assertFalse(self.ui.ui.add_folder_button.isVisible())

    def test_check_settings_button(self):
        """The 'check_settings_button' is visible by default."""
        self.assertTrue(self.ui.ui.check_settings_button.isVisible())


class RemoteFoldersPanelSubscriptionTestCase(FoldersPanelSubscriptionTestCase):
    """The test suite for the remote folder subscription."""

    class_ui = gui.RemoteFoldersPanel
    faked_volumes = volumes_with_music_unsubscribed()


class BaseLocalFoldersTestCase(BaseTestCase):
    """Test suite for the class implementing the LocalFolders feature."""

    @defer.inlineCallbacks
    def setUp(self):
        self.path = u'not-existing-dir'
        self.expected_size = self.build_test_dir(self.path)
        self.queue = Queue.Queue()
        yield super(BaseLocalFoldersTestCase, self).setUp()

    def build_test_dir(self, dir_path):
        """Build a testing directory hierarchy."""
        assert not os.path.exists(dir_path)

        os.makedirs(dir_path)
        self.addCleanup(shutil.rmtree,
            dir_path.encode(sys.getfilesystemencoding()))

        total_size = 0

        a_file = os.path.join(dir_path, 'test_file')
        with open(a_file, 'wb') as f:
            f.write('z' * 1000000)

        total_size += os.path.getsize(a_file)

        inner_dir = os.path.join(dir_path, 'test_dir')
        os.mkdir(inner_dir)

        other_file = os.path.join(dir_path, 'other_test_file')
        with open(other_file, 'wb') as f:
            f.write(' ' * 99999)

        total_size += os.path.getsize(other_file)

        empty_dir = os.path.join(dir_path, 'empty')
        os.mkdir(empty_dir)

        return total_size


class CalculateSizeTestCase(BaseLocalFoldersTestCase):
    """Test suite for the CalculateSize thread implementation."""

    @defer.inlineCallbacks
    def setUp(self):
        yield super(CalculateSizeTestCase, self).setUp()
        self.ui = gui.CalculateSize(path_name=self.path, queue=self.queue)
        self.patch(self.ui, 'start', lambda: None)

    def test_creation(self):
        """The created instance is correct."""
        self.assertEqual(self.ui.path_name, self.path)
        self.assertEqual(self.ui.queue, self.queue)
        self.assertTrue(self.ui.daemon)

    def test_run(self):
        """The run() method calculates the size for the given path."""
        self.ui.run()

        path, size = self.queue.get(block=True, timeout=0.5)

        self.assertEqual(path, self.path)
        self.assertEqual(size, self.expected_size)

    def test_run_handles_errors(self):
        """The run() method handles errors."""
        self.patch(gui.os, 'walk', helper_fail)
        self.ui.run()

        self.assertRaises(Queue.Empty, self.queue.get, block=True, timeout=0.5)


@skipIfOS("win32", "Windows does not easily support symlinks")
class CalculateSizeWithSymlinksTestCase(BaseLocalFoldersTestCase):
    """Test suite for the CalculateSize thread implementation."""

    def build_test_dir(self, dir_path):
        """Build a testing directory hierarchy."""
        total_size = super(CalculateSizeWithSymlinksTestCase,
                           self).build_test_dir(dir_path)

        a_file = os.path.join(dir_path, 'to_be_symlinked')
        with open(a_file, 'wb') as f:
            f.write('y' * 5000)

        total_size += os.path.getsize(a_file)

        # add a symlink to confirm those are avoided
        a_link = os.path.join(dir_path, 'some_link')
        os.symlink(a_file, a_link)

        return total_size


class FakedCalculateSize(object):
    """A faked CalculateSize thread."""

    def __init__(self, *args, **kwargs):
        self.started = False

    def start(self):
        """Fake start."""
        self.started = True


class CalculateSizeWithInvalidPath(CalculateSizeTestCase):
    """Test suite for the CalculateSize thread implementation."""

    def build_test_dir(self, dir_path):
        """Build a testing directory hierarchy."""
        total_size = super(CalculateSizeWithInvalidPath,
                           self).build_test_dir(dir_path)
        # Have to crete as bytes because it's an invalid path
        a_file = os.path.join(dir_path.encode(sys.getfilesystemencoding()),
            '\xe7\xa7')
        with open(a_file, 'wb') as f:
            f.write('y' * 5000)
        total_size += os.path.getsize(a_file)
        return total_size

    def test_run(self):
        """The run() method calculates the size for the given path."""
        self.ui.run()

        path, size = self.queue.get(block=True, timeout=0.5)

        self.assertEqual(path, self.path)
        self.assertEqual(size, self.expected_size)


class FolderItemTestCase(BaseLocalFoldersTestCase):
    """Test suite for the FolderItem widget."""

    @defer.inlineCallbacks
    def setUp(self):
        yield super(FolderItemTestCase, self).setUp()
        self.calculator = FakedCalculateSize(self.path, self.queue)
        self.patch(gui, 'CalculateSize', lambda *a, **kw: self.calculator)
        self.values = ['foo', 'bar']

        assert not self.calculator.started

    def test_no_params(self):
        """The creation with no params uses defaults."""
        item = gui.FolderItem()

        self.assertEqual(item.text(gui.LOCAL_SUBSCRIPTION_COL), '')
        self.assertEqual(item.text(gui.LOCAL_SPACE_COL), '')
        self.assertEqual(item.path, None)
        self.assertEqual(item.volume_id, None)
        self.assertEqual(item.thread, None)
        self.assertEqual(item.size, 0)
        self.assertEqual(item.checkState(gui.LOCAL_SUBSCRIPTION_COL),
                         gui.UNCHECKED)

    def test_values(self):
        """The creation with only values."""
        item = gui.FolderItem(values=self.values)

        self.assertEqual(item.text(gui.LOCAL_SUBSCRIPTION_COL), self.values[0])
        self.assertEqual(item.text(gui.LOCAL_SPACE_COL), self.values[1])
        self.assertEqual(item.path, None)
        self.assertEqual(item.volume_id, None)
        self.assertEqual(item.thread, None)
        self.assertEqual(item.size, 0)
        self.assertEqual(item.checkState(gui.LOCAL_SUBSCRIPTION_COL),
                         gui.UNCHECKED)

    def test_path(self):
        """The creation with only a path."""
        item = gui.FolderItem(path=self.path)

        self.assertEqual(item.text(gui.LOCAL_SUBSCRIPTION_COL), '')
        self.assertEqual(item.text(gui.LOCAL_SPACE_COL), '')
        self.assertEqual(item.path, self.path)
        self.assertEqual(item.volume_id, None)
        self.assertEqual(item.thread, None)
        self.assertEqual(item.size, 0)
        self.assertEqual(item.checkState(gui.LOCAL_SUBSCRIPTION_COL),
                         gui.UNCHECKED)

    def test_queue(self):
        """The creation with only a queue."""
        item = gui.FolderItem(queue=self.queue)

        self.assertEqual(item.text(gui.LOCAL_SUBSCRIPTION_COL), '')
        self.assertEqual(item.text(gui.LOCAL_SPACE_COL), '')
        self.assertEqual(item.path, None)
        self.assertEqual(item.volume_id, None)
        self.assertEqual(item.thread, None)
        self.assertEqual(item.size, 0)
        self.assertEqual(item.checkState(gui.LOCAL_SUBSCRIPTION_COL),
                         gui.UNCHECKED)

    def test_volume_id(self):
        """The creation with only a volume_id."""
        item = gui.FolderItem(volume_id='yadda')

        self.assertEqual(item.text(gui.LOCAL_SUBSCRIPTION_COL), '')
        self.assertEqual(item.text(gui.LOCAL_SPACE_COL), '')
        self.assertEqual(item.path, None)
        self.assertEqual(item.volume_id, 'yadda')
        self.assertEqual(item.thread, None)
        self.assertEqual(item.size, 0)
        self.assertEqual(item.checkState(gui.LOCAL_SUBSCRIPTION_COL),
                         gui.UNCHECKED)

    def test_path_and_volume_id(self):
        """The creation with only a volume_id."""
        item = gui.FolderItem(path=self.path, volume_id='yadda')

        self.assertEqual(item.text(gui.LOCAL_SUBSCRIPTION_COL), '')
        self.assertEqual(item.text(gui.LOCAL_SPACE_COL), '')
        self.assertEqual(item.path, self.path)
        self.assertEqual(item.volume_id, 'yadda')
        self.assertEqual(item.thread, None)
        self.assertEqual(item.size, 0)
        self.assertEqual(item.checkState(gui.LOCAL_SUBSCRIPTION_COL),
                         gui.CHECKED)

    def test_path_and_queue(self):
        """The creation with only a volume_id."""
        item = gui.FolderItem(path=self.path, queue=self.queue)

        self.assertEqual(item.text(gui.LOCAL_SUBSCRIPTION_COL), '')
        self.assertEqual(item.text(gui.LOCAL_SPACE_COL), '')
        self.assertEqual(item.path, self.path)
        self.assertEqual(item.volume_id, None)
        self.assertEqual(item.thread, self.calculator)
        self.assertEqual(item.size, None)
        self.assertEqual(item.checkState(gui.LOCAL_SUBSCRIPTION_COL),
                         gui.UNCHECKED)

        self.assertTrue(self.calculator.started)


class LocalFoldersPanelTestCase(UbuntuOneBinTestCase):
    """Test suite for the LocalFoldersPanel widget."""

    class_ui = gui.LocalFoldersPanel

    # TODO: add the test suite (LP: #959690).
