# -*- coding: utf-8 -*-

# Authors: Natalia B Bidart <natalia.bidart@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The test suite for the control panel for Ubuntu One."""

import os
import sys

from ubuntuone.devtools.testcase import TestCase as BaseTestCase

from ubuntuone.controlpanel.backend import (
    DEVICE_TYPE_COMPUTER, DEVICE_TYPE_PHONE,
)


if sys.platform == 'win32':
    USER_HOME = 'C:\\Users\\tester'
else:
    USER_HOME = '/home/tester'


TOKEN = {u'consumer_key': u'xQ7xDAz',
         u'consumer_secret': u'KzCJWCTNbbntwfyCKKjomJDzlgqxLy',
         u'name': u'Ubuntu One @ localhost',
         u'token': u'ABCDEF01234-localtoken',
         u'token_secret': u'qFYImEtlczPbsCnYyuwLoPDlPEnvNcIktZphPQklAWrvyfFMV'}

SAMPLE_ACCOUNT_JSON = """
{
    "username": "andrewpz",
    "openid": "https://login.launchpad.net/+id/abcdefg",
    "first_name": "Andrew P.",
    "last_name": "Zoilo",
    "couchdb": {
        "host": "https://couchdb.one.ubuntu.com",
        "root": "https://couchdb.one.ubuntu.com/u/abc/def/12345",
        "dbpath": "u/abc/def/12345"
    },
    "couchdb_root": "https://couchdb.one.ubuntu.com/u/abc/def/12345",
    "email": "andrewpz@protocultura.net",%s
    "nickname": "Andrew P. Zoilo",
    "id": 12345,
    "subscription": {
        "upgrade_available": false,
        "description": "Paid Plan, 50 GB of storage",
        "trial": false,
        "started": "2010-03-24T18:38:38Z",
        "is_paid": true,
        "expires": null,
        "qty": 1,
        "price": 0.0,
        "currency": null,
        "id": 654321,
        "name": "50 GB"
    }
}
"""

CURRENT_PLAN = "Ubuntu One Basic (2 GB) + 1 x 20-Pack with 20 GB (monthly)"
SAMPLE_CURRENT_PLAN = '\n    "current_plan": "%s",' % CURRENT_PLAN

SAMPLE_ACCOUNT_NO_CURRENT_PLAN = SAMPLE_ACCOUNT_JSON % ''
SAMPLE_ACCOUNT_WITH_CURRENT_PLAN = SAMPLE_ACCOUNT_JSON % SAMPLE_CURRENT_PLAN


SAMPLE_QUOTA_JSON = """
{
    "total": 53687091200,
    "used": 2350345156
}
"""

EXPECTED_ACCOUNT_INFO = {
    "quota_used": 2350345156,
    "quota_total": 53687091200,
    "type": "Paid Plan, 50 GB of storage",
    "name": "Andrew P. Zoilo",
    "email": "andrewpz@protocultura.net",
}

EXPECTED_ACCOUNT_INFO_WITH_CURRENT_PLAN = {
    "quota_used": 2350345156,
    "quota_total": 53687091200,
    "type": CURRENT_PLAN,
    "name": "Andrew P. Zoilo",
    "email": "andrewpz@protocultura.net",
}


# note that local computer is not first, do not change!
SAMPLE_DEVICES_JSON = """
[
    {
        "token": "ABCDEF01234token",
        "description": "Ubuntu One @ darkstar",
        "kind": "Computer"
    },
    {
        "token": "ABCDEF01234-localtoken",
        "description": "Ubuntu One @ localhost",
        "kind": "Computer"
    },
    {
        "kind": "Phone",
        "description": "Nokia E65",
        "id": 1000
    }
]
"""

EMPTY_DESCRIPTION_JSON = """
[
    {
        "token": "ABCDEF01234token",
        "description": null,
        "kind": "Computer"
    }
]
"""

LOCAL_DEVICE = {
    'is_local': True,
    'configurable': True,
    'device_id': 'ComputerABCDEF01234-localtoken',
    'limit_bandwidth': False,
    'autoconnect': True,
    'show_all_notifications': True,
    'share_autosubscribe': False,
    'udf_autosubscribe': False,
    'max_download_speed': -1,
    'max_upload_speed': -1,
    'name': 'Ubuntu One @ localhost',
    'type': 'Computer',
}

# note that local computer should be first, do not change!
EXPECTED_DEVICES_INFO = [
    LOCAL_DEVICE,
    {
        "device_id": "ComputerABCDEF01234token",
        "name": "Ubuntu One @ darkstar",
        "type": DEVICE_TYPE_COMPUTER,
        "is_local": False,
        "configurable": False,
    },
    {
        "device_id": "Phone1000",
        "name": "Nokia E65",
        "type": DEVICE_TYPE_PHONE,
        "configurable": False,
        "is_local": False,
    },
]

# note that local computer should be first, do not change!
EXPECTED_DEVICE_NAMES_INFO = [
    {
        'device_id': 'ComputerABCDEF01234-localtoken',
        'name': 'Ubuntu One @ localhost',
        'type': DEVICE_TYPE_COMPUTER,
        'is_local': True,
    },
    {
        'device_id': 'ComputerABCDEF01234token',
        'name': 'Ubuntu One @ darkstar',
        'type': DEVICE_TYPE_COMPUTER,
        'is_local': False,
    },
    {
        'device_id': 'Phone1000',
        'name': 'Nokia E65',
        'type': DEVICE_TYPE_PHONE,
        'is_local': False,
    },
]

SAMPLE_FOLDERS = [
    {u'generation': u'2',
     u'node_id': u'341da068-81d8-437a-8f75-5bb9d86455ba',
     u'path': os.path.join(USER_HOME, 'Public'),
     u'subscribed': u'True',
     u'suggested_path': u'~/Public',
     u'type': u'UDF',
     u'volume_id': u'9ea892f8-15fa-4201-bdbf-8de99fa5f588'},

    {u'generation': u'',
     u'node_id': u'11fbc86c-0d7a-49f5-ae83-8402caf66c6a',
     u'path': os.path.join(USER_HOME, 'Documents'),
     u'subscribed': u'',
     u'suggested_path': u'~/Documents',
     u'type': u'UDF',
     u'volume_id': u'2db262f5-a151-4c19-969c-bb5ced753c61'},

    {u'generation': u'24',
     u'node_id': u'9ee0e130-a7c7-4d76-a5e3-5df506221b48',
     u'path': os.path.join(USER_HOME, 'Pictures', 'Photos'),
     u'subscribed': u'True',
     u'suggested_path': u'~/Pictures/Photos',
     u'type': u'UDF',
     u'volume_id': u'1deb2874-3d28-46ae-9999-d5f48de9f460'},
]

ROOT_PATH = os.path.join(USER_HOME, 'My Ubuntu')
SHARES_PATH = os.path.join(USER_HOME, '.local', 'share', 'ubuntuone', 'shares')
SHARES_PATH_LINK = os.path.join(ROOT_PATH, 'Shared With Me')

SAMPLE_SHARES = [

    {u'accepted': u'True',
     u'subscribed': u'True',
     u'access_level': u'View',
     u'free_bytes': u'39892622746',
     u'generation': u'2704',
     u'name': u're',
     u'node_id': u'c483f419-ed28-490a-825d-a8c074e2d795',
     u'other_username': u'otheruser',
     u'other_visible_name': u'Other User',
     u'path': os.path.join(SHARES_PATH, 're from Other User'),
     u'type': u'Share',
     u'volume_id': u'4a1b263b-a2b3-4f66-9e66-4cd18050810d'},

    {u'accepted': u'True',
     u'subscribed': u'True',
     u'access_level': u'Modify',
     u'free_bytes': u'39892622746',
     u'generation': u'2704',
     u'name': u'do',
     u'node_id': u'84544ea4-aefe-4f91-9bb9-ed7b0a805baf',
     u'other_username': u'otheruser',
     u'other_visible_name': u'Other User',
     u'path': os.path.join(SHARES_PATH, 'do from Other User'),
     u'type': u'Share',
     u'volume_id': u'7d130dfe-98b2-4bd5-8708-9eeba9838ac0'},

    {u'name': u'yadda',
     u'subscribed': u'True',
     u'generation': u'36',
     u'other_username': u'https://login.ubuntu.com/+id/nHRnYmz',
     u'other_visible_name': u'',
     u'access_level': u'View',
     u'node_id': u'ca3a1cec-09d2-485e-9685-1a5180bd6441',
     u'volume_id': u'f1f1741f-ba49-46ef-b682-816c97e6e3d6',
     u'free_bytes': u'2146703403',
     u'path': os.path.join(SHARES_PATH, 'yadda from '),
     u'accepted': u'True',
     u'type': u'Share'},

    {u'name': u'images',
     u'subscribed': u'True',
     u'generation': u'36',
     u'other_username':
     u'https://login.ubuntu.com/+id/nHRnYmz',
     u'other_visible_name': u'',
     u'access_level': u'Modify',
     u'node_id':
     u'b932f0d3-6998-423f-9225-7683a4adbd6f',
     u'volume_id':
     u'a73f889d-ffd3-4447-b351-f0d8130d1e1a',
     u'free_bytes': u'2146703403',
     u'path': os.path.join(SHARES_PATH, 'images from '),
     u'accepted': u'True',
     u'type': u'Share'},

    {u'name': u'read-only',
     u'subscribed': u'True',
     u'generation': u'315',
     u'other_username': u'sharing-user',
     u'other_visible_name': u'A Sharing User',
     u'access_level': u'View',
     u'node_id': u'fd944823-e4c2-45a4-8c95-05997698bdbc',
     u'volume_id':
     u'64b43c96-6c7c-4135-994f-03a1a3774512',
     u'free_bytes': u'108786811673',
     u'path': os.path.join(SHARES_PATH, 'read-only from A Sharing User'),
     u'accepted': u'True',
     u'type': u'Share'},

    {u'name': u'read-write',
     u'subscribed': u'True',
     u'generation': u'314',
     u'other_username': u'sharing-user',
     u'other_visible_name': u'A Sharing User',
     u'access_level': u'Modify',
     u'node_id': u'e95662f7-9979-4745-8c21-8edaf893f143',
     u'volume_id': u'8896e8f8-57a3-4bf9-8558-fc54b7a3a777',
     u'free_bytes': u'108786811673',
     u'path': os.path.join(SHARES_PATH, 'read-write from A Sharing User'),
     u'accepted': u'True',
     u'type': u'Share'},

    {u'accepted': u'',
     u'access_level': u'View',
     u'free_bytes': u'',
     u'generation': u'',
     u'name': u'unaccepted',
     u'node_id': u'67b61c92-855c-49d8-8e09-6d201d15c999',
     u'other_username': u'bad guy',
     u'other_visible_name': u'',
     u'path': os.path.join(SHARES_PATH, 'unaccepted from '),
     u'subscribed': u'',
     u'type': u'Share',
     u'volume_id': u'19963c95-c684-48db-8668-ebe6d820d5c3'},
]

SAMPLE_SHARED = [
    {u'accepted': u'True',
     u'access_level': u'View',
     u'free_bytes': u'',
     u'generation': u'',
     u'name': u'bar',
     u'node_id': u'31e47530-9448-4f03-b4dc-4154fdf35225',
     u'other_username': u'otheruser',
     u'other_visible_name': u'Other User',
     u'path': os.path.join(ROOT_PATH, 'bar'),
     u'type': u'Shared',
     u'volume_id': u'79584900-517f-4dff-b2f3-20e8c1e79365'},
]


class TestCase(BaseTestCase):
    """Basics for testing."""

    assertIs = BaseTestCase.assertIdentical

    def setUp(self):
        self._called = False

    def tearDown(self):
        self._called = False

    def _set_called(self, *args, **kwargs):
        """Store 'args' and 'kwargs' for test assertions."""
        self._called = (args, kwargs)
