# -*- coding: utf-8 -*-

# Authors: Alejandro J. Cura <alecu@canonical.com>
# Authors: Natalia B. Bidart <natalia.bidart@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the DBus service when accessing SyncDaemon."""

import dbus

from ubuntuone.platform.linux import dbus_interface as sd_dbus_iface

from ubuntuone.controlpanel import sd_client
from ubuntuone.controlpanel.dbustests import DBusClientTestCase


SD_DBUS_IFACE_NAME = sd_dbus_iface.DBUS_IFACE_NAME
SD_DBUS_IFACE_STATUS_NAME = sd_dbus_iface.DBUS_IFACE_STATUS_NAME

# pylint, you have to go to decorator's school
# pylint: disable=C0322

# Access to a protected member of a client class
# pylint: disable=W0212


class StatusMockDBusSyncDaemon(dbus.service.Object):
    """A mock object that mimicks syncdaemon regarding the Status iface."""

    state_dict = {
        'name': 'TEST',
        'description': 'Some test state, nothing else.',
        'is_error': '',
        'is_connected': 'True',
        'is_online': '',
        'queues': 'GORGEOUS',
        'connection': '',
    }

    def _get_current_state(self):
        """Get the current status of the system."""
        return self.state_dict

    @dbus.service.method(SD_DBUS_IFACE_STATUS_NAME,
                         in_signature='', out_signature='a{ss}')
    def current_status(self):
        """Return the current faked status of the system."""
        return self._get_current_state()

    # pylint: disable=C0103
    # Invalid name "StatusChanged"

    @dbus.service.signal(SD_DBUS_IFACE_STATUS_NAME)
    def StatusChanged(self, status):
        """Fire a signal to notify that the status of the system changed."""

    def emit_status_changed(self, state=None):
        """Emit StatusChanged."""
        self.StatusChanged(self._get_current_state())


class StatusTestCase(DBusClientTestCase):
    """Test for the status dbus client methods."""

    def setUp(self):
        super(StatusTestCase, self).setUp()
        self.register_mockserver(SD_DBUS_IFACE_NAME,
                                 "/status", StatusMockDBusSyncDaemon)

    def test_set_status_changed_handler(self):
        """A proper callback can be connected to StatusChanged signal."""
        client = sd_client.SyncDaemonClient()
        _, sig = client.set_status_changed_handler(self._set_called)

        self.assertEqual(sig._handler, self._set_called)
        self.assertEqual(sig._member, 'StatusChanged')
        self.assertEqual(sig._path, '/status')
        self.assertEqual(sig._interface, SD_DBUS_IFACE_STATUS_NAME)
