/*
 * Syncdaemon API
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * Copyright 2010-2012 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 *
 */

#include "syncdaemon-share-info.h"

G_DEFINE_TYPE(SyncdaemonShareInfo, syncdaemon_share_info, G_TYPE_OBJECT)

struct _SyncdaemonShareInfoPrivate {
	gchar *name;
	SyncdaemonShareInfoType type;
	gchar *path;
	gchar *username;
	gchar *user_visible_name;
	gchar *node_id;
	gchar *volume_id;
	gboolean accepted;
	gboolean allow_modifications;
};

static void
syncdaemon_share_info_finalize (GObject *object)
{
	SyncdaemonShareInfo *sinfo = SYNCDAEMON_SHARE_INFO (object);

	if (sinfo->priv != NULL) {
		if (sinfo->priv->name != NULL)
			g_free (sinfo->priv->name);

		if (sinfo->priv->path != NULL)
			g_free (sinfo->priv->path);

		if (sinfo->priv->username != NULL)
			g_free (sinfo->priv->username);

		if (sinfo->priv->user_visible_name != NULL)
			g_free (sinfo->priv->user_visible_name);

		if (sinfo->priv->node_id != NULL)
			g_free (sinfo->priv->node_id);

		if (sinfo->priv->volume_id != NULL)
			g_free (sinfo->priv->volume_id);

		g_free (sinfo->priv);
	}

	G_OBJECT_CLASS (syncdaemon_share_info_parent_class)->finalize (object);
}

static void
syncdaemon_share_info_class_init (SyncdaemonShareInfoClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize = syncdaemon_share_info_finalize;
}

static void
syncdaemon_share_info_init (SyncdaemonShareInfo *sinfo)
{
	sinfo->priv = g_new0 (SyncdaemonShareInfoPrivate, 1);
}

/**
 * syncdaemon_share_info_new:
 */
SyncdaemonShareInfo *
syncdaemon_share_info_new (void)
{
	return g_object_new (SYNCDAEMON_TYPE_SHARE_INFO, NULL);
}

/**
 * syncdaemon_share_info_new_from_hash_table:
 */
SyncdaemonShareInfo *
syncdaemon_share_info_new_from_hash_table (GHashTable *hash)
{
	SyncdaemonShareInfo *sinfo;

	sinfo = g_object_new (SYNCDAEMON_TYPE_SHARE_INFO, NULL);
	if (hash != NULL) {
		gchar *str;

		syncdaemon_share_info_set_name (sinfo, g_hash_table_lookup (hash, "name"));
		syncdaemon_share_info_set_path (sinfo, g_hash_table_lookup (hash, "path"));
		syncdaemon_share_info_set_username (sinfo, g_hash_table_lookup (hash, "other_username"));
		syncdaemon_share_info_set_user_visible_name (
			sinfo,
			g_hash_table_lookup (hash, "other_visible_name"));
		syncdaemon_share_info_set_node_id (sinfo, g_hash_table_lookup (hash, "node_id"));
		syncdaemon_share_info_set_volume_id (sinfo, g_hash_table_lookup (hash, "volume_id"));
		syncdaemon_share_info_set_accepted (
			sinfo,
			g_strcmp0 (g_hash_table_lookup (hash, "accepted"), "True") == 0);
		syncdaemon_share_info_set_allow_modifications (
			sinfo,
			g_strcmp0 (g_hash_table_lookup (hash, "access_level"), "Modify") == 0);

		str = g_hash_table_lookup (hash, "type");
		if (g_strcmp0 (str, "Shared") == 0)
			syncdaemon_share_info_set_share_type (sinfo, SYNCDAEMON_SHARE_INFO_TYPE_SHARED);
		else if (g_strcmp0 (str, "Share") == 0)
			syncdaemon_share_info_set_share_type (sinfo, SYNCDAEMON_SHARE_INFO_TYPE_SHARE);
		else
			syncdaemon_share_info_set_share_type (sinfo, SYNCDAEMON_SHARE_INFO_TYPE_UNKNOWN);
	}

	return sinfo;
}

/**
 * syncdameon_share_info_get_name:
 */
const gchar *
syncdaemon_share_info_get_name (SyncdaemonShareInfo *sinfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_SHARE_INFO (sinfo), NULL);

	return (const gchar *) sinfo->priv->name;
}

/**
 * syncdameon_share_info_set_name:
 */
void
syncdaemon_share_info_set_name (SyncdaemonShareInfo *sinfo, const gchar *name)
{
	g_return_if_fail (SYNCDAEMON_IS_SHARE_INFO (sinfo));

	if (sinfo->priv->name != NULL)
		g_free (sinfo->priv->name);

	sinfo->priv->name = g_strdup (name);
}

/**
 * syncdameon_share_info_get_share_type:
 */
SyncdaemonShareInfoType
syncdaemon_share_info_get_share_type (SyncdaemonShareInfo *sinfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_SHARE_INFO (sinfo), SYNCDAEMON_SHARE_INFO_TYPE_UNKNOWN);

	return sinfo->priv->type;
}

/**
 * syncdameon_share_info_set_share_type:
 */
void
syncdaemon_share_info_set_share_type (SyncdaemonShareInfo *sinfo, SyncdaemonShareInfoType type)
{
	g_return_if_fail (SYNCDAEMON_IS_SHARE_INFO (sinfo));

	sinfo->priv->type = type;
}

/**
 * syncdameon_share_info_get_path:
 */
const gchar *
syncdaemon_share_info_get_path (SyncdaemonShareInfo *sinfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_SHARE_INFO (sinfo), NULL);

	return (const gchar *) sinfo->priv->path;
}

/**
 * syncdameon_share_info_set_path:
 */
void
syncdaemon_share_info_set_path (SyncdaemonShareInfo *sinfo, const gchar *path)
{
	g_return_if_fail (SYNCDAEMON_IS_SHARE_INFO (sinfo));

	if (sinfo->priv->path != NULL)
		g_free (sinfo->priv->path);

	sinfo->priv->path = g_strdup (path);
}

/**
 * syncdameon_share_info_get_username:
 */
const gchar *
syncdaemon_share_info_get_username (SyncdaemonShareInfo *sinfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_SHARE_INFO (sinfo), NULL);

	return (const gchar *) sinfo->priv->username;
}

/**
 * syncdameon_share_info_set_username:
 */
void
syncdaemon_share_info_set_username (SyncdaemonShareInfo *sinfo, const gchar *username)
{
	g_return_if_fail (SYNCDAEMON_IS_SHARE_INFO (sinfo));

	if (sinfo->priv->username != NULL)
		g_free (sinfo->priv->username);

	sinfo->priv->username = g_strdup (username);
}

/**
 * syncdameon_share_info_get_user_visible_name:
 */
const gchar *
syncdaemon_share_info_get_user_visible_name (SyncdaemonShareInfo *sinfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_SHARE_INFO (sinfo), NULL);

	return (const gchar *) sinfo->priv->user_visible_name;
}

/**
 * syncdameon_share_info_set_user_visible_name:
 */
void
syncdaemon_share_info_set_user_visible_name (SyncdaemonShareInfo *sinfo, const gchar *user_visible_name)
{
	g_return_if_fail (SYNCDAEMON_IS_SHARE_INFO (sinfo));

	if (sinfo->priv->user_visible_name != NULL)
		g_free (sinfo->priv->user_visible_name);

	sinfo->priv->user_visible_name = g_strdup (user_visible_name);
}

/**
 * syncdameon_share_info_get_node_id:
 */
const gchar *
syncdaemon_share_info_get_node_id (SyncdaemonShareInfo *sinfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_SHARE_INFO (sinfo), NULL);

	return (const gchar *) sinfo->priv->node_id;
}

/**
 * syncdameon_share_info_set_node_id:
 */
void
syncdaemon_share_info_set_node_id (SyncdaemonShareInfo *sinfo, const gchar *node_id)
{
	g_return_if_fail (SYNCDAEMON_IS_SHARE_INFO (sinfo));

	if (sinfo->priv->node_id != NULL)
		g_free (sinfo->priv->node_id);

	sinfo->priv->node_id = g_strdup (node_id);
}

/**
 * syncdameon_share_info_get_volume_id:
 */
const gchar *
syncdaemon_share_info_get_volume_id (SyncdaemonShareInfo *sinfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_SHARE_INFO (sinfo), NULL);

	return (const gchar *) sinfo->priv->volume_id;
}

/**
 * syncdameon_share_info_set_volume_id:
 */
void
syncdaemon_share_info_set_volume_id (SyncdaemonShareInfo *sinfo, const gchar *volume_id)
{
	g_return_if_fail (SYNCDAEMON_IS_SHARE_INFO (sinfo));

	if (sinfo->priv->volume_id != NULL)
		g_free (sinfo->priv->volume_id);

	sinfo->priv->volume_id = g_strdup (volume_id);
}

/**
 * syncdameon_share_info_get_accepted:
 */
gboolean
syncdaemon_share_info_get_accepted (SyncdaemonShareInfo *sinfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_SHARE_INFO (sinfo), FALSE);

	return sinfo->priv->accepted;
}

/**
 * syncdameon_share_info_set_accepted:
 */
void
syncdaemon_share_info_set_accepted (SyncdaemonShareInfo *sinfo, gboolean accepted)
{
	g_return_if_fail (SYNCDAEMON_IS_SHARE_INFO (sinfo));

	sinfo->priv->accepted = accepted;
}

/**
 * syncdameon_share_info_get_allow_modifications:
 */
gboolean
syncdaemon_share_info_get_allow_modifications (SyncdaemonShareInfo *sinfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_SHARE_INFO (sinfo), FALSE);

	return sinfo->priv->allow_modifications;
}

/**
 * syncdameon_share_info_set_allow_modifications:
 */
void
syncdaemon_share_info_set_allow_modifications (SyncdaemonShareInfo *sinfo, gboolean allow_modifications)
{
	g_return_if_fail (SYNCDAEMON_IS_SHARE_INFO (sinfo));

	sinfo->priv->allow_modifications = allow_modifications;
}
