/*
 * Syncdaemon API
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * Copyright 2010-2012 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 *
 */

#include "syncdaemon-folder-info.h"

G_DEFINE_TYPE(SyncdaemonFolderInfo, syncdaemon_folder_info, G_TYPE_OBJECT)

struct _SyncdaemonFolderInfoPrivate {
	gchar *path;
	gboolean subscribed;
	gchar *volume_id;
};

static void
syncdaemon_folder_info_finalize (GObject *object)
{
	SyncdaemonFolderInfo *finfo = SYNCDAEMON_FOLDER_INFO (object);

	if (finfo->priv != NULL) {
		if (finfo->priv->path != NULL)
			g_free (finfo->priv->path);

		if (finfo->priv->volume_id != NULL)
			g_free (finfo->priv->volume_id);

		g_free (finfo->priv);
	}

	G_OBJECT_CLASS (syncdaemon_folder_info_parent_class)->finalize (object);
}

static void
syncdaemon_folder_info_class_init (SyncdaemonFolderInfoClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize = syncdaemon_folder_info_finalize;
}

static void
syncdaemon_folder_info_init (SyncdaemonFolderInfo *finfo)
{
	finfo->priv = g_new0 (SyncdaemonFolderInfoPrivate, 1);
}

/**
 * syncdaemon_folder_info_new:
 */
SyncdaemonFolderInfo *
syncdaemon_folder_info_new (void)
{
	return g_object_new (SYNCDAEMON_TYPE_FOLDER_INFO, NULL);
}

/**
 * syncdaemon_folder_info_new_from_hash_table:
 */
SyncdaemonFolderInfo *
syncdaemon_folder_info_new_from_hash_table (GHashTable *hash)
{
	SyncdaemonFolderInfo *finfo;

	finfo = g_object_new (SYNCDAEMON_TYPE_FOLDER_INFO, NULL);
	if (hash != NULL) {
		syncdaemon_folder_info_set_path (finfo, g_hash_table_lookup (hash, "path"));
		syncdaemon_folder_info_set_subscribed (
			finfo,
			g_strcmp0 (g_hash_table_lookup (hash, "subscribed"), "True") == 0);
		syncdaemon_folder_info_set_volume_id (finfo, g_hash_table_lookup (hash, "volume_id"));
	}

	return finfo;
}

/**
 * syncdaemon_folder_info_get_path:
 */
const gchar *
syncdaemon_folder_info_get_path (SyncdaemonFolderInfo *finfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_FOLDER_INFO (finfo), NULL);

	return (const gchar *) finfo->priv->path;
}

/**
 * syncdaemon_folder_info_set_path:
 */
void
syncdaemon_folder_info_set_path (SyncdaemonFolderInfo *finfo, const gchar *path)
{
	g_return_if_fail (SYNCDAEMON_IS_FOLDER_INFO (finfo));

	if (finfo->priv->path != NULL)
		g_free (finfo->priv->path);

	finfo->priv->path = g_strdup (path);
}

/**
 * syncdaemon_folder_info_get_subscribed:
 */
gboolean
syncdaemon_folder_info_get_subscribed (SyncdaemonFolderInfo *finfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_FOLDER_INFO (finfo), FALSE);

	return finfo->priv->subscribed;
}

/**
 * syncdaemon_folder_info_set_subscribed:
 */
void
syncdaemon_folder_info_set_subscribed (SyncdaemonFolderInfo *finfo, gboolean subscribed)
{
	g_return_if_fail (SYNCDAEMON_IS_FOLDER_INFO (finfo));

	finfo->priv->subscribed = subscribed;
}

/**
 * syncdaemon_folder_info_get_volume_id:
 */
const gchar *
syncdaemon_folder_info_get_volume_id (SyncdaemonFolderInfo *finfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_FOLDER_INFO (finfo), NULL);

	return (const gchar *) finfo->priv->volume_id;
}

/**
 * syncdaemon_folder_info_set_volume_id:
 */
void
syncdaemon_folder_info_set_volume_id (SyncdaemonFolderInfo *finfo, const gchar *volume_id)
{
	g_return_if_fail (SYNCDAEMON_IS_FOLDER_INFO (finfo));

	if (finfo->priv->volume_id != NULL)
		g_free (finfo->priv->volume_id);

	finfo->priv->volume_id = g_strdup (volume_id);
}
