# -*- coding: utf-8 -*-
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# In addition, as a special exception, the copyright holders give
# permission to link the code of portions of this program with the
# OpenSSL library under certain conditions as described in each
# individual source file, and distribute linked combinations
# including the two.
# You must obey the GNU General Public License in all respects
# for all of the code used other than OpenSSL.  If you modify
# file(s) with this exception, you may extend this exception to your
# version of the file(s), but you are not obligated to do so.  If you
# do not wish to do so, delete this exception statement from your
# version.  If you delete this exception statement from all source
# files in the program, then also delete it here.
"""Tests for some tools for talking to the syncdaemon."""

import sys

from twisted.internet import defer
from twisted.trial.unittest import TestCase

from ubuntuone.platform.tools import windows


# ugly trick to stop pylint for complaining about
# WindowsError on Linux
if sys.platform != 'win32':
    WindowsError = None


class TestSyncDaemonTool(TestCase):
    """Various utility methods to test/play with the SyncDaemon."""

    @defer.inlineCallbacks
    def setUp(self):
        yield super(TestSyncDaemonTool, self).setUp()
        self.patch(windows.UbuntuOneClient, "connect", lambda _: defer.Deferred())
        self.sdtool = windows.SyncDaemonToolProxy()

    def test_call_after_connection(self):
        """Test the _call_after_connection method."""
        collected_calls = []
        test_method = lambda *args: collected_calls.append(args)
        test_method = self.sdtool._call_after_connection(test_method)
        test_method(123)
        self.assertEqual(len(collected_calls), 0)
        self.sdtool.connected.callback("got connected!")
        self.assertEqual(len(collected_calls), 1)

    def test_should_wrap(self):
        """Only some attributes should be wrapped."""
        test_function = "sample_function"
        assert test_function not in self.sdtool._DONT_VERIFY_CONNECTED
        self.assertTrue(self.sdtool._should_wrap(test_function))

    def test_should_not_wrap_listed_attributes(self):
        """Attributes listed in DONT_VERIFY_CONNECTED should not be wrapped."""
        for attribute_name in self.sdtool._DONT_VERIFY_CONNECTED:
            self.assertFalse(self.sdtool._should_wrap(attribute_name))

    def test_should_not_wrap_underscore_attributes(self):
        """Attributes starting with underscore should not be wrapped."""
        dunder_function = "__sample_attribute__"
        assert dunder_function not in self.sdtool._DONT_VERIFY_CONNECTED
        self.assertFalse(self.sdtool._should_wrap(dunder_function))

        under_function = "_sampleattribute"
        assert under_function not in self.sdtool._DONT_VERIFY_CONNECTED
        self.assertFalse(self.sdtool._should_wrap(under_function))

    def test_getattributes_wraps_methods(self):
        """The common methods are wrapped with connect."""
        for attr_name in dir(self.sdtool):
            if self.sdtool._should_wrap(attr_name):
                attr = getattr(self.sdtool, attr_name)
                func_name = getattr(attr, "__name__", None)
                self.assertEqual(func_name, "call_after_connection_inner")

    def test_getattributes_does_not_wrap_special(self):
        """The special methods and attributes are not wrapped."""
        for attr_name in dir(self.sdtool):
            if not self.sdtool._should_wrap(attr_name):
                attr = getattr(self.sdtool, attr_name)
                func_name = getattr(attr, "__name__", None)
                self.assertNotEqual(func_name, "call_after_connection_inner")

    def test_start_missing_exe(self):
        """Test starting the service when the exe is missing."""
        # file is missing
        self.patch(windows.os.path, 'exists', lambda f: False)
        key = 'key'
        path = 'path/to/exe'
        self.patch(windows, 'OpenKey', lambda k,p: key)
        self.patch(windows, 'QueryValueEx', lambda k,p: path)

        self.assertFailure(self.sdtool.start(), WindowsError)
