#
# Author: Manuel de la Pena <manuel@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Test the filesystem notifications on windows."""

import os
import tempfile
import time

from mocker import MockerTestCase, ANY

from contrib.testing.testcase import BaseTwistedTestCase
from ubuntuone.platform.windows.pyinotify import ProcessEvent
from ubuntuone.platform.windows.os_helper import LONG_PATH_PREFIX
from ubuntuone.platform.windows.filesystem_notifications import (
    WatchManager,
    Notifier,
    FILE_NOTIFY_CHANGE_FILE_NAME,
    FILE_NOTIFY_CHANGE_DIR_NAME,
    FILE_NOTIFY_CHANGE_ATTRIBUTES,
    FILE_NOTIFY_CHANGE_SIZE,
    FILE_NOTIFY_CHANGE_LAST_WRITE,
    FILE_NOTIFY_CHANGE_SECURITY,
    FILE_NOTIFY_CHANGE_LAST_ACCESS
)

class TestCaseHandler(ProcessEvent):
    """ProcessEvent used for test cases."""

    def my_init(self, **kwargs):
        """Init the event notifier."""
        self.processed_events = []

    def process_IN_CREATE(self, event):
        self.processed_events.append(event)

    def process_IN_DELETE(self, event):
        self.processed_events.append(event)

    def process_default(self, event):
        """Process events and added to the list."""
        self.processed_events.append(event)

class TestWatch(BaseTwistedTestCase):
    """Test the watch so that it returns the same events as pyinotify."""

    def setUp(self):
        """Set infor for the tests."""
        super(TestWatch, self).setUp()
        self.basedir = LONG_PATH_PREFIX + os.path.abspath(
            self.mktemp('test_root'))
        self.mask = FILE_NOTIFY_CHANGE_FILE_NAME | \
            FILE_NOTIFY_CHANGE_DIR_NAME | \
            FILE_NOTIFY_CHANGE_ATTRIBUTES | \
            FILE_NOTIFY_CHANGE_SIZE | \
            FILE_NOTIFY_CHANGE_LAST_WRITE | \
            FILE_NOTIFY_CHANGE_SECURITY | \
            FILE_NOTIFY_CHANGE_LAST_ACCESS

    def _perform_operations(self, path, mask, auto_add, actions, number_events):
        """Performs the file operations and returns the recorded events."""
        manager = WatchManager()
        manager.add_watch(path, mask, auto_add=auto_add)
        handler = TestCaseHandler()
        notifier = Notifier(manager, handler)
        # execution the actions
        actions()
        # process the recorded events
        while not len(handler.processed_events) >= number_events:
            notifier.process_events()
        events = handler.processed_events
        notifier.stop()
        return events

    def test_file_create(self):
        """Test that the correct event is returned on a file create."""
        file_name = os.path.join(self.basedir, 'test_file_create')

        def create_file():
            """Action used for the test."""
            # simply create a new file
            open(file_name, 'w').close()

        event = self._perform_operations(self.basedir, self.mask, False,
            create_file, 1)[0]
        self.assertFalse(event.dir)
        self.assertEqual(0x100, event.mask)
        self.assertEqual('IN_CREATE', event.maskname)
        self.assertEqual(os.path.split(file_name)[1], event.name)
        self.assertEqual('.', event.path)
        self.assertEqual(os.path.join(self.basedir, file_name), event.pathname)
        self.assertEqual(0, event.wd)

    def test_dir_create(self):
        """Test that the correct event is returned on a dir creation."""
        dir_name = os.path.join(self.basedir, 'test_dir_create')

        def create_dir():
            """Action for the test."""
            os.mkdir(dir_name)

        event = self._perform_operations(self.basedir, self.mask, False,
            create_dir, 1)[0]
        self.assertTrue(event.dir)
        self.assertEqual(0x40000100, event.mask)
        self.assertEqual('IN_CREATE|IN_ISDIR', event.maskname)
        self.assertEqual(os.path.split(dir_name)[1], event.name)
        self.assertEqual('.', event.path)
        self.assertEqual(os.path.join(self.basedir, dir_name), event.pathname)
        self.assertEqual(0, event.wd)

    def test_file_remove(self):
        """Test that the correct event is raised when a file is removed."""
        file_name = os.path.join(self.basedir, 'test_file_remove')
        # create the file before recording
        open(file_name, 'w').close()

        def remove_file():
            """Action for the test."""
            os.remove(file_name)

        event = self._perform_operations(self.basedir, self.mask, False,
            remove_file, 1)[0]
        self.assertFalse(event.dir)
        self.assertEqual(0x200, event.mask)
        self.assertEqual('IN_DELETE', event.maskname)
        self.assertEqual(os.path.split(file_name)[1], event.name)
        self.assertEqual('.', event.path)
        self.assertEqual(os.path.join(self.basedir, file_name), event.pathname)
        self.assertEqual(0, event.wd)

    def test_dir_remove(self):
        """Test that the correct event is raised when a dir is removed."""
        dir_name = os.path.join(self.basedir, 'test_dir_remove')
        # create the dir before recording
        os.mkdir(dir_name)

        def remove_dir():
            """Action for the test."""
            os.rmdir(dir_name)

        event = self._perform_operations(self.basedir, self.mask, False,
            remove_dir, 1)[0]
        self.assertTrue(event.dir)
        self.assertEqual(0x40000200, event.mask)
        self.assertEqual('IN_DELETE|IN_ISDIR', event.maskname)
        self.assertEqual('.', event.path)
        self.assertEqual(os.path.join(self.basedir, dir_name), event.pathname)
        self.assertEqual(0, event.wd)

    def test_file_write(self):
        """Test that the correct event is raised when a file is written."""
        file_name = os.path.join(self.basedir, 'test_file_write')
        # create the file before recording
        fd = open(file_name, 'w')
        def write_file():
            """Action for the test."""
            fd.write('test')
            fd.close()

        event = self._perform_operations(self.basedir, self.mask, False,
            write_file, 1)[0]
        self.assertFalse(event.dir)
        self.assertEqual(0x2, event.mask)
        self.assertEqual('IN_MODIFY', event.maskname)
        self.assertEqual(os.path.split(file_name)[1], event.name)
        self.assertEqual('.', event.path)
        self.assertEqual(os.path.join(self.basedir, file_name), event.pathname)
        self.assertEqual(0, event.wd)
        # clean behind us by removeing the file
        os.remove(file_name)

    def test_file_moved_to_watched_dir_same_watcher(self):
        """Test that the correct event is raised when a file is moved."""
        from_file_name = os.path.join(self.basedir,
            'test_file_moved_to_watched_dir_same_watcher')
        to_file_name = os.path.join(self.basedir,
            'test_file_moved_to_watched_dir_same_watcher_2')
        open(from_file_name, 'w').close()
        #create file before recording

        def move_file():
            """Action for the test."""
            os.rename(from_file_name, to_file_name)
        events = self._perform_operations(self.basedir,
            self.mask, False, move_file, 2)
        move_from_event = events[0]
        move_to_event = events[1]
        # first test the move from
        self.assertFalse(move_from_event.dir)
        self.assertEqual(0x40, move_from_event.mask)
        self.assertEqual('IN_MOVED_FROM', move_from_event.maskname)
        self.assertEqual(os.path.split(from_file_name)[1], move_from_event.name)
        self.assertEqual('.', move_from_event.path)
        self.assertEqual(os.path.join(self.basedir, from_file_name),
            move_from_event.pathname)
        self.assertEqual(0, move_from_event.wd)
        # test the move to
        self.assertFalse(move_to_event.dir)
        self.assertEqual(0x80, move_to_event.mask)
        self.assertEqual('IN_MOVED_TO', move_to_event.maskname)
        self.assertEqual(os.path.split(to_file_name)[1], move_to_event.name)
        self.assertEqual('.', move_to_event.path)
        self.assertEqual(os.path.join(self.basedir, to_file_name),
            move_to_event.pathname)
        self.assertEqual(os.path.split(from_file_name)[1],
            move_to_event.src_pathname)
        self.assertEqual(0, move_to_event.wd)
        # assert that both cookies are the same
        self.assertEqual(move_from_event.cookie, move_to_event.cookie)

    def test_file_moved_to_not_watched_dir(self):
        """Test that the correct event is raised when a file is moved."""
        from_file_name = os.path.join(self.basedir,
            'test_file_moved_to_not_watched_dir')
        open(from_file_name, 'w').close()

        def move_file():
            """Action for the test."""
            os.rename(from_file_name, os.path.join(tempfile.mkdtemp(),
                'test_file_moved_to_not_watched_dir'))

        # while on linux we will have to do some sort of magic like facundo
        # did, on windows we will get a deleted event which is much more
        # cleaner, first time ever windows is nicer!
        event = self._perform_operations(self.basedir, self.mask, False,
            move_file, 1)[0]
        self.assertFalse(event.dir)
        self.assertEqual(0x200, event.mask)
        self.assertEqual('IN_DELETE', event.maskname)
        self.assertEqual(os.path.split(from_file_name)[1], event.name)
        self.assertEqual('.', event.path)
        self.assertEqual(os.path.join(self.basedir, from_file_name), event.pathname)
        self.assertEqual(0, event.wd)

    def test_file_move_from_not_watched_dir(self):
        """Test that the correct event is raised when a file is moved."""
        from_file_name = os.path.join(tempfile.mkdtemp(),
            'test_file_move_from_not_watched_dir')
        to_file_name = os.path.join(self.basedir,
            'test_file_move_from_not_watched_dir')
        # create file before we record
        open(from_file_name, 'w').close()

        def move_files():
            """Action for the test."""
            os.rename(from_file_name, to_file_name)

        # while on linux we have to do some magic operations like facundo did
        # on windows we have a created file, hurray!
        event = self._perform_operations(self.basedir, self.mask, False,
            move_files, 1)[0]
        self.assertFalse(event.dir)
        self.assertEqual(0x100, event.mask)
        self.assertEqual('IN_CREATE', event.maskname)
        self.assertEqual(os.path.split(to_file_name)[1], event.name)
        self.assertEqual('.', event.path)
        self.assertEqual(os.path.join(self.basedir, to_file_name),
            event.pathname)
        self.assertEqual(0, event.wd)

    def test_dir_moved_to_watched_dir_same_watcher(self):
        """Test that the correct event is raised when a dir is moved."""
        from_dir_name = os.path.join(self.basedir,
            'test_dir_moved_to_watched_dir_same_watcher')
        to_dir_name = os.path.join(self.basedir,
            'test_dir_moved_to_watched_dir_same_watcher_2')
        os.mkdir(from_dir_name)
        def move_file():
            """Action for the test."""
            os.rename(from_dir_name, to_dir_name)

        move_from_event, move_to_event = self._perform_operations(self.basedir,
            self.mask, False, move_file, 2)
        # first test the move from
        self.assertTrue(move_from_event.dir)
        self.assertEqual(0x40000040, move_from_event.mask)
        self.assertEqual('IN_MOVED_FROM|IN_ISDIR', move_from_event.maskname)
        self.assertEqual(os.path.split(from_dir_name)[1], move_from_event.name)
        self.assertEqual('.', move_from_event.path)
        self.assertEqual(os.path.join(self.basedir, from_dir_name),
            move_from_event.pathname)
        self.assertEqual(0, move_from_event.wd)
        # test the move to
        self.assertTrue(move_to_event.dir)
        self.assertEqual(0x40000080, move_to_event.mask)
        self.assertEqual('IN_MOVED_TO|IN_ISDIR', move_to_event.maskname)
        self.assertEqual(os.path.split(to_dir_name)[1], move_to_event.name)
        self.assertEqual('.', move_to_event.path)
        self.assertEqual(os.path.join(self.basedir, to_dir_name),
            move_to_event.pathname)
        self.assertEqual(os.path.split(from_dir_name)[1], move_to_event.src_pathname)
        self.assertEqual(0, move_to_event.wd)
        # assert that both cookies are the same
        self.assertEqual(move_from_event.cookie, move_to_event.cookie)

    def test_dir_moved_to_not_watched_dir(self):
        """Test that the correct event is raised when a file is moved."""
        dir_name = os.path.join(self.basedir,
            'test_dir_moved_to_not_watched_dir')
        os.mkdir(dir_name)

        def move_dir():
            """Action for the test."""
            os.rename(dir_name, os.path.join(tempfile.mkdtemp(),
                'test_dir_moved_to_not_watched_dir'))

        # on windows a move to outside a watched dir translates to a remove
        event = self._perform_operations(self.basedir, self.mask, False,
            move_dir, 1)[0]
        self.assertTrue(event.dir)
        self.assertEqual(0x40000200, event.mask)
        self.assertEqual('IN_DELETE|IN_ISDIR', event.maskname)
        self.assertEqual('.', event.path)
        self.assertEqual(os.path.join(self.basedir, dir_name), event.pathname)
        self.assertEqual(0, event.wd)

    def test_dir_move_from_not_watched_dir(self):
        """Test that the correct event is raised when a file is moved."""
        from_dir_name = os.path.join(tempfile.mkdtemp(),
            'test_dir_move_from_not_watched_dir')
        to_dir_name = os.path.join(self.basedir,
            'test_dir_move_from_not_watched_dir')
        # create file before we record
        os.mkdir(from_dir_name)

        def move_dir():
            """Action for the test."""
            os.rename(from_dir_name, to_dir_name)

        event = self._perform_operations(self.basedir, self.mask, False,
            move_dir, 1)[0]
        self.assertTrue(event.dir)
        self.assertEqual(0x40000100, event.mask)
        self.assertEqual('IN_CREATE|IN_ISDIR', event.maskname)
        self.assertEqual(os.path.split(from_dir_name)[1], event.name)
        self.assertEqual('.', event.path)
        self.assertEqual(os.path.join(self.basedir, to_dir_name), event.pathname)
        self.assertEqual(0, event.wd)

    def test_exclude_filter(self):
        """Test that the exclude filter works as expectd."""
        manager = WatchManager()
        # add a watch that will always exclude all actions
        manager.add_watch(self.basedir, self.mask, auto_add=True,
            exclude_filter=lambda x: True )
        handler = TestCaseHandler()
        notifier = Notifier(manager, handler)
        # execution the actions
        file_name = os.path.join(self.basedir, 'test_file_create')
        open(file_name, 'w').close()
        # give some time for the system to get the events
        time.sleep(1)
        notifier.process_events()
        notifier.stop()
        self.assertEqual(0, len(handler.processed_events))


class TestWatchManager(MockerTestCase):
    """Test the watch manager."""

    def setUp(self):
        """Set each of the tests."""
        self.watch_factory = self.mocker.mock()
        self.watch = self.mocker.mock()
        self.manager = WatchManager(watch_factory=self.watch_factory)

    def test_stop(self):
        """Test that the different watches are stoppe."""
        self.manager._wdm = {1: self.watch}
        self.watch.path
        self.watch.stop_watching()
        self.mocker.replay()
        self.manager.stop()

    def test_get_present_watch(self):
        """Test that we can get a Watch using is wd."""
        self.manager._wdm = {1: self.watch}
        self.assertEqual(self.watch, self.manager.get_watch(1))

    def test_get_missing_watch(self):
        """Test that we get an error when trying to get a missing wd."""
        self.assertRaises(KeyError, self.manager.get_watch, (1,))

    def test_delete_present_watch(self):
        """Test that we can remove a present watch."""
        self.manager._wdm = {1: self.watch}
        self.watch.stop_watching()
        self.mocker.replay()
        self.mocker.replay()
        self.manager.del_watch(1)

    def test_add_single_watch(self):
        """Test the addition of a new single watch."""
        wd = 0
        path = 'path'
        mask = 'bit_mask'
        auto_add = True
        events_queue = self.manager._events_queue
        exclude_filter = lambda x: False
        proc_fun = lambda x: None
        # expect the factory to be called with the passed values
        self.watch_factory(wd, path, mask, auto_add,
            events_queue=events_queue, exclude_filter=exclude_filter,
            proc_fun=proc_fun)
        self.mocker.result(self.watch)
        self.watch.start_watching()
        self.mocker.replay()
        self.manager.add_watch(path, mask, proc_fun=proc_fun, auto_add=auto_add,
            exclude_filter=exclude_filter)
        self.assertEqual(1, len(self.manager._wdm))

    def test_update_present_watch(self):
        """Test the update of a present watch."""
        self.manager._wdm = {1: self.watch}
        mask = 'mask'
        proc_fun = lambda x: None
        self.watch.path
        self.watch.stop_watching()
        self.watch.update(mask, proc_fun=proc_fun, auto_add=False)
        self.watch.start_watching()
        self.mocker.replay()
        self.manager.update_watch(1, mask, proc_fun)

    def test_get_wath_present_wd(self):
        """Test that the correct path is returned."""
        self.manager._wdm = {1: self.watch}
        path = 'path'
        self.watch.path
        self.mocker.result(path)
        self.mocker.replay()
        self.assertEqual(path, self.manager.get_path(1))

    def test_get_wath_missing_wd(self):
        """Test that the correct path is returned."""
        self.assertEqual(None, self.manager.get_path(1))

    def test_get_wd_exact_path(self):
        """Test the wd is returned when there is a watch for the path."""
        self.manager._wdm = {1: self.watch}
        path = 'path'
        self.watch.path
        self.mocker.result(path)
        self.mocker.replay()
        self.assertEqual(1, self.manager.get_wd(path))

    def test_get_wd_child_path(self):
        """Test the wd is returned when we have a child path."""
        self.manager._wdm = {1: self.watch}
        path = 'path'
        self.watch.path
        self.mocker.result(path)
        self.watch.auto_add
        self.mocker.result(True)
        self.mocker.replay()
        self.assertEqual(1, self.manager.get_wd(os.path.join(path, 'child')))

    def test_rm_present_wd(self):
        """Test the removal of a present watch."""
        self.manager._wdm = {1: self.watch}
        self.watch.stop_watching()
        self.mocker.replay()
        self.manager.rm_watch(1)
        self.assertEqual(None, self.manager._wdm.get(1))

    def test_rm_root_path(self):
        """Test the removal of a root path."""
        self.manager._wdm = {1: self.watch}
        path = 'path'
        self.watch.path
        self.mocker.result(path)
        self.watch.path
        self.mocker.result(path)
        self.watch.stop_watching()
        self.mocker.replay()
        self.manager.rm_path(path)
        self.assertEqual(None, self.manager._wdm.get(1))

    def test_rm_child_path(self):
        """Test the removal of a child path."""
        self.manager._wdm = {1: self.watch}
        path = 'path'
        self.watch.path
        self.mocker.result(path)
        self.watch.path
        self.mocker.result(path)
        self.watch.auto_add
        self.mocker.result(True)
        self.watch.exclude_filter = ANY
        self.mocker.replay()
        self.manager.rm_path(os.path.join(path, 'child'))
        self.assertEqual(self.watch, self.manager._wdm[1])
