# tests.platform.linux.test_unity
#
# Author: Alejandro J. Cura <alecu@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Tests for the libunity interface."""

from twisted.trial.unittest import TestCase

from ubuntuone.platform.linux import unity


class FakeLauncherEntryProps(object):
    """A fake Unity.LauncherEntry.props"""

    progress = 0.0
    progress_visible = False
    urgent = False


class FakeLauncherEntry(object):
    """A fake Unity.LauncherEntry"""

    @staticmethod
    def get_for_desktop_id(dotdesktop):
        """Find the LauncherEntry for a given dotdesktop."""
        return FakeLauncherEntry()

    def __init__(self):
        """Initialize this fake instance."""
        self.props = FakeLauncherEntryProps()


class LauncherProgressbarEmblemTestCase(TestCase):
    """Test the Launcher progressbar and emblem."""

    skip = None if unity.use_libunity else "libunity not installed."

    def setUp(self):
        """Initialize this test instance."""
        self.patch(unity.Unity, "LauncherEntry", FakeLauncherEntry)
        self.launcher = unity.UbuntuOneLauncher()

    def test_progress_starts_hidden(self):
        """The progressbar starts hidden."""
        self.assertFalse(self.launcher.entry.props.progress_visible,
                         "The progressbar starts hidden.")

    def test_progress_shown(self):
        """The progressbar is shown."""
        self.launcher.show_progressbar()
        self.assertTrue(self.launcher.entry.props.progress_visible,
                        "The progressbar is shown.")

    def test_progress_hidden_after_shown(self):
        """The progressbar is hidden after being shown."""
        self.launcher.show_progressbar()
        self.launcher.hide_progressbar()
        self.assertFalse(self.launcher.entry.props.progress_visible,
                         "The progressbar is hidden.")

    def test_progress_is_updated(self):
        """The progress value is updated."""
        value = 0.5
        self.launcher.set_progress(value)
        self.assertEqual(self.launcher.entry.props.progress, value)

    def test_urgency_set(self):
        """The urgency of the launcher is set."""
        self.launcher.set_urgent()
        self.assertTrue(
            self.launcher.entry.props.urgent,
            "The launcher should be set to urgent.")
        self.launcher.set_urgent(False)
        self.assertFalse(
            self.launcher.entry.props.urgent,
            "The launcher should not be set to urgent.")
