# -*- coding: utf-8 -*-

# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""tests for the syncdaemon u1sdtool script """
import os
from StringIO import StringIO
from tests.syncdaemon.test_tools import TestToolsBase
from ubuntuone.syncdaemon.volume_manager import Share, UDF
from ubuntuone.syncdaemon.tools import (
    show_path_info,
    show_uploads,
    show_downloads,
    show_shares,
    show_shared,
    show_folders,
    show_state,
    show_waiting_content,
    show_waiting_metadata,
)
from tests.syncdaemon.test_dbus import FakeCommand
from twisted.internet import defer


class U1SDToolTests(TestToolsBase):
    """Tests for u1sdtool output"""

    def test_show_shares_empty(self):
        """test the output of --list-shared """
        out = StringIO()
        d = self.tool.get_shares()
        d.addCallback(lambda result: show_shares(result, out))
        def check(result):
            """check the output"""
            self.assertEquals('No shares\n', out.getvalue())
        d.addCallback(check)
        return d

    def test_show_shares(self):
        """test the output of --list-shared """
        out = StringIO()
        share_path = os.path.join(self.shares_dir, u"ñoño".decode('utf-8'))
        share = Share(path=share_path, name=u"ñoño", volume_id='share_id',
                      access_level='View', other_username='fake_user',
                      other_visible_name=u"ñoño")
        self.main.vm.add_share(share)
        expected = u"Shares list:\n id=share_id name=\xf1o\xf1o " + \
                "accepted=False access_level=View from=fake_user\n"
        d = self.tool.get_shares()
        d.addCallback(lambda result: show_shares(result, out))
        def check(result):
            """check the output"""
            self.assertEquals(out.getvalue(), expected)
        d.addCallback(check)
        return d

    def test_show_shared_empty(self):
        """test the output of --list-shared """
        out = StringIO()
        d = self.tool.list_shared()
        d.addCallback(lambda result: show_shared(result, out))
        def check(result):
            """check the output"""
            self.assertEquals('No shared\n', out.getvalue())
        d.addCallback(check)
        return d

    def test_show_shared(self):
        """test the output of --list-shared """
        path = os.path.join(self.root_dir, "ñoño")
        self.fs_manager.create(path, "")
        self.fs_manager.set_node_id(path, "node_id")
        # helper function, pylint: disable-msg=C0111
        def fake_create_share(node_id, user, name, access_level, marker):
            self.main.vm.handle_AQ_CREATE_SHARE_OK(share_id='share_id',
                                              marker=marker)
        self.main.action_q.create_share = fake_create_share
        self.main.vm.create_share(path, 'fake_user', 'shared_name', 'View')
        out = StringIO()
        expected = u"Shared list:\n  id=share_id name=shared_name " + \
                "accepted=False access_level=View to=fake_user " + \
                "path=%s\n" % path.decode('utf-8')
        d = self.tool.list_shared()
        d.addCallback(lambda result: show_shared(result, out))
        def check(result):
            """check the output"""
            self.assertEquals(out.getvalue(), expected)
        d.addCallback(check)
        return d

    def test_show_path_info_unicode(self):
        """test the output of --info with unicode paths """
        return self.generic_test_show_path_info_unicode('utf-8')

    def test_show_path_info_unicode_pipe(self):
        """test the output of --info with unicode paths going to e.g. a pipe"""
        return self.generic_test_show_path_info_unicode(None)

    def generic_test_show_path_info_unicode(self, encoding):
        """generic test for the output of --info with unicode paths """
        path = os.path.join(self.root_dir, "ñoño")
        mdid = self.fs_manager.create(path, "")
        self.fs_manager.set_node_id(path, "uuid1")
        mdobj = self.fs_manager.get_by_mdid(mdid)
        self.fs_manager.create_partial(mdobj.node_id, mdobj.share_id)
        fh = self.fs_manager.get_partial_for_writing(mdobj.node_id,
                                                     mdobj.share_id)
        fh.write("foobar")
        fh.close()
        self.fs_manager.commit_partial(mdobj.node_id, mdobj.share_id,
                                       "localhash")
        self.fs_manager.remove_partial("uuid1", "")
        d = self.tool.get_metadata(path)
        out = StringIO()
        out.encoding = encoding
        expected = """ File: %(path_info)s
  info_created: %(info_created)s
  info_is_partial: %(info_is_partial)s
  info_last_downloaded: %(info_last_downloaded)s
  info_last_partial_created: %(info_last_partial_created)s
  info_last_partial_removed: %(info_last_partial_removed)s
  info_node_id_assigned: %(info_node_id_assigned)s
  is_dir: %(is_dir)s
  local_hash: %(local_hash)s
  mdid: %(mdid)s
  node_id: %(node_id)s
  path: %(path)s
  server_hash: %(server_hash)s
  share_id: %(share_id)s
  stat: %(stat)s
"""
        # the callback, pylint: disable-msg=C0111
        def callback(result):
            if encoding is not None:
                result.update(dict(path_info=path.decode(encoding)))
            else:
                result.update(dict(path_info=path))
            for k, v in result.copy().items():
                result[k] = v.decode('utf-8')
            value = expected % result
            self.assertEquals(out.getvalue(), value)
        # helper callback, pylint: disable-msg=C0111
        def show(result):
            show_path_info(result, path, out)
            return result
        d.addCallback(show)
        d.addCallback(callback)
        return d

    def test_show_current_transfers_empty(self):
        """test the output of --current_transfers option """
        out = StringIO()
        d = self.tool.get_current_uploads()
        d.addCallback(lambda result: show_uploads(result, out))
        d.addCallback(lambda _: self.tool.get_current_downloads())
        d.addCallback(lambda result: show_downloads(result, out))
        expected = u'Current uploads: 0\nCurrent downloads: 0\n'
        def check(result):
            """check the output"""
            self.assertEquals(out.getvalue(), expected)
        d.addCallback(check)
        return d

    def test_show_current_transfers(self):
        """test the output of --current_transfers option with transfers in
        progress.
        """
        share_path = os.path.join(self.shares_dir, 'share')
        self.main.vm.add_share(Share(path=share_path, volume_id='share_id'))
        # create a download
        down_path = os.path.join(share_path, "down_path")
        self.fs_manager.create(down_path, "share_id")
        self.fs_manager.set_node_id(down_path, "down_node_id")
        self.action_q.downloading[('share_id', 'down_node_id')] = dict(
            deflated_size=10, size=100, n_bytes_read=1)
        # create a upload
        up_path = os.path.join(share_path, "up_path")
        self.fs_manager.create(up_path, "share_id")
        self.fs_manager.set_node_id(up_path, "node_id")
        self.action_q.uploading[('share_id', 'node_id')] = dict(
            deflated_size=100, n_bytes_written=10)
        out = StringIO()
        expected = u"Current uploads:\n  path: %(up_path)s\n    " + \
                "deflated size: 100\n    bytes written: 10\nCurrent " + \
                "downloads:\n  path: %(down_path)s\n    deflated size: " + \
                "10\n    bytes read: 1\n"
        expected = expected % dict(up_path=up_path, down_path=down_path)
        d = self.tool.get_current_uploads()
        d.addCallback(lambda result: show_uploads(result, out))
        d.addCallback(lambda _: self.tool.get_current_downloads())
        d.addCallback(lambda result: show_downloads(result, out))
        def check(result):
            """check the output"""
            self.assertEquals(out.getvalue(), expected)
        d.addCallback(check)
        return d

    def test_show_state(self):
        """test the output of --status """
        out = StringIO()
        expected = [
            "State: QUEUE_MANAGER",
            "connection: With User With Network",
            "description: processing queues",
            "is_connected: True",
            "is_error: False",
            "is_online: True",
            "queues: IDLE"
        ]
        d = self.tool.get_status()
        d.addCallback(lambda result: show_state(result, out))
        def check(result):
            """check the output"""
            info = [x.strip() for x in out.getvalue().split("\n") if x.strip()]
            self.assertEquals(info, expected)
        d.addCallback(check)
        return d

    def test_show_waiting_metadata(self):
        """Test the output of --waiting-metadata"""
        # inject the fake data
        cmd = FakeCommand("share_id", "node_id")
        cmd1 = FakeCommand("share_id_1", "node_id_1")
        self.action_q.meta_queue.waiting.extend([cmd, cmd1])
        out = StringIO()
        expected = " " + str(cmd) + "\n "+ str(cmd1) + "\n"
        d = self.tool.waiting_metadata()
        d.addCallback(lambda result: show_waiting_metadata(result, out))
        def check(result):
            """check the output"""
            self.assertEquals(out.getvalue(), expected)
        d.addCallback(check)
        return d

    def test_show_waiting_content(self):
        """Test the output of --waiting-content"""
        class FakeCommand2(FakeCommand):
            pass
        path = os.path.join(self.root_dir, "foo")
        self.fs_manager.create(path, "")
        self.fs_manager.set_node_id(path, "node_id")
        path_1 = os.path.join(self.root_dir, "bar")
        self.fs_manager.create(path_1, "")
        self.fs_manager.set_node_id(path_1, "node_id_1")
        path_2 = os.path.join(self.root_dir, "foo_1")
        self.fs_manager.create(path_2, "")
        self.fs_manager.set_node_id(path_2, "node_id_2")
        path_3 = os.path.join(self.root_dir, "bar_1")
        self.fs_manager.create(path_3, "")
        self.fs_manager.set_node_id(path_3, "node_id_3")
        # inject the fake data
        self.action_q.content_queue.waiting.extend([
                FakeCommand("", "node_id"),
                FakeCommand2("", "node_id_2"),
                FakeCommand("", "node_id_1"),
                FakeCommand2("", "node_id_3")])
        out = StringIO()
        expected = u"operation='FakeCommand' node_id='node_id' share_id='' " + \
                "path='%(path)s'\n" + \
                "operation='FakeCommand2' node_id='node_id_2' share_id='' " + \
                "path='%(path_2)s'\n" + \
                "operation='FakeCommand' node_id='node_id_1' share_id='' " + \
                "path='%(path_1)s'\n" + \
                "operation='FakeCommand2' node_id='node_id_3' share_id=''" + \
                " path='%(path_3)s'\n"
        expected = expected % dict(path=path, path_1=path_1,
                                   path_2=path_2, path_3=path_3)
        d = self.tool.waiting_content()
        d.addCallback(lambda result: show_waiting_content(result, out))
        def check(result):
            """check the output"""
            self.assertEquals(out.getvalue(), expected)
        d.addCallback(check)
        return d

    def test_show_folders_empty(self):
        """test the output of --list-folders """
        out = StringIO()
        d = self.tool.get_folders()
        d.addCallback(lambda result: show_folders(result, out))
        def check(result):
            """check the output"""
            self.assertEquals('No folders\n', out.getvalue())
        d.addCallback(check)
        return d

    @defer.inlineCallbacks
    def test_show_folders(self):
        """test the output of --list-folders """
        out = StringIO()
        path = 'ñoño'
        suggested_path = os.path.join("~", 'ñoño')

        udf = UDF("folder_id", "node_id", suggested_path, path,
                  subscribed=True)
        yield self.main.vm.add_udf(udf)
        expected = u"Folder list:\n  id=folder_id subscribed=True " + \
                   u"path=\xf1o\xf1o\n"
        d = self.tool.get_folders()
        d.addCallback(lambda result: show_folders(result, out))
        def check(result):
            """check the output"""
            self.assertEquals(out.getvalue(), expected)
        d.addCallback(check)
        yield d
