# -*- coding: utf-8 -*-
# Author: Manuel de la Pena <manuel@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Controllers with the logic of the UI."""

import os
import StringIO
import tempfile

# pylint: disable=F0401
try:
    from PIL import Image
except ImportError:
    import Image
# pylint: enable=F0401

from PyQt4.QtGui import QMessageBox, QWizard, QPixmap
from PyQt4.QtCore import QUrl
from twisted.internet.defer import inlineCallbacks, returnValue

from ubuntu_sso import NO_OP
from ubuntu_sso.logger import setup_logging
from ubuntu_sso.utils.ui import (
    CAPTCHA_LOAD_ERROR,
    CAPTCHA_SOLUTION_ENTRY,
    CAPTCHA_REQUIRED_ERROR,
    EMAIL1_ENTRY,
    EMAIL2_ENTRY,
    EMAIL_LABEL,
    EMAIL_MISMATCH,
    EMAIL_INVALID,
    ERROR,
    EXISTING_ACCOUNT_CHOICE_BUTTON,
    FORGOTTEN_PASSWORD_BUTTON,
    JOIN_HEADER_LABEL,
    LOGIN_PASSWORD_LABEL,
    NAME_ENTRY,
    PASSWORD1_ENTRY,
    PASSWORD2_ENTRY,
    PASSWORD_HELP,
    PASSWORD_MISMATCH,
    PASSWORD_TOO_WEAK,
    REQUEST_PASSWORD_TOKEN_LABEL,
    RESET_PASSWORD,
    RESET_CODE_ENTRY,
    REQUEST_PASSWORD_TOKEN_WRONG_EMAIL,
    REQUEST_PASSWORD_TOKEN_TECH_ERROR,
    SET_UP_ACCOUNT_CHOICE_BUTTON,
    SET_UP_ACCOUNT_BUTTON,
    SIGN_IN_BUTTON,
    SUCCESS,
    TC_BUTTON,
    TOS_LABEL,
    TRY_AGAIN_BUTTON,
    VERIFY_EMAIL_TITLE,
    VERIFY_EMAIL_CONTENT,
    YES_TO_TC,
    is_min_required_password,
    is_correct_email)


logger = setup_logging('ubuntu_sso.controllers')
FAKE_URL = '<a href="http://one.ubuntu.com">%s</a>'

# pylint: disable=W0511
# disabled warnings about TODO comments


# Based on the gtk implementation, but added error_message
# support, and rewritten.
def _build_general_error_message(errordict):
    """Concatenate __all__ and message from the errordict."""
    result = None
    msg1 = errordict.get('__all__')
    msg2 = errordict.get('message')
    if msg2 is None:
        # See the errordict in LP: 828417
        msg2 = errordict.get('error_message')
    if msg1 is not None and msg2 is not None:
        result = '\n'.join((msg1, msg2))
    elif msg1 is not None:
        result = msg1
    elif msg2 is not None:
        result = msg2
    else:
        # I give up.
        result = "Error: %r" % errordict
    return result


class BackendController(object):
    """Represent a controller that talks with the sso self.backend."""

    def __init__(self, title='', subtitle=''):
        """Create a new instance."""
        self.root = None
        self.view = None
        self.backend = None
        self._title = title
        self._subtitle = subtitle

    def __del__(self):
        """Clean the resources."""
        logger.info('Unregistering %s backend', self.backend)
        self.backend.unregister_to_signals()
        if self.root is not None:
            logger.info('Disconnecting from root.')
            self.root.disconnect()

    @inlineCallbacks
    def get_backend(self):
        """Return the backend used by the controller."""
        # get the back end from the root
        if self.root is None:
            from ubuntu_sso.main.windows import UbuntuSSOClient
            self.root = UbuntuSSOClient()
            self.root = yield self.root.connect()
        self.backend = self.root.sso_login
        yield self.backend.register_to_signals()
        returnValue(self.backend)

    #pylint: disable=C0103
    def pageInitialized(self):
        """Call to prepare the page just before it is shown."""
        pass
    #pylint: enable=C0103


class ChooseSignInController(BackendController):
    """Controlled to the ChooseSignIn view/widget."""

    def __init__(self, title='', subtitle=''):
        """Create a new instance to manage the view."""
        super(ChooseSignInController, self).__init__(title, subtitle)
        self.view = None

    # use an ugly name just so have a simlar api as found in PyQt
    #pylint: disable=C0103

    def setupUi(self, view):
        """Perform the required actions to set up the ui."""
        self.view = view
        self._set_up_translated_strings()
        self.view.header.set_title(self._title)
        self.view.header.set_subtitle(self._subtitle)
        self._connect_buttons()
    #pylint: enable=C0103

    def _set_up_translated_strings(self):
        """Set the correct strings for the UI."""
        logger.debug('ChooseSignInController._set_up_translated_strings')
        self.view.ui.existing_account_button.setText(
                                        EXISTING_ACCOUNT_CHOICE_BUTTON)
        self.view.ui.setup_account_button.setText(
                                        SET_UP_ACCOUNT_CHOICE_BUTTON)

    def _connect_buttons(self):
        """Connect the buttons to the actions to perform."""
        logger.debug('ChooseSignInController._connect_buttons')
        self.view.ui.existing_account_button.clicked.connect(
                                                    self._set_next_existing)
        self.view.ui.setup_account_button.clicked.connect(self._set_next_new)

    def _set_next_existing(self):
        """Set the next id and fire signal."""
        logger.debug('ChooseSignInController._set_next_existing')
        self.view.next = self.view.wizard().current_user_page_id
        self.view.wizard().next()

    def _set_next_new(self):
        """Set the next id and fire signal."""
        logger.debug('ChooseSignInController._set_next_new')
        self.view.next = self.view.wizard().setup_account_page_id
        self.view.wizard().next()


class CurrentUserController(BackendController):
    """Controller used in the view that is used to allow the signin."""

    def __init__(self, backend=None, title='', subtitle='', message_box=None):
        """Create a new instance."""
        super(CurrentUserController, self).__init__(title, subtitle)
        if message_box is None:
            message_box = QMessageBox
        self.message_box = message_box

    def _set_translated_strings(self):
        """Set the translated strings."""
        logger.debug('CurrentUserController._set_translated_strings')
        self.view.ui.email_label.setText(EMAIL_LABEL)
        self.view.ui.email_edit.setPlaceholderText(EMAIL1_ENTRY)
        self.view.ui.password_label.setText(LOGIN_PASSWORD_LABEL)
        self.view.ui.password_edit.setPlaceholderText(PASSWORD1_ENTRY)
        self.view.ui.forgot_password_label.setText(
                                    FAKE_URL % FORGOTTEN_PASSWORD_BUTTON)
        self.view.ui.sign_in_button.setText(SIGN_IN_BUTTON)

    def _connect_ui(self):
        """Connect the buttons to perform actions."""
        logger.debug('CurrentUserController._connect_buttons')
        self.view.ui.sign_in_button.clicked.connect(self.login)
        # lets add call backs to be execute for the calls we are interested
        self.backend.on_login_error_cb = lambda app, error:\
                                                    self.on_login_error(error)
        self.backend.on_logged_in_cb = self.on_logged_in
        self.view.ui.forgot_password_label.linkActivated.connect(
                                                    self.on_forgotten_password)
        self.view.ui.email_edit.textChanged.connect(self._validate)
        self.view.ui.password_edit.textChanged.connect(self._validate)

    def _validate(self):
        """Perform input validation."""
        valid = True
        if not is_correct_email(unicode(self.view.ui.email_edit.text())):
            valid = False
        elif not unicode(self.view.ui.password_edit.text()):
            valid = False
        self.view.ui.sign_in_button.setEnabled(valid)

    def login(self):
        """Perform the login using the self.backend."""
        logger.debug('CurrentUserController.login')
        # grab the data from the view and call the backend
        email = unicode(self.view.ui.email_edit.text())
        password = unicode(self.view.ui.password_edit.text())
        d = self.backend.login(self.view.wizard().app_name, email, password)
        d.addErrback(self.on_login_error)

    def on_login_error(self, error):
        """There was an error when login in."""
        # let the user know
        logger.error('Got error when login %s, error: %s',
                     self.view.wizard().app_name, error)
        self.message_box.critical(_build_general_error_message(error))

    def on_logged_in(self, app_name, result):
        """We managed to log in."""
        logger.info('Logged in for %s', app_name)
        email = unicode(self.view.ui.email_edit.text())
        self.view.wizard().loginSuccess.emit(app_name, email)
        logger.debug('Wizard.loginSuccess emitted.')

    def on_forgotten_password(self):
        """Show the user the forgotten password page."""
        logger.info('Forgotten password')
        self.view.next = self.view.wizard().forgotten_password_page_id
        self.view.wizard().next()

    # use an ugly name just so have a simlar api as found in PyQt
    #pylint: disable=C0103
    @inlineCallbacks
    def setupUi(self, view):
        """Setup the view."""
        self.view = view
        self.backend = yield self.get_backend()
        self.view.header.set_title(self._title)
        self.view.header.set_subtitle(self._subtitle)
        self._set_translated_strings()
        self._connect_ui()
    #pylint: enable=C0103


class SetUpAccountController(BackendController):
    """Conroller for the setup account view."""

    def __init__(self, message_box=None, title='', subtitle=''):
        """Create a new instance."""
        super(SetUpAccountController, self).__init__(title, subtitle)
        if message_box is None:
            message_box = QMessageBox
        self.message_box = message_box

    def _set_translated_strings(self):
        """Set the different gettext translated strings."""
        logger.debug('SetUpAccountController._set_translated_strings')
        # set the translated string
        self.view.ui.name_label.setText(NAME_ENTRY)
        self.view.ui.email_label.setText(EMAIL1_ENTRY)
        self.view.ui.confirm_email_label.setText(EMAIL2_ENTRY)
        self.view.ui.password_label.setText(PASSWORD1_ENTRY)
        self.view.ui.confirm_password_label.setText(PASSWORD2_ENTRY)
        self.view.ui.password_info_label.setText(PASSWORD_HELP)
        self.view.ui.captcha_solution_edit.setPlaceholderText(
                                                       CAPTCHA_SOLUTION_ENTRY)
        self.view.ui.terms_checkbox.setText(
                        YES_TO_TC % {'app_name': self.view.wizard().app_name})
        self.view.ui.terms_button.setText(TC_BUTTON)
        self.view.ui.set_up_button.setText(SET_UP_ACCOUNT_BUTTON)

    def _set_line_edits_validations(self):
        """Set the validations to be performed on the edits."""
        logger.debug('SetUpAccountController._set_line_edits_validations')
        self.view.set_line_edit_validation_rule(self.view.ui.email_edit,
                                                is_correct_email)
        # set the validation rule for the email confirmation
        self.view.set_line_edit_validation_rule(
                                            self.view.ui.confirm_email_edit,
                                            self.is_correct_email_confirmation)
        # connect the changed text of the password to trigger a changed text
        # in the confirm so that the validation is redone
        self.view.ui.email_edit.textChanged.connect(
                            self.view.ui.confirm_email_edit.textChanged.emit)
        self.view.set_line_edit_validation_rule(self.view.ui.password_edit,
                                                is_min_required_password)
        self.view.set_line_edit_validation_rule(
                                        self.view.ui.confirm_password_edit,
                                        self.is_correct_password_confirmation)
        # same as the above case, lets connect a signal to a signal
        self.view.ui.password_edit.textChanged.connect(
                        self.view.ui.confirm_password_edit.textChanged.emit)

    def _connect_ui_elements(self):
        """Set the connection of signals."""
        logger.debug('SetUpAccountController._connect_ui_elements')
        self.view.ui.terms_button.clicked.connect(self.set_next_tos)
        self.view.ui.set_up_button.clicked.connect(self.set_next_validation)
        self.view.ui.refresh_label.linkActivated.connect(lambda url: \
                                          self._refresh_captcha())
        # set the callbacks for the captcha generation
        self.backend.on_captcha_generated_cb = self.on_captcha_generated
        self.backend.on_captcha_generation_error_cb = lambda app, error: \
                            self.on_captcha_generation_error(error)
        self.backend.on_user_registered_cb = self.on_user_registered
        self.backend.on_user_registration_error_cb = \
                                            self.on_user_registration_error
        # We need to check if we enable the button on many signals
        self.view.ui.name_edit.textEdited.connect(self._enable_setup_button)
        self.view.ui.email_edit.textEdited.connect(self._enable_setup_button)
        self.view.ui.confirm_email_edit.textEdited.connect(
            self._enable_setup_button)
        self.view.ui.password_edit.textEdited.connect(
            self._enable_setup_button)
        self.view.ui.confirm_password_edit.textEdited.connect(
            self._enable_setup_button)
        self.view.ui.captcha_solution_edit.textEdited.connect(
            self._enable_setup_button)
        self.view.ui.terms_checkbox.stateChanged.connect(
            self._enable_setup_button)

    def _enable_setup_button(self):
        """Only enable the setup button if the form is valid."""
        email = unicode(self.view.ui.email_edit.text())
        confirm_email = unicode(self.view.ui.confirm_email_edit.text())
        password = unicode(self.view.ui.password_edit.text())
        confirm_password = unicode(self.view.ui.confirm_password_edit.text())
        captcha_solution = unicode(self.view.ui.captcha_solution_edit.text())

        enabled = self.view.ui.terms_checkbox.isChecked() and \
          captcha_solution and is_min_required_password(password) and \
          password == confirm_password and is_correct_email(email) and \
          email == confirm_email

        self.view.ui.set_up_button.setEnabled(enabled)
        self.view.ui.set_up_button.setProperty("DisabledState",
            not self.view.ui.set_up_button.isEnabled())
        self.view.ui.set_up_button.style().unpolish(self.view.ui.set_up_button)
        self.view.ui.set_up_button.style().polish(self.view.ui.set_up_button)

    def _refresh_captcha(self):
        """Refresh the captcha image shown in the ui."""
        logger.debug('SetUpAccountController._refresh_captcha')
        # lets clean behind us, do we have the old file arround?
        old_file = None
        if self.view.captcha_file and os.path.exists(self.view.captcha_file):
            old_file = self.view.captcha_file
        fd = tempfile.TemporaryFile(mode='r')
        file_name = fd.name
        self.view.captcha_file = file_name
        d = self.backend.generate_captcha(self.view.wizard().app_name,
                                          file_name)
        if old_file:
            d.addCallback(lambda x: os.unlink(old_file))
        d.addErrback(self.on_captcha_generation_error)

    def _set_titles(self):
        """Set the diff titles of the view."""
        logger.debug('SetUpAccountController._set_titles')
        self.view.header.set_title(
            JOIN_HEADER_LABEL % {'app_name': self.view.wizard().app_name})
        self.view.header.set_subtitle(self.view.wizard().help_text)

    def _register_fields(self):
        """Register the diff fields of the Ui."""
        self.view.registerField('email_address', self.view.ui.email_edit)
        self.view.registerField('password', self.view.ui.password_edit)

    def on_captcha_generated(self, app_name, result):
        """A new image was generated."""
        logger.debug('SetUpAccountController.on_captcha_generated')
        self.view.captcha_id = result
        # HACK: First, let me apologize before hand, you can mention my mother
        # if needed I would do the same (mandel)
        # In an ideal world we could use the Qt plug-in for the images so that
        # we could load jpgs etc.. but this won't work when the app has been
        # brozen win py2exe using bundle_files=1
        # The main issue is that Qt will complain about the thread not being
        # the correct one when performing a moveToThread operation which is
        # done either by a setParent or something within the qtreactor, PIL
        # in this case does solve the issue. Sorry :(
        pil_image = Image.open(self.view.captcha_file)
        string_io = StringIO.StringIO()
        pil_image.save(string_io, format='png')
        pixmap_image = QPixmap()
        pixmap_image.loadFromData(string_io.getvalue())
        self.view.captcha_image = pixmap_image

    def on_captcha_generation_error(self, error):
        """An error ocurred."""
        logger.debug('SetUpAccountController.on_captcha_generation_error')
        self.message_box.critical(CAPTCHA_LOAD_ERROR)

    def on_user_registration_error(self, app_name, error):
        """Let the user know we could not register."""
        logger.debug('SetUpAccountController.on_user_registration_error')
        # errors are returned as a dict with the data we want to show.
        self._refresh_captcha()
        self.message_box.critical(_build_general_error_message(error))

    def on_user_registered(self, app_name, result):
        """Execute when the user did register."""
        logger.debug('SetUpAccountController.on_user_registered')
        self.view.next = self.view.wizard().email_verification_page_id
        self.view.wizard().next()

    def set_next_tos(self):
        """Set the tos page as the next one."""
        logger.debug('SetUpAccountController.set_next_tos')
        self.view.next = self.view.wizard().tos_page_id
        self.view.wizard().next()

    def validate_form(self):
        """Validate the info of the form and return an error."""
        logger.debug('SetUpAccountController.validate_form')
        email = unicode(self.view.ui.email_edit.text())
        confirm_email = unicode(self.view.ui.confirm_email_edit.text())
        password = unicode(self.view.ui.password_edit.text())
        confirm_password = unicode(self.view.ui.confirm_password_edit.text())
        captcha_solution = unicode(self.view.ui.captcha_solution_edit.text())
        if not is_correct_email(email):
            self.message_box.critical(EMAIL_INVALID)
        if email != confirm_email:
            self.message_box.critical(EMAIL_MISMATCH)
            return False
        if not is_min_required_password(password):
            self.message_box.critical(PASSWORD_TOO_WEAK)
            return False
        if password != confirm_password:
            self.message_box.critical(PASSWORD_MISMATCH)
            return False
        if not captcha_solution:
            self.message_box.critical(CAPTCHA_REQUIRED_ERROR)
            return False
        return True

    def set_next_validation(self):
        """Set the validation as the next page."""
        logger.debug('SetUpAccountController.set_next_validation')
        email = unicode(self.view.ui.email_edit.text())
        password = unicode(self.view.ui.password_edit.text())
        name = unicode(self.view.ui.name_edit.text())
        captcha_id = self.view.captcha_id
        captcha_solution = unicode(self.view.ui.captcha_solution_edit.text())
        # validate the current info of the form, try to perform the action
        # to register the user, and then move foward
        if self.validate_form():
            self.backend.register_user(self.view.wizard().app_name, email,
                                       password, name, captcha_id,
                                       captcha_solution)
        # Update validation page's title, which contains the email
        p_id = self.view.wizard().email_verification_page_id
        self.view.wizard().page(p_id).controller.set_titles()

    def is_correct_email(self, email_address):
        """Return if the email is correct."""
        logger.debug('SetUpAccountController.is_correct_email')
        return '@' in email_address

    def is_correct_email_confirmation(self, email_address):
        """Return that the email is the same."""
        logger.debug('SetUpAccountController.is_correct_email_confirmation')
        return unicode(self.view.ui.email_edit.text()) == email_address

    def is_correct_password_confirmation(self, password):
        """Return that the passwords are correct."""
        logger.debug('SetUpAccountController.is_correct_password_confirmation')
        return unicode(self.view.ui.password_edit.text()) == password

    #pylint: disable=C0103
    @inlineCallbacks
    def setupUi(self, view):
        """Setup the view."""
        self.view = view
        # request the backend to be used with the ui
        self.backend = yield self.get_backend()
        self._connect_ui_elements()
        self._refresh_captcha()
        self._set_titles()
        self.view.header.set_title(self._title)
        self.view.header.set_subtitle(self._subtitle)
        self._set_translated_strings()
        self._set_line_edits_validations()
        self._register_fields()
    #pylint: enable=C0103


class TosController(BackendController):
    """Controller used for the tos page."""

    def __init__(self, title='', subtitle='', tos_url=''):
        """Create a new instance."""
        super(TosController, self).__init__(title, subtitle)
        self.view = None
        self._tos_url = tos_url

    #pylint: disable=C0103
    def setupUi(self, view):
        """Set up the ui."""
        self.view = view
        self.view.header.set_title(self._title)
        self.view.header.set_subtitle(self._subtitle)
        # load the tos page
        self.view.ui.terms_webkit.load(QUrl(self._tos_url))
        self.view.ui.tos_link_label.setText(
            TOS_LABEL %
            {'url': self._tos_url})
    #pylint: enable=C0103


class EmailVerificationController(BackendController):
    """Controller used for the verification page."""

    def __init__(self, message_box=None, title='', subtitle=''):
        """Create a new instance."""
        super(EmailVerificationController, self).__init__(title, subtitle)
        if message_box is None:
            message_box = QMessageBox
        self.message_box = message_box

    def _set_translated_strings(self):
        """Set the trnaslated strings."""
        logger.debug('EmailVerificationController._set_translated_strings')
        self.view.ui.verification_code_edit.setPlaceholderText(
                                                        VERIFY_EMAIL_TITLE)

    def _connect_ui_elements(self):
        """Set the connection of signals."""
        logger.debug('EmailVerificationController._connect_ui_elements')
        self.view.next_button.clicked.connect(self.validate_email)
        self.backend.on_email_validated_cb = lambda app, result: \
                            self.on_email_validated(app)
        self.backend.on_email_validation_error_cb = \
                                                self.on_email_validation_error

    def _set_titles(self):
        """Set the different titles."""
        logger.debug('EmailVerificationController._set_titles')
        self.view.header.set_title(VERIFY_EMAIL_TITLE)
        self.view.header.set_subtitle(VERIFY_EMAIL_CONTENT % {
            "app_name": self.view.wizard().app_name,
            "email": self.view.wizard().field("email_address").toString(),
        })

    def set_titles(self):
        """This class needs to have a public set_titles.

        Since the subtitle contains data that is only known after SetupAccount
        and _set_titles is only called on initialization.
        """
        self._set_titles()

    #pylint: disable=C0103
    @inlineCallbacks
    def setupUi(self, view):
        """Setup the view."""
        self.view = view
        self.backend = yield self.get_backend()
        self._set_titles()
        self._set_translated_strings()
        self._connect_ui_elements()
    #pylint: enable=C0103

    def validate_email(self):
        """Call the next action."""
        logger.debug('EmailVerificationController.validate_email')
        email = unicode(self.view.wizard().field('email_address').toString())
        password = unicode(self.view.wizard().field('password').toString())
        code = unicode(self.view.ui.verification_code_edit.text())
        self.backend.validate_email(self.view.wizard().app_name, email,
                                    password, code)

    def on_email_validated(self, app_name):
        """Signal thrown after the email is validated."""
        logger.info('EmailVerificationController.on_email_validated')
        email = self.view.wizard().field('email_address').toString()
        self.view.wizard().registrationSuccess.emit(app_name, email)

    def on_email_validation_error(self, app_name, error):
        """Signal thrown when there's a problem validating the email."""
        msg = error.get('email_token')
        if msg is None:
            msg = _build_general_error_message(error)
        self.message_box.critical(msg)

    #pylint: disable=C0103
    def pageInitialized(self):
        """Called to prepare the page just before it is shown."""
        self.view.next_button.setDefault(True)
        self.view.next_button.style().unpolish(
            self.view.next_button)
        self.view.next_button.style().polish(
            self.view.next_button)
    #pylint: enable=C0103


class ErrorController(BackendController):
    """Controller used for the error page."""

    def __init__(self, title='', subtitle=''):
        """Create a new instance."""
        super(ErrorController, self).__init__(title, subtitle)
        self.view = None

    #pylint: disable=C0103
    def setupUi(self, view):
        """Setup the view."""
        self.view = view
        self.view.next = -1
        self.view.ui.error_message_label.setText(ERROR)
        self.view.header.set_title(self._title)
        self.view.header.set_subtitle(self._subtitle)
    #pylint: enable=C0103


class ForgottenPasswordController(BackendController):
    """Controller used to deal with the forgotten pwd page."""

    def __init__(self, message_box=None, title='', subtitle=''):
        """Create a new instance."""
        super(ForgottenPasswordController, self).__init__()
        if message_box is None:
            message_box = QMessageBox
        self.message_box = message_box
        super(ForgottenPasswordController, self).__init__(title, subtitle)

    def _register_fields(self):
        """Register the fields of the wizard page."""
        self.view.registerField('email_address',
                                self.view.email_address_line_edit)

    def _set_translated_strings(self):
        """Set the translated strings in the view."""
        self.view.forgotted_password_intro_label.setText(
                                    REQUEST_PASSWORD_TOKEN_LABEL % {'app_name':
                                    self.view.wizard().app_name})
        self.view.email_address_label.setText(EMAIL_LABEL)
        self.view.send_button.setText(RESET_PASSWORD)
        self.view.try_again_button.setText(TRY_AGAIN_BUTTON)

    def _set_enhanced_line_edit(self):
        """Set the extra logic to the line edits."""
        self.view.set_line_edit_validation_rule(
                                           self.view.email_address_line_edit,
                                           is_correct_email)

    def _connect_ui(self):
        """Connect the diff signals from the Ui."""
        self.view.email_address_line_edit.textChanged.connect(self._validate)
        self.view.send_button.clicked.connect(
                            lambda: self.backend.request_password_reset_token(
                                                self.view.wizard().app_name,
                                                self.view.email_address))
        self.view.try_again_button.clicked.connect(self.on_try_again)
        # set the backend callbacks to be used
        self.backend.on_password_reset_token_sent_cb = lambda app, result:\
                                    self.on_password_reset_token_sent()
        self.backend.on_password_reset_error_cb = self.on_password_reset_error

    def _validate(self):
        """Validate that we have an email."""
        email = unicode(self.view.email_address_line_edit.text())
        self.view.send_button.setEnabled(is_correct_email(email))

    def on_try_again(self):
        """Set back the widget to the initial state."""
        self.view.error_label.setVisible(False)
        self.view.try_again_widget.setVisible(False)
        self.view.email_widget.setVisible(True)

    def on_password_reset_token_sent(self):
        """Action taken when we managed to get the password reset done."""
        # ignore the result and move to the reset page
        self.view.next = self.view.wizard().reset_password_page_id
        self.view.wizard().next()

    def on_password_reset_error(self, app_name, error):
        """Action taken when there was an error requesting the reset."""
        if error['errtype'] == 'ResetPasswordTokenError':
            # the account provided is wrong, lets tell the user to try
            # again
            self.view.error_label.setText(REQUEST_PASSWORD_TOKEN_WRONG_EMAIL)
            self.view.error_label.setVisible(True)
        else:
            # ouch, I dont know what went wrong, tell the user to try later
            self.view.email_widget.setVisible(False)
            self.view.forgotted_password_intro_label.setVisible(False)
            self.view.try_again_wisget.setVisible(True)
            # set the error message
            self.view.error_label.setText(REQUEST_PASSWORD_TOKEN_TECH_ERROR)

    #pylint: disable=C0103
    @inlineCallbacks
    def setupUi(self, view):
        """Setup the view."""
        self.view = view
        self.backend = yield self.get_backend()
        # hide the error label
        self.view.error_label.setVisible(False)
        self.view.try_again_widget.setVisible(False)
        self._set_translated_strings()
        self._connect_ui()
        self._set_enhanced_line_edit()
        self._register_fields()
        self.view.header.set_title(self._title)
        self.view.header.set_subtitle(self._subtitle)
    #pylint: enable=C0103


class ResetPasswordController(BackendController):
    """Controller used to deal with reseintg the password."""

    def __init__(self, title='', subtitle='', message_box=None):
        """Create a new instance."""
        if message_box is None:
            message_box = QMessageBox
        self.message_box = message_box
        super(ResetPasswordController, self).__init__(title, subtitle)

    def _set_translated_strings(self):
        """Translate the diff strings used in the app."""
        self.view.ui.reset_code_line_edit.setPlaceholderText(RESET_CODE_ENTRY)
        self.view.ui.password_line_edit.setPlaceholderText(PASSWORD1_ENTRY)
        self.view.ui.confirm_password_line_edit.setPlaceholderText(
                                                        PASSWORD2_ENTRY)
        self.view.ui.reset_password_button.setText(RESET_PASSWORD)
        self.view.setSubTitle(PASSWORD_HELP)

    def _connect_ui(self):
        """Connect the different ui signals."""
        self.view.ui.reset_password_button.clicked.connect(
                                                    self.set_new_password)
        self.backend.on_password_changed_cb = self.on_password_changed
        self.backend.on_password_change_error_cb = \
                                                self.on_password_change_error
        self.view.ui.reset_code_line_edit.textChanged.connect(self._validate)
        self.view.ui.password_line_edit.textChanged.connect(self._validate)
        self.view.ui.confirm_password_line_edit.textChanged.connect(
            self._validate)

    def _validate(self):
        """Enable the submit button if data is valid."""
        enabled = True
        code = unicode(self.view.ui.reset_code_line_edit.text())
        password = unicode(self.view.ui.password_line_edit.text())
        confirm_password = unicode(
            self.view.ui.confirm_password_line_edit.text())
        if not is_min_required_password(password):
            enabled = False
        elif not self.is_correct_password_confirmation(confirm_password):
            enabled = False
        elif not code:
            enabled = False
        self.view.ui.reset_password_button.setEnabled(enabled)

    def _add_line_edits_validations(self):
        """Add the validations to be use by the line edits."""
        self.view.set_line_edit_validation_rule(
                                           self.view.ui.password_line_edit,
                                           is_min_required_password)
        self.view.set_line_edit_validation_rule(
                                    self.view.ui.confirm_password_line_edit,
                                    self.is_correct_password_confirmation)
        # same as the above case, lets connect a signal to a signal
        self.view.ui.password_line_edit.textChanged.connect(
                     self.view.ui.confirm_password_line_edit.textChanged.emit)

    def on_password_changed(self, app_name, result):
        """Let user know that the password was changed."""

    def on_password_change_error(self, app_name, error):
        """Let the user know that there was an error."""
        logger.error('Got error changing password for %s, error: %s',
                     self.view.wizard().app_name, error)
        self.message_box.critical(_build_general_error_message(error))

    def set_new_password(self):
        """Request a new password to be set."""
        app_name = self.view.wizard().app_name
        email = unicode(self.view.wizard().field('email_address').toString())
        code = unicode(self.view.ui.reset_code_line_edit.text())
        password = unicode(self.view.ui.password_line_edit.text())
        logger.info('Settig new password for %s and email %s with code %s',
                    app_name, email, code)
        self.backend.set_new_password(app_name, email, code, password)

    def is_correct_password_confirmation(self, password):
        """Return if the password is correct."""
        return unicode(self.view.ui.password_line_edit.text()) == password

    #pylint: disable=C0103
    @inlineCallbacks
    def setupUi(self, view):
        """Setup the view."""
        self.view = view
        self.backend = yield self.get_backend()
        self._set_translated_strings()
        self._connect_ui()
        self._add_line_edits_validations()
        self.view.header.set_title(self._title)
        self.view.header.set_subtitle(self._subtitle)
    #pylint: enable=C0103


class SuccessController(BackendController):
    """Controller used for the success page."""

    def __init__(self, title='', subtitle=''):
        """Create a new instance."""
        super(SuccessController, self).__init__(title, subtitle)
        self.view = None

    #pylint: disable=C0103
    def setupUi(self, view):
        """Setup the view."""
        self.view = view
        self.view.next = -1
        self.view.ui.success_message_label.setText(SUCCESS)
        self.view.header.set_title(self._title)
        self.view.header.set_subtitle(self._subtitle)
    #pylint: enable=C0103


class UbuntuSSOWizardController(object):
    """Controller used for the overall wizard."""

    def __init__(self, login_success_callback=NO_OP,
                 registration_success_callback=NO_OP,
                 user_cancellation_callback=NO_OP):
        """Create a new instance."""
        self.view = None
        self.login_success_callback = login_success_callback
        self.registration_success_callback = registration_success_callback
        self.user_cancellation_callback = user_cancellation_callback

    def on_user_cancelation(self):
        """Process the cancel action."""
        logger.debug('UbuntuSSOWizardController.on_user_cancelation')
        self.user_cancellation_callback(self.view.app_name)
        self.view.close()

    @inlineCallbacks
    def on_login_success(self, app_name, email):
        """Process the success of a login."""
        logger.debug('UbuntuSSOWizardController.on_login_success')
        result = yield self.login_success_callback(
            unicode(app_name), unicode(email))
        logger.debug('Result from callback is %s', result)
        if result == 0:
            logger.info('Success in calling the given success_callback')
            self.show_success_message()
        else:
            logger.info('Error in calling the given success_callback')
            self.show_error_message()

    @inlineCallbacks
    def on_registration_success(self, app_name, email):
        """Process the success of a registration."""
        logger.debug('UbuntuSSOWizardController.on_registration_success')
        result = yield self.registration_success_callback(unicode(app_name),
                                                          unicode(email))
        # TODO: what to do?
        logger.debug('Result from callback is %s', result)
        if result == 0:
            logger.info('Success in calling the given registration_callback')
            self.show_success_message()
        else:
            logger.info('Success in calling the given registration_callback')
            self.show_error_message()

    def show_success_message(self):
        """Show the success message in the view."""
        logger.info('Showing success message.')
        # get the id of the success page, set it as the next id of the
        # current page and let the wizard move to the next step
        self.view.currentPage().next = self.view.success_page_id
        self.view.next()
        # show the finish button but with a close message
        buttons_layout = []
        buttons_layout.append(QWizard.Stretch)
        buttons_layout.append(QWizard.FinishButton)
        self.view.setButtonLayout(buttons_layout)

    def show_error_message(self):
        """Show the error page in the view."""
        logger.info('Showing error message.')
        # similar to the success page but using the error id
        self.view.currentPage().next = self.view.error_page_id
        self.view.next()
        # show the finish button but with a close message
        buttons_layout = []
        buttons_layout.append(QWizard.Stretch)
        buttons_layout.append(QWizard.FinishButton)
        self.view.setButtonLayout(buttons_layout)

    #pylint: disable=C0103
    def setupUi(self, view):
        """Setup the view."""
        self.view = view
        self.view.setWizardStyle(QWizard.ModernStyle)
        self.view.button(QWizard.CancelButton).clicked.connect(
                                                    self.on_user_cancelation)
        self.view.loginSuccess.connect(self.on_login_success)
        self.view.registrationSuccess.connect(self.on_registration_success)
    #pylint: enable=C0103
