/*
 * Fast index for tag data
 *
 * Copyright (C) 2006--2011  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <tests/test-utils.h>
#include <tagcoll/coll/intdiskindex.h>
#include <string>
#include <map>

using namespace std;

namespace tagcoll {
namespace coll {

class BigMap
{
	int seq;
	map<int, string> tostring;
	map<string, int> toint;
public:
	BigMap() : seq(0) {}

	void assign(int a, const string& b)
	{
		tostring[a] = b;
		toint[b] = a;
	}

	int get(const string& val)
	{
		std::map<string, int>::const_iterator i = toint.find(val);
		if (i == toint.end())
		{
			assign(seq, val);
			return seq++;
		} else
			return i->second;
	}
	
	string get(int val)
	{
		std::map<int, string>::const_iterator i = tostring.find(val);
		ensure(i != tostring.end());
		return i->second;
	}

	template<typename ITEMS>
	std::set<std::string> fromInt(const ITEMS& items)
	{
		std::set<std::string> res;
		for (typename ITEMS::const_iterator i = items.begin(); i != items.end(); ++i)
			res.insert(get(*i));
		return res;
	}
	template<typename ITEMS>
	std::set<int> toInt(const ITEMS& items)
	{
		std::set<int> res;
		for (typename ITEMS::const_iterator i = items.begin(); i != items.end(); ++i)
			res.insert(get(*i));
		return res;
	}
};


class StringDiskIndex;

template<>
struct coll_traits< StringDiskIndex >
{
	typedef std::string item_type;
	typedef std::string tag_type;
	typedef std::set<std::string> tagset_type;
	typedef std::set<std::string> itemset_type;
};

class StringDiskIndex : public coll::ReadonlyCollection< StringDiskIndex >
{
protected:
	IntDiskIndex index;
	mutable BigMap& items;
	mutable BigMap& tags;

	std::string itemconv(const int& item) const
	{
		return this->items.get(item);
	}
	std::set<std::string> itemconv(const std::set<int>& items) const
	{
		return this->items.fromInt(items);
	}
	int itemconv(const std::string& item) const
	{
		return this->items.get(item);
	}
	std::set<int> itemconv(const std::set<std::string>& items) const
	{
		return this->items.toInt(items);
	}
	std::string tagconv(const int& tag) const
	{
		return this->tags.get(tag);
	}
	std::set<std::string> tagconv(const std::set<int>& tags) const
	{
		return this->tags.fromInt(tags);
	}
	int tagconv(const std::string& tag) const
	{
		return this->tags.get(tag);
	}
	std::set<int> tagconv(const std::set<std::string>& tags) const
	{
		return this->tags.toInt(tags);
	}


public:
	class const_iterator
	{
		const StringDiskIndex& index;
		int idx;
		mutable std::pair< std::string, std::set<std::string> >* cached;

	public:
		// Builds an iterator
		const_iterator(const StringDiskIndex& index, int idx)
			: index(index), idx(idx), cached(0) {}
		// Builds the end iterator
		const_iterator(const StringDiskIndex& index)
			: index(index), idx(index.index.itemCount()), cached(0) {}
		~const_iterator() { if (cached) delete cached; }

		std::pair< std::string, std::set<std::string> > operator*() const
		{
			return std::make_pair(index.itemconv(idx), index.tagconv(index.index.getTagsOfItem(idx)));
		}
		std::pair< std::string, std::set<std::string> >* operator->() const
		{
			if (!cached)
				cached = new std::pair< std::string, std::set<std::string> >(operator*());
			return cached;
		}

		const_iterator operator++()
		{
			++idx;
			if (cached) { delete cached; cached = 0; }
			return *this;
		}
		bool operator!=(const const_iterator& iter)
		{
			return idx != iter.idx;
		}
	};
	const_iterator begin() const { return const_iterator(*this, 0); }
	const_iterator end() const { return const_iterator(*this); }

	StringDiskIndex(const diskindex::MasterMMap& master, int pkgindex, int tagindex, BigMap& items, BigMap& tags)
		: index(master, pkgindex, tagindex), items(items), tags(tags) {}

	std::set<std::string> getItemsHavingTag(const std::string& tag) const
	{
		return itemconv(index.getItemsHavingTag(tagconv(tag)));
	}
	std::set<std::string> getItemsHavingTags(const std::set<std::string>& tags) const
	{
		return itemconv(index.getItemsHavingTags(tagconv(tags)));
	}
	std::set<std::string> getTagsOfItem(const std::string& item) const
	{
		return tagconv(index.getTagsOfItem(itemconv(item)));
	}
	std::set<std::string> getTagsOfItems(const std::set<std::string>& items) const
	{
		return tagconv(index.getTagsOfItems(itemconv(items)));
	}

    bool hasTag(const std::string& tag) const
	{
		return index.hasTag(tagconv(tag));
	}

	std::set<std::string> getTaggedItems() const
	{
		return itemconv(index.getTaggedItems());
	}

	std::set<std::string> getAllTags() const
	{
		return tagconv(index.getAllTags());
	}

	unsigned int itemCount() const { return index.itemCount(); }
	unsigned int tagCount() const { return index.tagCount(); }

	unsigned int getCardinality(const std::string& tag) const
	{
		return index.getCardinality(tagconv(tag));
	}

	std::set<std::string> getCompanionTags(const std::set<std::string>& tags) const
	{
		return tagconv(index.getCompanionTags(tagconv(tags)));
	}
};

template<typename OUT>
class BigMapSerializer : public wibble::mixin::OutputIterator< BigMapSerializer<OUT> >
{
	BigMap& itemMap;
	BigMap& tagMap;
	OUT out;

public:
	BigMapSerializer(BigMap& itemMap, BigMap& tagMap, const OUT& out) :
		itemMap(itemMap), tagMap(tagMap), out(out) {}

	template<typename ITEMS, typename TAGS>
	BigMapSerializer<OUT>& operator=(const std::pair<ITEMS, TAGS>& data)
	{
		if (!data.second.empty())
		{
			*out = make_pair(itemMap.toInt(data.first), tagMap.toInt(data.second));
			++out;
		}
		return *this;
	}
};

template<typename OUT>
BigMapSerializer<OUT> bigMapSerializer(BigMap& itemMap, BigMap& tagMap, const OUT& out)
{
	return BigMapSerializer<OUT>(itemMap, tagMap, out);
}


}
}

namespace tut {

using namespace std;
using namespace tagcoll;
using namespace tagcoll::tests;
using namespace tagcoll::coll;

static const char* fname = "tagcoll_intdiskindex.tmp";

struct tagcoll_coll_intdiskindex_shar {
	BigMap itemMap;
	BigMap tagMap;

	tagcoll_coll_intdiskindex_shar()
	{
		diskindex::MasterMMapIndexer master(fname);

		IntDiskIndexer indexer;
		output_test_collection(bigMapSerializer(itemMap, tagMap, inserter(indexer)));
		
		master.append(indexer.pkgIndexer());
		master.append(indexer.tagIndexer());
		master.commit();
	}
	~tagcoll_coll_intdiskindex_shar()
	{
		unlink(fname);
	}
};
TESTGRP(tagcoll_coll_intdiskindex);

#include <iostream>

#if 0
static void outts(const std::set<string>& s)
{
	for (std::set<string>::const_iterator i = s.begin(); i != s.end(); i++)
		if (i == s.begin())
			cerr << *i;
		else
			cerr << ", " << *i;
}
#endif

template<> template<>
void to::test<1>()
{
	diskindex::MasterMMap master(fname);

	StringDiskIndex idx(master, 0, 1, itemMap, tagMap);

#if 0
	cerr << "Items: ";
	std::set<string> s = idx.getTaggedItems();
	outts(s);
	cerr << endl;
	for (std::set<string>::const_iterator i = s.begin(); i != s.end(); i++)
	{
		cerr << "  " << *i << ": ";
		outts(idx.getTags(*i));
		cerr << endl;
	}


	cerr << "Tags: ";
	s = idx.getAllTags();
	outts(s);
	cerr << endl;
	for (std::set<string>::const_iterator i = s.begin(); i != s.end(); i++)
	{
		cerr << "  " << *i << ": ";
		outts(idx.getItems(*i));
		cerr << endl;
	}
#endif
	
	test_readonly_collection(idx);
}

// If we don't open the mmapped index, the collection should still work,
// although empty
template<> template<>
void to::test<2>()
{
	IntDiskIndex coll;
	set<int> res;

	ensure(coll.begin() == coll.end());

	res = coll.getTaggedItems();
	ensure_equals(res.size(), 0u);

	res = coll.getAllTags();
	ensure_equals(res.size(), 0u);

	vector<int> res1 = coll.getAllTagsAsVector();
	ensure_equals(res1.size(), 0u);

	ensure_equals(coll.itemCount(), 0u);
	ensure_equals(coll.tagCount(), 0u);
}

}

#include <tagcoll/coll/simple.tcc>
#include <tagcoll/tests/test-utils.tcc>

// vim:set ts=4 sw=4:
