/*  Screem:  skel-plugin.c
 *
 *  Copyright (C) 2004 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 */
#include <config.h>

#include <glib/gi18n.h>
#include <gtk/gtk.h>

#include <string.h>

#include "screem-skel-plugin.h"

/* setup your plugin here */

/* plugin name should only include a-zA-Z */
static const gchar *plugin_name = "ScreemFormWizard";

static const gchar *authors[] = {
	"David A Knight <david@screem.org>",
	NULL
};
static const gchar *displayed_name = N_( "Screem Form Wizard" );
static const gchar *description = N_( "A wizard for inserting a form" );
static const gchar *version = "2.0.0";

/* add any per instance data items here */
struct ScreemSkelPluginPrivate {
	GtkWidget *dialog;

	GtkWidget *method;
	GtkWidget *action;
	GtkWidget *enctype;
};

static void form_wizard_display( GtkAction *action, gpointer user_data );
static void form_wizard_clicked( GtkWidget *widget, 
		gint button, gpointer data );
static void create_dialog( ScreemPlugin *plugin );

/**
 * setup:
 * 
 * this function will be called once for each window,
 * you should add any actions / ui here, eg.
 * 
 * screem_plugin_add_action( plugin, name, label, tip, stock_id,
 * 				callback, error );
 * screem_plugin_add_menu( plugin, path, action, error );
 * screem_plugin_add_toolbar( plugin, path, action, error );
 *
 *
 * to insert text into the current page being edited your callbacks
 * should make use of
 * screem_plugin_get_cursor_position( plugin )
 * screem_plugin_set_cursor_position( plugin, pos )
 * screem_plugin_insert( plugin, pos, text, length, indent )
 * 
 **/
static gboolean setup( ScreemPlugin *plugin )
{
	GError *error;
	gboolean ret;

	error = NULL;
	
	ret = screem_plugin_add_interface( plugin, "FormWizard",
				_( "Form Wizard" ),
				_( "Create a Form" ),
				GTK_STOCK_JUSTIFY_FILL,
				G_CALLBACK( form_wizard_display ),
				&error  );
	if( ! ret ) {
		g_print( "Add interface error: %s\n", error->message );
		g_error_free( error );
	} 	
	return ret;

}

/**
 * cleanup:
 *
 * this function will be called once for each window when
 * it is closed, you should cleanup any data items you
 * have in ScreemSkelPluginPrivate here
 **/
static void cleanup( ScreemSkelPluginPrivate *priv )
{
	gtk_widget_destroy( priv->dialog );
}

static void form_wizard_display( GtkAction *action, gpointer user_data )
{
	ScreemPlugin *plugin;
	ScreemSkelPluginPrivate *priv;
	ScreemPage *page;

	plugin = SCREEM_PLUGIN( user_data );
	priv = SCREEM_SKEL_PLUGIN( plugin )->priv;
	
	page = screem_plugin_get_current_document( plugin );

	if( page ) {
		create_dialog( plugin );
		if( ! GTK_WIDGET_VISIBLE( priv->dialog ) ) {
			screem_plugin_restore_from_session( plugin,
					priv->dialog );
		}
		gtk_widget_show_all( priv->dialog );
		gdk_window_raise( priv->dialog->window );
	}
}

static void form_wizard_clicked( GtkWidget *widget, 
		gint button, gpointer data )
{
	ScreemPlugin *plugin;
	ScreemSkelPluginPrivate *priv;
	GtkWidget *entry;
	const gchar *method;
	const gchar *action;
	const gchar *enctype;

	gchar *open;
	gchar *close;

	plugin = SCREEM_PLUGIN( data );
	priv = SCREEM_SKEL_PLUGIN( plugin )->priv;
	
	if( button == GTK_RESPONSE_OK || button == GTK_RESPONSE_APPLY ) {
		entry = GTK_BIN( priv->method )->child;
		method = gtk_entry_get_text( GTK_ENTRY( entry ) );
		entry = GTK_BIN( priv->action )->child;
		action = gtk_entry_get_text( GTK_ENTRY( entry ) );
		entry = GTK_BIN( priv->enctype )->child;
		enctype = gtk_entry_get_text( GTK_ENTRY( entry ) );
		if( ! strcmp( "", enctype ) )
			enctype = NULL;

		if( enctype )
			open = g_strdup_printf( "<form method=\"%s\" action=\"%s\" enctype=\"%s\">\n", method, action, enctype );
		else
			open = g_strdup_printf( "<form method=\"%s\" action=\"%s\">\n", method, action );
		
		close = "\n</form>";

		screem_plugin_insert_markup( plugin, open, close, TRUE );

		g_free( open );
	}

	screem_plugin_store_in_session( plugin, widget );
	
	if( button != GTK_RESPONSE_OK ) {
		gtk_widget_hide( widget );
	}
}

static void create_dialog( ScreemPlugin *plugin )
{
	ScreemSkelPluginPrivate *priv;
	GtkWidget *dialog;
	GtkWidget *table;
	GtkWidget *label;

	static const gchar *methods[] = {
		"POST", "GET", NULL
	};
	static const gchar *actions[] = {
		NULL
	};
	static const gchar *types[] = {
		"", "application/x-www-form-urlencoded",
		"multipart/form-data", "image/jpeg",
		"image/gif", "image/png", "text/plain", "text/html",
		"text/xml", NULL
	};
	gint i;
	
	priv = SCREEM_SKEL_PLUGIN( plugin )->priv;
	
	if( ! priv->dialog ) {
		dialog = gtk_dialog_new_with_buttons(_("Insert a Form - Screem"),
				NULL, 0, GTK_STOCK_CLOSE, 
				GTK_RESPONSE_CLOSE, GTK_STOCK_APPLY,
				GTK_RESPONSE_APPLY, GTK_STOCK_OK,
				GTK_RESPONSE_OK, NULL );
		gtk_dialog_set_has_separator( GTK_DIALOG( dialog ), 
				FALSE );
		gtk_container_set_border_width( GTK_CONTAINER( dialog ), 6 );
		gtk_box_set_spacing( GTK_BOX( GTK_DIALOG( dialog )->vbox ),
				12 );
	
		table = gtk_table_new( 3, 2, FALSE );
		gtk_table_set_col_spacings( GTK_TABLE( table ), 12 );
		gtk_table_set_row_spacings( GTK_TABLE( table ), 6 );
	
		label = gtk_label_new( "method=" );
		gtk_table_attach( GTK_TABLE( table ), label,
				  0, 1, 0, 1, 0, 0, 0, 0 );
		label = gtk_label_new( "action=" );
		gtk_table_attach( GTK_TABLE( table ), label,
				  0, 1, 1, 2, 0, 0, 0, 0 );
		label = gtk_label_new( "enctype=" );
		gtk_table_attach( GTK_TABLE( table ), label,
				  0, 1, 2, 3, 0, 0, 0, 0 );

		/* methods combo */

		priv->method = gtk_combo_box_entry_new_text();
		
		for( i = 0; methods[ i ]; ++ i ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( priv->method ),
					methods[ i ] );
		}
		gtk_table_attach( GTK_TABLE( table ), priv->method,
				  1, 2, 0, 1, 
				  GTK_EXPAND | GTK_FILL,
				  GTK_EXPAND | GTK_FILL,
				  0, 0 );

		/* action combo */
		priv->action = gtk_combo_box_entry_new_text();
		
		for( i = 0; actions[ i ]; ++ i ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( priv->action ),
					actions[ i ] );
		}
		gtk_table_attach( GTK_TABLE( table ), priv->action,
				  1, 2, 1, 2, 				 
				  GTK_EXPAND | GTK_FILL,
				  GTK_EXPAND | GTK_FILL,
				  0, 0 );

		/* enctype combo */
		priv->enctype = gtk_combo_box_entry_new_text();
		
		for( i = 0; types[ i ]; ++ i ) {
			gtk_combo_box_append_text( GTK_COMBO_BOX( priv->enctype ),
					types[ i ] );
		}
		gtk_table_attach( GTK_TABLE( table ), priv->enctype,
				  1, 2, 2, 3, 
				  GTK_EXPAND | GTK_FILL,
				  GTK_EXPAND | GTK_FILL,
				  0, 0 );
	
		gtk_box_pack_start( GTK_BOX( GTK_DIALOG( dialog )->vbox ), table,
				    TRUE, TRUE, 0 );

		g_signal_connect( G_OBJECT( dialog ), "response", 
				  G_CALLBACK( form_wizard_clicked ), 
				  plugin );
	
		gtk_window_set_wmclass(GTK_WINDOW( dialog ), 
				"form_wizard", "Screem");
		gtk_window_set_role( GTK_WINDOW( dialog ), 
				"screem_form_wizard" );

		g_signal_connect( G_OBJECT( dialog ), "delete_event",
				  G_CALLBACK( gtk_widget_hide ), NULL );

		priv->dialog = dialog;
	}
}

/* There should be no need to change any code below here */
enum {
	ARG_0
};

static void screem_skel_plugin_class_init( ScreemSkelPluginClass *klass );
static void screem_skel_plugin_init( ScreemSkelPlugin *skel_plugin );
static void screem_skel_plugin_finalize( GObject *object );

/* G Object stuff */
#define PARENT_TYPE SCREEM_TYPE_PLUGIN

static gpointer parent_class;

static void screem_skel_plugin_class_init( ScreemSkelPluginClass *klass )
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS( klass );

	object_class->finalize = screem_skel_plugin_finalize;
	parent_class = g_type_class_peek_parent( klass );
}

static void screem_skel_plugin_init( ScreemSkelPlugin *skel_plugin )
{
	skel_plugin->priv = g_new0( ScreemSkelPluginPrivate, 1 );
	SCREEM_PLUGIN( skel_plugin )->setup = setup;
}

static void screem_skel_plugin_finalize( GObject *object )
{
	ScreemSkelPlugin *skel_plugin;
	ScreemSkelPluginPrivate *priv;
	
	skel_plugin = SCREEM_SKEL_PLUGIN( object );
	priv = skel_plugin->priv;

	cleanup( priv );
	
	g_free( priv );
	
	G_OBJECT_CLASS( parent_class )->finalize( object );
}

static GType screem_skel_plugin_get_type()
{
	static GType type = 0;
	
	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemSkelPluginClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_skel_plugin_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemSkelPlugin ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_skel_plugin_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       plugin_name,
					       &info, 0 );
	}

	return type;
}

static ScreemSkelPlugin *screem_skel_plugin_new( void )
{
	ScreemSkelPlugin *skel_plugin;

	skel_plugin = SCREEM_SKEL_PLUGIN( g_object_new( SCREEM_TYPE_SKEL_PLUGIN, 
				"name", plugin_name,
				NULL ) );

	return skel_plugin;
}

G_MODULE_EXPORT void get_details( ScreemPluginDetails **ret )
{
	ScreemPluginDetails *details;

	details = g_new0( ScreemPluginDetails, 1 );
	details->name = plugin_name;
	details->displayed_name = displayed_name;
	details->authors = authors;
	details->description = description;
	details->version = version;
	details->create = screem_skel_plugin_new;
	details->api_version = SCREEM_PLUGIN_REQUIRED_VERSION;

	*ret = details;
}

