# -*- mode: python; coding: utf-8 -*-
#
# Pigment Python binding cloning example
#
# Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
#
# Author: Loïc Molinari <loic@fluendo.com>

import sys, pgm, gobject, gst
from pgm.timing import controller, modifier, keyframe

# Terminate the mainloop on a delete event
def on_delete(viewport, event, pipeline):
    pipeline.set_state(gst.STATE_NULL)
    pgm.main_quit()

# Terminate the mainloop on 'Escape' or 'q' key press event
def on_key_press(viewport, event, pipeline):
    if event.keyval == pgm.keysyms.q or event.keyval == pgm.keysyms.Escape:
        pgm.main_quit()

def main(args):
    # Argument check
    if len(args) < 1:
        print 'usage: %s uri' % sys.argv[0]
        return -1

    # OpenGL viewport creation
    gl = pgm.viewport_factory_make('opengl')
    gl.title = 'Video cloning'

    # Canvas and image drawables creation
    cvs = pgm.Canvas()
    img1 = pgm.Image()
    img2 = pgm.Image()
    img3 = pgm.Image()

    # Bind the canvas to the OpenGL viewport
    gl.set_canvas(cvs)

    # Images properties
    img1.position = (0.1, 0.0, 0.0)
    img1.size = (1.8, 2.0)
    img1.fg_color = (255, 100, 100, 255)
    img1.bg_a = 0
    img2.position = (1.1, 0.5, 0.0)
    img2.size = (1.8, 2.0)
    img2.fg_color = (100, 255, 100, 155)
    img2.bg_a = 0
    img3.position = (2.0, 1.0, 0.0)
    img3.size = (1.8, 2.0)
    img3.fg_color = (100, 100, 255, 255)
    img3.bg_a = 0

    # Drawables need to be shown
    img1.show()
    img2.show()
    img3.show()

    # Add them to the middle layer of the canvas
    cvs.add(pgm.DRAWABLE_MIDDLE, img1, img3, img2)

    # GStreamer pipeline setup
    sink = gst.element_factory_make('pgmimagesink')
    pipeline = gst.element_factory_make('playbin')
    pipeline.set_property('uri', sys.argv[1])
    pipeline.set_property('video-sink', sink)
    sink.set_property('image', img1)
    pipeline.set_state(gst.STATE_PLAYING)

    # Cloning from the first image
    img2.set_from_image(img1)
    img3.set_from_image(img1)

    # Create a animation position timeline
    tml = [keyframe.KeyFrame(1.0, 0.0),
           keyframe.KeyFrame(0.0, 1.0)]

    # Create the animation modifiers
    mdf1 = modifier.Modifier([img1], 'y', tml)
    mdf2 = modifier.Modifier([img2], 'y', tml)
    mdf3 = modifier.Modifier([img3], 'y', tml)

    # Create the animation controllers
    ctrl1 = controller.Controller(duration=700, resolution=7,
                                 repeat_behavior=controller.REVERSE,
                                 repeat_count=controller.INFINITE,
                                 transformation=controller.DECELERATE,
                                 modifiers=[mdf1])
    ctrl2 = controller.Controller(duration=800, resolution=7,
                                 repeat_behavior=controller.REVERSE,
                                 repeat_count=controller.INFINITE,
                                 transformation=controller.DECELERATE,
                                 modifiers=[mdf2])
    ctrl3 = controller.Controller(duration=900, resolution=7,
                                 repeat_behavior=controller.REVERSE,
                                 repeat_count=controller.INFINITE,
                                 transformation=controller.DECELERATE,
                                 modifiers=[mdf3])

    # Start the controllers
    ctrl1.start()
    ctrl2.start()
    ctrl3.start()

    # Let's start the mainloop
    gl.connect('delete-event', on_delete, pipeline)
    gl.connect('key-press-event', on_key_press, pipeline)
    gl.show()
    pgm.main()

if __name__ == '__main__':
    sys.exit(main(sys.argv[1:]))
