/**
  @file state.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2006 Nokia Corporation. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <glib.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "log.h"
#include "btcond-signals.h"
#include "btcond-bt.h"
#include "btcond-hci.h"
#include "state.h"

#define DEV_NUMBER 0

struct connection_info {
    gchar  *bda;
    GArray *handles;
};

static GSList *connections = NULL;

static gboolean _bt_down = FALSE;
#ifdef USE_MCE
static gboolean _cover_closed = FALSE;
static gboolean _inactive = FALSE;
#endif

static void clear_connections(void)
{
    GSList *l;

    if (!connections)
        return;

    debug("Emptying connections list");

    for (l = connections; l != NULL; l = l->next) {
        struct connection_info *info = l->data;
        g_free(info->bda);
        g_array_free(info->handles, TRUE);
        g_free(info);
    }

    g_slist_free(connections);
    connections = NULL;
}

#ifdef USE_MCE
static void disable_bt(void)
{
    bt_disconnect(NULL, TRUE);
    hci_bt_down();
    clear_connections();

}

static void enable_bt(void)
{
    hci_bt_up();
}
#endif

#if 0
static void free_info(struct connection_info *info, gpointer user_data)
{
    g_free(info->bda);
    g_array_free(info->handles, TRUE);
    g_free(info);
}
#endif

static gint cmp_info(struct connection_info *info, const gchar *bda)
{
    return g_ascii_strcasecmp(info->bda, bda);
}

static gint cmp_handle(struct connection_info *info, uint16_t *handle)
{
    int i;
    for (i = 0; i < info->handles->len; i++) {
        if (g_array_index(info->handles, uint16_t, i) == *handle)
            return 0;
    }
    return 1;
}

static struct connection_info *get_connection_info(const gchar *bda)
{
    GSList *element;
    element = g_slist_find_custom(connections, bda, (GCompareFunc)cmp_info);
    if (element == NULL)
        return NULL;
    else
        return element->data;
}

static struct connection_info *get_connection_info_by_handle(uint16_t handle)
{
    GSList *element;
    element = g_slist_find_custom(connections, &handle, (GCompareFunc)cmp_handle);
    if (element == NULL)
        return NULL;
    else
        return element->data;
}

static void add_handle(con_evt_t *event)
{
    struct connection_info *info;
    info = get_connection_info(event->bda);
    if (info == NULL) {
        info = g_new0(struct connection_info, 1);
        info->bda = g_strdup(event->bda);
        connections = g_slist_append(connections, info);
    }

    if (info->handles == NULL) {
        info->handles = g_array_new(FALSE, FALSE, sizeof(uint16_t));
        send_dbus_connection_status(event);
    }

    info->handles = g_array_append_val(info->handles, event->handle);
    debug("new handle %u added (usage count: %d)",
            event->handle, info->handles->len);
}

static void remove_handle(con_evt_t *event)
{
    struct connection_info *info;
    int i;

    info = get_connection_info_by_handle(event->handle);
    if (info == NULL) {
        debug("Connection info for handle %u not found", event->handle);
        return;
    }

    for (i = 0; i < info->handles->len; i++) {
        if (g_array_index(info->handles, uint16_t, i) == event->handle)
            break; 
    }
    g_assert(i != info->handles->len);

    info->handles = g_array_remove_index(info->handles, i);
    debug("handle %u removed (usage count: %d)", event->handle, info->handles->len);

    if (info->handles->len == 0) {
        g_array_free(info->handles, TRUE);
        send_dbus_connection_status(event);
        g_free(info->bda);
        connections = g_slist_remove(connections, info);
        g_free(info);
    }

#ifdef USE_MCE
    if (_inactive && !connections)
        disable_bt();
#endif
}

gboolean handle_is_connected(uint16_t handle)
{
    if (get_connection_info_by_handle(handle) == NULL)
        return FALSE;
    else
        return TRUE;
}

ConnStatus connection_status(const gchar *bda)
{
    if (_bt_down)
        return CONN_STATUS_DISABLED;

    if (get_connection_info(bda))
        return CONN_STATUS_CONNECTED;
    else
        return CONN_STATUS_DISCONNECTED;
}

const gchar *get_handle_bda(uint16_t handle)
{
    struct connection_info *info;

    info = get_connection_info_by_handle(handle);
    if (info == NULL)
        return NULL;

    return info->bda;
}

#ifdef USE_MCE
# define SYSFS_COVER_FILE "/sys/devices/platform/gpio-switch/prot_shell/cover_switch"
void update_cover_state()
{
    gchar* buf = NULL;
    GError *err = NULL;

    g_file_get_contents(SYSFS_COVER_FILE, &buf, NULL, &err);
    if (err != NULL) {
        error("couldn't read cover state file: %s", err->message);
        g_error_free(err);
        return; /* buf is not allocated */
    }

    if (strncmp(buf, "closed", strlen("closed")) == 0) {
        debug("Cover state: closed");
	disable_bt();
        _cover_closed = TRUE;
    } else {
        debug("Cover state: open");
        _cover_closed = FALSE;
    }

    g_free(buf);
}
#endif

#ifdef USE_MCE
void update_inactive_state(gboolean inactive)
{
    _inactive = inactive;

    debug("update_inactive_state: inactive=%d, _bt_down=%d, _cover_closed=%d",
            inactive, _bt_down, _cover_closed);

    if (inactive) {
        if (!connections)
            disable_bt();
    }
    else {
        if (_bt_down && !_cover_closed)
            enable_bt();
    }
}
#endif

void update_state(con_evt_t *event)
{
    switch (event->type) {
        case EVT_TYPE_CC:
            debug("Connection complete (handle %u, bda %s)",
                    event->handle, event->bda);
            add_handle(event);
            break;
        case EVT_TYPE_DC:
            debug("Disconnection complete (handle %u)", event->handle);
            remove_handle(event);
            break;
        default:
            debug("update_state: invalid event type (%d)", event->type);
            break;
    }
}

void update_dev_state(int dev, gboolean down)
{
    if (dev != DEV_NUMBER) {
        error("Device event for unknown device %d", dev);
        return;
    }

    _bt_down = down;

    if (down) {
        bt_disconnect(NULL, TRUE);
        clear_connections();
    }
}


