/**
  @file dbus-helper.h

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2006 Nokia Corporation. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

*/
#ifndef _DBUS_HELPER_H_
#define _DBUS_HELPER_H_

#include <stdarg.h>
#include <glib.h>
#include <dbus/dbus.h>

typedef void (*name_cb)(const char *name, gpointer user_data);

void append_dbus_args(DBusMessage *message, int first_arg_type, ...);

gboolean get_basic_args(DBusMessageIter *iter, int first_arg_type, ...);

gboolean send_and_unref(DBusConnection *connection, DBusMessage *message);

gboolean send_invalid_args(DBusConnection *connection, DBusMessage *message);

DBusMessage *new_dbus_signal(const char *path,
                             const char *interface,
                             const char *name,
                             const char *destination);

DBusMessage *new_dbus_method_call(const char *service,
                                  const char *path,
                                  const char *interface,
                                  const char *method);

DBusMessage *new_dbus_method_return(DBusMessage *message);

DBusMessage *new_dbus_error(DBusMessage *message, const char *name, const char *fmt, ...);

DBusMessage *new_dbus_error_gerr(DBusMessage *message, GError *err);

/** Get device mode from MCE
 * @param connection D-BUS connection to use
 * @returns device mode (as reported by MCE, should be "flight" or "normal")
 */
gchar *get_device_mode(DBusConnection *connection);

/** Add a function to be called when device mode changes (reported by a D-BUS
 * signal from MCE)
 * @param connection D-BUS connection to listen for MCE signals on
 * @param mode_cb function to call when the mode changes
 * @returns TRUE on success, FALSE on failure
 */
#ifdef USE_MCE
gboolean add_mode_listener(DBusConnection *connection,
                           void (*mode_cb)(const gchar *mode));

#endif

gboolean add_name_listener(DBusConnection *connection, const char *name,
                           name_cb func, gpointer user_data);

gboolean remove_name_listener(DBusConnection *connection, const char *name,
                              name_cb func, gpointer user_data);


#endif /* _DBUS_HELPER_H_ */
