/**
  @file btsearch-dbus.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2006 Nokia Corporation. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <glib.h>

#include <dbus/dbus.h>

#include "log.h"
#include "btsearch-bt.h"
#include "devlist.h"
#include "bt-dbus.h"
#include "dbus-helper.h"
#include "btsearch-dbus.h"

#define EXIT_TIMEOUT 5000 /* milliseconds */

extern GMainLoop *event_loop;

static GSource *exit_timer = NULL;

static gboolean exit_cb(gpointer user_data)
{
    error("btsearch running but no requests received.");
    g_main_loop_quit(event_loop);
    return FALSE;
}

static void set_exit_timer(void)
{
    if (exit_timer)
        g_source_destroy(exit_timer);
    exit_timer = g_timeout_source_new(EXIT_TIMEOUT);
    g_source_set_callback(exit_timer, exit_cb, NULL, NULL);
    (void) g_source_attach(exit_timer, NULL);
}

static void remove_exit_timer(void)
{
    if (exit_timer) {
        g_source_destroy(exit_timer);
        exit_timer = NULL;
    }
}

static DBusHandlerResult start_search_request(DBusMessage    *message,
                                              DBusConnection *connection)
{
    DBusMessage *reply;
    const char *sender;

    sender = dbus_message_get_sender(message);
    if (sender == NULL) {
        error("Unable to get sender of message");
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    if (search_in_progress()) {
        reply = new_dbus_error(message, BTSEARCH_ERROR_SEARCH_IN_PROGRESS, NULL);
        if (!send_and_unref(connection, reply))
            error("Unable to send search_in_progress error reply");
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    remove_exit_timer();

    if (!bt_ok()) {
        reply = new_dbus_error(message, BTSEARCH_ERROR_BT_DISABLED, NULL);
        if (!send_and_unref(connection, reply))
            error("Unable to send bt_disabled error reply");
        g_main_loop_quit(event_loop);
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    init_search(sender);

    if (dbus_message_get_no_reply(message))
        return DBUS_HANDLER_RESULT_HANDLED;

    /* Send empty reply to indicate "OK". Is this necessary? */
    reply = new_dbus_method_return(message);
    if (!send_and_unref(connection, reply))
        error("Unable to send start_search request reply");

    return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult stop_search_request(DBusMessage    *message,
                                             DBusConnection *connection)
{
    DBusMessage *reply;

    if (!search_in_progress()) {
        reply = new_dbus_error(message, BTSEARCH_ERROR_NO_SEARCH, NULL);
        if (!send_and_unref(connection, reply))
            error("Unable to send no_search error reply");
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    stop_search();

    if (dbus_message_get_no_reply(message))
        return DBUS_HANDLER_RESULT_HANDLED;

    /* Send empty reply to indicate "OK". Is this necessary? */
    reply = new_dbus_method_return(message);
    if (!send_and_unref(connection, reply))
        error("Unable to send stop_search reply");

    return DBUS_HANDLER_RESULT_HANDLED;
}


/* Handler for D-Bus requests */
static DBusHandlerResult btsearch_req_handler(DBusConnection     *connection,
                                              DBusMessage        *message,
                                              void               *user_data)
{
    const char *dest;

    if (dbus_message_get_type(message) != DBUS_MESSAGE_TYPE_METHOD_CALL)
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

    dest = dbus_message_get_destination(message);
    if (!g_str_equal(dest, BTSEARCH_SERVICE)) {
        debug("Received D-Bus message not addressed to me.");
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    if (dbus_message_is_method_call(message,
                                    BTSEARCH_REQ_INTERFACE,
                                    BTSEARCH_START_SEARCH_REQ))
        return start_search_request(message, connection);
    else if (dbus_message_is_method_call(message,
                                         BTSEARCH_REQ_INTERFACE,
                                         BTSEARCH_STOP_SEARCH_REQ))
        return stop_search_request(message, connection);

    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static DBusObjectPathVTable btsearch_req_vtable = {
    .message_function    = btsearch_req_handler,
    .unregister_function = NULL
};

/* Create bindings for D-Bus handlers */
void init_dbus_handlers(DBusConnection *connection)
{
    dbus_bool_t ret;

    ret = dbus_connection_register_object_path(connection,
                                               BTSEARCH_REQ_PATH,
                                               &btsearch_req_vtable,
                                               NULL);
    if (ret == FALSE)
        error("register_object_path(req) failed");

    set_exit_timer();
}

void destroy_dbus_handlers(DBusConnection *connection)
{
    dbus_connection_unregister_object_path(connection, BTSEARCH_REQ_PATH);
}
