/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Kernel_p.h"

#include "GTLCore/Parameter.h"
#include "GTLCore/CompilationMessage.h"
#include "GTLCore/Function.h"
#include "GTLCore/ModuleData_p.h"
#include "GTLCore/Macros_p.h"
#include "GTLCore/Type.h"
#include "GTLCore/TypesManager.h"
#include "GTLCore/VirtualMachine_p.h"

#include "GTLCore/AST/Tree.h"

#include "GTLFragment/Library_p.h"

#include "Debug.h"

#include "CodeGenerator_p.h"
#include "Library_p.h"
#include "Wrapper_p.h"

using namespace OpenShiva;

KernelPrivate::KernelPrivate()
{
}

KernelPrivate::~KernelPrivate()
{
}

void KernelPrivate::preCompilation()
{
  if (self->GTLFragment::Library::d->parameters_name2id.find("IMAGE_WIDTH") == self->GTLFragment::Library::d->parameters_name2id.end())
  {
    self->setParameter( Kernel::IMAGE_WIDTH, 800.0f );
  }
  if (self->GTLFragment::Library::d->parameters_name2id.find("IMAGE_HEIGHT") == self->GTLFragment::Library::d->parameters_name2id.end())
  {
    self->setParameter( Kernel::IMAGE_HEIGHT, 600.0f );
  }
  std::vector<GTLCore::Value> values;
  values.push_back( self->GTLFragment::Library::d->parameters_info[ self->parameterId("IMAGE_WIDTH")].value);
  values.push_back( self->GTLFragment::Library::d->parameters_info[ self->parameterId("IMAGE_HEIGHT")].value);
  self->setParameter( "IMAGE_SIZE", GTLCore::Value(values, GTLCore::TypesManager::getVector(GTLCore::Type::Float32, 2) ) );
}

void KernelPrivate::cleanup()
{
  evaluatePixelsFunction = 0; // It's deleted by the moduleData
}

void KernelPrivate::postCompilation()
{
  std::list<GTLCore::String> keepVisible;
  
  // Functions
  keepVisible.push_back("evaluateDependents");
  keepVisible.push_back("evaluatePixels");
  keepVisible.push_back("runTest");
  keepVisible.push_back("changed");
  keepVisible.push_back("needed");
  keepVisible.push_back("generated");
  
  // Parmeters
  foreach(const GTLFragment::ParameterInfo& pi, self->GTLFragment::Library::d->parameters_info)
  {
    GTLCore::String symbol = GTLCore::AST::GlobalConstantDeclaration::nameToSymbol(GTLCore::ScopedName(self->name(), pi.name));
    keepVisible.push_back(symbol);
  }
  self->GTLFragment::Library::d->m_moduleData->hideAllSymbolsBut(keepVisible);
  
  // Create a wrapper
  determineTypes();
  // Create the generateEvaluatePixels LLVM function
  evaluatePixelsFunction = CodeGenerator::generateEvaluatePixels( self, self->GTLFragment::Library::d->m_moduleData->llvmLinkedModule(), self->GTLFragment::Library::d->count_channels_generic );
  needRunEvealuateDependents = true;
  runEvaluateDependentsIfNeeded();
}

void KernelPrivate::runEvaluateDependentsIfNeeded()
{
  if(not needRunEvealuateDependents) return;
  // Call evaluateDepends as needed
  const std::list<GTLCore::Function*>* evaluateDependentsFuncs = self->GTLFragment::Library::d->m_moduleData->function( self->name(), "evaluateDependents");
  if( evaluateDependentsFuncs )
  {
    GTLCore::Function* evaluateDependentsFunc = evaluateDependentsFuncs->front();
    void (*func)() = (void (*)())GTLCore::VirtualMachine::instance()->getPointerToFunction( evaluateDependentsFunc);
    (*func)();
  }
}

GTLCore::ModuleData* KernelPrivate::moduleData()
{
  return self->GTLFragment::Library::d->m_moduleData;
}

void KernelPrivate::determineTypes()
{
  m_inputsTypes.clear();
  const std::list<GTLCore::Function*>*  ePFunctions = self->GTLFragment::Library::d->m_moduleData->function( self->name(), "evaluatePixel" );
  GTL_ASSERT(ePFunctions);
  GTLCore::Function* ePFunction = ePFunctions->front();
  bool hasOutput = false;
  foreach( GTLCore::Parameter arg, ePFunction->parameters() )
  {
    if( arg.isOutput() )
    {
      SHIVA_ASSERT(not hasOutput );
      hasOutput = true;
      m_outputPixelType = arg.type();
      SHIVA_ASSERT( m_outputPixelType->dataType() == GTLCore::Type::STRUCTURE );
      SHIVA_ASSERT( m_outputPixelType->structName().head( 5 ) == "pixel" );
      if(  m_outputPixelType->structName() == "pixel" )
      {
        m_outputImageType = self->GTLFragment::Library::d->m_moduleData->typesManager()->getStructure( "image" );
      } else {
        m_outputImageType = self->GTLFragment::Library::d->m_moduleData->typesManager()->getStructure( "image" + m_outputPixelType->structName().tail(1) );
      }
      GTL_ASSERT( m_outputImageType );
    } else {
      SHIVA_ASSERT( arg.type()->dataType() == GTLCore::Type::STRUCTURE );
      SHIVA_ASSERT( arg.type()->structName().head( 5 ) == "image" );
      m_inputsTypes.push_back( arg.type() );
    }
  }
  SHIVA_ASSERT( hasOutput );
}

const std::list< const GTLCore::Type* >& KernelPrivate::inputsTypes()
{
  return m_inputsTypes;
}

const GTLCore::Type* KernelPrivate::outputPixelType()
{
  return m_outputPixelType;
}

const GTLCore::Type* KernelPrivate::outputImageType()
{
  return m_outputImageType;
}

void KernelPrivate::createWrapper(GTLCore::ModuleData* _moduleData, llvm::Module* module, int _channels, GTLFragment::WhichStdLibrary _library)
{
  self->Library::d->m_wrapper = new Wrapper(self, _moduleData, module, _channels, _library != GTLFragment::ShivaWrappersLibrary);
}

GTLFragment::Wrapper* KernelPrivate::wrapper()
{
  GTL_ABORT("should not happen");
}

void KernelPrivate::addOperatorOverloads(GTLCore::OperatorOverloadCenter* _center)
{
  GTL_ABORT("should not happen");
}
