/*
 *  Copyright (c) 2008,2009,2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Kernel.h"

#include <sstream>
#include <fstream>

#include <llvm/Module.h>

#include "GTLCore/CompilationMessage.h"
#include "GTLCore/Function.h"
#include "GTLCore/ModuleData_p.h"
#include "GTLCore/PixelDescription.h"
#include "GTLCore/Region.h"
#include "GTLCore/Region.h"
#include "GTLCore/Type.h"
#include "GTLCore/TypesManager.h"
#include "GTLCore/Value.h"
#include "GTLCore/Value_p.h"
#include "GTLCore/VirtualMachine_p.h"
#include "GTLCore/wrappers/Allocate.h"

#include "GTLFragment/Library_p.h"

#include "Debug.h"
#include "Library_p.h"

#include "Kernel_p.h"
#include "Wrapper_p.h"
#include "wrappers/ImageWrap_p.h"
#include "wrappers/RegionWrap_p.h"
#include <GTLCore/AST/Tree.h>
#include <llvm/DerivedTypes.h>

using namespace OpenShiva;

Kernel::Kernel(int _channelsNb ) : Library( GTLFragment::Library::SHIVA_KERNEL, _channelsNb), d(new KernelPrivate )
{
  d->self = this;
  d->evaluatePixelsFunction = 0;
  Library::d->libraryCompilation = d;
}

Kernel::~Kernel()
{
  cleanup();
  Library::d->libraryCompilation = 0;
  delete d;
}

void Kernel::evaluatePixels( const GTLCore::RegionI& _region, const std::list< const GTLCore::AbstractImage* >& _inputImages, GTLCore::AbstractImage* _outputImage, GTLCore::ProgressReport* report, const GTLCore::Transform& transfo, const GTLCore::ChannelsFlags& flags) const
{
  d->runEvaluateDependentsIfNeeded();
  SHIVA_DEBUG( _region.x() << " " << _region.y() << " " << _region.columns() << " " << _region.rows());
  SHIVA_ASSERT( d->evaluatePixelsFunction );
  
  // Wrap input images
  SHIVA_ASSERT( Library::d->m_wrapper );
  const void** inputImages = new const void*[ _inputImages.size() ];
  int i = 0;
  for( std::list< const GTLCore::AbstractImage* >::const_iterator it = _inputImages.begin();
       it != _inputImages.end(); ++it)
  {
    inputImages[i] = (const void*)Library::d->m_wrapper->wrapImage( const_cast<GTLCore::AbstractImage*>( *it ) ); // TODO no const_cast
  }
  
  // Wrap output image
  ImageWrap* owrap = Library::d->m_wrapper->wrapImage( _outputImage );
  
  // Call function
  SHIVA_DEBUG("Call function");
  void (*func)(int, int, int, int, const void**, void*, GTLCore::ProgressReport*, const GTLCore::Transform*, gtl_uint64 )
   = ( void(*)(int, int, int, int, const void**, void*, GTLCore::ProgressReport*, const GTLCore::Transform*, gtl_uint64 ))
   GTLCore::VirtualMachine::instance()->getPointerToFunction( d->evaluatePixelsFunction);
  SHIVA_ASSERT(func);
  (*func)( _region.x(), _region.y(), _region.columns(), _region.rows(), inputImages, owrap, report, &transfo, flags.value() );
  for( std::size_t i = 0; i < _inputImages.size(); ++i)
  {
    delete (ImageWrap*)inputImages[i];
  }
  delete[] inputImages;
  delete owrap;
  SHIVA_DEBUG( "done" );
}

int Kernel::runTest() const
{
  d->runEvaluateDependentsIfNeeded();
  SHIVA_ASSERT( isCompiled() );
  const std::list<GTLCore::Function*>* fs = GTLFragment::Library::d->m_moduleData->function( name(), "runTest");
  SHIVA_ASSERT( fs );
  GTLCore::Function* f = fs->front();
  GTLCore::Value v = f->call( std::vector< GTLCore::Value >() );
  return v.asInt32();
}

bool Kernel::hasTestFunction() const
{
  return GTLFragment::Library::d->m_moduleData->function( name(), "runTest");
}

GTLCore::RegionF Kernel::needed( GTLCore::RegionI output_region, int input_index, const std::list< GTLCore::RegionI>& input_DOD, const GTLCore::Transform& transfo)
{
  d->runEvaluateDependentsIfNeeded();
  const std::list<GTLCore::Function*>* neededFunctions = GTLFragment::Library::d->m_moduleData->function( name(),"needed");
  GTL_ASSERT( neededFunctions );
  GTLCore::Function* neededFunction = neededFunctions->front();
  RegionWrap* (*func)( RegionWrap*, int, ArrayWrap* ) = ( RegionWrap* (*)( RegionWrap*, int, ArrayWrap* ) )GTLCore::VirtualMachine::instance()->getPointerToFunction( neededFunction);
  RegionWrap* rwrap = (*func)( regionToRegionWrap( transfo.map(output_region.toRegionF()) ), input_index, regionListToArrayWrap( input_DOD ) );
  
  GTLCore::RegionF region = regionWrapToRegionF( rwrap );
  gtlFree( rwrap );
  return region;
}

bool Kernel::hasNeededFunction() const
{
  return GTLFragment::Library::d->m_moduleData->function( name(), "needed");
}

GTLCore::RegionF Kernel::changed( GTLCore::RegionI changed_input_region, int input_index, const std::list< GTLCore::RegionI>& input_DOD, const GTLCore::Transform& transfo)
{
  TEST_GUARD_READ(GTLFragment::Library::d)
  d->runEvaluateDependentsIfNeeded();
  const std::list<GTLCore::Function*>* changedFunctions = GTLFragment::Library::d->m_moduleData->function( name(), "changed");
  GTL_ASSERT(changedFunctions);
  GTLCore::Function* changedFunction = changedFunctions->front();
  RegionWrap* (*func)( RegionWrap*, int, ArrayWrap* ) = ( RegionWrap* (*)( RegionWrap*, int, ArrayWrap* ) )GTLCore::VirtualMachine::instance()->getPointerToFunction( changedFunction);
  ArrayWrap* aw = regionListToArrayWrap( input_DOD );
  RegionWrap* rwrap = (*func)( regionToRegionWrap( changed_input_region ), input_index, aw ); // TODO leak
  
  GTLCore::RegionF region = regionWrapToRegionF( rwrap );
  gtlFree( rwrap );
  gtlFreeAllArray<RegionWrap*>( aw );
  return transfo.invert().map(region);
}

bool Kernel::hasChangedFunction() const
{
  TEST_GUARD_READ(GTLFragment::Library::d)
  return GTLFragment::Library::d->m_moduleData->function( name(), "changed");
}

GTLCore::RegionF Kernel::generated(const GTLCore::Transform& transfo)
{
  TEST_GUARD_READ(GTLFragment::Library::d)
  d->runEvaluateDependentsIfNeeded();
  const std::list<GTLCore::Function*>*  fs = GTLFragment::Library::d->m_moduleData->function( name(), "generated");
  SHIVA_ASSERT( fs );
  GTLCore::Function* f = fs->front();

  RegionWrap* (*func)() = (RegionWrap* (*)())GTLCore::VirtualMachine::instance()->getPointerToFunction( f);
  
  RegionWrap* rwrap = (*func)();
  GTLCore::RegionF region = regionWrapToRegionF( rwrap );
  gtlFree( rwrap );
  return transfo.invert().map(GTLCore::RegionF(region));
}

bool Kernel::hasGeneratedFunction() const
{
  return GTLFragment::Library::d->m_moduleData->function( name(), "generated");
}

void Kernel::setParameter( StandardParameter _hint, const GTLCore::Value& _value)
{
  switch( _hint )
  {
    case IMAGE_WIDTH:
    {
      GTL_ASSERT( _value.type()->dataType() == GTLCore::Type::FLOAT32 );
      setParameter("IMAGE_WIDTH", _value);
    }
      break;
    case IMAGE_HEIGHT:
    {
      GTL_ASSERT( _value.type()->dataType() == GTLCore::Type::FLOAT32 );
      setParameter("IMAGE_HEIGHT", _value);
    }
      break;
  }
}


void Kernel::setParameter( const GTLCore::String& _name, const GTLCore::Value& _value )
{
  std::map< GTLCore::String, int >::iterator it = GTLFragment::Library::d->parameters_name2id.find(_name);
  if(it == GTLFragment::Library::d->parameters_name2id.end())
  {
    setParameter(GTLFragment::Library::d->createParameter(_name, _value), _value);
  } else {
    setParameter(it->second, _value);
  }
}

void Kernel::setParameter( std::size_t _id, const GTLCore::Value& _value)
{
  TEST_GUARD_WRITE(GTLFragment::Library::d)
  GTL_ASSERT(_value.isValid());
  GTLFragment::ParameterInfo& pi = GTLFragment::Library::d->parameters_info[ _id ];
  pi.value = _value;
  if(isCompiled())
  {
    GTLCore::String symbol = GTLCore::AST::GlobalConstantDeclaration::nameToSymbol(GTLCore::ScopedName(name(), pi.name));
    llvm::GlobalVariable* variable = d->moduleData()->llvmLinkedModule()->getGlobalVariable((std::string)symbol);
    GTL_ASSERT(variable);
    void* ptr = GTLCore::VirtualMachine::instance()->getGlobalVariablePointer(variable);
    GTL_ASSERT(ptr);
    GTLCore::copyValueToPtr(_value, variable->getType()->getElementType(), d->moduleData()->llvmModule()->getContext(), ptr);
    d->needRunEvealuateDependents = true;
  }
}

std::size_t Kernel::parameterId( const GTLCore::String& _name )
{
  return GTLFragment::Library::d->parameters_name2id[_name];
}

void Kernel::setParameters( const std::map< GTLCore::String, GTLCore::Value>&  _values )
{
  for( std::map< GTLCore::String, GTLCore::Value>::const_iterator it = _values.begin();
       it != _values.end(); ++it)
  {
    setParameter(it->first, it->second);
  }
}
