/*
 *  Copyright (c) 2009 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include <iostream>
#include <cstdlib>

#include <GTLCore/CompilationMessages.h>
#include <GTLCore/Type.h>
#include <OpenCTL/Version.h>
#include <map>
#include <OpenCTL/Template.h>

#define ARG_IS(a,b) argv[ai] == GTLCore::String(a) or argv[ai] == GTLCore::String(b)

void printVersion()
{
  std::cout << OpenCTL::LibraryShortName() << " - " << OpenCTL::LibraryName() << " - " << OpenCTL::LibraryVersionString() << std::endl;
  std::cout << OpenCTL::LibraryCopyright() << std::endl;
  std::cout << "CTL Version : " << OpenCTL::LanguageVersion() << std::endl;
}
void printHelp()
{
  std::cout << "Usage : ctltc [option] fileName.ctlt" << std::endl;
  std::cout << std::endl;
  std::cout << "Options : " << std::endl;
  std::cout << "  -c --channels [count]   set the number of channels" << std::endl;
  std::cout << "  -t --channeltype [type] set the type of the channels (uint8, uint16, half, float)" << std::endl;
  std::cout << "  -a --alpha [bool]       enable or disable the alpha channel" << std::endl;
  std::cout << "  -nt --namedtype [name] [type]       define a named type" << std::endl;
  std::cout << std::endl;
  std::cout << "  -h --help               print this message" << std::endl;
  std::cout << "  -v --version            print the version information" << std::endl;
}


const GTLCore::Type* getChannelType(const GTLCore::String& channelType)
{
  if( channelType == "uint16" )
  {
    return GTLCore::Type::UnsignedInteger16;
  } else if( channelType == "half" )
  {
    return GTLCore::Type::Float16;
  } else if( channelType == "float" )
  {
    return GTLCore::Type::Float32;
  } else if( channelType == "uint8" )
  {
    return GTLCore::Type::UnsignedInteger8;
  }
  std::cerr << "Unknown type: '" << channelType << "'." << std::endl;
  exit(EXIT_FAILURE);
}

int main(int argc, char** argv)
{
  GTLCore::String fileName = "";
  int channelsCount = 4;
  // Select the type of the resulting image
  const GTLCore::Type* typeChannelImage = GTLCore::Type::UnsignedInteger8;
  bool userChannelType = false;
  bool hasAlpha = true;
  std::map<GTLCore::String, const GTLCore::Type* > typesMap;
  for(int ai = 1; ai < argc; ai++)
  {
    if(ARG_IS("-h","--help"))
    {
      printHelp();
      return EXIT_SUCCESS;
    } else if(ARG_IS("-v","--version"))
    {
      printVersion();
      return EXIT_SUCCESS;
    }  else if(ARG_IS("-c", "--channels")) {
      if( ai == argc - 1 )
      {
        std::cerr << "Expected count after -c --channels." << std::endl;
        return EXIT_FAILURE;
      } else {
        ++ai;
        channelsCount = GTLCore::String( argv[ai] ).toInt();
      }
    } else if(ARG_IS("-t", "--channeltype")) {
      if( ai == argc - 1 )
      {
        std::cerr << "Expected type after -t --channeltype." << std::endl;
        return EXIT_FAILURE;
      } else {
        ++ai;
        GTLCore::String channelType = argv[ai];
        userChannelType = true;
        typeChannelImage = getChannelType(channelType);
      }
    } else if(ARG_IS("-a", "--alpha")) {
      if( ai == argc - 1 )
      {
        std::cerr << "Expected bool after -a --alpha." << std::endl;
        return EXIT_FAILURE;
      } else {
        ++ai;
        hasAlpha = (GTLCore::String(argv[ai]) == "true");
      }
    } else if(ARG_IS("-nt", "--namedtype")) {
      if( ai >= argc - 2 )
      {
        std::cerr << "Expected name and type." << std::endl;
        return EXIT_FAILURE;
      } else {
        ++ai;
        GTLCore::String name = argv[ai];
        ++ai;
        GTLCore::String channelType = argv[ai];
        const GTLCore::Type* type = getChannelType(channelType);
        typesMap[channelType] = type;
      }
    } else {
      if( ai != argc - 1)
      {
        std::cerr << "Invalid command line parameters." << std::endl;
      } else {
        fileName = argv[ai];
      }
    }
  }
  if( fileName == "")
  {
    printHelp();
  } else {
    OpenCTL::Template ctltemplate;
    ctltemplate.loadFromFile(fileName);
    ctltemplate.compile();
    if(ctltemplate.isCompiled())
    {
      int alphaPos = -1;
      if(hasAlpha) {
        alphaPos = channelsCount - 1;
      }
      std::cout << ctltemplate.generateSource(typesMap, GTLCore::PixelDescription(typeChannelImage, channelsCount, alphaPos)) << std::endl;
    } else {
      std::cout << "Error: " << std::endl << ctltemplate.compilationMessages().toString() << std::endl;
    }
  }
}
