/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "ImageDC.h"

#include <algorithm>

#include <llvm/System/Path.h>

#include <GTLCore/String.h>

using namespace GTLImageIO;

struct ImageDC::Private {
  std::list< GTLCore::String > readExtension;
  std::list< GTLCore::String > writeExtension;
  GTLCore::String decodableFilter;
  GTLCore::String encodableFilter;
};

ImageDC::ImageDC() : d(new Private)
{
}

ImageDC::~ImageDC()
{
  delete d;
}

bool ImageDC::canDecodeImage( const GTLCore::String& _fileName ) const
{
  GTLCore::String extension_ = extension( _fileName );
  return std::find( d->readExtension.begin(),
                    d->readExtension.end(),
                    extension_ ) != d->readExtension.end();
}

bool ImageDC::canEncodeImage( const GTLCore::String& _fileName ) const
{
  GTLCore::String extension_ = extension( _fileName );
  return std::find( d->writeExtension.begin(),
                    d->writeExtension.end(),
                    extension_ ) != d->writeExtension.end();
}

GTLCore::String ImageDC::extension( const GTLCore::String& _fileName ) const
{
  GTLCore::String ext = llvm::sys::Path( _fileName ).getSuffix();
  return ext.toLower();
}

void ImageDC::addReadExtension( const GTLCore::String& _extension )
{
  d->readExtension.push_back( _extension );
  d->decodableFilter += " *." + _extension;
}

void ImageDC::addWriteExtension( const GTLCore::String& _extension )
{
  d->writeExtension.push_back( _extension );
  d->encodableFilter += " *." + _extension;
}

void ImageDC::addReadWriteExtension( const GTLCore::String& _extension )
{
  addReadExtension(_extension);
  addWriteExtension(_extension);
}

const GTLCore::String& ImageDC::decodableFilter() const
{
  return d->decodableFilter;
}

const GTLCore::String& ImageDC::encodableFilter() const
{
  return d->encodableFilter;
}
