/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "../Metadata/Entry.h"
#include "../Metadata/Factory_p.h"
#include "../Metadata/Group.h"
#include "../Metadata/ParameterEntry.h"
#include "../Metadata/TextEntry.h"
#include "../Metadata/ValueEntry.h"

class TestEntry : public GTLTest::Case {
  public:
    TestEntry() : GTLTest::Case("Entry")
    {
    }
    virtual void runTest()
    {
      const Metadata::Entry* entry = Metadata::Factory::createEntry( "test");
      GTLTEST_CHECK_EQUAL( entry->name(), "test" );
      GTLTEST_CHECK_EQUAL( entry->asGroup(), 0 );
      GTLTEST_CHECK_EQUAL( entry->asParameterEntry(), 0 );
      GTLTEST_CHECK_EQUAL( entry->asTextEntry(), 0 );
      Metadata::Factory::deleteEntry( entry );
    }
};

class TestTextEntry : public GTLTest::Case {
  public:
    TestTextEntry() : GTLTest::Case("TextEntry")
    {
    }
    virtual void runTest()
    {
      const Metadata::TextEntry* entry = Metadata::Factory::createTextEntry( "test", "text" );
      GTLTEST_CHECK_EQUAL( entry->name(), "test" );
      GTLTEST_CHECK_EQUAL( entry->text(), "text" );
      GTLTEST_CHECK_EQUAL( entry->asGroup(), 0 );
      GTLTEST_CHECK_EQUAL( entry->asParameterEntry(), 0 );
      GTLTEST_CHECK_EQUAL( entry->asTextEntry(), entry );
      Metadata::Factory::deleteEntry( entry );
    }
};

class TestGroup : public GTLTest::Case {
  public:
    TestGroup() : GTLTest::Case("TestGroup")
    {
    }
    virtual void runTest()
    {
      const Metadata::TextEntry* textEntry = Metadata::Factory::createTextEntry( "test", "text" );
      std::list< const Metadata::Entry* > entries;
      entries.push_back( textEntry );
      const Metadata::Group* entry = Metadata::Factory::createGroup( "testGroup", entries );
      GTLTEST_CHECK_EQUAL( entry->name(), "testGroup" );
      GTLTEST_CHECK_EQUAL( entry->entries().size(), 1 );
      GTLTEST_CHECK_EQUAL( *entry->entries().begin(), textEntry );
      GTLTEST_CHECK_EQUAL( entry->asGroup(), entry );
      GTLTEST_CHECK_EQUAL( entry->asParameterEntry(), 0 );
      GTLTEST_CHECK_EQUAL( entry->asTextEntry(), 0 );
      Metadata::Factory::deleteEntry( entry );
    }
};

#define TEST_PARAMETER_ENTRY_CONSISTENT_TYPE(entry) \
  GTLTEST_CHECK_EQUAL( entry->minimumValue().type(), entry->valueType() ); \
  GTLTEST_CHECK_EQUAL( entry->maximumValue().type(), entry->valueType() ); \
  GTLTEST_CHECK_EQUAL( entry->defaultValue().type(), entry->valueType() );

#define TEST_PARAMETER_ENTRY(entry) \
  GTLTEST_CHECK_EQUAL( entry->asGroup(), entry); \
  GTLTEST_CHECK_EQUAL( entry->asParameterEntry(), entry ); \
  GTLTEST_CHECK_EQUAL( entry->asTextEntry(), 0 );

class TestParameterEntry : public GTLTest::Case {
  public:
    TestParameterEntry() : GTLTest::Case("TestParameterEntry")
    {
    }
    virtual void runTest()
    {
      {
        std::list< const Metadata::Entry* > entries;
        entries.push_back( Metadata::Factory::createTextEntry( "type", "int" ) );
        entries.push_back( Metadata::Factory::createValueEntry( "minValue", -10 ) );
        entries.push_back( Metadata::Factory::createValueEntry( "maxValue", 20 ) );
        entries.push_back( Metadata::Factory::createValueEntry( "defaultValue", 10 ) );
        const Metadata::ParameterEntry* entry = Metadata::Factory::createParameterEntry( "int", entries );
        GTLTEST_CHECK_EQUAL( entry->name(), "int" );
        TEST_PARAMETER_ENTRY_CONSISTENT_TYPE(entry);
        TEST_PARAMETER_ENTRY(entry);
        GTLTEST_CHECK_EQUAL( GTLCore::Type::Integer32, entry->valueType() );
        GTLTEST_CHECK_EQUAL( entry->minimumValue().asInt32(), -10 );
        GTLTEST_CHECK_EQUAL( entry->maximumValue().asInt32(), 20 );
        GTLTEST_CHECK_EQUAL( entry->defaultValue().asInt32(), 10 );
      }
      {
        std::list< const Metadata::Entry* > entries;
        entries.push_back( Metadata::Factory::createTextEntry( "type", "float" ) );
        entries.push_back( Metadata::Factory::createValueEntry( "minValue", -0.1f ) );
        entries.push_back( Metadata::Factory::createValueEntry( "maxValue", 1.0f ) );
        entries.push_back( Metadata::Factory::createValueEntry( "defaultValue", 0.5f ) );
        const Metadata::ParameterEntry* entry = Metadata::Factory::createParameterEntry( "float", entries );
        GTLTEST_CHECK_EQUAL( entry->name(), "float" );
        GTLTEST_CHECK_EQUAL( GTLCore::Type::Float, entry->valueType() );
        TEST_PARAMETER_ENTRY_CONSISTENT_TYPE(entry);
        TEST_PARAMETER_ENTRY(entry);
        GTLTEST_CHECK_EQUAL( entry->minimumValue().asFloat(), -0.1f );
        GTLTEST_CHECK_EQUAL( entry->maximumValue().asFloat(), 1.0f );
        GTLTEST_CHECK_EQUAL( entry->defaultValue().asFloat(), 0.5f );
      }
      {
        std::list< const Metadata::Entry* > entries;
        entries.push_back( Metadata::Factory::createTextEntry( "type", "int" ) );
        entries.push_back( Metadata::Factory::createValueEntry( "defaultValue", 10 ) );
        const Metadata::ParameterEntry* entry = Metadata::Factory::createParameterEntry( "int", entries );
        GTLTEST_CHECK_EQUAL( entry->name(), "int" );
        TEST_PARAMETER_ENTRY_CONSISTENT_TYPE(entry);
        TEST_PARAMETER_ENTRY(entry);
        GTLTEST_CHECK_EQUAL( GTLCore::Type::Integer32, entry->valueType() );
        GTLTEST_CHECK_EQUAL( entry->minimumValue().asInt32(), 0 );
        GTLTEST_CHECK_EQUAL( entry->maximumValue().asInt32(), 100 );
        GTLTEST_CHECK_EQUAL( entry->defaultValue().asInt32(), 10 );
      }
      {
        std::list< const Metadata::Entry* > entries;
        entries.push_back( Metadata::Factory::createTextEntry( "type", "float" ) );
        entries.push_back( Metadata::Factory::createValueEntry( "defaultValue", 0.5f ) );
        const Metadata::ParameterEntry* entry = Metadata::Factory::createParameterEntry( "float", entries);
        GTLTEST_CHECK_EQUAL( entry->name(), "float" );
        GTLTEST_CHECK_EQUAL( GTLCore::Type::Float, entry->valueType() );
        TEST_PARAMETER_ENTRY_CONSISTENT_TYPE(entry);
        TEST_PARAMETER_ENTRY(entry);
        GTLTEST_CHECK_EQUAL( entry->minimumValue().asFloat(), 0.0f );
        GTLTEST_CHECK_EQUAL( entry->maximumValue().asFloat(), 1.0f );
        GTLTEST_CHECK_EQUAL( entry->defaultValue().asFloat(), 0.5f );
      }
    }
};

class TestMetadata : public GTLTest::Suite {
  public:
    TestMetadata() : GTLTest::Suite("Metdata")
    {
      addCase( new TestEntry );
      addCase( new TestTextEntry );
      addCase( new TestGroup );
      addCase( new TestParameterEntry );
    }
};
