/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Visitor_p.h"

// LLVM
#include <llvm/BasicBlock.h>
#include <llvm/Constants.h>
#include <llvm/Function.h>
#include <llvm/Instructions.h>
#include <llvm/Module.h>
#include <llvm/GlobalVariable.h>

// GTLCore
#include "CodeGenerator_p.h"
#include "Debug.h"
#include <GTLCore/Macros.h>
#include "Macros_p.h"
#include "Type.h"
#include "Type_p.h"
#include "ErrorMessage.h"
#include "ErrorMessages_p.h"
#include "ExpressionResult_p.h"
#include "VariableNG_p.h"
#include "Utils_p.h"

#include "AST/Expression.h"
#include "wrappers/ArrayWrap.h"
#include "ExpressionGenerationContext_p.h"

using namespace GTLCore;

PrimitiveVisitor* primitiveVisitor = 0;
ArrayVisitor* arrayVisitor = 0;
StructureVisitor* structureVisitor = 0;
VectorVisitor* vectorVisitor = 0;
STATIC_INITIALISATION( Visitors )
{
  primitiveVisitor = new PrimitiveVisitor;
  arrayVisitor = new ArrayVisitor;
  structureVisitor = new StructureVisitor;
  vectorVisitor = new VectorVisitor;
}

//--------- Visitor ---------///

Visitor::Visitor()
{
}

Visitor::~Visitor()
{
  
}

const Visitor* Visitor::getVisitorFor(const Type* _type)
{
  GTL_ASSERT( _type );
  if( _type->d->visitor() )
  {
    return _type->d->visitor();
  } else if( _type->dataType() == Type::ARRAY )
  {
    return arrayVisitor;
  } else if( _type->dataType() == Type::STRUCTURE ) {
    return structureVisitor;
  } else if( _type->dataType() == Type::VECTOR ) {
    return vectorVisitor;
  } else {
    return primitiveVisitor;
  }
  return 0;
}

//--------- PrimitiveVisitor ---------///

PrimitiveVisitor::PrimitiveVisitor() : Visitor( )
{
}

PrimitiveVisitor::~PrimitiveVisitor()
{
}

const Type* PrimitiveVisitor::pointerToIndexType( const Type* ) const
{
  return 0;
}

llvm::Value* PrimitiveVisitor::pointerToIndex(GenerationContext& _generationContext, ExpressionGenerationContext& _expressionGenerationContext, llvm::Value* _pointer, const Type* _type, llvm::Value* _index) const
{
  GTL_ABORT("Primitive type doesn't allow access using indexes"); //TODO except vectors
}

ExpressionResult PrimitiveVisitor::get( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType) const
{
  return ExpressionResult( new llvm::LoadInst( _pointer, "PrimitiveVisitor::get", _currentBlock), _pointerType);
}

llvm::BasicBlock* PrimitiveVisitor::set( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* _value, const Type* _valueType, bool _allocatedInMemory ) const
{
  GTL_ASSERT( _pointerType->dataType() != Type::STRUCTURE and _pointerType->dataType() != Type::ARRAY and _pointerType->dataType() != Type::VECTOR );
  GTL_DEBUG( *_pointer );
  new llvm::StoreInst(
          _generationContext.codeGenerator()->convertValueTo( _currentBlock, _value, _valueType, _pointerType ),
          _pointer, "PrimitiveVisitor::set", _currentBlock);
  return _currentBlock;
}

llvm::BasicBlock* PrimitiveVisitor::initialise(GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, const std::list< llvm::Value*>& _sizes, bool _allocatedInMemory) const
{
  // Don't do nothing
  return _currentBlock;
}

llvm::BasicBlock* PrimitiveVisitor::cleanUp( GenerationContext& , llvm::BasicBlock* _currentBlock, llvm::Value* , const Type* , llvm::Value* , bool, bool /*_ignoreCount*/, bool /*_deletePointer*/ ) const
{
  return _currentBlock;
}

llvm::BasicBlock* PrimitiveVisitor::mark( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* increment ) const
{
  return _currentBlock;
}

llvm::Constant* PrimitiveVisitor::createStaticVariable( llvm::Module* module, const Type* type, const std::list< int>& /*_sizes*/ ) const
{
  return CodeGenerator::convertConstantTo( CodeGenerator::integerToConstant( module->getContext(), 0), Type::Integer32, type );
}

//--------- ArrayVisitor ---------///

ArrayVisitor::ArrayVisitor() : Visitor(  )
{
}

ArrayVisitor::~ArrayVisitor()
{
}

const Type* ArrayVisitor::pointerToIndexType( const Type* _type) const
{
  return _type->embeddedType();
}

llvm::Value* ArrayVisitor::pointerToIndex(GenerationContext& _generationContext, ExpressionGenerationContext& _expressionGenerationContext, llvm::Value* _pointer, const Type* _type, llvm::Value* _index) const
{
  return _generationContext.codeGenerator()->accessArrayValueClamp( _generationContext, _expressionGenerationContext, _pointer, _index);
}

ExpressionResult ArrayVisitor::get( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType ) const
{
  return ExpressionResult(_pointer, _pointerType);
}

llvm::BasicBlock* ArrayVisitor::set( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* _value, const Type* _valueType, bool _allocatedInMemory) const
{
    // Check if size are identical or update the size of this array
    {
      llvm::Value* test= _generationContext.codeGenerator()->createDifferentExpression( _currentBlock, getSize( _generationContext, _currentBlock, _pointer ), Type::Integer32, getSize( _generationContext, _currentBlock, _value ), Type::Integer32);
      llvm::BasicBlock* ifContent = llvm::BasicBlock::Create( _generationContext.llvmContext(), "ifContent");
      _generationContext.llvmFunction()->getBasicBlockList().push_back( ifContent );
      std::list<llvm::Value*> sizes;
      sizes.push_back( getSize( _generationContext, ifContent, _value ) );
      
      llvm::BasicBlock* endIfContent = cleanUp(_generationContext, ifContent, _pointer, _pointerType, 0, _allocatedInMemory, true, false );
      
      endIfContent = initialise( _generationContext, endIfContent, _pointer, _pointerType, sizes, _allocatedInMemory );
      
      llvm::BasicBlock* afterIf = llvm::BasicBlock::Create( _generationContext.llvmContext() );
      _generationContext.llvmFunction()->getBasicBlockList().push_back( afterIf);
      _generationContext.codeGenerator()->createIfStatement( _currentBlock, test, Type::Boolean, ifContent, endIfContent, afterIf );
      _currentBlock = afterIf;
    
    }
    //   int i = 0;
    VariableNG* index = new VariableNG( Type::Integer32, false, false);
    index->initialise( _generationContext, _currentBlock, ExpressionResult( _generationContext.codeGenerator()->integerToConstant(_generationContext.llvmContext(), 0), Type::Integer32), std::list<llvm::Value*>());
    
    // Construct the body of the for loop
    llvm::BasicBlock* bodyBlock = llvm::BasicBlock::Create(_generationContext.llvmContext(), "bodyBlock");
    _generationContext.llvmFunction()->getBasicBlockList().push_back( bodyBlock);
    GTL_DEBUG( " value = " << *_pointer << " type = " << *_pointer->getType() << " " << *_pointerType->embeddedType() << " " << *_pointerType );
    GTL_DEBUG( " value = " << *_value << " type = " << *_value->getType() );
    const Visitor* visitor = Visitor::getVisitorFor( _pointerType->embeddedType() );
    llvm::BasicBlock* endBodyBlock = visitor->set( 
          _generationContext,
          bodyBlock, 
          _generationContext.codeGenerator()->accessArrayValueNoClamp( bodyBlock, _pointer, index->get( _generationContext, bodyBlock  ) ),
          _pointerType->embeddedType(),
          visitor->get( _generationContext, bodyBlock, 
                  _generationContext.codeGenerator()->accessArrayValueNoClamp(
                                    bodyBlock, _value, index->get( _generationContext, bodyBlock ) ), _pointerType->embeddedType() ).value(),
          _valueType->embeddedType(), _allocatedInMemory );
    
    // Create the for statement
    llvm::BasicBlock* returnBlock = CodeGenerator::createIterationForStatement(
                    _generationContext,
                    _currentBlock,
                    index,
                    getSize( _generationContext, _currentBlock, _pointer),
                    Type::Integer32,
                    bodyBlock,
                    endBodyBlock );
    delete index;
    return returnBlock;
}

llvm::BasicBlock* ArrayVisitor::setSize(GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* _size, bool _allocatedInMemory) const
{
  GTL_ASSERT( _pointerType->dataType() == Type::ARRAY );
  GTL_DEBUG( *_size );
  std::vector<llvm::Value*> indexes;
  indexes.push_back( llvm::ConstantInt::get(llvm::Type::getInt32Ty(_generationContext.llvmContext()), 0)); // Access the structure of the array
  indexes.push_back( llvm::ConstantInt::get(llvm::Type::getInt32Ty(_generationContext.llvmContext()), ArrayWrap::POS_SIZE)); // Access the size of the array
  { // Init the size
    GTL_DEBUG( *_pointer );
    llvm::Value* ptr = llvm::GetElementPtrInst::Create( _pointer, indexes.begin(), indexes.end(), "ArrayVisitor::setSize", _currentBlock);
    new llvm::StoreInst(_generationContext.codeGenerator()->convertValueTo( _currentBlock, _size, Type::Integer32, Type::Integer32 ), ptr, "", _currentBlock);
  }
  // Allocate the Array
  indexes[1] = llvm::ConstantInt::get(llvm::Type::getInt32Ty(_generationContext.llvmContext()), ArrayWrap::POS_DATA);
  {
    llvm::Value* ptr = llvm::GetElementPtrInst::Create( _pointer, indexes.begin(), indexes.end(), "ArrayVisitor::setSize", _currentBlock);
    llvm::Value* array = 0;
    if( _allocatedInMemory )
    {
      array = CodeGenerator::allocateMemory( _generationContext,
                    _pointerType->embeddedType()->d->type(_generationContext.llvmContext()), _size, _currentBlock);
    } else {
      array = new llvm::AllocaInst(
                    _pointerType->embeddedType()->d->type(_generationContext.llvmContext()), _size, "ArrayVisitor::setSize", _currentBlock);
    }
    new llvm::StoreInst( array, ptr, "ArrayVisitor::setSize", _currentBlock);
  }
  return _currentBlock;
}

llvm::Value* ArrayVisitor::getSize(GenerationContext& _generationContext, llvm::BasicBlock* currentBlock, llvm::Value* _pointer ) const
{
  return CodeGenerator::accessArraySize(currentBlock, _pointer);
}

llvm::BasicBlock* ArrayVisitor::initialise(GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, const std::list< llvm::Value*>& _sizes, bool _allocatedInMemory) const
{
  GTL_ASSERT( _pointerType->dataType() == Type::ARRAY );
  GTL_DEBUG( _sizes.empty() );
  CodeGenerator::setCountFieldOf( _currentBlock, _pointer, CodeGenerator::integerToConstant( _generationContext.llvmContext(), 1 ) );
  if( not _sizes.empty())
  {
    llvm::Value* currentSize = _sizes.front();
    _currentBlock = setSize( _generationContext, _currentBlock, _pointer, _pointerType, currentSize, _allocatedInMemory );
    std::list< llvm::Value*> sizeAfter = _sizes;
    sizeAfter.pop_front();
    //   int i = 0;
    VariableNG* index = new VariableNG( Type::Integer32, false, false);
    index->initialise( _generationContext, _currentBlock, ExpressionResult(_generationContext.codeGenerator()->integerToConstant(_generationContext.llvmContext(), 0), Type::Integer32), std::list<llvm::Value*>());
    
    // Construct the body of the for loop
    llvm::BasicBlock* bodyBlock = llvm::BasicBlock::Create(_generationContext.llvmContext(), "bodyBlock");
    _generationContext.llvmFunction()->getBasicBlockList().push_back( bodyBlock);
    
    const Visitor* visitor = Visitor::getVisitorFor( _pointerType->embeddedType() );
    llvm::BasicBlock* endBodyBlock = visitor->initialise(
            _generationContext,
            bodyBlock,
            _generationContext.codeGenerator()->accessArrayValueNoClamp(
                      bodyBlock, _pointer, index->get( _generationContext, bodyBlock ) ),
            _pointerType->embeddedType(),
            sizeAfter, _allocatedInMemory );
    
    // Create the for statement
    llvm::BasicBlock* returnBB = CodeGenerator::createIterationForStatement(
                    _generationContext,
                    _currentBlock,
                    index,
                    currentSize,
                    Type::Integer32,
                    bodyBlock,
                    endBodyBlock );
    delete index;
    return returnBB;
  } else {
    std::vector<llvm::Value*> indexes;
    indexes.push_back( llvm::ConstantInt::get(llvm::Type::getInt32Ty(_generationContext.llvmContext()), 0)); // Access the structure of the array
    indexes.push_back( llvm::ConstantInt::get(llvm::Type::getInt32Ty(_generationContext.llvmContext()), ArrayWrap::POS_SIZE)); // Access the size of the array
    // Init the size
    llvm::Value* ptr = llvm::GetElementPtrInst::Create( _pointer, indexes.begin(), indexes.end(), "ArrayVisitor::initialise", _currentBlock);
    new llvm::StoreInst( GTLCore::CodeGenerator::integerToConstant( _generationContext.llvmContext(), 0 ), ptr, "ArrayVisitor::initialise", _currentBlock);
  }
  return _currentBlock;
}

llvm::BasicBlock* ArrayVisitor::cleanUp( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* _donttouch, bool _allocatedInMemory, bool _ignoreCount, bool _deletePointer ) const
{
  // TODO use _donttouch in the array as well
    //   int i = 0;
#ifdef OPENGTL_ENABLE_DEBUG_OUTPUT
  if( _pointer )
  {
    GTL_DEBUG( _pointer << " " << *_pointer );
  }
  GTL_DEBUG( _donttouch );
  if( _donttouch )
  {
    GTL_DEBUG( _donttouch  << " " << *_donttouch );
  }
#endif
  if( _pointer != _donttouch )
  {
    llvm::Value* arraySize = getSize( _generationContext, _currentBlock, _pointer);
    llvm::Value* test1 = CodeGenerator::createStrictInferiorExpression( _currentBlock, CodeGenerator::getCountFieldOf( _currentBlock, _pointer), Type::Integer32, CodeGenerator::integerToConstant(_generationContext.llvmContext(), 1 ), Type::Integer32 );
    llvm::Value* test2 = CodeGenerator::createDifferentExpression( _currentBlock, arraySize, Type::Integer32, CodeGenerator::integerToConstant( _generationContext.llvmContext(), 0 ), Type::Integer32 );
    llvm::Value* testIf = CodeGenerator::createAndExpression( _currentBlock, test1, Type::Boolean, test2, Type::Boolean );
    llvm::BasicBlock* firstIfBlock = llvm::BasicBlock::Create(_generationContext.llvmContext(), "firstIfBlock");
    _generationContext.llvmFunction()->getBasicBlockList().push_back( firstIfBlock);
    
    
    VariableNG* index = new VariableNG( Type::Integer32, false, false);
    index->initialise( _generationContext, firstIfBlock, ExpressionResult( _generationContext.codeGenerator()->integerToConstant(_generationContext.llvmContext(), 0), Type::Integer32), std::list<llvm::Value*>());
    
    // Construct the body of the for loop
    llvm::BasicBlock* bodyBlock = llvm::BasicBlock::Create(_generationContext.llvmContext(), "bodyBlock");
    _generationContext.llvmFunction()->getBasicBlockList().push_back( bodyBlock);
    GTL_DEBUG( " value = " << *_pointer << " type = " << *_pointer->getType() << " " << *_pointerType->embeddedType() << " " << *_pointerType );
    const Visitor* visitor = Visitor::getVisitorFor( _pointerType->embeddedType() );
    llvm::BasicBlock* endBodyBlock = visitor->cleanUp( 
          _generationContext,
          bodyBlock, 
          _generationContext.codeGenerator()->accessArrayValueNoClamp( bodyBlock, _pointer, index->get( _generationContext, bodyBlock  ) ),
          _pointerType->embeddedType(), _donttouch, _allocatedInMemory, _ignoreCount, false );
    // Create the for statement
    llvm::BasicBlock*  afterBlock = CodeGenerator::createIterationForStatement(
                    _generationContext,
                    firstIfBlock,
                    index,
                    arraySize,
                    Type::Integer32,
                    bodyBlock,
                    endBodyBlock );
    if( _allocatedInMemory )
    {
      std::vector<llvm::Value*> indexes;
      indexes.push_back( llvm::ConstantInt::get(llvm::Type::getInt32Ty(_generationContext.llvmContext()), 0)); // Access the structure of the array
      indexes.push_back( llvm::ConstantInt::get(llvm::Type::getInt32Ty(_generationContext.llvmContext()), ArrayWrap::POS_DATA )); // Access the data of the array
      // Free the array
      llvm::Value* ptrToData = llvm::GetElementPtrInst::Create( _pointer, indexes.begin(), indexes.end(), "ArrayVisitor::cleanUp", afterBlock);
      CodeGenerator::freeMemory(_generationContext, new llvm::LoadInst( ptrToData, "ArrayVisitor::cleanUp", afterBlock ), afterBlock );
      if( _deletePointer )
      {
        CodeGenerator::freeMemory( _generationContext, _pointer, afterBlock );
      }
    }
    
    llvm::BasicBlock* afterIfBlock = llvm::BasicBlock::Create(_generationContext.llvmContext(), "afterIfBlock");
    _generationContext.llvmFunction()->getBasicBlockList().push_back( afterIfBlock);
    
    CodeGenerator::createIfStatement( _currentBlock, testIf, Type::Boolean, firstIfBlock, afterBlock, afterIfBlock );
    
    delete index;
    return afterIfBlock;
  } else {
    return _currentBlock;
  }
}

llvm::BasicBlock* ArrayVisitor::mark( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* _increment ) const
{
  CodeGenerator::incrementCountFieldOf( _currentBlock, _pointer, _increment );
    
  //   int i = 0;
  VariableNG* index = new VariableNG( Type::Integer32, false, false);
  index->initialise( _generationContext, _currentBlock, ExpressionResult( _generationContext.codeGenerator()->integerToConstant(_generationContext.llvmContext(), 0), Type::Integer32), std::list<llvm::Value*>());
    
  // Construct the body of the for loop
  llvm::BasicBlock* bodyBlock = llvm::BasicBlock::Create(_generationContext.llvmContext(), "bodyBlock");
  _generationContext.llvmFunction()->getBasicBlockList().push_back( bodyBlock);
  GTL_DEBUG( " value = " << *_pointer << " type = " << *_pointer->getType() << " " << *_pointerType->embeddedType() << " " << *_pointerType );
  const Visitor* visitor = Visitor::getVisitorFor( _pointerType->embeddedType() );
  llvm::BasicBlock* endBodyBlock = visitor->mark( 
        _generationContext,
        bodyBlock, 
        _generationContext.codeGenerator()->accessArrayValueNoClamp( bodyBlock, _pointer, index->get( _generationContext, bodyBlock  ) ),
        _pointerType->embeddedType(), _increment );
  
  llvm::BasicBlock*  afterBlock = CodeGenerator::createIterationForStatement(
                  _generationContext,
                  _currentBlock,
                  index,
                  getSize( _generationContext, _currentBlock, _pointer),
                  Type::Integer32,
                  bodyBlock,
                  endBodyBlock );
  delete index;
  return afterBlock;
}

llvm::Constant* ArrayVisitor::createStaticVariable( llvm::Module* _module, const Type* type, const std::list< int>& _sizes ) const
{
  // Init size
  int currentSize = _sizes.front();
  std::list< int> sizeAfter = _sizes;
  sizeAfter.pop_front();
 
  const GTLCore::Type* embedded = type->embeddedType();
  //
  std::vector<llvm::Constant*> initialisers;
  // Initialize the count
  initialisers.push_back( CodeGenerator::integerToConstant( _module->getContext(), 1 ) );
  // Initialize the current size
  initialisers.push_back( CodeGenerator::integerToConstant( _module->getContext(), currentSize ) );
  // Initialize the data
  const Visitor* visitor = Visitor::getVisitorFor( embedded );
  std::vector<llvm::Constant*> arrayInitialisers;
  for( int i = 0; i < currentSize; ++i)
  {
    arrayInitialisers.push_back( visitor->createStaticVariable( _module, embedded, sizeAfter ) );
  }
  // Initialize the pointer to the data
  llvm::ArrayType* arrType = llvm::ArrayType::get(embedded->d->type(_module->getContext()), currentSize);
  llvm::Constant* arrayData = new llvm::GlobalVariable( *_module, arrType, false, llvm::GlobalValue::InternalLinkage,  llvm::ConstantArray::get(arrType ,arrayInitialisers), "ArrayVisitor::createStaticVariable");
  // Get the pointer to the data
  
  std::vector<llvm::Constant*> indices;
  indices.push_back( CodeGenerator::integerToConstant( _module->getContext(), 0 ) );
  indices.push_back( CodeGenerator::integerToConstant( _module->getContext(), 0 ) );
  initialisers.push_back( llvm::ConstantExpr::getGetElementPtr(arrayData, &indices[0], indices.size() ) );
  
  GTL_DEBUG(* initialisers[2] << " " <<  *type->d->type(_module->getContext()));
  return llvm::ConstantStruct::get( dynamic_cast<const llvm::StructType*>(type->d->type(_module->getContext())), initialisers );
}

//--------- VectorVisitor ---------///


VectorVisitor::VectorVisitor( )
{}
VectorVisitor::~VectorVisitor()
{}
const Type* VectorVisitor::pointerToIndexType( const Type* _type ) const
{
  return _type->embeddedType();
}

llvm::Value* VectorVisitor::pointerToIndex(GenerationContext& _generationContext, ExpressionGenerationContext& _expressionGenerationContext, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* _index) const
{
  GTL_ASSERT( _pointerType->dataType() == Type::VECTOR );
  
  return llvm::GetElementPtrInst::Create( 
              CodeGenerator::convertPointerTo( _expressionGenerationContext.currentBasicBlock(), _pointer, _pointerType->embeddedType()->d->type(_generationContext.llvmContext()) ),
              _index, "VectorVisitor::pointerToIndex", _expressionGenerationContext.currentBasicBlock());
}

ExpressionResult VectorVisitor::get( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType) const
{
  return ExpressionResult(new llvm::LoadInst( _pointer, "VectorVisitor::get", _currentBlock), _pointerType );
}

llvm::BasicBlock* VectorVisitor::set(GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* _value, const Type* _valueType, bool _allocatedInMemory) const
{
  GTL_ASSERT( _pointerType->dataType() == Type::VECTOR );
  GTL_DEBUG( "value = " << *_value << " type = " << *_valueType );
  new llvm::StoreInst(
          _generationContext.codeGenerator()->convertValueTo( _currentBlock, _value, _valueType, _pointerType ),
          _pointer, "VectorVisitor::set", _currentBlock);
  return _currentBlock;
}

llvm::BasicBlock* VectorVisitor::initialise( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, const std::list< llvm::Value*>& _sizes, bool _allocatedInMemory) const
{
  // Don't do nothing
  return _currentBlock;
}

llvm::BasicBlock* VectorVisitor::cleanUp( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* _donttouch, bool _allocatedInMemory, bool /*_ignoreCount*/, bool /*_deletePointer*/ ) const
{
  // Don't do nothing
  return _currentBlock;
}

llvm::BasicBlock* VectorVisitor::mark( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* increment ) const
{
  return _currentBlock;
}

llvm::Constant* VectorVisitor::createStaticVariable( llvm::Module* _module, const Type* _type, const std::list< int>& /*_sizes*/ ) const
{
  GTL_ASSERT( _type->dataType() == Type::VECTOR );
  std::vector< llvm::Constant* > constants;
  const Visitor* visitor = Visitor::getVisitorFor( _type->embeddedType() );
  for( std::size_t i  = 0; i < _type->vectorSize(); ++i)
  {
    constants.push_back( visitor->createStaticVariable( _module, _type->embeddedType(), std::list<int>() ) );
  }
  GTL_ASSERT( not constants.empty());
  return llvm::ConstantVector::get( dynamic_cast<const llvm::VectorType*>(_type->d->type(_module->getContext()) ), constants);
}

//--------- StructureVisitor ---------///

StructureVisitor::StructureVisitor() : Visitor( )
{
}

StructureVisitor::~StructureVisitor()
{
}

const Type* StructureVisitor::pointerToIndexType( const Type* ) const
{
  return 0;
}

llvm::Value* StructureVisitor::pointerToValue( GenerationContext& _generationContext,
                                                llvm::BasicBlock* _currentBlock,
                                                llvm::Value* _pointer, int _index ) const
{
  std::vector<llvm::Value*> indexes;
  indexes.push_back( llvm::ConstantInt::get(llvm::Type::getInt32Ty(_generationContext.llvmContext()), 0));
  indexes.push_back( llvm::ConstantInt::get(llvm::Type::getInt32Ty(_generationContext.llvmContext()), _index + STRUCT_FIRST_ELEMENT ));
  return llvm::GetElementPtrInst::Create( _pointer, indexes.begin(), indexes.end(), "StructureVisitor::pointerToValue", _currentBlock);
}

llvm::Value* StructureVisitor::pointerToIndex(GenerationContext& _generationContext, ExpressionGenerationContext& _expressionGenerationContext, llvm::Value* _pointer, const Type* _type, llvm::Value* _index) const
{
  GTL_ABORT("Structure doesn't allow access using indexes");
}

ExpressionResult StructureVisitor::get( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType ) const
{
  return ExpressionResult(_pointer, _pointerType);
}

llvm::BasicBlock* StructureVisitor::set( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* _value, const Type* _valueType, bool _allocatedInMemory ) const
{
  GTL_ASSERT( _pointerType->dataType() == Type::STRUCTURE );
  for(uint i = 0; i < _pointerType->countStructDataMembers(); ++i)
  {
    const Type* type = _pointerType->structDataMember(i).type();
    llvm::Value* nptrToOwnMember = pointerToValue(_generationContext, _currentBlock, _pointer, i);
    llvm::Value* nptrToValueMember = pointerToValue( _generationContext, _currentBlock, _value, i);
    const Visitor* visitor = Visitor::getVisitorFor( type );
    llvm::Value* memberValue = visitor->get( _generationContext, _currentBlock, nptrToValueMember, type ).value();
    _currentBlock = visitor->set( _generationContext, _currentBlock, nptrToOwnMember, type, memberValue, type, _allocatedInMemory );
  }
  return _currentBlock;
}

llvm::BasicBlock* StructureVisitor::initialise(GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, const std::list< llvm::Value*>& _sizes, bool _allocatedInMemory) const
{
  GTL_ASSERT( _pointerType->dataType() == Type::STRUCTURE );
  
  for( uint i = 0; i < _pointerType->countStructDataMembers(); ++i)
  {
    std::list< llvm::Value* > sizes;
    const Type::StructDataMember& sdm = _pointerType->structDataMember(i);
    for(std::list<int>::const_iterator it = sdm.initialSizes().begin();
        it != sdm.initialSizes().end(); ++it)
    {
      sizes.push_back( _generationContext.codeGenerator()->integerToConstant( _generationContext.llvmContext(), *it ) );
    }
    const Type* type = sdm.type();
    const Visitor* visitor = Visitor::getVisitorFor( type );
    _currentBlock = visitor->initialise( _generationContext, _currentBlock,
                            pointerToValue( _generationContext, _currentBlock, _pointer, i ),
                            type, sizes, _allocatedInMemory );
  }
  CodeGenerator::setCountFieldOf( _currentBlock, _pointer, CodeGenerator::integerToConstant( _generationContext.llvmContext(), 1 ) );
  return _currentBlock;
}

llvm::BasicBlock* StructureVisitor::cleanUp( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* _donttouch, bool _allocatedInMemory, bool _ignoreCount, bool _deletePointer ) const
{
  // TODO use ignoreCount
#ifdef OPENGTL_ENABLE_DEBUG_OUTPUT
  if( _pointer )
  {
    GTL_DEBUG( _pointer << " " << *_pointer );
  }
  if( _donttouch )
  {
    GTL_DEBUG( _donttouch  << " " << *_donttouch );
  }
#endif
  if( _pointer == _donttouch ) return _currentBlock;
  
  llvm::Value* test = CodeGenerator::createStrictInferiorExpression( _currentBlock, CodeGenerator::getCountFieldOf( _currentBlock, _pointer), Type::Integer32, CodeGenerator::integerToConstant(_generationContext.llvmContext(),  1 ), Type::Integer32 );
  llvm::BasicBlock* firstIfBlock = llvm::BasicBlock::Create(_generationContext.llvmContext(), "firstIfBlockStructureVisitorCleanUp");
  _generationContext.llvmFunction()->getBasicBlockList().push_back( firstIfBlock);
  llvm::BasicBlock* lastIfBlock = firstIfBlock;
  
  GTL_ASSERT( _pointerType->dataType() == Type::STRUCTURE );
  
  for( uint i = 0; i < _pointerType->countStructDataMembers(); ++i)
  {
    const Type* type = _pointerType->structDataMember(i).type();
    const Visitor* visitor = Visitor::getVisitorFor( type );
    lastIfBlock = visitor->cleanUp( _generationContext, lastIfBlock, pointerToValue( _generationContext, lastIfBlock, _pointer, i ), type, _donttouch, _allocatedInMemory, _ignoreCount, false );
  }
  
  llvm::BasicBlock* afterIfBlock = llvm::BasicBlock::Create(_generationContext.llvmContext(), "afterIfBlockStructureVisitorCleanUp");
  _generationContext.llvmFunction()->getBasicBlockList().push_back( afterIfBlock);
  if( _allocatedInMemory and _deletePointer )
  {
    CodeGenerator::freeMemory( _generationContext, _pointer, lastIfBlock );
  }
  CodeGenerator::createIfStatement( _currentBlock, test, Type::Boolean, firstIfBlock, lastIfBlock, afterIfBlock );
  
  return afterIfBlock;
}

llvm::BasicBlock* StructureVisitor::mark( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* _increment ) const
{
  CodeGenerator::incrementCountFieldOf( _currentBlock, _pointer, _increment );
  for( uint i = 0; i < _pointerType->countStructDataMembers(); ++i)
  {
    const Type* type = _pointerType->structDataMember(i).type();
    const Visitor* visitor = Visitor::getVisitorFor( type );
    _currentBlock = visitor->mark( _generationContext, _currentBlock, pointerToValue( _generationContext, _currentBlock, _pointer, i ), type, _increment );
  }
  return _currentBlock;
}

llvm::Constant* StructureVisitor::createStaticVariable( llvm::Module* _module, const Type* _type, const std::list< int>& /*_sizes */) const
{
  GTL_ASSERT( _type->dataType() == Type::STRUCTURE );
  std::vector< llvm::Constant* > initialisers;
  initialisers.push_back( CodeGenerator::integerToConstant( _module->getContext(), 1 ) );
  for( uint i = 0; i < _type->countStructDataMembers(); ++i)
  {
    const Type* type = _type->structDataMember(i).type();
    const Visitor* visitor = Visitor::getVisitorFor( type );
    initialisers.push_back( visitor->createStaticVariable( _module, type, _type->structDataMember(i).initialSizes() ) );
  }
  return llvm::ConstantStruct::get( dynamic_cast<const llvm::StructType*>(_type->d->type(_module->getContext())), initialisers );
}
