/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "VirtualMachine_p.h"

#include "Function_p.h"
#include "Value.h"

#include "Debug.h"

// JIT
#include <llvm/ExecutionEngine/ExecutionEngine.h>
#include <llvm/Module.h>
#include <llvm/ModuleProvider.h>
#include <llvm/DerivedTypes.h>
#include <llvm/Target/TargetOptions.h>
#include <llvm/Target/TargetSelect.h>

#include "llvm/Support/CommandLine.h"
#include "llvm/System/Threading.h"

#include "Macros_p.h"
#include "Optimiser.h"
#include "Optimiser_p.h"

using namespace GTLCore;

struct VirtualMachine::Private {
  llvm::ExecutionEngine* executionEngine;
  static VirtualMachine* virtualMachine;
};

VirtualMachine* VirtualMachine::Private::virtualMachine = 0;

VirtualMachine::VirtualMachine() : d(new Private)
{
  d->executionEngine = 0;
  llvm::llvm_start_multithreaded();
#ifdef OPENGTL_32_BITS
 char** argv = new char*[2];
 argv[0] = const_cast<char*>( "GTLVM");
 argv[1] = const_cast<char*>( "-mattr=-3dnow,-3dnowa,-mmx" ); // http://llvm.org/bugs/show_bug.cgi?id=3287
 llvm::cl::ParseCommandLineOptions(2, argv, "");
 delete[] argv;
#endif

  GTLCore::String errorMessage;
  llvm::LLVMContext& dummyContext = llvm::getGlobalContext(); // TODO someday (llvm 2.7 ?) use own context
  llvm::InitializeNativeTarget();
  
  llvm::CodeGenOpt::Level OLvl = llvm::CodeGenOpt::Default;
  switch (Optimiser::instance()->d->level()) {
    case 0: OLvl = llvm::CodeGenOpt::None; break;
    case 3: OLvl = llvm::CodeGenOpt::Aggressive; break;
    case 1:
    case 2:
    default:
      OLvl = llvm::CodeGenOpt::Default; break;
  }
  
  
  d->executionEngine = llvm::ExecutionEngine::createJIT( new llvm::ExistingModuleProvider( new llvm::Module("dummy", dummyContext ) ), &errorMessage);
//   d->executionEngine->DisableLazyCompilation(true); // does not work until llvm 2.7, which might have that by default anyway
  GTL_DEBUG("Error while creating execution engine : " << errorMessage);
  GTL_ASSERT(d->executionEngine);
}

VirtualMachine::~VirtualMachine()
{
  delete d->executionEngine;
  delete d;
}

void VirtualMachine::registerModule( llvm::ModuleProvider* mp)
{
  GTL_ASSERT(d->executionEngine);
  d->executionEngine->addModuleProvider(mp);
}

void VirtualMachine::unregisterModule( llvm::ModuleProvider* mp)
{
  if(d->executionEngine)
  {
    llvm::Module* module = mp->getModule();
    d->executionEngine->removeModuleProvider(mp);
    for( llvm::Module::FunctionListType::iterator it = module->getFunctionList().begin();
         it != module->getFunctionList().end(); ++it)
    {
//      d->executionEngine->freeMachineCodeForFunction(it);
      d->executionEngine->updateGlobalMapping( it, 0);
    }
    for( llvm::Module::GlobalListType::iterator it = module->getGlobalList().begin();
         it != module->getGlobalList().end(); ++it)
    {
      d->executionEngine->updateGlobalMapping( it, 0);
    }
  }
}

void *VirtualMachine::getPointerToFunction(llvm::Function *F)
{
  return d->executionEngine->getPointerToFunction(F);
}

void *VirtualMachine::getPointerToFunction(Function *F)
{
  return getPointerToFunction( F->d->data->function( ) );
}

void* VirtualMachine::getGlobalVariablePointer( llvm::GlobalVariable* GV)
{
    return d->executionEngine->getPointerToGlobal( GV );
}

#if 0
void VirtualMachine::setGlobalVariable( llvm::GlobalVariable *GV, const GTLCore::Value& value)
{
  void* ptr = d->executionEngine->getPointerToGlobal( GV );
  if( GV->getType()->getElementType() == llvm::Type::Int32Ty )
  {
    *(int*)ptr = value.asInt32();
  } else if( GV->getType()->getElementType() == llvm::Type::Int1Ty )
  {
    *(bool*)ptr = value.asBoolean();
  } else if( GV->getType()->getElementType() == llvm::Type::FloatTy )
  {
    GTL_DEBUG("Set " << value.asFloat() << " on ptr " << ptr << " from value = " << *(float*)ptr);
    *(float*)ptr = value.asFloat();
    GTL_DEBUG( *(float*)ptr );
  }
}

GTLCore::Value VirtualMachine::getGlobalVariable( llvm::GlobalVariable *GV)
{
  void* ptr = d->executionEngine->getPointerToGlobal( GV );
  if( GV->getType()->getElementType() == llvm::Type::Int32Ty )
  {
    return GTLCore::Value( *(int*)ptr);
  }
  if( GV->getType()->getElementType() == llvm::Type::Int1Ty )
  {
    return GTLCore::Value( *(bool*)ptr);
  }
  if( GV->getType()->getElementType() == llvm::Type::FloatTy )
  {
    return GTLCore::Value( *(float*)ptr);
  }
  return GTLCore::Value();
}
#endif

VirtualMachine* VirtualMachine::instance()
{
  if( not Private::virtualMachine)
  {
    Private::virtualMachine = new VirtualMachine;
  }
  return Private::virtualMachine;
}

llvm::ExecutionEngine* VirtualMachine::executionEngine()
{
  return d->executionEngine;
}
