/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "VariableNG_p.h"

#include <llvm/Constants.h>
#include <llvm/Instructions.h>

#include "CodeGenerator_p.h"
#include "Debug.h"
#include "ExpressionResult_p.h"
#include "GenerationContext_p.h"
#include "Type.h"
#include "Type_p.h"
#include "Visitor_p.h"

using namespace GTLCore;


struct VariableNG::Private
{
  const Visitor* visitor;
  const GTLCore::Type* type;
  bool constant;
  bool isInitialised;
  llvm::Value* pointer;
  bool directlyOnTheStack;
  bool allocatedInMemory;
  bool constantPointer;
  bool dependant;
#ifdef OPENGTL_ENABLE_DEBUG_OUTPUT
  bool dependantWasChangedToConst;
#endif
};

VariableNG::VariableNG(const GTLCore::Type* _type, bool _constant, bool _dependant ) : d(new Private)
{
  d->type = _type;
  d->constant = _constant;
  d->isInitialised = false;
  d->visitor = Visitor::getVisitorFor( _type );
  d->pointer = 0;
  d->constantPointer = false;
  d->allocatedInMemory = false;
  setAllocatedInMemory( true ); // Currently it's not possible to not allocate array/structure in memory until function return a structure and not a pointer, and structure/arrays knows wether they are allocated on stack or not
  d->dependant = _dependant;
#ifdef OPENGTL_ENABLE_DEBUG_OUTPUT
  d->dependantWasChangedToConst = false;
#endif
}

VariableNG::~VariableNG()
{
  delete d;
}
bool VariableNG::constant() const
{
  return d->constant;
}

void VariableNG::setConstant( bool v )
{
  GTL_ASSERT( d->dependant );
#ifdef OPENGTL_ENABLE_DEBUG_OUTPUT
  if( not v )
  {
    GTL_ASSERT( not d->dependantWasChangedToConst );
  }
  d->dependantWasChangedToConst = true;
#endif
  d->constant = v;
}

const GTLCore::Type* VariableNG::type() const
{
  return d->type;
}

llvm::BasicBlock* VariableNG::initialise( GTLCore::GenerationContext& _generationContext, llvm::BasicBlock* _bb, const ExpressionResult& _initialiser, const std::list<llvm::Value*>& _initialSize )
{
  GTL_DEBUG( _initialSize.size() );
  llvm::Value* pointer_ = 0;
  if( _initialiser.value() and _initialiser.functionResult()
      and (type()->dataType() == Type::ARRAY or type()->dataType() == Type::STRUCTURE ) )
  {
    initialise( _generationContext, _bb, _initialiser.value() );
    _bb = d->visitor->mark( _generationContext, _bb, pointer(_bb), d->type, CodeGenerator::integerToConstant( _generationContext.llvmContext(), 1 ) );
  } else {
    if( d->allocatedInMemory )
    {
      pointer_ = CodeGenerator::allocateMemory(_generationContext, d->type->d->type(_generationContext.llvmContext()), CodeGenerator::integerToConstant(_generationContext.llvmContext(), 1), _bb);
    } else {
      pointer_ = new llvm::AllocaInst( d->type->d->type(_generationContext.llvmContext()), llvm::ConstantInt::get(llvm::Type::getInt32Ty(_generationContext.llvmContext()), 1), "Variable", _bb);
    }
    initialise( _generationContext, _bb, pointer_ );
    _bb = d->visitor->initialise( _generationContext, _bb, pointer_, d->type, _initialSize, d->allocatedInMemory);
    
    if( _initialiser.value() )
    {
      _bb = d->visitor->set( _generationContext, _bb, pointer_, d->type, _initialiser.value(), _initialiser.type() , d->allocatedInMemory);
    }
  }
  d->constantPointer = false;
  return _bb;
}

void VariableNG::initialise( GTLCore::GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer )
{
  GTL_ASSERT( _pointer->getType()->getTypeID() == llvm::Type::PointerTyID );
  GTL_ASSERT(not d->isInitialised);
  d->isInitialised = true;
  d->directlyOnTheStack = _currentBlock and ( d->type->dataType() == Type::ARRAY or d->type->dataType() == Type::STRUCTURE );
  if( d->directlyOnTheStack )
  {
    d->pointer = new llvm::AllocaInst( llvm::PointerType::get( d->type->d->type(_generationContext.llvmContext()), 0 ), "Variable Pointer", _currentBlock );
    GTL_DEBUG( *d->pointer << " " << *_pointer );
    new llvm::StoreInst( _pointer, d->pointer, "Variable Pointer", _currentBlock );
  } else {
    d->pointer = _pointer;
  }
  d->constantPointer = true;
}

llvm::Value* VariableNG::get(GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock)
{
  llvm::Value* v = d->visitor->get( _generationContext, _currentBlock, d->pointer, d->type ).value();
  return v;
}

llvm::BasicBlock* VariableNG::set(GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _value, const GTLCore::Type* _valueType )
{
  GTL_ASSERT(not constant() );
  return d->visitor->set( _generationContext, _currentBlock, d->pointer, d->type, _value, _valueType, d->allocatedInMemory);
}

llvm::BasicBlock* VariableNG::replacePointer( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer)
{
  GTL_ASSERT( _pointer->getType() == d->pointer->getType() );
  GTL_ASSERT( not d->constantPointer );
  GTL_ASSERT( d->allocatedInMemory );
  _currentBlock = cleanUp( _generationContext, _currentBlock, 0 );
  d->pointer = _pointer;
  _currentBlock = d->visitor->mark( _generationContext, _currentBlock, d->pointer, d->type, CodeGenerator::integerToConstant( _generationContext.llvmContext(), 1 ) );
  return _currentBlock;
}

llvm::Value* VariableNG::pointer(llvm::BasicBlock* _currentBlock)
{
  GTL_DEBUG( *d->pointer << " " << d->directlyOnTheStack );
  if( d->directlyOnTheStack )
  {
    return new llvm::LoadInst( d->pointer, "Load Variable Pointer", _currentBlock );
  } else {
    return d->pointer;
  }
}

bool VariableNG::allocatedInMemory() const
{
  return d->allocatedInMemory;
}

void VariableNG::setAllocatedInMemory( bool _allocated )
{
  if( d->type->dataType() == Type::ARRAY or d->type->dataType() == Type::STRUCTURE )
  {
    d->allocatedInMemory = _allocated;
  }
}


bool VariableNG::constantPointer() const
{
  return d->constantPointer;
}

llvm::BasicBlock* VariableNG::cleanUp( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _donttouch )
{
  if( d->allocatedInMemory and not d->constantPointer )
  {
    llvm::Value* pointer_ = pointer( _currentBlock);
    _currentBlock = d->visitor->mark( _generationContext, _currentBlock, pointer_, d->type, CodeGenerator::integerToConstant( _generationContext.llvmContext(), -1 ) );
    // pointer_ is deallocated inside cleanup
    _currentBlock = d->visitor->cleanUp( _generationContext, _currentBlock, pointer_, d->type, _donttouch, d->allocatedInMemory, true, true );
  }
  return _currentBlock;
}
