/*
 *  Copyright (c) 2007-2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "ExpressionResult_p.h"

#include "llvm/Constant.h"
#include "llvm/DerivedTypes.h"

#include "Type.h"
#include "Type_p.h"

#include "Debug.h"

using namespace GTLCore;

struct ExpressionResult::Private
{
  union {
    llvm::Value* value;
    llvm::Constant* constant;
  } valueOrConstant;
  bool isConstant;
  bool functionResult;
  const GTLCore::Type* type;
};

ExpressionResult::ExpressionResult() : d(new Private)
{
  d->valueOrConstant.value = 0;
  d->isConstant = false;
  d->functionResult = false;
  d->type = Type::Undefined;
}

ExpressionResult::ExpressionResult(llvm::Value* _value, const GTLCore::Type* _type, bool _functionResult) : d(new Private)
{
  GTL_ASSERT( _value );
  GTL_ASSERT( _type );
  GTL_DEBUG( *_value->getType() << " " << *_type->d->type(_value->getContext()) << " " << *_type->d->asArgumentType(_value->getContext()) );
  GTL_ASSERT( _value->getType() == _type->d->type(_value->getContext()) or _value->getType() == _type->d->asArgumentType(_value->getContext()) );
  d->valueOrConstant.value = _value;
  d->isConstant = false;
  d->functionResult = _functionResult;
  d->type = _type;
}

ExpressionResult::ExpressionResult(llvm::Constant* _constant, const GTLCore::Type* _type, bool _functionResult) : d(new Private)
{
  GTL_ASSERT( _constant );
  GTL_ASSERT( _type );
  GTL_DEBUG( *_constant->getType() << " " << *_type );
  GTL_ASSERT( _constant->getType() == _type->d->type(_constant->getContext())
           or _constant->getType() == _type->d->asArgumentType(_constant->getContext())
           or llvm::isa<llvm::ArrayType>(_constant->getType()) );
  d->valueOrConstant.constant = _constant;
  d->isConstant = true;
  d->functionResult = _functionResult;
  d->type = _type;
}

ExpressionResult::ExpressionResult(const ExpressionResult& rhs ) : d(new Private(*rhs.d))
{
}

ExpressionResult ExpressionResult::operator=(const ExpressionResult& rhs )
{
  *d = *rhs.d;
  return *this;
}

ExpressionResult::~ExpressionResult()
{
  delete d;
}

bool ExpressionResult::isConstant() const
{
  return d->isConstant;
}

llvm::Value* ExpressionResult::value() const
{
  return d->valueOrConstant.value;
}

void ExpressionResult::setValue( llvm::Value* v)
{
  d->valueOrConstant.value = v;
  d->isConstant = false;
}

llvm::Constant* ExpressionResult::constant() const
{
  GTL_ASSERT( dynamic_cast<llvm::Constant*>(d->valueOrConstant.value));
  if(d->isConstant) return d->valueOrConstant.constant;
  return 0;
}

void ExpressionResult::setConstant( llvm::Constant* v)
{
  d->valueOrConstant.constant = v;
  d->isConstant = true;
}

const GTLCore::Type* ExpressionResult::type() const
{
  return d->type;
}

bool ExpressionResult::functionResult() const
{
  return d->functionResult;
}
