/*
 *  Copyright (c) 2007-2009 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "CodeGenerator_p.h"

#include <llvm/CallingConv.h>
#include <llvm/Constants.h>
#include <llvm/Function.h>
#include <llvm/Instructions.h>
#include <llvm/Module.h>
#include <llvm/Value.h>

#include "Debug.h"
#include "ExpressionResult_p.h"
#include "Function_p.h"
#include "GenerationContext_p.h"
#include "Macros_p.h"
#include "Parameter.h"
#include "Type.h"
#include "Type_p.h"
#include "Value.h"
#include "VariableNG_p.h"
#include "ModuleData_p.h"
#include "TypesManager_p.h"
#include "ExpressionGenerationContext_p.h"

#include "AST/AccessorExpression.h"
#include "wrappers/ArrayWrap.h"
#include "MemoryModelConfig_p.h"
#include "GenerationContext_p.h"
#include "Visitor_p.h"
#include "Visitor_p.h"

#define UNIFORMIZE_TYPES( _v1_, _v2_) \
  GTL_ASSERT( _v1_.value() ); \
  GTL_ASSERT( _v2_.value() ); \
  GTL_CHECK_EQUAL( lhsType, rhsType); \
  GTLCore::ExpressionResult v1 = _v1_; \
  GTLCore::ExpressionResult v2 = _v2_; \
  GTL_ASSERT( v1.value() ); \
  GTL_ASSERT( v2.value() );
  
#define UNIFORMIZE_VALUE_TYPES( _v1_, _v2_ ) \
  GTL_CHECK_EQUAL( lhsType, rhsType ); \
  llvm::Value* v1 = _v1_; \
  llvm::Value* v2 = _v2_; \
  GTL_ASSERT( v1 ); \
  GTL_ASSERT( v2 );

#define UNIFORMIZE_CONSTANT_TYPES( _v1_, _v2_ ) \
  GTL_CHECK_EQUAL( lhsType, rhsType ); \
  llvm::Constant* v1 = _v1_; \
  llvm::Constant* v2 = _v2_; \
  GTL_ASSERT( v1 ); \
  GTL_ASSERT( v2 );

#define MAKE_VALUE_BOOLEAN( _v1_, _v2_ ) \
  llvm::Value* v1 = convertValueTo( currentBlock, _v1_, llvm::IntegerType::get(1)); \
  llvm::Value* v2 = convertValueTo( currentBlock, _v2_, llvm::IntegerType::get(1)); \
  GTL_ASSERT( v1 ); \
  GTL_ASSERT( v2 );

#define MAKE_CONSTANT_BOOLEAN( _v1_, _v2_ ) \
  llvm::Constant* v1 = convertConstantTo( _v1_, llvm::IntegerType::get(1)); \
  llvm::Constant* v2 = convertConstantTo( _v2_, llvm::IntegerType::get(1)); \
  GTL_ASSERT( v1 ); \
  GTL_ASSERT( v2 );

using namespace GTLCore;

struct CodeGenerator::Private {
  ModuleData* module;
};

CodeGenerator::CodeGenerator(ModuleData* module) : d(new Private)
{
  d->module = module;
}

CodeGenerator::~CodeGenerator()
{
  delete d;
}

llvm::Constant* CodeGenerator::integerToConstant(llvm::LLVMContext& _llvmContext, int v)
{
  return llvm::ConstantInt::get(llvm::Type::getInt32Ty(_llvmContext), v);
}

llvm::Constant* CodeGenerator::boolToConstant(llvm::LLVMContext& _llvmContext, bool v)
{
  return llvm::ConstantInt::get(llvm::IntegerType::getInt1Ty( _llvmContext), v);
}

llvm::Constant* CodeGenerator::floatToConstant(llvm::LLVMContext& _llvmContext, float v)
{
  return llvm::ConstantFP::get(_llvmContext, llvm::APFloat(v));
}

llvm::Constant* CodeGenerator::nullPointerToConstant(llvm::LLVMContext& _llvmContext)
{
  return llvm::ConstantPointerNull::get( static_cast<const llvm::PointerType*>( GTLCore::Type::Pointer->d->type(_llvmContext) ) );
}

llvm::Constant* CodeGenerator::valueToConstant( GenerationContext& _gc, const GTLCore::Value& v)
{
  const Type* type = v.type();
  if( type == GTLCore::Type::Integer32 or type == GTLCore::Type::UnsignedInteger32 )
  {
    return integerToConstant( _gc.llvmContext(), v.asInt32() );
  } else if( type == GTLCore::Type::Float ) {
    return floatToConstant( _gc.llvmContext(), v.asFloat() );
  } else if( type == GTLCore::Type::Boolean ) {
    return boolToConstant( _gc.llvmContext(), v.asBoolean() );
  } else if( type->dataType() == Type::ARRAY or type->dataType() == Type::VECTOR or type->isStructure() ) {
    std::vector< llvm::Constant* > members;
    GTL_ASSERT(v.asArray());
    if(type->isStructure())
    {
      members.push_back( CodeGenerator::integerToConstant( _gc.llvmContext(), 0 ) ); // GC constant      
    }
    foreach( const GTLCore::Value& val, *v.asArray())
    {
      members.push_back(valueToConstant(_gc, val));
      GTL_ASSERT( (type->dataType() != Type::ARRAY and type->dataType() != Type::VECTOR) or val.type() == type->embeddedType());
      GTL_ASSERT( not type->isStructure() or val.type() == type->structDataMember(members.size() - 1 - STRUCT_FIRST_ELEMENT).type() );
    }
    if( type->dataType() == Type::ARRAY )
    {
      return CodeGenerator::constantsToArray(_gc, members, v.type());
    } else if( type->dataType() == Type::VECTOR )
    {
      return CodeGenerator::constantsToVector(_gc, members, v.type());      
    } else {
      GTL_ASSERT(v.type()->isStructure());
      return CodeGenerator::constantsToStructure(_gc, members, v.type());
    }
  } else {
    GTL_ABORT("Unimplemented");
  }
}

llvm::Constant* CodeGenerator::constantsToArray( GenerationContext& _gc, const std::vector< llvm::Constant* >& _constants, const GTLCore::Type* _type )
{
  const GTLCore::Type* arrayType = _type->embeddedType();
  std::vector<llvm::Constant*> arrayStruct;
  arrayStruct.push_back( _gc.codeGenerator()->integerToConstant( _gc.llvmContext(), 0 ) );
  arrayStruct.push_back( _gc.codeGenerator()->integerToConstant( _gc.llvmContext(), _constants.size() ) );
  llvm::Constant* constant = llvm::ConstantArray::get(
                llvm::ArrayType::get( arrayType->d->type(_gc.llvmContext()), _constants.size()), _constants );
  llvm::GlobalVariable* gvar = new llvm::GlobalVariable( *_gc.llvmModule(), constant->getType(), true, llvm::GlobalValue::ExternalLinkage, constant, "" );
  
  llvm::Constant* zero = llvm::ConstantInt::get(llvm::Type::getInt32Ty(_gc.llvmContext()), 0); // Access the structure of the array
  llvm::Constant *Ops[] = {zero, zero };
  llvm::Constant* iptr = llvm::ConstantExpr::getGetElementPtr( gvar, Ops, 2);
  
  arrayStruct.push_back( iptr );
  
  const llvm::StructType* structType = dynamic_cast<const llvm::StructType*>( _type->d->type(_gc.llvmContext()) );
  GTL_ASSERT( structType );
  return llvm::ConstantStruct::get( structType, arrayStruct );
}

llvm::Constant* CodeGenerator::constantsToStructure( GenerationContext& _gc, const std::vector< llvm::Constant* >& _constants, const GTLCore::Type* _type )
{
  GTL_ASSERT(_constants.size() == _type->countStructDataMembers() + STRUCT_FIRST_ELEMENT);
  const llvm::StructType* structType = dynamic_cast<const llvm::StructType*>( _type->d->type(_gc.llvmContext()) );
  return llvm::ConstantStruct::get( structType, _constants );
}

llvm::Constant* CodeGenerator::constantsToVector( GenerationContext& _gc, const std::vector< llvm::Constant* >& _constants, const GTLCore::Type* _type )
{
  const llvm::VectorType* vectorType = dynamic_cast<const llvm::VectorType*>( _type->d->type(_gc.llvmContext()) );
  GTL_ASSERT( vectorType );
  GTL_ASSERT( not _constants.empty());
  return llvm::ConstantVector::get( vectorType, _constants );
}

GTLCore::Value CodeGenerator::constantToValue(llvm::Constant* constant, const GTLCore::Type* _type)
{
  if( constant->getType() == llvm::Type::getInt32Ty(constant->getContext() ) )
  {
    GTL_ASSERT(_type == GTLCore::Type::Integer32 or _type == GTLCore::Type::UnsignedInteger32 );
    return (int)dynamic_cast<llvm::ConstantInt* >( constant )->getValue().getLimitedValue();
  } else if ( constant->getType() == llvm::Type::getFloatTy(constant->getContext()) )
  {
    GTL_ASSERT(_type == GTLCore::Type::Float32 or _type == GTLCore::Type::Float16 );
    return dynamic_cast<llvm::ConstantFP* >( constant )->getValueAPF().convertToFloat();
  } else if ( constant->getType() == llvm::Type::getInt1Ty(constant->getContext()) ) {
    GTL_ASSERT(_type == GTLCore::Type::Boolean );
    return (bool)(dynamic_cast<llvm::ConstantInt* >( constant )->getValue().getLimitedValue());
  } else if( constant->getType()->getTypeID() == llvm::Type::StructTyID ) {
    llvm::ConstantStruct* cs = dynamic_cast<llvm::ConstantStruct* >( constant );
    switch(_type->dataType())
    {
      case Type::STRUCTURE:
      {
        GTL_ASSERT(_type->countStructDataMembers() + STRUCT_FIRST_ELEMENT == cs->getNumOperands() );
        std::vector< GTLCore::Value > values;
        for(std::size_t i = STRUCT_FIRST_ELEMENT; i < cs->getNumOperands(); ++i)
        {
          values.push_back(constantToValue( cs->getOperand(i), _type->structDataMember(i - 1).type() ) );
        }
        return GTLCore::Value(values, _type);
      }
      case Type::ARRAY:
      {
        GTL_ASSERT( cs->getNumOperands() == 2 + STRUCT_FIRST_ELEMENT );
        GTL_ASSERT( cs->getOperand(1)->getType() == llvm::Type::getInt32Ty(constant->getContext()) );
        std::size_t sizeArray = dynamic_cast<llvm::ConstantInt* >( cs->getOperand(1))->getValue().getLimitedValue();
        llvm::ConstantExpr* expr = dynamic_cast<llvm::ConstantExpr*>(cs->getOperand(2));
        GTL_ASSERT(expr);
        llvm::GlobalVariable* arrayGV = dynamic_cast<llvm::GlobalVariable*>(expr->getOperand(0));
        GTL_ASSERT(arrayGV);
        llvm::ConstantArray* array = dynamic_cast<llvm::ConstantArray*>(arrayGV->getInitializer());
        GTL_ASSERT(array);
        GTL_ASSERT(array->getNumOperands() == sizeArray);
        std::vector< GTLCore::Value > values;
        for(std::size_t i = 0; i < sizeArray; ++i)
        {
          values.push_back(constantToValue( array->getOperand(i), _type->embeddedType() ) );  
        }
        return GTLCore::Value(values, _type);
      }
      default:
        GTL_ABORT("Unsupported type");
    }
  } else if( constant->getType()->getTypeID() == llvm::Type::PointerTyID) {
    return constantToValue( dynamic_cast<llvm::GlobalVariable*>(constant)->getInitializer() , _type);
  } else {
    GTL_ABORT("Unimplemented");
  }
}

llvm::Value* CodeGenerator::createVector( llvm::LLVMContext& _llvmContext, const GTLCore::Type* _vecType)
{
  llvm::Constant* _const = 0;
  switch( _vecType->embeddedType()->dataType() )
  {
    case Type::UNSIGNED_INTEGER32:
    case Type::INTEGER32:
      _const = integerToConstant( _llvmContext, 0 );
      break;
    case Type::FLOAT:
    case Type::HALF:
      _const = floatToConstant( _llvmContext, 0.0 );
      break;
    default:  
      break;
  }
  GTL_ASSERT( _const );
  return createVector( _vecType, _const );
}

llvm::Value* CodeGenerator::createVector( const GTLCore::Type* _vecType, llvm::Constant* _val)
{
  GTL_ASSERT( _vecType->dataType() == Type::VECTOR );
  std::vector<llvm::Constant*> inits;
  for( std::size_t i = 0; i < _vecType->vectorSize(); ++i)
  {
    inits.push_back( _val );
  }
  GTL_ASSERT( not inits.empty());
  return llvm::ConstantVector::get( static_cast<const llvm::VectorType*>( _vecType->d->type( _val->getContext() )), inits );
}

llvm::Value* CodeGenerator::createVector( int _size, llvm::Constant* _val, const GTLCore::Type* _valType)
{
  std::vector<llvm::Constant*> inits;
  for( int i = 0; i < _size; ++i)
  {
    inits.push_back( _val );
  }
  GTL_ASSERT( not inits.empty());
  return llvm::ConstantVector::get( static_cast<const llvm::VectorType*>( GTLCore::TypesManager::getVector( _valType, _size)->d->type( _val->getContext() )), inits );
}

llvm::Value* CodeGenerator::convertPointerToCharP(llvm::BasicBlock* _currentBlock, llvm::Value* _value)
{
  return convertPointerTo( _currentBlock, _value, llvm::Type::getInt8Ty(_value->getContext()) );
}

llvm::Value* CodeGenerator::convertPointerTo(llvm::BasicBlock* _currentBlock, llvm::Value* _value, const llvm::Type* _type)
{
  return new llvm::BitCastInst( _value, llvm::PointerType::get( _type, 0 ), "", _currentBlock );
}

llvm::Value* CodeGenerator::convertValueTo(llvm::BasicBlock* _currentBlock, llvm::Value* _value, const Type* _valueType, const Type* _targetType)
{
  GTL_DEBUG("Convert value " << *_value << " from " << *_valueType << " to " << *_targetType);
  llvm::Constant* _constant = dynamic_cast<llvm::Constant*>(_value);
  if(_constant)
  {
    return convertConstantTo( _constant, _valueType, _targetType);
  }
  GTL_ASSERT(_value);
  GTL_ASSERT(_valueType);
  GTL_ASSERT(_targetType);
  if(_value->getType() == _targetType->d->type(_value->getContext()) )
  { // Don't cast the value as it has allready the correct type
    return _value;
  }
  if( _targetType->dataType() == Type::VECTOR and _valueType->dataType() != Type::VECTOR )
  {
    GTL_DEBUG("Convert value to a vector");
    // Create a vector
    llvm::Value* resultLoad = createVector( _targetType, floatToConstant( _value->getContext(), 0.0 ) );
    llvm::Value* vecElt = convertValueTo( _currentBlock, _value, _valueType, _targetType->embeddedType() );
    // Affect the same value to each element of the vector
    for(unsigned int i = 0; i < _targetType->vectorSize(); ++i)
    {
      resultLoad = llvm::InsertElementInst::Create( resultLoad, vecElt, integerToConstant(_value->getContext(), i), "", _currentBlock );
    }
    GTL_DEBUG("Done");
    return resultLoad;
  }
  GTL_DEBUG("Create cast instruction");
  GTL_DEBUG( *_value << " to "  << *_targetType );
  GTL_ASSERT(_targetType->d->type(_value->getContext())->isFirstClassType());
  GTL_ASSERT(_value->getType()->isFirstClassType());
  GTL_ASSERT( _targetType->dataType() != Type::ARRAY );
  GTL_ASSERT( _targetType->dataType() != Type::STRUCTURE );
  return llvm::CastInst::Create(
                      llvm::CastInst::getCastOpcode(_value, _valueType->isSigned(), _targetType->d->type(_value->getContext()), _targetType->isSigned()), _value, _targetType->d->type(_value->getContext()), "", _currentBlock );
}

llvm::Constant* CodeGenerator::convertConstantTo(llvm::Constant* _constant, const Type* _constantType, const Type* _targetType)
{
  const llvm::Type* dstType = _targetType->d->type(_constant->getContext());
  GTL_DEBUG( *_constant->getType() << " " << *dstType );
  if(_constant->getType() == dstType )
  { // Don't cast the value as it has allready the correct type
    return _constant;
  }
  if( _targetType->dataType() == Type::VECTOR and _constantType->dataType() != Type::VECTOR )
  {
    llvm::Constant* vecElt = convertConstantTo( _constant, _constantType, _targetType->embeddedType() );
    // Affect the same value to each element of the vector
    std::vector<llvm::Constant*> constants;
    for(unsigned int i = 0; i < _targetType->vectorSize(); ++i)
    {
      constants.push_back( vecElt );
    }
    // store back
    GTL_ASSERT( not constants.empty());
    return llvm::ConstantVector::get( llvm::cast<const llvm::VectorType>(dstType), constants );
  }
  GTL_ASSERT( _targetType->d->type(_constant->getContext())->isFirstClassType() );
  GTL_ASSERT( _constant->getType()->isFirstClassType() );
  GTL_DEBUG( "dstType = " << *dstType << " " << dstType << " srcType = " << *_constant->getType() << " " << _constant->getType());
  unsigned t = (unsigned)llvm::CastInst::getCastOpcode(_constant, _constantType->isSigned(), dstType, _targetType->isSigned());
  return llvm::ConstantExpr::getCast( t, _constant, dstType );
}



llvm::Value* CodeGenerator::convertToHalf( GenerationContext& _generationContext, llvm::BasicBlock* currentBlock, llvm::Value* value, const GTLCore::Type* _valueType )
{
  std::vector<const llvm::Type*> llvmArguments;
  llvmArguments.push_back( llvm::Type::getFloatTy(_generationContext.llvmContext()) );
  llvm::Function* func = dynamic_cast<llvm::Function*>( _generationContext.llvmModule()->getOrInsertFunction( "_Z10float2halff",
      llvm::FunctionType::get( llvm::Type::getInt16Ty(_generationContext.llvmContext()) , llvmArguments, false ) ) );
  std::vector<llvm::Value*> convertedParams;
  convertedParams.push_back( convertValueTo( currentBlock, value, _valueType, GTLCore::Type::Float ) );
  return llvm::CallInst::Create( func, convertedParams.begin(), convertedParams.end(), "", currentBlock );
}

llvm::Value* CodeGenerator::convertFromHalf( GenerationContext& _generationContext, llvm::BasicBlock* currentBlock, llvm::Value* value)
{
  std::vector<const llvm::Type*> llvmArguments;
  llvmArguments.push_back( llvm::Type::getInt16Ty(_generationContext.llvmContext()) );
  llvm::Function* func = dynamic_cast<llvm::Function*>( _generationContext.llvmModule()->getOrInsertFunction( "_Z10half2floatt",
      llvm::FunctionType::get( llvm::Type::getFloatTy(_generationContext.llvmContext()) , llvmArguments, false ) ) );
  std::vector<llvm::Value*> convertedParams;
  convertedParams.push_back( convertValueTo( currentBlock, value, GTLCore::Type::Half, GTLCore::Type::Integer16 ) );
  return llvm::CallInst::Create( func, convertedParams.begin(), convertedParams.end(), "", currentBlock );
  
}

llvm::Value* CodeGenerator::convertArrayToVector(GenerationContext& gc, ExpressionGenerationContext& _egc, llvm::Value* value, const Type* valueType, const Type* type)
{
  // int arraySize = value->size;
  // if( arraySize > type->vectorSize() ) arraySize = type->vectorSize();
  // 'type' result;
  // for(int i = 0; i < arraySize; ++i )
  // {
  //   result[i] = array[i];
  // }
  
  // int arraySize = value->size;
  llvm::Value* arraySize = accessArraySize(_egc.currentBasicBlock(), value);
  llvm::Value* vectorSize = integerToConstant( gc.llvmContext(), type->vectorSize() );
  llvm::Value* arraySizeAlloc = new llvm::AllocaInst( llvm::Type::getInt32Ty(gc.llvmContext()), "", _egc.currentBasicBlock() );
  
  // arraySize > type->vectorSize()
  llvm::Value* aStVComp = createStrictSupperiorExpression(_egc.currentBasicBlock(),
                                      arraySize, GTLCore::Type::Integer32, vectorSize, GTLCore::Type::Integer32);
  // arraySize = type->vectorSize
  llvm::BasicBlock* block1 = gc.createBasicBlock();
  new llvm::StoreInst( vectorSize, arraySizeAlloc, block1 );
  // arraySize = arraySize
  llvm::BasicBlock* block2 = gc.createBasicBlock();
  new llvm::StoreInst( arraySize, arraySizeAlloc, block2 );
  // if( arraySize > type->vectorSize() ) arraySize = type->vectorSize();
  llvm::BasicBlock* after = gc.createBasicBlock();
  createIfElseStatement( _egc.currentBasicBlock(), aStVComp, Type::Boolean, block1, block1, block2, block2, after);

  arraySize = new llvm::LoadInst( arraySizeAlloc, "", after );
  llvm::Value* result = createVector( type, floatToConstant( gc.llvmContext(), 0.0 ) );
  
  // int i = 0;
  VariableNG idx(GTLCore::Type::Integer32, false, false );
  idx.initialise( gc, after, 
                     GTLCore::ExpressionResult(integerToConstant( gc.llvmContext(), 0), GTLCore::Type::Integer32),
                     std::list<llvm::Value*>());

  //   result[i] = array[i];
  llvm::BasicBlock* forBlock = gc.createBasicBlock();
  llvm::Value* cidx = idx.get( gc, forBlock );
  llvm::Value* val = accessArrayValueNoClamp( forBlock, value, cidx );
  llvm::InsertElementInst::Create(result, val, cidx, "", forBlock );
  
  // for(int i = 0; i < arraySize; ++i )
  llvm::BasicBlock* afterFor = createIterationForStatement(gc, after, &idx, arraySize, GTLCore::Type::Integer32, forBlock, forBlock);
  
  _egc.setCurrentBasicBlock(afterFor);
  
  return result;
}

llvm::Constant* CodeGenerator::convertConstantArrayToVector(llvm::Constant* constant, const Type* constantType, const Type* type)
{
  GTL_DEBUG(*constant);
  GTL_DEBUG(*constant->getType());
  llvm::ConstantStruct* arrayConstant = dynamic_cast<llvm::ConstantStruct*>(constant->getOperand(0));
  GTL_ASSERT(arrayConstant);
  GTL_ASSERT(constantType->dataType() == Type::ARRAY );
  GTL_ASSERT(type->dataType() == Type::VECTOR );
  GTL_ASSERT(arrayConstant->getNumOperands() == 3 );
  llvm::Constant* values = arrayConstant->getOperand(2)->getOperand(0)->getOperand(0);
  GTL_DEBUG(*values);
  GTL_ASSERT(dynamic_cast<llvm::ConstantArray*>(values) );
  std::vector<llvm::Constant*> constants;
  for(unsigned int i = 0; i < values->getNumOperands(); ++i )
  {
    GTL_DEBUG(*values->getOperand(i));
    constants.push_back(values->getOperand(i));
    if( constants.size() == type->vectorSize() )
    {
      break;
    }
  }
  GTL_ASSERT( constants.size() == type->vectorSize() );
  GTL_ASSERT( not constants.empty());
  return llvm::ConstantVector::get( static_cast<const llvm::VectorType*>( type->d->type(constant->getContext()) ), constants);
}

llvm::Value* CodeGenerator::vectorValueAt( llvm::BasicBlock* _currentBlock, llvm::Value* _vector, llvm::Value* _index)
{
  return llvm::ExtractElementInst::Create( _vector, _index, "", _currentBlock );
}

llvm::Value* CodeGenerator::createRound( llvm::BasicBlock* _currentBlock, llvm::Value* _val )
{
  _val = GTLCore::CodeGenerator::createAdditionExpression( _currentBlock, _val, GTLCore::Type::Float,
                          GTLCore::CodeGenerator::floatToConstant( _val->getContext(), 0.5 ), GTLCore::Type::Float );
  return GTLCore::CodeGenerator::convertValueTo( _currentBlock, _val,
        GTLCore::Type::Float, GTLCore::Type::Integer32 );
}


void CodeGenerator::createIfStatement( llvm::BasicBlock* before, llvm::Value* test, const Type* testType, llvm::BasicBlock* firstAction, llvm::BasicBlock* lastAction, llvm::BasicBlock* after)
{
  GTL_ASSERT( test->getType() == testType->d->type(before->getContext()));
  if( not lastAction->getTerminator() )
  {
    llvm::BranchInst::Create( after, lastAction );
  }
  llvm::BranchInst::Create( firstAction, after, convertValueTo( before, test, testType, GTLCore::Type::Boolean ), before);
}

void CodeGenerator::createIfElseStatement( llvm::BasicBlock* before, llvm::Value* test, const Type* testType, llvm::BasicBlock* firstAction, llvm::BasicBlock* lastAction, llvm::BasicBlock* firstElseAction, llvm::BasicBlock* lastElseAction, llvm::BasicBlock* after)
{
  if( not lastAction->getTerminator() )
  {
      llvm::BranchInst::Create( after, lastAction );
  }
  llvm::BranchInst::Create( firstAction, firstElseAction, convertValueTo( before, test, testType, GTLCore::Type::Boolean ), before);
  if( not lastElseAction->getTerminator() )
  {
      llvm::BranchInst::Create( after, lastElseAction );
  }
}

void CodeGenerator::createForStatement(llvm::BasicBlock* before, llvm::BasicBlock* beginTest, llvm::BasicBlock* endTest, llvm::Value* testResult, const Type* testType, llvm::BasicBlock* update, llvm::BasicBlock* firstAction, llvm::BasicBlock* lastAction, llvm::BasicBlock* after)
{
  llvm::BranchInst::Create( beginTest, before );
  llvm::BranchInst::Create( firstAction, after, convertValueTo( before, testResult, testType, GTLCore::Type::Boolean ), endTest );
  llvm::BranchInst::Create( update, lastAction);
  llvm::BranchInst::Create( beginTest, update);
}

llvm::BasicBlock* CodeGenerator::createIterationForStatement( GenerationContext& generationContext, llvm::BasicBlock* before, GTLCore::VariableNG* variable, llvm::Value* maxValue, const Type* maxValueType, llvm::BasicBlock* firstAction, llvm::BasicBlock* lastAction)
{
  // i < size
  llvm::BasicBlock* forTestBlock = llvm::BasicBlock::Create( generationContext.llvmContext(), "forTestBlock");
  generationContext.llvmFunction()->getBasicBlockList().push_back( forTestBlock);
  llvm::Value* forTest = createStrictInferiorExpression(forTestBlock, variable->get( generationContext, forTestBlock ), variable->type(), maxValue, maxValueType);
  GTL_ASSERT(forTest);
  // i += pixelSize
  llvm::BasicBlock* updateBlock = llvm::BasicBlock::Create(generationContext.llvmContext(), "updateBlock");
  generationContext.llvmFunction()->getBasicBlockList().push_back( updateBlock);
  createIncrementExpression( generationContext, updateBlock, variable );
  // Exit block
  llvm::BasicBlock* finBlock = llvm::BasicBlock::Create(generationContext.llvmContext(), "finBlock");
  generationContext.llvmFunction()->getBasicBlockList().push_back( finBlock);
  // Create for statement
  createForStatement(before, forTestBlock, forTestBlock, forTest, Type::Boolean, updateBlock, firstAction, lastAction, finBlock);
  return finBlock;
}

void CodeGenerator::createWhileStatement( llvm::BasicBlock* before, llvm::BasicBlock* test, llvm::Value* testResult, const Type* testType, llvm::BasicBlock* firstAction, llvm::BasicBlock* lastAction,  llvm::BasicBlock* after)
{
  llvm::BranchInst::Create( test, before);
  llvm::BranchInst::Create( firstAction, after, convertValueTo( test, testResult, testType, GTLCore::Type::Boolean ), test);
  llvm::BranchInst::Create( test, lastAction );

}


llvm::Function* CodeGenerator::createFunction( llvm::FunctionType* type, const GTLCore::String& name)
{
  return llvm::Function::Create( type, llvm::GlobalValue::ExternalLinkage,
                  name, d->module->llvmModule() );
}

void CodeGenerator::createIncrementExpression( GenerationContext& gc, llvm::BasicBlock* currentBlock, VariableNG* var)
{
  var->set( gc, currentBlock, createAdditionExpression( currentBlock, var->get(gc, currentBlock), var->type(), integerToConstant(gc.llvmContext(), 1), Type::Integer32), var->type());
}

void CodeGenerator::createIncrementExpression( llvm::BasicBlock* _currentBlock, llvm::Value* _pointer)
{
  new llvm::StoreInst(
      createAdditionExpression( _currentBlock,
                                new llvm::LoadInst( _pointer, "", _currentBlock ),
                                Type::Integer32, integerToConstant( _pointer->getContext(), 1), Type::Integer32 ),
                      _pointer, "", _currentBlock );
}

void CodeGenerator::createDecrementExpression( GenerationContext& gc, llvm::BasicBlock* currentBlock, VariableNG* var)
{
  var->set(gc, currentBlock, createSubstractionExpression( currentBlock, var->get(gc, currentBlock), var->type(), integerToConstant( gc.llvmContext(),1), Type::Integer32), var->type());
}

void CodeGenerator::createDecrementExpression( GenerationContext& gc, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer)
{
  new llvm::StoreInst(
      createSubstractionExpression( _currentBlock,
                                new llvm::LoadInst( _pointer, "", _currentBlock ),
                                Type::Integer32, integerToConstant( gc.llvmContext(), 1), Type::Integer32 ),
                                _pointer, "", _currentBlock );
}

#define CREATE_BINARY_OPERATOR( _creator_, _v1_, _v2_) \
  if( _v1_.isConstant() and _v2_.isConstant() ) \
  { \
    return GTLCore::ExpressionResult( _creator_(_v1_.constant(), lhsType, _v2_.constant(), rhsType ), lhsType); \
  } else { \
    return GTLCore::ExpressionResult( _creator_( currentBlock, _v1_.value(), lhsType, _v2_.value(), rhsType ), lhsType); \
  }

#define CREATE_BINARY_EXPRESSION(_GTL_Function_Name_, _LLVM_Constant_Expr_, _LLVM_Binary_Operator ) \
  llvm::Value* CodeGenerator::_GTL_Function_Name_(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const Type* lhsType, llvm::Value* rhs, const Type* rhsType) \
  { \
    UNIFORMIZE_VALUE_TYPES( lhs, rhs ) \
    return llvm::BinaryOperator::_LLVM_Binary_Operator( v1, v2, #_GTL_Function_Name_, currentBlock ); \
  } \
  llvm::Constant* CodeGenerator::_GTL_Function_Name_( llvm::Constant* lhs, const Type* lhsType, llvm::Constant* rhs, const Type* rhsType) \
  { \
    UNIFORMIZE_CONSTANT_TYPES( lhs, rhs ) \
    return llvm::ConstantExpr::_LLVM_Constant_Expr_( v1, v2 ); \
  } \
  ExpressionResult CodeGenerator::_GTL_Function_Name_( llvm::BasicBlock* currentBlock, ExpressionResult lhs, ExpressionResult rhs) \
  { \
    const Type* lhsType = lhs.type(); \
    const Type* rhsType = rhs.type(); \
    GTL_DEBUG( "lhs = " << lhs << " lhsType = " << *lhsType << " rhs = " << rhs << " rhsType = " << *rhsType ); \
    UNIFORMIZE_TYPES(lhs, rhs); \
    CREATE_BINARY_OPERATOR(  _GTL_Function_Name_, v1, v2); \
  }

CREATE_BINARY_EXPRESSION(createAdditionExpression, getAdd, CreateAdd)
CREATE_BINARY_EXPRESSION(createSubstractionExpression, getSub, CreateSub)
CREATE_BINARY_EXPRESSION(createMultiplicationExpression, getMul, CreateMul)
CREATE_BINARY_EXPRESSION(createRightShiftExpression, getAShr, CreateAShr)
CREATE_BINARY_EXPRESSION(createLeftShiftExpression, getShl, CreateShl)


llvm::Value* CodeGenerator::createDivisionExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const Type* lhsType, llvm::Value* rhs, const Type* rhsType)
{
  UNIFORMIZE_VALUE_TYPES( lhs, rhs );
  if( v1->getType()->isFloatingPoint()
      or ( lhsType->dataType() == GTLCore::Type::VECTOR
           and lhsType->embeddedType()->dataType() == GTLCore::Type::FLOAT ) )
  {
    return llvm::BinaryOperator::CreateFDiv(v1, v2, "", currentBlock );
  } else {
    return llvm::BinaryOperator::CreateSDiv(v1, v2, "", currentBlock );
  }
}

llvm::Constant* CodeGenerator::createDivisionExpression( llvm::Constant* lhs, const Type* lhsType, llvm::Constant* rhs, const Type* rhsType)
{
  UNIFORMIZE_CONSTANT_TYPES( lhs, rhs );
  if( v1->getType()->isFloatingPoint() )
  {
    return llvm::ConstantExpr::getFDiv( v1, v2);
  } else {
    return llvm::ConstantExpr::getSDiv( v1, v2);
  }
}

ExpressionResult CodeGenerator::createDivisionExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, ExpressionResult rhs)
{
  const Type* lhsType = lhs.type();
  const Type* rhsType = rhs.type();
  UNIFORMIZE_TYPES(lhs, rhs);
  CREATE_BINARY_OPERATOR(  createDivisionExpression, v1, v2);
}

llvm::Value* CodeGenerator::createModuloExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const Type* lhsType, llvm::Value* rhs, const Type* rhsType)
{
  UNIFORMIZE_VALUE_TYPES( lhs, rhs );
  return llvm::BinaryOperator::CreateSRem( v1, v2, "", currentBlock );
}

llvm::Constant* CodeGenerator::createModuloExpression( llvm::Constant* lhs, const Type* lhsType, llvm::Constant* rhs, const Type* rhsType)
{
  UNIFORMIZE_CONSTANT_TYPES( lhs, rhs );
  return llvm::ConstantExpr::getSRem( v1, v2);
}

ExpressionResult CodeGenerator::createModuloExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, const Type* lhsType, ExpressionResult rhs, const Type* rhsType)
{
  UNIFORMIZE_TYPES(lhs, rhs);
  CREATE_BINARY_OPERATOR(  createModuloExpression, v1, v2);
}


#define CREATE_BINARY_BOOLEAN_EXPRESSION(_GTL_Function_Name_, _LLVM_Constant_Expr_, _LLVM_Binary_Operator ) \
  llvm::Value* CodeGenerator::_GTL_Function_Name_(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const Type* lhsType, llvm::Value* rhs, const Type* rhsType) \
  { \
    MAKE_BOOLEAN_VALUE( lhs, rhs ) \
    return llvm::BinaryOperator::_LLVM_Binary_Operator( v1, v2, "", currentBlock ); \
  } \
  llvm::Constant* CodeGenerator::_GTL_Function_Name_( llvm::Constant* lhs, const Type* lhsType, llvm::Constant* rhs, const Type* rhsType) \
  { \
    MAKE_BOOLEAN_CONSTANT( lhs, rhs ) \
    return llvm::ConstantExpr::_LLVM_Constant_Expr_( v1, v2 ); \
  } \
  ExpressionResult CodeGenerator::_GTL_Function_Name_( llvm::BasicBlock* currentBlock, ExpressionResult lhs, const Type* lhsType, ExpressionResult rhs, const Type* rhsType) \
  { \
    UNIFORMIZE_TYPES(lhs, rhs); \
    CREATE_BINARY_OPERATOR(  _GTL_Function_Name_, v1, v2); \
  }

CREATE_BINARY_EXPRESSION(createOrExpression, getOr, CreateOr)
CREATE_BINARY_EXPRESSION(createAndExpression, getAnd, CreateAnd)

#define CREATE_BINARY_INTEGER_EXPRESSION(_GTL_Function_Name_, _LLVM_Constant_Expr_, _LLVM_Binary_Operator ) \
  llvm::Value* CodeGenerator::_GTL_Function_Name_(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const Type* lhsType, llvm::Value* rhs, const Type* rhsType) \
  { \
    UNIFORMIZE_VALUE_TYPES( lhs, rhs ) \
    GTL_ASSERT( v1->getType()->isInteger() ); \
    GTL_ASSERT( v2->getType()->isInteger() ); \
    return llvm::BinaryOperator::_LLVM_Binary_Operator( v1, v2, "", currentBlock ); \
  } \
  llvm::Constant* CodeGenerator::_GTL_Function_Name_( llvm::Constant* lhs, const Type* lhsType, llvm::Constant* rhs, const Type* rhsType) \
  { \
    UNIFORMIZE_CONSTANT_TYPES( lhs, rhs ) \
    GTL_ASSERT( v1->getType()->isInteger() ); \
    GTL_ASSERT( v2->getType()->isInteger() ); \
    return llvm::ConstantExpr::_LLVM_Constant_Expr_( v1, v2 ); \
  } \
  ExpressionResult CodeGenerator::_GTL_Function_Name_( llvm::BasicBlock* currentBlock, ExpressionResult lhs, const Type* lhsType, ExpressionResult rhs, const Type* rhsType) \
  { \
    UNIFORMIZE_TYPES(lhs, rhs); \
    CREATE_BINARY_OPERATOR(  _GTL_Function_Name_, v1, v2); \
  }

CREATE_BINARY_INTEGER_EXPRESSION(createBitXorExpression, getXor, CreateXor )
CREATE_BINARY_INTEGER_EXPRESSION(createBitOrExpression, getOr, CreateOr )
CREATE_BINARY_INTEGER_EXPRESSION(createBitAndExpression, getAnd, CreateAnd )


#define CREATE_COMPARISON_EXPRESSION(_GTL_Function_Name_, _LLVM_UInt_Expr_, _LLVM_SInt_Expr_, _LLVM_Float_Expr_ ) \
  llvm::Value* CodeGenerator::_GTL_Function_Name_(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const Type* lhsType, llvm::Value* rhs, const Type* rhsType) \
  { \
    return createComparisonExpression( currentBlock, lhs, lhsType, rhs, rhsType, llvm::ICmpInst::_LLVM_UInt_Expr_, llvm::ICmpInst::_LLVM_SInt_Expr_, llvm::FCmpInst::_LLVM_Float_Expr_); \
  } \
  llvm::Constant* CodeGenerator::_GTL_Function_Name_( llvm::Constant* lhs, const Type* lhsType, llvm::Constant* rhs, const Type* rhsType) \
  { \
    return createComparisonExpression( lhs, lhsType, rhs, rhsType,  llvm::ICmpInst::_LLVM_UInt_Expr_, llvm::ICmpInst::_LLVM_SInt_Expr_, llvm::FCmpInst::_LLVM_Float_Expr_); \
  } \
  ExpressionResult CodeGenerator::_GTL_Function_Name_( llvm::BasicBlock* currentBlock, ExpressionResult lhs, const Type* lhsType, ExpressionResult rhs, const Type* rhsType) \
  { \
    return createComparisonExpression( currentBlock, lhs, lhsType, rhs, rhsType,  llvm::ICmpInst::_LLVM_UInt_Expr_, llvm::ICmpInst::_LLVM_SInt_Expr_, llvm::FCmpInst::_LLVM_Float_Expr_); \
  }

CREATE_COMPARISON_EXPRESSION(createEqualExpression, ICMP_EQ, ICMP_EQ, FCMP_OEQ);
CREATE_COMPARISON_EXPRESSION(createDifferentExpression, ICMP_NE, ICMP_NE, FCMP_ONE);
CREATE_COMPARISON_EXPRESSION(createStrictInferiorExpression, ICMP_ULT, ICMP_SLT, FCMP_OLT);
CREATE_COMPARISON_EXPRESSION(createInferiorOrEqualExpression, ICMP_ULE, ICMP_SLE, FCMP_OLE);
CREATE_COMPARISON_EXPRESSION(createStrictSupperiorExpression, ICMP_UGT, ICMP_SGT, FCMP_OGT);
CREATE_COMPARISON_EXPRESSION(createSupperiorOrEqualExpression, ICMP_UGE, ICMP_SGE, FCMP_OGE);

llvm::Value* CodeGenerator::createComparisonExpression(llvm::BasicBlock* currentBlock, llvm::Value* lhs, const Type* lhsType, llvm::Value* rhs, const Type* rhsType, unsigned int unsignedIntegerPred, unsigned int signedIntegerPred, unsigned int floatPred)
{
  GTL_DEBUG(*lhs << " type:" << *lhs->getType() << " " << *rhs << " type: " << *rhs->getType());
  GTL_DEBUG(lhs->getType() << " " << rhs->getType());
  UNIFORMIZE_VALUE_TYPES( lhs, rhs );
  if( v1->getType()->isFloatingPoint() )
  {
    return new llvm::FCmpInst( *currentBlock, (llvm::FCmpInst::Predicate)floatPred, v1, v2 );
  } else if( v2->getType()->isInteger() or v2->getType()->getTypeID() == llvm::Type::PointerTyID ) {
    if( rhsType->isSigned() )
    {
      return new llvm::ICmpInst( *currentBlock, (llvm::ICmpInst::Predicate)signedIntegerPred, v1, v2);
    } else {
      return new llvm::ICmpInst( *currentBlock, (llvm::ICmpInst::Predicate)unsignedIntegerPred, v1, v2);
    }
  }
  else {
    GTL_ABORT("Invalid comparison");
    return 0;
  }
}

llvm::Constant* CodeGenerator::createComparisonExpression( llvm::Constant* lhs, const Type* lhsType, llvm::Constant* rhs, const Type* rhsType, unsigned int unsignedIntegerPred, unsigned int signedIntegerPred, unsigned int floatPred)
{
  UNIFORMIZE_CONSTANT_TYPES( lhs, rhs );
  unsigned op = v1->getType()->isFloatingPoint() ? floatPred : 
      ( rhsType->isSigned() ? signedIntegerPred : unsignedIntegerPred);
  return (llvm::Constant*)llvm::ConstantExpr::getCompare( op, v1, v2);
}

ExpressionResult CodeGenerator::createComparisonExpression( llvm::BasicBlock* currentBlock, ExpressionResult lhs, const Type* lhsType, ExpressionResult rhs, const Type* rhsType, unsigned int unsignedIntegerPred, unsigned int signedIntegerPred, unsigned int floatPred)
{
  UNIFORMIZE_TYPES(lhs, rhs);
  if( v1.isConstant() and v2.isConstant() )
  {
    return GTLCore::ExpressionResult(createComparisonExpression( v1.constant(), lhsType, v2.constant(), rhsType, unsignedIntegerPred, signedIntegerPred, floatPred), Type::Boolean);
  } else {
    return GTLCore::ExpressionResult(createComparisonExpression( currentBlock, v1.value(), lhsType, v2.value(), rhsType, unsignedIntegerPred, signedIntegerPred, floatPred), Type::Boolean);
  }
}

#define CREATE_UNARY_OPERATOR( _creator_, _v1_) \
  if( _v1_.isConstant() ) \
  { \
    return GTLCore::ExpressionResult( _creator_(_v1_.constant(), rhsType), rhsType); \
  } else { \
    return GTLCore::ExpressionResult( _creator_( currentBlock, _v1_.value(), rhsType ), rhsType ); \
  }

ExpressionResult CodeGenerator::createMinusExpression( llvm::BasicBlock* currentBlock, ExpressionResult rhs, const Type* rhsType)
{
  CREATE_UNARY_OPERATOR( createMinusExpression, rhs);
}

llvm::Value* CodeGenerator::createMinusExpression(llvm::BasicBlock* currentBlock, llvm::Value* rhs, const Type* rhsType)
{
  return llvm::BinaryOperator::CreateNeg( rhs, "", currentBlock );
}

llvm::Constant* CodeGenerator::createMinusExpression( llvm::Constant* rhs, const Type* rhsType)
{
  return llvm::ConstantExpr::getNeg( rhs);
}

ExpressionResult CodeGenerator::createNotExpression( llvm::BasicBlock* currentBlock, ExpressionResult rhs, const Type* rhsType)
{
  CREATE_UNARY_OPERATOR( createNotExpression, rhs);
}

llvm::Value* CodeGenerator::createNotExpression(llvm::BasicBlock* currentBlock, llvm::Value* rhs, const Type* rhsType)
{
  return llvm::BinaryOperator::CreateNot( rhs, "", currentBlock );
}

llvm::Constant* CodeGenerator::createNotExpression( llvm::Constant* rhs, const Type* rhsType)
{
  return llvm::ConstantExpr::getNot( rhs);
}

ExpressionResult CodeGenerator::createTildeExpression( llvm::BasicBlock* currentBlock, ExpressionResult rhs, const Type* rhsType)
{
  CREATE_UNARY_OPERATOR( createTildeExpression, rhs);
}

llvm::Value* CodeGenerator::createTildeExpression(llvm::BasicBlock* currentBlock, llvm::Value* rhs, const Type* rhsType)
{
  return llvm::BinaryOperator::CreateXor( rhs, integerToConstant(rhs->getContext(), 0xFFFFFFFF ), "", currentBlock );
}

llvm::Constant* CodeGenerator::createTildeExpression( llvm::Constant* rhs, const Type* rhsType)
{
  return llvm::ConstantExpr::getXor(rhs, integerToConstant(rhs->getContext(), 0xFFFFFFFF ));
}

llvm::Value* CodeGenerator::accessArrayValueClamp( GTLCore::GenerationContext& generationContext, GTLCore::ExpressionGenerationContext& _expressionGenerationContext, llvm::Value* _pointer, llvm::Value* _index )
{
  llvm::BasicBlock* currentBlock = _expressionGenerationContext.currentBasicBlock();
  // Clamp
  llvm::Value* indexValuePtr = new llvm::AllocaInst(llvm::Type::getInt32Ty(generationContext.llvmContext()), "ClampedPlacehold", currentBlock);
  new llvm::StoreInst(_index, indexValuePtr, "Store for clamping", currentBlock);
  currentBlock = createClampExpression(generationContext, currentBlock, indexValuePtr, GTLCore::Type::Integer32, integerToConstant(generationContext.llvmContext(), 0), 
                                       createSubstractionExpression(currentBlock, accessArraySize(currentBlock, _pointer), Type::Integer32, integerToConstant(generationContext.llvmContext(), 1), Type::Integer32) );
  _expressionGenerationContext.setCurrentBasicBlock(currentBlock);
  _index = new llvm::LoadInst(indexValuePtr, "Loadnewindex", currentBlock);
  return accessArrayValueNoClamp(currentBlock, _pointer, _index);
}

llvm::Value* CodeGenerator::accessArrayValueNoClamp( llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, llvm::Value* _index )
{
  GTL_DEBUG( *_pointer << " " << *_index );
  std::vector<llvm::Value*> indexes;
  indexes.push_back( llvm::ConstantInt::get(llvm::Type::getInt32Ty(_pointer->getContext()), 0)); // Access the structure of the array
  indexes.push_back( llvm::ConstantInt::get(llvm::Type::getInt32Ty(_pointer->getContext()), ArrayWrap::POS_DATA)); // Access the data of the array
  
  // Select the pointer
  llvm::Value* iptr = llvm::GetElementPtrInst::Create( _pointer, indexes.begin(), indexes.end(), "", _currentBlock);
  // pointer on the values
  llvm::Value* iptr_val = new llvm::LoadInst( iptr, "load array data pointer", _currentBlock);
  return llvm::GetElementPtrInst::Create( iptr_val, _index, "point to the value in the data pointer", _currentBlock);
}

llvm::Value* CodeGenerator::accessArraySizePointer( llvm::BasicBlock* _currentBlock, llvm::Value* _pointer )
{
  std::vector<llvm::Value*> indexes;
  indexes.push_back( llvm::ConstantInt::get(llvm::Type::getInt32Ty(_pointer->getContext()), 0)); // Access the structure of the array
  indexes.push_back( llvm::ConstantInt::get(llvm::Type::getInt32Ty(_pointer->getContext()), ArrayWrap::POS_SIZE)); // Access the size of the array
  return llvm::GetElementPtrInst::Create( _pointer, indexes.begin(), indexes.end(), "", _currentBlock);
}

llvm::Value* CodeGenerator::accessArraySize( llvm::BasicBlock* _currentBlock, llvm::Value* _pointer )
{
  return new llvm::LoadInst( accessArraySizePointer(_currentBlock, _pointer), "", _currentBlock);
}

GTLCore::ExpressionResult CodeGenerator::callFunction( GenerationContext& _gc, ExpressionGenerationContext& egc, const GTLCore::Function* _function, const std::list<AST::Expression*>& _arguments )
{
  // Function Type
  GTL_ASSERT( _function );
  llvm::Function* llvmFunction = _function->d->data->function( );
#ifdef OPENGTL_ENABLE_DEBUG_OUTPUT
    const llvm::FunctionType *FTy = llvm::cast<llvm::FunctionType>(llvm::cast<llvm::PointerType>( llvmFunction->getType())->getElementType());
    GTL_DEBUG( *FTy );
#endif
  GTL_ASSERT( llvmFunction );
  std::vector< Parameter >::const_iterator oparam_it = _function->parameters().begin();
  std::vector<llvm::Value*> convertedParams;
  
  llvm::Value* returnPtr = 0;
  if(_function->d->isReturnedAsPointer())
  {
    returnPtr = new llvm::AllocaInst( _function->returnType()->d->type(_gc.llvmContext()), "", egc.currentBasicBlock() );
    convertedParams.push_back(returnPtr);
  }
  
  for( std::list< AST::Expression* >::const_iterator it = _arguments.begin();
       it != _arguments.end(); ++it, ++oparam_it)
  {
    if(oparam_it->isOutput())
    {
      AST::AccessorExpression* aexp = dynamic_cast<AST::AccessorExpression*>( *it );
      GTL_ASSERT( aexp );
      llvm::Value* pointer = aexp->pointer( _gc, egc );
      GTL_ASSERT( pointer->getType() == llvm::PointerType::get( oparam_it->type()->d->type(_gc.llvmContext()), 0) );
      convertedParams.push_back( pointer );
      GTL_DEBUG(*pointer->getType() << " " << *oparam_it->type()->d->type(_gc.llvmContext()) << " " << pointer );
    } else {
      llvm::Value* value = 0;
      if( (*it)->type()->dataType() == GTLCore::Type::ARRAY or (*it)->type()->dataType() == GTLCore::Type::STRUCTURE )
      {
        AST::AccessorExpression* aexp = dynamic_cast<AST::AccessorExpression*>( *it );
        if( aexp and not dynamic_cast<AST::FunctionMemberAccessorExpression*>( aexp ) )
        {
          value = aexp->pointer( _gc, egc );
        } else if( dynamic_cast<AST::FunctionCallExpression*>( *it )
                   or dynamic_cast<AST::GlobalDataExpression*>( *it ) or dynamic_cast<AST::FunctionMemberAccessorExpression*>( aexp ) ) {
            value = (*it)->generateValue(_gc, egc).value();
        } else {
          GTL_ABORT("Can't pass an array or a structure");
        }
      } else {
        value = (*it)->generateValue(_gc, egc).value();
      }
      GTL_ASSERT( value );
      GTL_DEBUG( "param's type = " << *oparam_it->type()->d->type(_gc.llvmContext()) << " value = " << *value );
//       if( oparam_it->type()->d->type()->isFirstClassType() )
//       {
//         value = _gc.codeGenerator()-> convertValueTo( bb, value, (*it)->type(), oparam_it->type() );
//       }
      convertedParams.push_back( value );
    }
  }
  for(unsigned int i = convertedParams.size(); i < _function->d->data->maximumParameters(); ++i)
  {
    llvm::Constant* val = valueToConstant( _gc, _function->d->data->parameters()[i].defaultValue() );
    GTL_ASSERT( val );
    if( _function->d->data->parameters()[i].type()->isStructure()
        or _function->d->data->parameters()[i].type()->dataType() == Type::ARRAY )
    {
      val = new llvm::GlobalVariable( *_gc.llvmModule(), val->getType(), true, llvm::GlobalValue::ExternalLinkage, val, "" );
    }
    convertedParams.push_back( val );
  }
#ifdef OPENGTL_ENABLE_DEBUG_OUTPUT
  {
    const llvm::FunctionType *FTy =
      llvm::cast<llvm::FunctionType>(llvm::cast<llvm::PointerType>(llvmFunction->getType())->getElementType());
    
    GTL_ASSERT( convertedParams.size() == FTy->getNumParams() or
            (FTy->isVarArg() and convertedParams.size() > FTy->getNumParams()) );
    
    for (unsigned i = 0; i < convertedParams.size(); ++i) {
      if( i < FTy->getNumParams() and (FTy->getParamType(i) != convertedParams[i]->getType()) )
      {
        GTL_DEBUG( "Wrong parameter " << i << " : " << FTy->getParamType(i) << " => " << *FTy->getParamType(i) << " but got " << *convertedParams[i] << " => " << *convertedParams[i]->getType() );
      } else {
        GTL_DEBUG( "Parameter " << i << " : " << FTy->getParamType(i) << " => " << *FTy->getParamType(i) << " but got " << convertedParams[i]->getType() << " => " << *convertedParams[i]->getType() );
      }
    }
  }
#endif
  
  llvm::CallInst *CallInst = llvm::CallInst::Create(llvmFunction, convertedParams.begin(), convertedParams.end(), "", egc.currentBasicBlock());
  CallInst->setTailCall(false);
  llvm::Value* result = CallInst;
  if(_function->d->isReturnedAsPointer())
  {
    GTL_ASSERT(returnPtr);
    result = new llvm::LoadInst(returnPtr, "", egc.currentBasicBlock());
  }
  return GTLCore::ExpressionResult( result, _function->returnType(), true );
}

llvm::Value* CodeGenerator::countFieldPointer( llvm::BasicBlock* _currentBlock, llvm::Value* _pointer )
{
  std::vector<llvm::Value*> indexes;
  indexes.push_back( llvm::ConstantInt::get(llvm::Type::getInt32Ty(_pointer->getContext()), 0)); // Access the structure of the array/structure
  indexes.push_back( llvm::ConstantInt::get(llvm::Type::getInt32Ty(_pointer->getContext()), 0)); // Access the count field of the array/structure
  // Select the pointer
  return llvm::GetElementPtrInst::Create( _pointer, indexes.begin(), indexes.end(), "", _currentBlock);
}

void CodeGenerator::setCountFieldOf( llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, llvm::Value* _value )
{
  new llvm::StoreInst( _value, countFieldPointer( _currentBlock, _pointer ), "", _currentBlock);
}

llvm::Value* CodeGenerator::getCountFieldOf( llvm::BasicBlock* _currentBlock, llvm::Value* _pointer )
{
  return new llvm::LoadInst( countFieldPointer( _currentBlock, _pointer ), "", _currentBlock );
}

llvm::Value* CodeGenerator::incrementCountFieldOf( llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, llvm::Value* _increment )
{
  llvm::Value* pointerCF = countFieldPointer(_currentBlock, _pointer);
  llvm::Value* val = createAdditionExpression( _currentBlock,
                                new llvm::LoadInst( pointerCF, "CodeGenerator::incrementCountFieldOf", _currentBlock ),
                                Type::Integer32, _increment, Type::Integer32 );
  new llvm::StoreInst( val, pointerCF, "CodeGenerator::incrementCountFieldOf", _currentBlock );
  return val;
}

#ifdef _USE_GTL_MM_

llvm::Function* createGtlMallocFunction( GenerationContext& _gc, llvm::Module* _module )
{
  std::vector<const llvm::Type*> memcpyTyArgs;
  memcpyTyArgs.push_back(llvm::Type::getInt32Ty(_gc.llvmContext()));
  llvm::FunctionType* memcpyTy = llvm::FunctionType::get( llvm::PointerType::get(llvm::IntegerType::getInt8Ty(_gc.llvmContext()), 0), memcpyTyArgs, false);
  
  llvm::Function* func_llvm_memcpy_i32 = (llvm::Function*)_module->getOrInsertFunction(
      "gtl_malloc", memcpyTy); // (external, no body)
  func_llvm_memcpy_i32->setCallingConv(llvm::CallingConv::C);
  return func_llvm_memcpy_i32;
}

llvm::Function* createGtlFreeFunction( GenerationContext& _gc, llvm::Module* _module )
{
  std::vector<const llvm::Type*> memcpyTyArgs;
  memcpyTyArgs.push_back(llvm::PointerType::get(llvm::IntegerType::getInt8Ty(_gc.llvmContext()), 0));
  llvm::FunctionType* memcpyTy = llvm::FunctionType::get( llvm::Type::getVoidTy(_gc.llvmContext()), memcpyTyArgs, false);
  
  llvm::Function* func_llvm_memcpy_i32 = (llvm::Function*)_module->getOrInsertFunction(
      "gtl_free", memcpyTy); // (external, no body)
  func_llvm_memcpy_i32->setCallingConv(llvm::CallingConv::C);
  return func_llvm_memcpy_i32;
}
#endif

llvm::Value* CodeGenerator::allocateMemory( GenerationContext& _gc, const llvm::Type* _type, llvm::Value* _size, llvm::BasicBlock* _bb)
{
#ifdef _USE_GTL_MM_
  llvm::Function* func = createGtlMallocFunction( _gc, _gc.module()->llvmModule() );
  std::vector<llvm::Value*> params;
  GTL_DEBUG( *_size << "  * " << *llvm::ConstantExpr::getSizeOf( _type ) );
  params.push_back( llvm::BinaryOperator::CreateMul( 
    _size,
    llvm::CastInst::CreateIntegerCast( llvm::ConstantExpr::getSizeOf( _type ),
                                       llvm::Type::getInt32Ty(_gc.llvmContext()), false, "", _bb ), "", _bb  ) );
  GTL_DEBUG( *func );
  llvm::CallInst *CallFunc = llvm::CallInst::Create(func, params.begin(), params.end(), "", _bb);
  CallFunc->setTailCall(false);
  return convertPointerTo( _bb, CallFunc, _type );
#else
#ifndef _USE_OS_MM_
#error _USE_OS_MM_ or _USE_GTL_MM_ should be defined
#endif
  return new llvm::MallocInst( _type, _size, "", _bb);
#endif
}

void CodeGenerator::freeMemory( GenerationContext& _gc, llvm::Value* _ptr, llvm::BasicBlock* _bb)
{
#ifdef _USE_GTL_MM_
  llvm::Function* func = createGtlFreeFunction( _gc, _gc.module()->llvmModule() );
  std::vector<llvm::Value*> params;
  params.push_back( convertPointerTo(_bb, _ptr, llvm::IntegerType::getInt8Ty(_gc.llvmContext()) ) );
  GTL_DEBUG(*params[0]);
  llvm::CallInst *CallFunc = llvm::CallInst::Create(func, params.begin(), params.end(), "", _bb);
  CallFunc->setTailCall(false);
#else
#ifndef _USE_OS_MM_
#error _USE_OS_MM_ or _USE_GTL_MM_ should be defined
#endif
  new llvm::FreeInst( _ptr, _bb );
#endif
}

llvm::Function* createProgressReportFunction( llvm::Module* _module, const GTLCore::String& name )
{
  std::vector<const llvm::Type*> memcpyTyArgs;
  memcpyTyArgs.push_back(llvm::PointerType::get(llvm::Type::getInt8Ty(_module->getContext()), 0));
  llvm::FunctionType* memcpyTy = llvm::FunctionType::get( llvm::Type::getVoidTy(_module->getContext()), memcpyTyArgs, false);
  
  llvm::Function* func_llvm_memcpy_i32 = (llvm::Function*)_module->getOrInsertFunction(
      name, memcpyTy); // (external, no body)
  func_llvm_memcpy_i32->setCallingConv(llvm::CallingConv::C);
  return func_llvm_memcpy_i32;
}

llvm::BasicBlock* callProgressReport( GenerationContext& _gc, llvm::Function* _function, llvm::Value* _a, llvm::BasicBlock* _bb)
{
  llvm::BasicBlock* ifblock = _gc.createBasicBlock();
  
  // _a != 0
  llvm::Value* test = CodeGenerator::createDifferentExpression(_bb, _a, Type::Pointer, CodeGenerator::nullPointerToConstant(_gc.llvmContext()), Type::Pointer );
  
  std::vector<llvm::Value*> params;
  params.push_back( _a );
  llvm::CallInst *CallFunc = llvm::CallInst::Create(_function, params.begin(), params.end(), "", ifblock);
  CallFunc->setTailCall(false);
  
  llvm::BasicBlock* afterBlock = _gc.createBasicBlock();
  
  
  CodeGenerator::createIfStatement(_bb, test, Type::Boolean, ifblock, ifblock, afterBlock);
  
  return afterBlock;  
}

llvm::BasicBlock* CodeGenerator::callProgressReportNextRow( GenerationContext& _gc, llvm::Value* _a, llvm::BasicBlock* _bb)
{
  llvm::Function* function = createProgressReportFunction( _gc.llvmModule(), "gtl_progress_report_next_row" );
  // if(_a) gtl_progress_report_next_row(_a);
  return callProgressReport(_gc, function, _a, _bb);
}

llvm::BasicBlock* CodeGenerator::callProgressReportNextPixel( GenerationContext& _gc, llvm::Value* _a, llvm::BasicBlock* _bb)
{
  llvm::Function* function = createProgressReportFunction( _gc.llvmModule(), "gtl_progress_report_next_pixel" );
  // if(_a) gtl_progress_report_next_pixel(_a);
  return callProgressReport(_gc, function, _a, _bb);  
}


llvm::BasicBlock* CodeGenerator::createClampExpression( GenerationContext& _context, llvm::BasicBlock* _currentBlock, llvm::Value* _valuePtr, const Type* _type, llvm::Value* _min, llvm::Value* _max)
{
  GTL_ASSERT(llvm::isa<llvm::PointerType>(_valuePtr->getType()));
  llvm::Value* _value = new llvm::LoadInst(_valuePtr, "load for clamping", _currentBlock);
  // if( v < min ) v = min;
  // else if( v > max ) v = max
  
  // v < min
  llvm::Value* inferiorTo0 = createStrictInferiorExpression( _currentBlock, _value, _type, _min, _type );
  
  // v = min
  llvm::BasicBlock* inferiorTo0Block = _context.createBasicBlock();
  new llvm::StoreInst( _min, _valuePtr, "store min", inferiorTo0Block);
  
  // else
  llvm::BasicBlock* firstElseBlock = _context.createBasicBlock();
  llvm::BasicBlock* lastElseBlock = _context.createBasicBlock();

  // v > max
  llvm::Value* superiorTo1 = createStrictSupperiorExpression(firstElseBlock, _value, _type, _max, _type);
  // v = max
  llvm::BasicBlock* superiorTo1Block = _context.createBasicBlock();
  new llvm::StoreInst( _max, _valuePtr, "store max", superiorTo1Block);
  // if v > maxs
  createIfStatement(firstElseBlock, superiorTo1, Type::Boolean, superiorTo1Block, superiorTo1Block, lastElseBlock);
  
  // if else
  llvm::BasicBlock* lastBlock = _context.createBasicBlock();
  createIfElseStatement(_currentBlock, inferiorTo0, Type::Boolean, inferiorTo0Block, inferiorTo0Block, firstElseBlock, lastElseBlock, lastBlock );

  return lastBlock;
}
