/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "CoumpoundExpression.h"

#include <llvm/Constants.h>
#include <llvm/DerivedTypes.h>
#include <llvm/GlobalVariable.h>
#include <llvm/Instructions.h>

#include "../CodeGenerator_p.h"
#include "../Debug.h"
#include "../Type.h"
#include "../Type_p.h"
#include "../ExpressionResult_p.h"
#include "../ExpressionGenerationContext_p.h"

using namespace GTLCore::AST;

CoumpoundExpression::CoumpoundExpression( const GTLCore::Type* _type, const std::vector<Expression*>& _expressions) : m_expressions(_expressions), m_type(_type)
{
  GTL_ASSERT(_type);
}

CoumpoundExpression::~CoumpoundExpression( )
{
  for( std::vector<Expression*>::iterator it = m_expressions.begin();
       it != m_expressions.end(); ++it)
  {
    delete *it;
  }
}

const GTLCore::Type* CoumpoundExpression::type() const
{
  return m_type;
}

bool CoumpoundExpression::isConstant() const
{
  return m_constant;
}

GTLCore::ExpressionResult CoumpoundExpression::generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const
{
  GTL_ASSERT( _egc.currentBasicBlock() );
  GTL_DEBUG( *m_type << " " << m_type->dataType() << " " << Type::ARRAY << " " << Type::STRUCTURE << " " << Type::VECTOR );
  
  switch( m_type->dataType() )
  {
    case Type::VECTOR:
    {
      GTL_DEBUG( m_expressions.size() << " " << m_type->vectorSize() );
      GTL_ASSERT( m_expressions.size() == m_type->vectorSize() );
      llvm::Value* resultLoad = CodeGenerator::createVector( _gc.llvmContext(), m_type );
      // Affect the same value to each element of the vector
      for(std::size_t i = 0; i < m_expressions.size(); ++i)
      {
        llvm::Value* val = m_expressions[i]->generateValue( _gc, _egc ).value();
        GTL_DEBUG( i << " " << *val );
        resultLoad = llvm::InsertElementInst::Create( resultLoad, val, CodeGenerator::integerToConstant( _gc.llvmContext(), i ), "", _egc.currentBasicBlock() );
      }
      return GTLCore::ExpressionResult( resultLoad, m_type );
    }
      break;
    case Type::ARRAY:
      GTL_ABORT("Unimplemented");
    case Type::STRUCTURE:
      GTL_ABORT("Unimplemented");
    default:
      GTL_ABORT("Unsupported");
  }
}

void CoumpoundExpression::markAsReturnExpression()
{
  for(std::size_t i = 0; i < m_expressions.size(); ++i)
  {
    m_expressions[i]->markAsReturnExpression();
  }
}
