/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "ConstantCoumpoundExpression.h"

#include <llvm/Constants.h>
#include <llvm/DerivedTypes.h>
#include <llvm/GlobalVariable.h>
#include <llvm/Instructions.h>

#include "../CodeGenerator_p.h"
#include "../Debug.h"
#include "../Type.h"
#include "../Type_p.h"
#include "../ExpressionResult_p.h"
#include "../ExpressionGenerationContext_p.h"

using namespace GTLCore::AST;

ConstantCoumpoundExpression::ConstantCoumpoundExpression( const GTLCore::Type* _type, const std::vector<Expression*>& _expressions) : m_expressions(_expressions), m_type(_type)
{
  GTL_ASSERT(_type);
  GTL_ASSERT(not m_expressions.empty());
}

ConstantCoumpoundExpression::~ConstantCoumpoundExpression( )
{
  for( std::vector<Expression*>::iterator it = m_expressions.begin();
       it != m_expressions.end(); ++it)
  {
    delete *it;
  }
}

const GTLCore::Type* ConstantCoumpoundExpression::type() const
{
  return m_type;
}

bool ConstantCoumpoundExpression::isConstant() const
{
  return m_constant;
}

GTLCore::ExpressionResult ConstantCoumpoundExpression::generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const
{
  GTL_DEBUG( m_type->dataType() << " " << Type::ARRAY << " " << Type::STRUCTURE );
  if( m_type->dataType() == Type::ARRAY or m_type->dataType() == Type::VECTOR)
  {
    const GTLCore::Type* arrayType = m_type->embeddedType();
    std::vector< llvm::Constant* > members;
    for(  std::vector<Expression*>::const_iterator it = m_expressions.begin();
       it != m_expressions.end(); ++it)
    {
      members.push_back( _gc.codeGenerator()->convertConstantTo( (*it)->generateValue( _gc, _egc).constant(), (*it)->type(), arrayType ) );
    }
    GTL_ASSERT(members.size() == m_expressions.size());
    if( m_type->dataType() == Type::ARRAY )
    {
      return GTLCore::ExpressionResult( CodeGenerator::constantsToArray(_gc, members, m_type) , type() );
    } else if( m_type->dataType() == Type::VECTOR )
    {
      llvm::Constant* constant = CodeGenerator::constantsToVector(_gc, members, m_type);
      GTL_DEBUG( "Coumpound constant = " << *constant << " type = " << *m_type );
      return GTLCore::ExpressionResult( constant, m_type );
    }
  } else if( m_type->dataType() == Type::STRUCTURE )
  {
    std::vector<llvm::Constant*> members;
    members.push_back( CodeGenerator::integerToConstant( _gc.llvmContext(), 0 ) ); // GC constant
    for( uint i = 0; i < m_expressions.size(); ++i)
    {
      members.push_back( _gc.codeGenerator()->convertConstantTo( m_expressions[i]->generateValue( _gc, _egc).constant(), m_expressions[i]->type(), m_type->structDataMember(i).type() ) );
    }
    return GTLCore::ExpressionResult( CodeGenerator::constantsToStructure(_gc, members, m_type), type() );
  }
  GTL_ABORT("Can't generate coumpound value, neither array, neither vector.");
  return GTLCore::ExpressionResult();
}

void ConstantCoumpoundExpression::markAsReturnExpression()
{
  GTL_ABORT("ConstantCoumpoundExpression can't be in a return statement");
}

Expression* ConstantCoumpoundExpression::expressionAt(unsigned int _idx )
{
  return m_expressions[ _idx ];
}
