/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _AST_ACCESSOR_EXPRESSION_H_
#define _AST_ACCESSOR_EXPRESSION_H_

#include "Expression.h"
#include "../Type.h"

namespace GTLCore {
  class ExpressionResult;
  namespace AST {
    /**
     * @internal
     * This is the base class for \ref Expression that give acces to a part of a structure/array.
     * @ingroup GTLCore_AST
     */
    class AccessorExpression : public Expression {
      public:
        virtual GTLCore::ExpressionResult generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const;
      public:
        virtual llvm::Value* pointer(GenerationContext& _gc, ExpressionGenerationContext& _egc) const = 0;
        virtual bool allocatedInMemory() const = 0;
        virtual llvm::BasicBlock* affect( GenerationContext& _gc, ExpressionGenerationContext& _egc, const ExpressionResult& _value );
    };
    /**
     * @internal
     * Access the size of an array.
     * @ingroup GTLCore_AST
     */
    class ArraySizeAccessorExpression : public AccessorExpression {
      public:
        ArraySizeAccessorExpression(AccessorExpression* _parent );
        virtual bool isConstant() const;
        virtual const GTLCore::Type* type() const;
        virtual llvm::Value* pointer(GenerationContext& _gc, ExpressionGenerationContext& _egc) const;
        virtual void markAsReturnExpression();
        virtual bool allocatedInMemory() const { return m_parent->allocatedInMemory(); }
      private:
        AccessorExpression * m_parent;
    };
    /**
     * @internal
     * Access a member of a structure.
     * @ingroup GTLCore_AST
     */
    class StructAccessorExpression : public AccessorExpression {
      public:
        StructAccessorExpression(AccessorExpression* _parent, unsigned int _index);
        ~StructAccessorExpression();
        virtual bool isConstant() const;
        virtual const GTLCore::Type* type() const;
      public:
        virtual llvm::Value* pointer(GenerationContext& _gc, ExpressionGenerationContext& _egc) const;
        virtual void markAsReturnExpression();
        virtual bool allocatedInMemory() const { return m_parent->allocatedInMemory(); }
      private:
        AccessorExpression * m_parent;
        unsigned int m_index;
    };
    /**
     * @internal
     * Access an index of an array.
     * @ingroup GTLCore_AST
     */
    class ArrayAccessorExpression : public AccessorExpression {
      public:
        ArrayAccessorExpression(AccessorExpression * _parent, Expression* _index);
        ~ArrayAccessorExpression();
        virtual bool isConstant() const;
        virtual const GTLCore::Type* type() const;
      public:
        virtual llvm::Value* pointer(GenerationContext& _gc, ExpressionGenerationContext& _egc) const;
        virtual void markAsReturnExpression();
        virtual bool allocatedInMemory() const { return m_parent->allocatedInMemory(); }
      private:
        AccessorExpression * m_parent;
        Expression* m_index;
    };
    /**
     * @internal
     * Access the content of a variable.
     * @ingroup GTLCore_AST
     */
    class VariableAccessorExpression : public AccessorExpression {
      public:
        VariableAccessorExpression(GTLCore::VariableNG* _variable );
        virtual bool isConstant() const;
        virtual const GTLCore::Type* type() const;
      public:
        virtual llvm::Value* pointer(GenerationContext& _gc, ExpressionGenerationContext& _egc) const;
        virtual void markAsReturnExpression();
        virtual bool allocatedInMemory() const;
        virtual llvm::BasicBlock* affect( GenerationContext& _gc, ExpressionGenerationContext& _egc, const ExpressionResult& _value );
      private:
        GTLCore::VariableNG* m_variable;
    };
    /**
     * @internal
     * Call a function member of a structure.
     * @ingroup GTLCore_AST
     */
    class FunctionMemberAccessorExpression : public AccessorExpression {
      public:
        FunctionMemberAccessorExpression( AccessorExpression * _parent, 
                                          const Type::StructFunctionMember* _member,
                                          const std::list<AST::Expression*>& _arguments );
        ~FunctionMemberAccessorExpression();
        virtual bool isConstant() const;
        virtual const GTLCore::Type* type() const;
      public:
        virtual GTLCore::ExpressionResult generateValue( GenerationContext& _gc, ExpressionGenerationContext& _egc ) const;
        virtual llvm::Value* pointer(GenerationContext& _gc, ExpressionGenerationContext& _egc) const;
        virtual void markAsReturnExpression();
        virtual bool allocatedInMemory() const;
      private:
        AccessorExpression* m_parent;
        const Type::StructFunctionMember* m_member;
        std::list<AST::Expression*> m_arguments;
    };
  }
}

#endif
