/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "nl-folders-source.h"

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <clutter/clutter.h>
#include <clutter-gtk/clutter-gtk.h>
#include <launcher/launcher.h>
#include <netbook-launcher/netbook-launcher.h>

G_DEFINE_TYPE (NlFoldersSource, nl_folders_source, G_TYPE_OBJECT);

#define NL_FOLDERS_SOURCE_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  NL_TYPE_FOLDERS_SOURCE, \
  NlFoldersSourcePrivate))

struct _NlFoldersSourcePrivate
{
  NlShell *shell;
  LauncherFolderBookmarks *folders;
  NlPixbufCache *pixbuf_cache;

  ClutterActor *vbox;
  ClutterActor *text;
  ClutterActor *iconview;
};

enum
{
  PROP_0,
  PROP_SHELL
};

/* Forwards */
static gboolean on_folders_changed (NlFoldersSource *source);

/* GObject stuff */
static void
set_property (GObject      *object,
              guint         prop_id,
              const GValue *value,
              GParamSpec   *pspec)
{
  NlFoldersSourcePrivate *priv;

  g_return_if_fail (NL_IS_FOLDERS_SOURCE (object));
  priv = NL_FOLDERS_SOURCE_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_SHELL:
      priv->shell = g_value_get_pointer (value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
get_property (GObject      *object,
              guint         prop_id,
              GValue       *value,
              GParamSpec   *pspec)
{
  NlFoldersSourcePrivate *priv;

  g_return_if_fail (NL_IS_FOLDERS_SOURCE (object));
  priv = NL_FOLDERS_SOURCE_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_SHELL:
      g_value_set_pointer (value, priv->shell);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}
static void
nl_folders_source_finalize (GObject *object)
{
  NlFoldersSourcePrivate *priv;

  priv = NL_FOLDERS_SOURCE_GET_PRIVATE (object);

  if (priv->folders)
    {
      g_object_unref (priv->folders);
      priv->folders = NULL;
    }

  if (priv->pixbuf_cache)
    {
      g_object_unref (priv->pixbuf_cache);
      priv->pixbuf_cache = NULL;
    }
  G_OBJECT_CLASS (nl_folders_source_parent_class)->finalize (object);
}

static void
nl_folders_source_constructed (GObject *object)
{
  NlFoldersSourcePrivate *priv;

  priv = NL_FOLDERS_SOURCE_GET_PRIVATE (object);

  /* Init the LauncherFolderBookmarks object */
  priv->folders = launcher_folder_bookmarks_get_default ();
  g_signal_connect_swapped (priv->folders, "bookmarks-changed",
                            G_CALLBACK (on_folders_changed), object);

  /* Build the view */
  priv->vbox = ctk_vbox_new (6);
  g_object_ref_sink (priv->vbox);

  priv->text = ctk_text_new (_("<big><b>Folders</b></big>"));
  ctk_text_set_alignment (CTK_TEXT (priv->text), PANGO_ALIGN_CENTER);
  ctk_box_pack (CTK_BOX (priv->vbox), priv->text, FALSE,FALSE);

  priv->iconview = ctk_icon_view_new ();
  ctk_box_pack (CTK_BOX (priv->vbox), priv->iconview, FALSE,FALSE);

  nl_shell_add_places_source (priv->shell, priv->vbox);

  g_idle_add ((GSourceFunc)on_folders_changed, object);
}

static void
nl_folders_source_class_init (NlFoldersSourceClass *klass)
{
  GObjectClass *obj_class = G_OBJECT_CLASS (klass);
  GParamSpec   *pspec;

  obj_class->finalize     = nl_folders_source_finalize;
  obj_class->constructed  = nl_folders_source_constructed;
  obj_class->set_property = set_property;
  obj_class->get_property = get_property;

  pspec = g_param_spec_pointer ("shell", "shell", "shell",
                                G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_SHELL, pspec);

  g_type_class_add_private (obj_class, sizeof (NlFoldersSourcePrivate));
}

static void
nl_folders_source_init (NlFoldersSource *manager)
{
  NlFoldersSourcePrivate *priv;

  priv = manager->priv = NL_FOLDERS_SOURCE_GET_PRIVATE (manager);

  priv->pixbuf_cache = nl_pixbuf_cache_get_default ();
}

/*
 * Public methods
 */
GObject *
nl_folders_source_new (NlShell *shell)

{
  GObject *folders_source;

  g_return_val_if_fail (NL_IS_SHELL (shell), NULL);

  folders_source = g_object_new (NL_TYPE_FOLDERS_SOURCE,
                                 "shell", shell,
                                 NULL);

  return folders_source;
}

/*
 * Private methods
 */
static void
remove_child (ClutterActor *child, ClutterContainer *container)
{
  clutter_container_remove_actor (container, child);
}

static void
on_folder_clicked (ClutterActor *actor, LauncherFolder *folder)
{
  GError *error = NULL;

  g_return_if_fail (folder);

  launcher_folder_activate (folder, &error);


  if (error)
    {
      GtkWidget *dialog;
      gchar     *title;

      dialog = gtk_message_dialog_new (NULL,
                                       0,
                                       GTK_MESSAGE_ERROR,
                                       GTK_BUTTONS_CLOSE,
                                       "%s",
                                       error->message);

      title = g_strdup_printf (_("Unable to open %s"),
                               launcher_folder_get_name (folder));
      gtk_window_set_title (GTK_WINDOW (dialog), title);
      g_free (title);

      g_signal_connect (dialog, "close", G_CALLBACK (gtk_widget_destroy), NULL);
      g_signal_connect (dialog, "response", G_CALLBACK(gtk_widget_destroy),NULL);

      gtk_widget_show (dialog);

      g_warning ("Unable to show folder %s: %s",
                 launcher_folder_get_name (folder),
                 error->message);

      g_error_free (error);
    }
  else
    {
      ClutterActor *image;

      if (CTK_IS_BUTTON (actor) && 0)
        image = CLUTTER_ACTOR (ctk_button_get_image (CTK_BUTTON (actor)));
      else
        {
          ClutterActor  *stage;
          GdkPixbuf     *pixbuf;
          NlPixbufCache *cache;

          stage = clutter_actor_get_parent(CLUTTER_ACTOR(nl_notify_get_default()));

          cache = nl_pixbuf_cache_get_default ();
          pixbuf = nl_pixbuf_cache_icon_for_name (cache,
                                                  launcher_folder_get_icon_name (folder),
                                                  64);
          image = ctk_image_new_from_pixbuf (64, pixbuf);

          g_object_unref (pixbuf);
          g_object_unref (cache);
        }

      nl_notify_popup (nl_notify_get_default (),
                       launcher_folder_get_name (folder),
                       CTK_IMAGE (image),
                       0);
    }
}

static void
show_menu (ClutterActor *actor, guint32 event_time, LauncherFolder *folder)
{
  GtkWidget *menu, *item;

  g_return_if_fail (folder);

  menu = gtk_menu_new ();

  item = gtk_image_menu_item_new_from_stock (GTK_STOCK_OPEN, NULL);
  gtk_widget_show (item);
  g_signal_connect(item, "activate",
                   G_CALLBACK (on_folder_clicked), folder);
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);

  gtk_menu_popup (GTK_MENU (menu),
                  NULL, NULL,
                  NULL, NULL,
                  3, event_time);
}

static void
add_launcher_to_view (NlFoldersSource *source, 
              LauncherFolder *folder)
{
  g_return_if_fail (NL_IS_FOLDERS_SOURCE (source));
  
  NlFoldersSourcePrivate *priv;
  GdkPixbuf *pixbuf;
  ClutterActor *tile;
  
  priv = source->priv;
  
  pixbuf = nl_pixbuf_cache_icon_for_name (priv->pixbuf_cache,
                                          launcher_folder_get_icon_name (folder),
                                          64);
  tile = nl_icon_tile_new (launcher_folder_get_name (folder), " ", pixbuf);
  clutter_container_add_actor (CLUTTER_CONTAINER (priv->iconview), tile);
  g_signal_connect (tile, "clicked", G_CALLBACK (on_folder_clicked), folder);
  g_signal_connect (tile, "show-context-menu", G_CALLBACK (show_menu), folder);
  if (pixbuf)
    g_object_unref (pixbuf);
}

static gboolean
on_folders_changed (NlFoldersSource *source)
{
  NlFoldersSourcePrivate *priv;
  static LauncherFolder  *desktop;
  static LauncherFolder  *download; 
  static LauncherFolder  *home; 
  static LauncherFolder  *examples; 
  GSList      *f;
  GList       *children;
  gchar       *name, *path;
  gboolean     have_download = FALSE;
  const gchar *desktop_uri;
  const gchar *download_uri;
  const gchar *home_uri;
  const gchar *examples_uri;
  const gchar *examples_url_uri;
  GKeyFile       *keyfile;
  GKeyFileFlags   flags;


  g_return_val_if_fail (NL_IS_FOLDERS_SOURCE (source), FALSE);
  priv = source->priv;

  children = clutter_container_get_children((CLUTTER_CONTAINER (priv->iconview)));
  g_list_foreach (children, (GFunc)remove_child, priv->iconview);
  g_list_free (children);
  
  desktop_uri = g_filename_to_uri (g_get_user_special_dir (
                                    G_USER_DIRECTORY_DESKTOP),
                                   NULL,
                                   NULL);
  download_uri = g_filename_to_uri (g_get_user_special_dir (
                                      G_USER_DIRECTORY_DOWNLOAD),
                                    NULL,
                                    NULL);
  home_uri = g_filename_to_uri (g_get_home_dir (),
                                    NULL,
                                    NULL);
  examples_uri = g_build_filename (home_uri, "examples.desktop", NULL);

  if (g_strcmp0 (desktop_uri, download_uri) == 0)
    have_download = TRUE;

  if (!home && home_uri)
    {
      path = g_filename_from_uri (home_uri, NULL, NULL);
      name = g_filename_display_basename (path);
      home = launcher_folder_new (name , home_uri);
      g_free (path);
    }
  add_launcher_to_view (source, home);

  if (!desktop && desktop_uri)
    {
      path = g_filename_from_uri (desktop_uri, NULL, NULL);
      name = g_filename_display_basename (path);
      desktop = launcher_folder_new (name , desktop_uri);
      g_free (path);
    }
  add_launcher_to_view (source, desktop);

  if (((desktop_uri && g_strcmp0 (desktop_uri, download_uri) != 0) || !desktop_uri) &&
      ((home_uri && g_strcmp0 (home_uri, download_uri) != 0) || !home_uri))
    {
     if (!download && download_uri)
       {
         path = g_filename_from_uri (download_uri, NULL, NULL);
         name = g_filename_display_basename (path);
         download = launcher_folder_new (name , download_uri);
         g_free (path);
       }
     add_launcher_to_view (source, download);
    }

  path = g_filename_from_uri (examples_uri, NULL, NULL);
  if (g_file_test (path, G_FILE_TEST_IS_REGULAR))
    {
      // otherwise, open with gedit
      keyfile = g_key_file_new ();
      flags = G_KEY_FILE_KEEP_COMMENTS | G_KEY_FILE_KEEP_TRANSLATIONS;
      if (g_key_file_load_from_file (keyfile, path, flags, NULL))
       {
          examples_url_uri = g_key_file_get_string (keyfile, "Desktop Entry", 
                                                    "URL", NULL);
          name = g_key_file_get_string (keyfile, "Desktop Entry",
                                        "Name", NULL);
          if (examples_url_uri && name)
            {
              examples = launcher_folder_new (name , examples_url_uri);
              add_launcher_to_view (source, examples);
            }
          g_key_file_free (keyfile);
       }
    }
  g_free (path);

  for (f = launcher_folder_bookmarks_get_bookmarks (priv->folders);
       f;
       f = f->next)
    {
      LauncherFolder *folder = f->data;

      if ((g_strcmp0 (launcher_folder_get_uri (folder), desktop_uri) != 0) &&
           (g_strcmp0 (launcher_folder_get_uri (folder), download_uri) != 0) &&
           (g_strcmp0 (launcher_folder_get_uri (folder), home_uri) != 0))
         add_launcher_to_view (source, folder);
    }


  return FALSE;
}

/*
 * Public Methods
 */

