//
// IntegerFormatterTest.cs - NUnit Test Cases for System.IntegerFormatter
//
// Author:
//   Gonzalo Paniagua Javier (gonzalo@ximian.com)
//
// (C) Ximian, Inc. http://www.ximian.com
//

using NUnit.Framework;
using System;
using System.Globalization;
using System.IO;
using System.Threading;

namespace MonoTests.System
{

public class IntegerFormatterTest : TestCase
{
	private const int INT32 = 0;
	private const int INT64 = 1;
	private const int UINT32 = 2;
	private const int UINT64 = 3;
	private int testType;

	public IntegerFormatterTest () {}

	protected override void SetUp () {}
	
	protected override void TearDown() {}

	public void TestStandard ()
	{
		// Make the test Culture insensitive
		NumberFormatInfo nfi = new NumberFormatInfo ();
		nfi.CurrencySymbol = "$";

		StringReader sr = new StringReader (tutti);

		string inputLine;
		while ((inputLine = sr.ReadLine ()) != null) {
			if (inputLine.IndexOf ('(') != -1)
				ProcessLine (inputLine, nfi);
			else
				ChangeType (inputLine);
		}

		sr.Close ();
	}

	private void ProcessLine (string testLine, NumberFormatInfo nfi)
	{
		string number = "0";
		string format = "X";
		string expected = "XXX";
		int idxStart;
		int idxEnd;

		idxStart = testLine.IndexOf ('(');
		if (idxStart != -1){
			idxStart++;
			idxEnd = testLine.IndexOf (')');
			number = testLine.Substring (idxStart,
						     idxEnd - idxStart);
		}

		idxStart = testLine.IndexOf ('(', idxStart);
		if (idxStart != -1) {
			idxStart++;
			idxEnd = testLine.IndexOf (')', idxStart);
			format = testLine.Substring (idxStart,
						     idxEnd - idxStart);
		}

		idxStart = testLine.IndexOf ('(', idxStart);
		if (idxStart != -1) {
			idxStart++;
			idxEnd = testLine.LastIndexOf (')');
			expected = testLine.Substring (idxStart,
						       idxEnd - idxStart);
		}

		DoTest (number, format, expected, nfi);
	}

	private void DoTest (string number,
				   string format, 
				   string expected,
				   NumberFormatInfo nfi)
	{
		string id = number + " " + format;

		switch (testType) {
		case INT32:
			int i32 = Int32.Parse (number, nfi);
			AssertEquals(id, expected, i32.ToString (format, nfi));
			break;
		case INT64:
			long i64 = Int64.Parse (number, nfi);
			AssertEquals(id, expected, i64.ToString (format, nfi));
			break;
		case UINT32:
			uint ui32 = UInt32.Parse (number, nfi);
			AssertEquals(id, expected, ui32.ToString (format, nfi));
			break;
		case UINT64:
			ulong ui64 = UInt64.Parse (number, nfi);
			AssertEquals(id, expected, ui64.ToString (format, nfi));
			break;
		}
	}

	private void ChangeType (string newType)
	{
		if (newType == "Int32")
			testType = INT32;
		else if (newType == "Int64")
			testType = INT64;
		else if (newType == "UInt32")
			testType = UINT32;
		else if (newType == "UInt64")
			testType = UINT64;
		else throw new Exception ("Test of type '" + newType + "' not supported.");

	}

	private static string tutti = "Int32\n" +
"(-2147483648) (C) (($2,147,483,648.00))\n" +
"(-2147483648) (C0) (($2,147,483,648))\n" +
"(-2147483648) (C1) (($2,147,483,648.0))\n" +
"(-2147483648) (C2) (($2,147,483,648.00))\n" +
"(-2147483648) (C3) (($2,147,483,648.000))\n" +
"(-2147483648) (C4) (($2,147,483,648.0000))\n" +
"(-2147483648) (C5) (($2,147,483,648.00000))\n" +
"(-2147483648) (C6) (($2,147,483,648.000000))\n" +
"(-2147483648) (C7) (($2,147,483,648.0000000))\n" +
"(-2147483648) (C8) (($2,147,483,648.00000000))\n" +
"(-2147483648) (C9) (($2,147,483,648.000000000))\n" +
"(-2147483648) (C10) (($2,147,483,648.0000000000))\n" +
"(-2147483648) (C99) (($2,147,483,648.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000))\n" +
"(-2147483648) (D) (-2147483648)\n" +
"(-2147483648) (D0) (-2147483648)\n" +
"(-2147483648) (D1) (-2147483648)\n" +
"(-2147483648) (D2) (-2147483648)\n" +
"(-2147483648) (D3) (-2147483648)\n" +
"(-2147483648) (D4) (-2147483648)\n" +
"(-2147483648) (D5) (-2147483648)\n" +
"(-2147483648) (D6) (-2147483648)\n" +
"(-2147483648) (D7) (-2147483648)\n" +
"(-2147483648) (D8) (-2147483648)\n" +
"(-2147483648) (D9) (-2147483648)\n" +
"(-2147483648) (D10) (-2147483648)\n" +
"(-2147483648) (D99) (-000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002147483648)\n" +
"(-2147483648) (E) (-2.147484E+009)\n" +
"(-2147483648) (E0) (-2E+009)\n" +
"(-2147483648) (E1) (-2.1E+009)\n" +
"(-2147483648) (E2) (-2.15E+009)\n" +
"(-2147483648) (E3) (-2.147E+009)\n" +
"(-2147483648) (E4) (-2.1475E+009)\n" +
"(-2147483648) (E5) (-2.14748E+009)\n" +
"(-2147483648) (E6) (-2.147484E+009)\n" +
"(-2147483648) (E7) (-2.1474836E+009)\n" +
"(-2147483648) (E8) (-2.14748365E+009)\n" +
"(-2147483648) (E9) (-2.147483648E+009)\n" +
"(-2147483648) (E10) (-2.1474836480E+009)\n" +
"(-2147483648) (E99) (-2.147483648000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000E+009)\n" +
"(-2147483648) (F) (-2147483648.00)\n" +
"(-2147483648) (F0) (-2147483648)\n" +
"(-2147483648) (F1) (-2147483648.0)\n" +
"(-2147483648) (F2) (-2147483648.00)\n" +
"(-2147483648) (F3) (-2147483648.000)\n" +
"(-2147483648) (F4) (-2147483648.0000)\n" +
"(-2147483648) (F5) (-2147483648.00000)\n" +
"(-2147483648) (F6) (-2147483648.000000)\n" +
"(-2147483648) (F7) (-2147483648.0000000)\n" +
"(-2147483648) (F8) (-2147483648.00000000)\n" +
"(-2147483648) (F9) (-2147483648.000000000)\n" +
"(-2147483648) (F10) (-2147483648.0000000000)\n" +
"(-2147483648) (F99) (-2147483648.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(-2147483648) (G) (-2147483648)\n" +
"(-2147483648) (G0) (-2147483648)\n" +
"(-2147483648) (G1) (-2E+09)\n" +
"(-2147483648) (G2) (-2.1E+09)\n" +
"(-2147483648) (G3) (-2.15E+09)\n" +
"(-2147483648) (G4) (-2.147E+09)\n" +
"(-2147483648) (G5) (-2.1475E+09)\n" +
"(-2147483648) (G6) (-2.14748E+09)\n" +
"(-2147483648) (G7) (-2.147484E+09)\n" +
"(-2147483648) (G8) (-2.1474836E+09)\n" +
"(-2147483648) (G9) (-2.14748365E+09)\n" +
"(-2147483648) (G10) (-2147483648)\n" +
"(-2147483648) (G99) (-2147483648)\n" +
"(-2147483648) (N) (-2,147,483,648.00)\n" +
"(-2147483648) (N0) (-2,147,483,648)\n" +
"(-2147483648) (N1) (-2,147,483,648.0)\n" +
"(-2147483648) (N2) (-2,147,483,648.00)\n" +
"(-2147483648) (N3) (-2,147,483,648.000)\n" +
"(-2147483648) (N4) (-2,147,483,648.0000)\n" +
"(-2147483648) (N5) (-2,147,483,648.00000)\n" +
"(-2147483648) (N6) (-2,147,483,648.000000)\n" +
"(-2147483648) (N7) (-2,147,483,648.0000000)\n" +
"(-2147483648) (N8) (-2,147,483,648.00000000)\n" +
"(-2147483648) (N9) (-2,147,483,648.000000000)\n" +
"(-2147483648) (N10) (-2,147,483,648.0000000000)\n" +
"(-2147483648) (N99) (-2,147,483,648.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(-2147483648) (P) (-214,748,364,800.00 %)\n" +
"(-2147483648) (P0) (-214,748,364,800 %)\n" +
"(-2147483648) (P1) (-214,748,364,800.0 %)\n" +
"(-2147483648) (P2) (-214,748,364,800.00 %)\n" +
"(-2147483648) (P3) (-214,748,364,800.000 %)\n" +
"(-2147483648) (P4) (-214,748,364,800.0000 %)\n" +
"(-2147483648) (P5) (-214,748,364,800.00000 %)\n" +
"(-2147483648) (P6) (-214,748,364,800.000000 %)\n" +
"(-2147483648) (P7) (-214,748,364,800.0000000 %)\n" +
"(-2147483648) (P8) (-214,748,364,800.00000000 %)\n" +
"(-2147483648) (P9) (-214,748,364,800.000000000 %)\n" +
"(-2147483648) (P10) (-214,748,364,800.0000000000 %)\n" +
"(-2147483648) (P99) (-214,748,364,800.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000 %)\n" +
"(-2147483648) (X) (80000000)\n" +
"(-2147483648) (X0) (80000000)\n" +
"(-2147483648) (X1) (80000000)\n" +
"(-2147483648) (X2) (80000000)\n" +
"(-2147483648) (X3) (80000000)\n" +
"(-2147483648) (X4) (80000000)\n" +
"(-2147483648) (X5) (80000000)\n" +
"(-2147483648) (X6) (80000000)\n" +
"(-2147483648) (X7) (80000000)\n" +
"(-2147483648) (X8) (80000000)\n" +
"(-2147483648) (X9) (080000000)\n" +
"(-2147483648) (X10) (0080000000)\n" +
"(-2147483648) (X99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000080000000)\n" +
"(2147483647) (C) ($2,147,483,647.00)\n" +
"(2147483647) (C0) ($2,147,483,647)\n" +
"(2147483647) (C1) ($2,147,483,647.0)\n" +
"(2147483647) (C2) ($2,147,483,647.00)\n" +
"(2147483647) (C3) ($2,147,483,647.000)\n" +
"(2147483647) (C4) ($2,147,483,647.0000)\n" +
"(2147483647) (C5) ($2,147,483,647.00000)\n" +
"(2147483647) (C6) ($2,147,483,647.000000)\n" +
"(2147483647) (C7) ($2,147,483,647.0000000)\n" +
"(2147483647) (C8) ($2,147,483,647.00000000)\n" +
"(2147483647) (C9) ($2,147,483,647.000000000)\n" +
"(2147483647) (C10) ($2,147,483,647.0000000000)\n" +
"(2147483647) (C99) ($2,147,483,647.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(2147483647) (D) (2147483647)\n" +
"(2147483647) (D0) (2147483647)\n" +
"(2147483647) (D1) (2147483647)\n" +
"(2147483647) (D2) (2147483647)\n" +
"(2147483647) (D3) (2147483647)\n" +
"(2147483647) (D4) (2147483647)\n" +
"(2147483647) (D5) (2147483647)\n" +
"(2147483647) (D6) (2147483647)\n" +
"(2147483647) (D7) (2147483647)\n" +
"(2147483647) (D8) (2147483647)\n" +
"(2147483647) (D9) (2147483647)\n" +
"(2147483647) (D10) (2147483647)\n" +
"(2147483647) (D99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002147483647)\n" +
"(2147483647) (E) (2.147484E+009)\n" +
"(2147483647) (E0) (2E+009)\n" +
"(2147483647) (E1) (2.1E+009)\n" +
"(2147483647) (E2) (2.15E+009)\n" +
"(2147483647) (E3) (2.147E+009)\n" +
"(2147483647) (E4) (2.1475E+009)\n" +
"(2147483647) (E5) (2.14748E+009)\n" +
"(2147483647) (E6) (2.147484E+009)\n" +
"(2147483647) (E7) (2.1474836E+009)\n" +
"(2147483647) (E8) (2.14748365E+009)\n" +
"(2147483647) (E9) (2.147483647E+009)\n" +
"(2147483647) (E10) (2.1474836470E+009)\n" +
"(2147483647) (E99) (2.147483647000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000E+009)\n" +
"(2147483647) (F) (2147483647.00)\n" +
"(2147483647) (F0) (2147483647)\n" +
"(2147483647) (F1) (2147483647.0)\n" +
"(2147483647) (F2) (2147483647.00)\n" +
"(2147483647) (F3) (2147483647.000)\n" +
"(2147483647) (F4) (2147483647.0000)\n" +
"(2147483647) (F5) (2147483647.00000)\n" +
"(2147483647) (F6) (2147483647.000000)\n" +
"(2147483647) (F7) (2147483647.0000000)\n" +
"(2147483647) (F8) (2147483647.00000000)\n" +
"(2147483647) (F9) (2147483647.000000000)\n" +
"(2147483647) (F10) (2147483647.0000000000)\n" +
"(2147483647) (F99) (2147483647.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(2147483647) (G) (2147483647)\n" +
"(2147483647) (G0) (2147483647)\n" +
"(2147483647) (G1) (2E+09)\n" +
"(2147483647) (G2) (2.1E+09)\n" +
"(2147483647) (G3) (2.15E+09)\n" +
"(2147483647) (G4) (2.147E+09)\n" +
"(2147483647) (G5) (2.1475E+09)\n" +
"(2147483647) (G6) (2.14748E+09)\n" +
"(2147483647) (G7) (2.147484E+09)\n" +
"(2147483647) (G8) (2.1474836E+09)\n" +
"(2147483647) (G9) (2.14748365E+09)\n" +
"(2147483647) (G10) (2147483647)\n" +
"(2147483647) (G99) (2147483647)\n" +
"(2147483647) (N) (2,147,483,647.00)\n" +
"(2147483647) (N0) (2,147,483,647)\n" +
"(2147483647) (N1) (2,147,483,647.0)\n" +
"(2147483647) (N2) (2,147,483,647.00)\n" +
"(2147483647) (N3) (2,147,483,647.000)\n" +
"(2147483647) (N4) (2,147,483,647.0000)\n" +
"(2147483647) (N5) (2,147,483,647.00000)\n" +
"(2147483647) (N6) (2,147,483,647.000000)\n" +
"(2147483647) (N7) (2,147,483,647.0000000)\n" +
"(2147483647) (N8) (2,147,483,647.00000000)\n" +
"(2147483647) (N9) (2,147,483,647.000000000)\n" +
"(2147483647) (N10) (2,147,483,647.0000000000)\n" +
"(2147483647) (N99) (2,147,483,647.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(2147483647) (P) (214,748,364,700.00 %)\n" +
"(2147483647) (P0) (214,748,364,700 %)\n" +
"(2147483647) (P1) (214,748,364,700.0 %)\n" +
"(2147483647) (P2) (214,748,364,700.00 %)\n" +
"(2147483647) (P3) (214,748,364,700.000 %)\n" +
"(2147483647) (P4) (214,748,364,700.0000 %)\n" +
"(2147483647) (P5) (214,748,364,700.00000 %)\n" +
"(2147483647) (P6) (214,748,364,700.000000 %)\n" +
"(2147483647) (P7) (214,748,364,700.0000000 %)\n" +
"(2147483647) (P8) (214,748,364,700.00000000 %)\n" +
"(2147483647) (P9) (214,748,364,700.000000000 %)\n" +
"(2147483647) (P10) (214,748,364,700.0000000000 %)\n" +
"(2147483647) (P99) (214,748,364,700.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000 %)\n" +
"(2147483647) (X) (7FFFFFFF)\n" +
"(2147483647) (X0) (7FFFFFFF)\n" +
"(2147483647) (X1) (7FFFFFFF)\n" +
"(2147483647) (X2) (7FFFFFFF)\n" +
"(2147483647) (X3) (7FFFFFFF)\n" +
"(2147483647) (X4) (7FFFFFFF)\n" +
"(2147483647) (X5) (7FFFFFFF)\n" +
"(2147483647) (X6) (7FFFFFFF)\n" +
"(2147483647) (X7) (7FFFFFFF)\n" +
"(2147483647) (X8) (7FFFFFFF)\n" +
"(2147483647) (X9) (07FFFFFFF)\n" +
"(2147483647) (X10) (007FFFFFFF)\n" +
"(2147483647) (X99) (00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000007FFFFFFF)\n" +
"(0) (C) ($0.00)\n" +
"(0) (C0) ($0)\n" +
"(0) (C1) ($0.0)\n" +
"(0) (C2) ($0.00)\n" +
"(0) (C3) ($0.000)\n" +
"(0) (C4) ($0.0000)\n" +
"(0) (C5) ($0.00000)\n" +
"(0) (C6) ($0.000000)\n" +
"(0) (C7) ($0.0000000)\n" +
"(0) (C8) ($0.00000000)\n" +
"(0) (C9) ($0.000000000)\n" +
"(0) (C10) ($0.0000000000)\n" +
"(0) (C99) ($0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (D) (0)\n" +
"(0) (D0) (0)\n" +
"(0) (D1) (0)\n" +
"(0) (D2) (00)\n" +
"(0) (D3) (000)\n" +
"(0) (D4) (0000)\n" +
"(0) (D5) (00000)\n" +
"(0) (D6) (000000)\n" +
"(0) (D7) (0000000)\n" +
"(0) (D8) (00000000)\n" +
"(0) (D9) (000000000)\n" +
"(0) (D10) (0000000000)\n" +
"(0) (D99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (E) (0.000000E+000)\n" +
"(0) (E0) (0E+000)\n" +
"(0) (E1) (0.0E+000)\n" +
"(0) (E2) (0.00E+000)\n" +
"(0) (E3) (0.000E+000)\n" +
"(0) (E4) (0.0000E+000)\n" +
"(0) (E5) (0.00000E+000)\n" +
"(0) (E6) (0.000000E+000)\n" +
"(0) (E7) (0.0000000E+000)\n" +
"(0) (E8) (0.00000000E+000)\n" +
"(0) (E9) (0.000000000E+000)\n" +
"(0) (E10) (0.0000000000E+000)\n" +
"(0) (E99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000E+000)\n" +
"(0) (F) (0.00)\n" +
"(0) (F0) (0)\n" +
"(0) (F1) (0.0)\n" +
"(0) (F2) (0.00)\n" +
"(0) (F3) (0.000)\n" +
"(0) (F4) (0.0000)\n" +
"(0) (F5) (0.00000)\n" +
"(0) (F6) (0.000000)\n" +
"(0) (F7) (0.0000000)\n" +
"(0) (F8) (0.00000000)\n" +
"(0) (F9) (0.000000000)\n" +
"(0) (F10) (0.0000000000)\n" +
"(0) (F99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (G) (0)\n" +
"(0) (G0) (0)\n" +
"(0) (G1) (0)\n" +
"(0) (G2) (0)\n" +
"(0) (G3) (0)\n" +
"(0) (G4) (0)\n" +
"(0) (G5) (0)\n" +
"(0) (G6) (0)\n" +
"(0) (G7) (0)\n" +
"(0) (G8) (0)\n" +
"(0) (G9) (0)\n" +
"(0) (G10) (0)\n" +
"(0) (G99) (0)\n" +
"(0) (N) (0.00)\n" +
"(0) (N0) (0)\n" +
"(0) (N1) (0.0)\n" +
"(0) (N2) (0.00)\n" +
"(0) (N3) (0.000)\n" +
"(0) (N4) (0.0000)\n" +
"(0) (N5) (0.00000)\n" +
"(0) (N6) (0.000000)\n" +
"(0) (N7) (0.0000000)\n" +
"(0) (N8) (0.00000000)\n" +
"(0) (N9) (0.000000000)\n" +
"(0) (N10) (0.0000000000)\n" +
"(0) (N99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (P) (0.00 %)\n" +
"(0) (P0) (0 %)\n" +
"(0) (P1) (0.0 %)\n" +
"(0) (P2) (0.00 %)\n" +
"(0) (P3) (0.000 %)\n" +
"(0) (P4) (0.0000 %)\n" +
"(0) (P5) (0.00000 %)\n" +
"(0) (P6) (0.000000 %)\n" +
"(0) (P7) (0.0000000 %)\n" +
"(0) (P8) (0.00000000 %)\n" +
"(0) (P9) (0.000000000 %)\n" +
"(0) (P10) (0.0000000000 %)\n" +
"(0) (P99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000 %)\n" +
"(0) (X) (0)\n" +
"(0) (X0) (0)\n" +
"(0) (X1) (0)\n" +
"(0) (X2) (00)\n" +
"(0) (X3) (000)\n" +
"(0) (X4) (0000)\n" +
"(0) (X5) (00000)\n" +
"(0) (X6) (000000)\n" +
"(0) (X7) (0000000)\n" +
"(0) (X8) (00000000)\n" +
"(0) (X9) (000000000)\n" +
"(0) (X10) (0000000000)\n" +
"(0) (X99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(-100) (C) (($100.00))\n" +
"(-100) (C0) (($100))\n" +
"(-100) (C1) (($100.0))\n" +
"(-100) (C2) (($100.00))\n" +
"(-100) (C3) (($100.000))\n" +
"(-100) (C4) (($100.0000))\n" +
"(-100) (C5) (($100.00000))\n" +
"(-100) (C6) (($100.000000))\n" +
"(-100) (C7) (($100.0000000))\n" +
"(-100) (C8) (($100.00000000))\n" +
"(-100) (C9) (($100.000000000))\n" +
"(-100) (C10) (($100.0000000000))\n" +
"(-100) (C99) (($100.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000))\n" +
"(-100) (D) (-100)\n" +
"(-100) (D0) (-100)\n" +
"(-100) (D1) (-100)\n" +
"(-100) (D2) (-100)\n" +
"(-100) (D3) (-100)\n" +
"(-100) (D4) (-0100)\n" +
"(-100) (D5) (-00100)\n" +
"(-100) (D6) (-000100)\n" +
"(-100) (D7) (-0000100)\n" +
"(-100) (D8) (-00000100)\n" +
"(-100) (D9) (-000000100)\n" +
"(-100) (D10) (-0000000100)\n" +
"(-100) (D99) (-000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000100)\n" +
"(-100) (E) (-1.000000E+002)\n" +
"(-100) (E0) (-1E+002)\n" +
"(-100) (E1) (-1.0E+002)\n" +
"(-100) (E2) (-1.00E+002)\n" +
"(-100) (E3) (-1.000E+002)\n" +
"(-100) (E4) (-1.0000E+002)\n" +
"(-100) (E5) (-1.00000E+002)\n" +
"(-100) (E6) (-1.000000E+002)\n" +
"(-100) (E7) (-1.0000000E+002)\n" +
"(-100) (E8) (-1.00000000E+002)\n" +
"(-100) (E9) (-1.000000000E+002)\n" +
"(-100) (E10) (-1.0000000000E+002)\n" +
"(-100) (E99) (-1.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000E+002)\n" +
"(-100) (F) (-100.00)\n" +
"(-100) (F0) (-100)\n" +
"(-100) (F1) (-100.0)\n" +
"(-100) (F2) (-100.00)\n" +
"(-100) (F3) (-100.000)\n" +
"(-100) (F4) (-100.0000)\n" +
"(-100) (F5) (-100.00000)\n" +
"(-100) (F6) (-100.000000)\n" +
"(-100) (F7) (-100.0000000)\n" +
"(-100) (F8) (-100.00000000)\n" +
"(-100) (F9) (-100.000000000)\n" +
"(-100) (F10) (-100.0000000000)\n" +
"(-100) (F99) (-100.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(-100) (G) (-100)\n" +
"(-100) (G0) (-100)\n" +
"(-100) (G1) (-1E+02)\n" +
"(-100) (G2) (-1E+02)\n" +
"(-100) (G3) (-100)\n" +
"(-100) (G4) (-100)\n" +
"(-100) (G5) (-100)\n" +
"(-100) (G6) (-100)\n" +
"(-100) (G7) (-100)\n" +
"(-100) (G8) (-100)\n" +
"(-100) (G9) (-100)\n" +
"(-100) (G10) (-100)\n" +
"(-100) (G99) (-100)\n" +
"(-100) (N) (-100.00)\n" +
"(-100) (N0) (-100)\n" +
"(-100) (N1) (-100.0)\n" +
"(-100) (N2) (-100.00)\n" +
"(-100) (N3) (-100.000)\n" +
"(-100) (N4) (-100.0000)\n" +
"(-100) (N5) (-100.00000)\n" +
"(-100) (N6) (-100.000000)\n" +
"(-100) (N7) (-100.0000000)\n" +
"(-100) (N8) (-100.00000000)\n" +
"(-100) (N9) (-100.000000000)\n" +
"(-100) (N10) (-100.0000000000)\n" +
"(-100) (N99) (-100.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(-100) (P) (-10,000.00 %)\n" +
"(-100) (P0) (-10,000 %)\n" +
"(-100) (P1) (-10,000.0 %)\n" +
"(-100) (P2) (-10,000.00 %)\n" +
"(-100) (P3) (-10,000.000 %)\n" +
"(-100) (P4) (-10,000.0000 %)\n" +
"(-100) (P5) (-10,000.00000 %)\n" +
"(-100) (P6) (-10,000.000000 %)\n" +
"(-100) (P7) (-10,000.0000000 %)\n" +
"(-100) (P8) (-10,000.00000000 %)\n" +
"(-100) (P9) (-10,000.000000000 %)\n" +
"(-100) (P10) (-10,000.0000000000 %)\n" +
"(-100) (P99) (-10,000.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000 %)\n" +
"(-100) (X) (FFFFFF9C)\n" +
"(-100) (X0) (FFFFFF9C)\n" +
"(-100) (X1) (FFFFFF9C)\n" +
"(-100) (X2) (FFFFFF9C)\n" +
"(-100) (X3) (FFFFFF9C)\n" +
"(-100) (X4) (FFFFFF9C)\n" +
"(-100) (X5) (FFFFFF9C)\n" +
"(-100) (X6) (FFFFFF9C)\n" +
"(-100) (X7) (FFFFFF9C)\n" +
"(-100) (X8) (FFFFFF9C)\n" +
"(-100) (X9) (0FFFFFF9C)\n" +
"(-100) (X10) (00FFFFFF9C)\n" +
"(-100) (X99) (0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000FFFFFF9C)\n" +
"(100) (C) ($100.00)\n" +
"(100) (C0) ($100)\n" +
"(100) (C1) ($100.0)\n" +
"(100) (C2) ($100.00)\n" +
"(100) (C3) ($100.000)\n" +
"(100) (C4) ($100.0000)\n" +
"(100) (C5) ($100.00000)\n" +
"(100) (C6) ($100.000000)\n" +
"(100) (C7) ($100.0000000)\n" +
"(100) (C8) ($100.00000000)\n" +
"(100) (C9) ($100.000000000)\n" +
"(100) (C10) ($100.0000000000)\n" +
"(100) (C99) ($100.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(100) (D) (100)\n" +
"(100) (D0) (100)\n" +
"(100) (D1) (100)\n" +
"(100) (D2) (100)\n" +
"(100) (D3) (100)\n" +
"(100) (D4) (0100)\n" +
"(100) (D5) (00100)\n" +
"(100) (D6) (000100)\n" +
"(100) (D7) (0000100)\n" +
"(100) (D8) (00000100)\n" +
"(100) (D9) (000000100)\n" +
"(100) (D10) (0000000100)\n" +
"(100) (D99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000100)\n" +
"(100) (E) (1.000000E+002)\n" +
"(100) (E0) (1E+002)\n" +
"(100) (E1) (1.0E+002)\n" +
"(100) (E2) (1.00E+002)\n" +
"(100) (E3) (1.000E+002)\n" +
"(100) (E4) (1.0000E+002)\n" +
"(100) (E5) (1.00000E+002)\n" +
"(100) (E6) (1.000000E+002)\n" +
"(100) (E7) (1.0000000E+002)\n" +
"(100) (E8) (1.00000000E+002)\n" +
"(100) (E9) (1.000000000E+002)\n" +
"(100) (E10) (1.0000000000E+002)\n" +
"(100) (E99) (1.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000E+002)\n" +
"(100) (F) (100.00)\n" +
"(100) (F0) (100)\n" +
"(100) (F1) (100.0)\n" +
"(100) (F2) (100.00)\n" +
"(100) (F3) (100.000)\n" +
"(100) (F4) (100.0000)\n" +
"(100) (F5) (100.00000)\n" +
"(100) (F6) (100.000000)\n" +
"(100) (F7) (100.0000000)\n" +
"(100) (F8) (100.00000000)\n" +
"(100) (F9) (100.000000000)\n" +
"(100) (F10) (100.0000000000)\n" +
"(100) (F99) (100.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(100) (G) (100)\n" +
"(100) (G0) (100)\n" +
"(100) (G1) (1E+02)\n" +
"(100) (G2) (1E+02)\n" +
"(100) (G3) (100)\n" +
"(100) (G4) (100)\n" +
"(100) (G5) (100)\n" +
"(100) (G6) (100)\n" +
"(100) (G7) (100)\n" +
"(100) (G8) (100)\n" +
"(100) (G9) (100)\n" +
"(100) (G10) (100)\n" +
"(100) (G99) (100)\n" +
"(100) (N) (100.00)\n" +
"(100) (N0) (100)\n" +
"(100) (N1) (100.0)\n" +
"(100) (N2) (100.00)\n" +
"(100) (N3) (100.000)\n" +
"(100) (N4) (100.0000)\n" +
"(100) (N5) (100.00000)\n" +
"(100) (N6) (100.000000)\n" +
"(100) (N7) (100.0000000)\n" +
"(100) (N8) (100.00000000)\n" +
"(100) (N9) (100.000000000)\n" +
"(100) (N10) (100.0000000000)\n" +
"(100) (N99) (100.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(100) (P) (10,000.00 %)\n" +
"(100) (P0) (10,000 %)\n" +
"(100) (P1) (10,000.0 %)\n" +
"(100) (P2) (10,000.00 %)\n" +
"(100) (P3) (10,000.000 %)\n" +
"(100) (P4) (10,000.0000 %)\n" +
"(100) (P5) (10,000.00000 %)\n" +
"(100) (P6) (10,000.000000 %)\n" +
"(100) (P7) (10,000.0000000 %)\n" +
"(100) (P8) (10,000.00000000 %)\n" +
"(100) (P9) (10,000.000000000 %)\n" +
"(100) (P10) (10,000.0000000000 %)\n" +
"(100) (P99) (10,000.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000 %)\n" +
"(100) (X) (64)\n" +
"(100) (X0) (64)\n" +
"(100) (X1) (64)\n" +
"(100) (X2) (64)\n" +
"(100) (X3) (064)\n" +
"(100) (X4) (0064)\n" +
"(100) (X5) (00064)\n" +
"(100) (X6) (000064)\n" +
"(100) (X7) (0000064)\n" +
"(100) (X8) (00000064)\n" +
"(100) (X9) (000000064)\n" +
"(100) (X10) (0000000064)\n" +
"(100) (X99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000064)\n" +
"Int64\n" +
"(-9223372036854775808) (C) (($9,223,372,036,854,775,808.00))\n" +
"(-9223372036854775808) (C0) (($9,223,372,036,854,775,808))\n" +
"(-9223372036854775808) (C1) (($9,223,372,036,854,775,808.0))\n" +
"(-9223372036854775808) (C2) (($9,223,372,036,854,775,808.00))\n" +
"(-9223372036854775808) (C3) (($9,223,372,036,854,775,808.000))\n" +
"(-9223372036854775808) (C4) (($9,223,372,036,854,775,808.0000))\n" +
"(-9223372036854775808) (C5) (($9,223,372,036,854,775,808.00000))\n" +
"(-9223372036854775808) (C6) (($9,223,372,036,854,775,808.000000))\n" +
"(-9223372036854775808) (C7) (($9,223,372,036,854,775,808.0000000))\n" +
"(-9223372036854775808) (C8) (($9,223,372,036,854,775,808.00000000))\n" +
"(-9223372036854775808) (C9) (($9,223,372,036,854,775,808.000000000))\n" +
"(-9223372036854775808) (C10) (($9,223,372,036,854,775,808.0000000000))\n" +
"(-9223372036854775808) (C99) (($9,223,372,036,854,775,808.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000))\n" +
"(-9223372036854775808) (D) (-9223372036854775808)\n" +
"(-9223372036854775808) (D0) (-9223372036854775808)\n" +
"(-9223372036854775808) (D1) (-9223372036854775808)\n" +
"(-9223372036854775808) (D2) (-9223372036854775808)\n" +
"(-9223372036854775808) (D3) (-9223372036854775808)\n" +
"(-9223372036854775808) (D4) (-9223372036854775808)\n" +
"(-9223372036854775808) (D5) (-9223372036854775808)\n" +
"(-9223372036854775808) (D6) (-9223372036854775808)\n" +
"(-9223372036854775808) (D7) (-9223372036854775808)\n" +
"(-9223372036854775808) (D8) (-9223372036854775808)\n" +
"(-9223372036854775808) (D9) (-9223372036854775808)\n" +
"(-9223372036854775808) (D10) (-9223372036854775808)\n" +
"(-9223372036854775808) (D99) (-000000000000000000000000000000000000000000000000000000000000000000000000000000009223372036854775808)\n" +
"(-9223372036854775808) (E) (-9.223372E+018)\n" +
"(-9223372036854775808) (E0) (-9E+018)\n" +
"(-9223372036854775808) (E1) (-9.2E+018)\n" +
"(-9223372036854775808) (E2) (-9.22E+018)\n" +
"(-9223372036854775808) (E3) (-9.223E+018)\n" +
"(-9223372036854775808) (E4) (-9.2234E+018)\n" +
"(-9223372036854775808) (E5) (-9.22337E+018)\n" +
"(-9223372036854775808) (E6) (-9.223372E+018)\n" +
"(-9223372036854775808) (E7) (-9.2233720E+018)\n" +
"(-9223372036854775808) (E8) (-9.22337204E+018)\n" +
"(-9223372036854775808) (E9) (-9.223372037E+018)\n" +
"(-9223372036854775808) (E10) (-9.2233720369E+018)\n" +
"(-9223372036854775808) (E99) (-9.223372036854775808000000000000000000000000000000000000000000000000000000000000000000000000000000000E+018)\n" +
"(-9223372036854775808) (F) (-9223372036854775808.00)\n" +
"(-9223372036854775808) (F0) (-9223372036854775808)\n" +
"(-9223372036854775808) (F1) (-9223372036854775808.0)\n" +
"(-9223372036854775808) (F2) (-9223372036854775808.00)\n" +
"(-9223372036854775808) (F3) (-9223372036854775808.000)\n" +
"(-9223372036854775808) (F4) (-9223372036854775808.0000)\n" +
"(-9223372036854775808) (F5) (-9223372036854775808.00000)\n" +
"(-9223372036854775808) (F6) (-9223372036854775808.000000)\n" +
"(-9223372036854775808) (F7) (-9223372036854775808.0000000)\n" +
"(-9223372036854775808) (F8) (-9223372036854775808.00000000)\n" +
"(-9223372036854775808) (F9) (-9223372036854775808.000000000)\n" +
"(-9223372036854775808) (F10) (-9223372036854775808.0000000000)\n" +
"(-9223372036854775808) (F99) (-9223372036854775808.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(-9223372036854775808) (G) (-9223372036854775808)\n" +
"(-9223372036854775808) (G0) (-9223372036854775808)\n" +
"(-9223372036854775808) (G1) (-9E+18)\n" +
"(-9223372036854775808) (G2) (-9.2E+18)\n" +
"(-9223372036854775808) (G3) (-9.22E+18)\n" +
"(-9223372036854775808) (G4) (-9.223E+18)\n" +
"(-9223372036854775808) (G5) (-9.2234E+18)\n" +
"(-9223372036854775808) (G6) (-9.22337E+18)\n" +
"(-9223372036854775808) (G7) (-9.223372E+18)\n" +
"(-9223372036854775808) (G8) (-9.223372E+18)\n" +
"(-9223372036854775808) (G9) (-9.22337204E+18)\n" +
"(-9223372036854775808) (G10) (-9.223372037E+18)\n" +
"(-9223372036854775808) (G99) (-9223372036854775808)\n" +
"(-9223372036854775808) (N) (-9,223,372,036,854,775,808.00)\n" +
"(-9223372036854775808) (N0) (-9,223,372,036,854,775,808)\n" +
"(-9223372036854775808) (N1) (-9,223,372,036,854,775,808.0)\n" +
"(-9223372036854775808) (N2) (-9,223,372,036,854,775,808.00)\n" +
"(-9223372036854775808) (N3) (-9,223,372,036,854,775,808.000)\n" +
"(-9223372036854775808) (N4) (-9,223,372,036,854,775,808.0000)\n" +
"(-9223372036854775808) (N5) (-9,223,372,036,854,775,808.00000)\n" +
"(-9223372036854775808) (N6) (-9,223,372,036,854,775,808.000000)\n" +
"(-9223372036854775808) (N7) (-9,223,372,036,854,775,808.0000000)\n" +
"(-9223372036854775808) (N8) (-9,223,372,036,854,775,808.00000000)\n" +
"(-9223372036854775808) (N9) (-9,223,372,036,854,775,808.000000000)\n" +
"(-9223372036854775808) (N10) (-9,223,372,036,854,775,808.0000000000)\n" +
"(-9223372036854775808) (N99) (-9,223,372,036,854,775,808.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(-9223372036854775808) (P) (-922,337,203,685,477,580,800.00 %)\n" +
"(-9223372036854775808) (P0) (-922,337,203,685,477,580,800 %)\n" +
"(-9223372036854775808) (P1) (-922,337,203,685,477,580,800.0 %)\n" +
"(-9223372036854775808) (P2) (-922,337,203,685,477,580,800.00 %)\n" +
"(-9223372036854775808) (P3) (-922,337,203,685,477,580,800.000 %)\n" +
"(-9223372036854775808) (P4) (-922,337,203,685,477,580,800.0000 %)\n" +
"(-9223372036854775808) (P5) (-922,337,203,685,477,580,800.00000 %)\n" +
"(-9223372036854775808) (P6) (-922,337,203,685,477,580,800.000000 %)\n" +
"(-9223372036854775808) (P7) (-922,337,203,685,477,580,800.0000000 %)\n" +
"(-9223372036854775808) (P8) (-922,337,203,685,477,580,800.00000000 %)\n" +
"(-9223372036854775808) (P9) (-922,337,203,685,477,580,800.000000000 %)\n" +
"(-9223372036854775808) (P10) (-922,337,203,685,477,580,800.0000000000 %)\n" +
"(-9223372036854775808) (P99) (-922,337,203,685,477,580,800.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000 %)\n" +
"(-9223372036854775808) (X) (8000000000000000)\n" +
"(-9223372036854775808) (X0) (8000000000000000)\n" +
"(-9223372036854775808) (X1) (8000000000000000)\n" +
"(-9223372036854775808) (X2) (8000000000000000)\n" +
"(-9223372036854775808) (X3) (8000000000000000)\n" +
"(-9223372036854775808) (X4) (8000000000000000)\n" +
"(-9223372036854775808) (X5) (8000000000000000)\n" +
"(-9223372036854775808) (X6) (8000000000000000)\n" +
"(-9223372036854775808) (X7) (8000000000000000)\n" +
"(-9223372036854775808) (X8) (8000000000000000)\n" +
"(-9223372036854775808) (X9) (8000000000000000)\n" +
"(-9223372036854775808) (X10) (8000000000000000)\n" +
"(-9223372036854775808) (X99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000008000000000000000)\n" +
"(9223372036854775807) (C) ($9,223,372,036,854,775,807.00)\n" +
"(9223372036854775807) (C0) ($9,223,372,036,854,775,807)\n" +
"(9223372036854775807) (C1) ($9,223,372,036,854,775,807.0)\n" +
"(9223372036854775807) (C2) ($9,223,372,036,854,775,807.00)\n" +
"(9223372036854775807) (C3) ($9,223,372,036,854,775,807.000)\n" +
"(9223372036854775807) (C4) ($9,223,372,036,854,775,807.0000)\n" +
"(9223372036854775807) (C5) ($9,223,372,036,854,775,807.00000)\n" +
"(9223372036854775807) (C6) ($9,223,372,036,854,775,807.000000)\n" +
"(9223372036854775807) (C7) ($9,223,372,036,854,775,807.0000000)\n" +
"(9223372036854775807) (C8) ($9,223,372,036,854,775,807.00000000)\n" +
"(9223372036854775807) (C9) ($9,223,372,036,854,775,807.000000000)\n" +
"(9223372036854775807) (C10) ($9,223,372,036,854,775,807.0000000000)\n" +
"(9223372036854775807) (C99) ($9,223,372,036,854,775,807.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(9223372036854775807) (D) (9223372036854775807)\n" +
"(9223372036854775807) (D0) (9223372036854775807)\n" +
"(9223372036854775807) (D1) (9223372036854775807)\n" +
"(9223372036854775807) (D2) (9223372036854775807)\n" +
"(9223372036854775807) (D3) (9223372036854775807)\n" +
"(9223372036854775807) (D4) (9223372036854775807)\n" +
"(9223372036854775807) (D5) (9223372036854775807)\n" +
"(9223372036854775807) (D6) (9223372036854775807)\n" +
"(9223372036854775807) (D7) (9223372036854775807)\n" +
"(9223372036854775807) (D8) (9223372036854775807)\n" +
"(9223372036854775807) (D9) (9223372036854775807)\n" +
"(9223372036854775807) (D10) (9223372036854775807)\n" +
"(9223372036854775807) (D99) (000000000000000000000000000000000000000000000000000000000000000000000000000000009223372036854775807)\n" +
"(9223372036854775807) (E) (9.223372E+018)\n" +
"(9223372036854775807) (E0) (9E+018)\n" +
"(9223372036854775807) (E1) (9.2E+018)\n" +
"(9223372036854775807) (E2) (9.22E+018)\n" +
"(9223372036854775807) (E3) (9.223E+018)\n" +
"(9223372036854775807) (E4) (9.2234E+018)\n" +
"(9223372036854775807) (E5) (9.22337E+018)\n" +
"(9223372036854775807) (E6) (9.223372E+018)\n" +
"(9223372036854775807) (E7) (9.2233720E+018)\n" +
"(9223372036854775807) (E8) (9.22337204E+018)\n" +
"(9223372036854775807) (E9) (9.223372037E+018)\n" +
"(9223372036854775807) (E10) (9.2233720369E+018)\n" +
"(9223372036854775807) (E99) (9.223372036854775807000000000000000000000000000000000000000000000000000000000000000000000000000000000E+018)\n" +
"(9223372036854775807) (F) (9223372036854775807.00)\n" +
"(9223372036854775807) (F0) (9223372036854775807)\n" +
"(9223372036854775807) (F1) (9223372036854775807.0)\n" +
"(9223372036854775807) (F2) (9223372036854775807.00)\n" +
"(9223372036854775807) (F3) (9223372036854775807.000)\n" +
"(9223372036854775807) (F4) (9223372036854775807.0000)\n" +
"(9223372036854775807) (F5) (9223372036854775807.00000)\n" +
"(9223372036854775807) (F6) (9223372036854775807.000000)\n" +
"(9223372036854775807) (F7) (9223372036854775807.0000000)\n" +
"(9223372036854775807) (F8) (9223372036854775807.00000000)\n" +
"(9223372036854775807) (F9) (9223372036854775807.000000000)\n" +
"(9223372036854775807) (F10) (9223372036854775807.0000000000)\n" +
"(9223372036854775807) (F99) (9223372036854775807.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(9223372036854775807) (G) (9223372036854775807)\n" +
"(9223372036854775807) (G0) (9223372036854775807)\n" +
"(9223372036854775807) (G1) (9E+18)\n" +
"(9223372036854775807) (G2) (9.2E+18)\n" +
"(9223372036854775807) (G3) (9.22E+18)\n" +
"(9223372036854775807) (G4) (9.223E+18)\n" +
"(9223372036854775807) (G5) (9.2234E+18)\n" +
"(9223372036854775807) (G6) (9.22337E+18)\n" +
"(9223372036854775807) (G7) (9.223372E+18)\n" +
"(9223372036854775807) (G8) (9.223372E+18)\n" +
"(9223372036854775807) (G9) (9.22337204E+18)\n" +
"(9223372036854775807) (G10) (9.223372037E+18)\n" +
"(9223372036854775807) (G99) (9223372036854775807)\n" +
"(9223372036854775807) (N) (9,223,372,036,854,775,807.00)\n" +
"(9223372036854775807) (N0) (9,223,372,036,854,775,807)\n" +
"(9223372036854775807) (N1) (9,223,372,036,854,775,807.0)\n" +
"(9223372036854775807) (N2) (9,223,372,036,854,775,807.00)\n" +
"(9223372036854775807) (N3) (9,223,372,036,854,775,807.000)\n" +
"(9223372036854775807) (N4) (9,223,372,036,854,775,807.0000)\n" +
"(9223372036854775807) (N5) (9,223,372,036,854,775,807.00000)\n" +
"(9223372036854775807) (N6) (9,223,372,036,854,775,807.000000)\n" +
"(9223372036854775807) (N7) (9,223,372,036,854,775,807.0000000)\n" +
"(9223372036854775807) (N8) (9,223,372,036,854,775,807.00000000)\n" +
"(9223372036854775807) (N9) (9,223,372,036,854,775,807.000000000)\n" +
"(9223372036854775807) (N10) (9,223,372,036,854,775,807.0000000000)\n" +
"(9223372036854775807) (N99) (9,223,372,036,854,775,807.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(9223372036854775807) (P) (922,337,203,685,477,580,700.00 %)\n" +
"(9223372036854775807) (P0) (922,337,203,685,477,580,700 %)\n" +
"(9223372036854775807) (P1) (922,337,203,685,477,580,700.0 %)\n" +
"(9223372036854775807) (P2) (922,337,203,685,477,580,700.00 %)\n" +
"(9223372036854775807) (P3) (922,337,203,685,477,580,700.000 %)\n" +
"(9223372036854775807) (P4) (922,337,203,685,477,580,700.0000 %)\n" +
"(9223372036854775807) (P5) (922,337,203,685,477,580,700.00000 %)\n" +
"(9223372036854775807) (P6) (922,337,203,685,477,580,700.000000 %)\n" +
"(9223372036854775807) (P7) (922,337,203,685,477,580,700.0000000 %)\n" +
"(9223372036854775807) (P8) (922,337,203,685,477,580,700.00000000 %)\n" +
"(9223372036854775807) (P9) (922,337,203,685,477,580,700.000000000 %)\n" +
"(9223372036854775807) (P10) (922,337,203,685,477,580,700.0000000000 %)\n" +
"(9223372036854775807) (P99) (922,337,203,685,477,580,700.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000 %)\n" +
"(9223372036854775807) (X) (7FFFFFFFFFFFFFFF)\n" +
"(9223372036854775807) (X0) (7FFFFFFFFFFFFFFF)\n" +
"(9223372036854775807) (X1) (7FFFFFFFFFFFFFFF)\n" +
"(9223372036854775807) (X2) (7FFFFFFFFFFFFFFF)\n" +
"(9223372036854775807) (X3) (7FFFFFFFFFFFFFFF)\n" +
"(9223372036854775807) (X4) (7FFFFFFFFFFFFFFF)\n" +
"(9223372036854775807) (X5) (7FFFFFFFFFFFFFFF)\n" +
"(9223372036854775807) (X6) (7FFFFFFFFFFFFFFF)\n" +
"(9223372036854775807) (X7) (7FFFFFFFFFFFFFFF)\n" +
"(9223372036854775807) (X8) (7FFFFFFFFFFFFFFF)\n" +
"(9223372036854775807) (X9) (7FFFFFFFFFFFFFFF)\n" +
"(9223372036854775807) (X10) (7FFFFFFFFFFFFFFF)\n" +
"(9223372036854775807) (X99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000007FFFFFFFFFFFFFFF)\n" +
"(0) (C) ($0.00)\n" +
"(0) (C0) ($0)\n" +
"(0) (C1) ($0.0)\n" +
"(0) (C2) ($0.00)\n" +
"(0) (C3) ($0.000)\n" +
"(0) (C4) ($0.0000)\n" +
"(0) (C5) ($0.00000)\n" +
"(0) (C6) ($0.000000)\n" +
"(0) (C7) ($0.0000000)\n" +
"(0) (C8) ($0.00000000)\n" +
"(0) (C9) ($0.000000000)\n" +
"(0) (C10) ($0.0000000000)\n" +
"(0) (C99) ($0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (D) (0)\n" +
"(0) (D0) (0)\n" +
"(0) (D1) (0)\n" +
"(0) (D2) (00)\n" +
"(0) (D3) (000)\n" +
"(0) (D4) (0000)\n" +
"(0) (D5) (00000)\n" +
"(0) (D6) (000000)\n" +
"(0) (D7) (0000000)\n" +
"(0) (D8) (00000000)\n" +
"(0) (D9) (000000000)\n" +
"(0) (D10) (0000000000)\n" +
"(0) (D99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (E) (0.000000E+000)\n" +
"(0) (E0) (0E+000)\n" +
"(0) (E1) (0.0E+000)\n" +
"(0) (E2) (0.00E+000)\n" +
"(0) (E3) (0.000E+000)\n" +
"(0) (E4) (0.0000E+000)\n" +
"(0) (E5) (0.00000E+000)\n" +
"(0) (E6) (0.000000E+000)\n" +
"(0) (E7) (0.0000000E+000)\n" +
"(0) (E8) (0.00000000E+000)\n" +
"(0) (E9) (0.000000000E+000)\n" +
"(0) (E10) (0.0000000000E+000)\n" +
"(0) (E99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000E+000)\n" +
"(0) (F) (0.00)\n" +
"(0) (F0) (0)\n" +
"(0) (F1) (0.0)\n" +
"(0) (F2) (0.00)\n" +
"(0) (F3) (0.000)\n" +
"(0) (F4) (0.0000)\n" +
"(0) (F5) (0.00000)\n" +
"(0) (F6) (0.000000)\n" +
"(0) (F7) (0.0000000)\n" +
"(0) (F8) (0.00000000)\n" +
"(0) (F9) (0.000000000)\n" +
"(0) (F10) (0.0000000000)\n" +
"(0) (F99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (G) (0)\n" +
"(0) (G0) (0)\n" +
"(0) (G1) (0)\n" +
"(0) (G2) (0)\n" +
"(0) (G3) (0)\n" +
"(0) (G4) (0)\n" +
"(0) (G5) (0)\n" +
"(0) (G6) (0)\n" +
"(0) (G7) (0)\n" +
"(0) (G8) (0)\n" +
"(0) (G9) (0)\n" +
"(0) (G10) (0)\n" +
"(0) (G99) (0)\n" +
"(0) (N) (0.00)\n" +
"(0) (N0) (0)\n" +
"(0) (N1) (0.0)\n" +
"(0) (N2) (0.00)\n" +
"(0) (N3) (0.000)\n" +
"(0) (N4) (0.0000)\n" +
"(0) (N5) (0.00000)\n" +
"(0) (N6) (0.000000)\n" +
"(0) (N7) (0.0000000)\n" +
"(0) (N8) (0.00000000)\n" +
"(0) (N9) (0.000000000)\n" +
"(0) (N10) (0.0000000000)\n" +
"(0) (N99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (P) (0.00 %)\n" +
"(0) (P0) (0 %)\n" +
"(0) (P1) (0.0 %)\n" +
"(0) (P2) (0.00 %)\n" +
"(0) (P3) (0.000 %)\n" +
"(0) (P4) (0.0000 %)\n" +
"(0) (P5) (0.00000 %)\n" +
"(0) (P6) (0.000000 %)\n" +
"(0) (P7) (0.0000000 %)\n" +
"(0) (P8) (0.00000000 %)\n" +
"(0) (P9) (0.000000000 %)\n" +
"(0) (P10) (0.0000000000 %)\n" +
"(0) (P99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000 %)\n" +
"(0) (X) (0)\n" +
"(0) (X0) (0)\n" +
"(0) (X1) (0)\n" +
"(0) (X2) (00)\n" +
"(0) (X3) (000)\n" +
"(0) (X4) (0000)\n" +
"(0) (X5) (00000)\n" +
"(0) (X6) (000000)\n" +
"(0) (X7) (0000000)\n" +
"(0) (X8) (00000000)\n" +
"(0) (X9) (000000000)\n" +
"(0) (X10) (0000000000)\n" +
"(0) (X99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(-100) (C) (($100.00))\n" +
"(-100) (C0) (($100))\n" +
"(-100) (C1) (($100.0))\n" +
"(-100) (C2) (($100.00))\n" +
"(-100) (C3) (($100.000))\n" +
"(-100) (C4) (($100.0000))\n" +
"(-100) (C5) (($100.00000))\n" +
"(-100) (C6) (($100.000000))\n" +
"(-100) (C7) (($100.0000000))\n" +
"(-100) (C8) (($100.00000000))\n" +
"(-100) (C9) (($100.000000000))\n" +
"(-100) (C10) (($100.0000000000))\n" +
"(-100) (C99) (($100.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000))\n" +
"(-100) (D) (-100)\n" +
"(-100) (D0) (-100)\n" +
"(-100) (D1) (-100)\n" +
"(-100) (D2) (-100)\n" +
"(-100) (D3) (-100)\n" +
"(-100) (D4) (-0100)\n" +
"(-100) (D5) (-00100)\n" +
"(-100) (D6) (-000100)\n" +
"(-100) (D7) (-0000100)\n" +
"(-100) (D8) (-00000100)\n" +
"(-100) (D9) (-000000100)\n" +
"(-100) (D10) (-0000000100)\n" +
"(-100) (D99) (-000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000100)\n" +
"(-100) (E) (-1.000000E+002)\n" +
"(-100) (E0) (-1E+002)\n" +
"(-100) (E1) (-1.0E+002)\n" +
"(-100) (E2) (-1.00E+002)\n" +
"(-100) (E3) (-1.000E+002)\n" +
"(-100) (E4) (-1.0000E+002)\n" +
"(-100) (E5) (-1.00000E+002)\n" +
"(-100) (E6) (-1.000000E+002)\n" +
"(-100) (E7) (-1.0000000E+002)\n" +
"(-100) (E8) (-1.00000000E+002)\n" +
"(-100) (E9) (-1.000000000E+002)\n" +
"(-100) (E10) (-1.0000000000E+002)\n" +
"(-100) (E99) (-1.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000E+002)\n" +
"(-100) (F) (-100.00)\n" +
"(-100) (F0) (-100)\n" +
"(-100) (F1) (-100.0)\n" +
"(-100) (F2) (-100.00)\n" +
"(-100) (F3) (-100.000)\n" +
"(-100) (F4) (-100.0000)\n" +
"(-100) (F5) (-100.00000)\n" +
"(-100) (F6) (-100.000000)\n" +
"(-100) (F7) (-100.0000000)\n" +
"(-100) (F8) (-100.00000000)\n" +
"(-100) (F9) (-100.000000000)\n" +
"(-100) (F10) (-100.0000000000)\n" +
"(-100) (F99) (-100.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(-100) (G) (-100)\n" +
"(-100) (G0) (-100)\n" +
"(-100) (G1) (-1E+02)\n" +
"(-100) (G2) (-1E+02)\n" +
"(-100) (G3) (-100)\n" +
"(-100) (G4) (-100)\n" +
"(-100) (G5) (-100)\n" +
"(-100) (G6) (-100)\n" +
"(-100) (G7) (-100)\n" +
"(-100) (G8) (-100)\n" +
"(-100) (G9) (-100)\n" +
"(-100) (G10) (-100)\n" +
"(-100) (G99) (-100)\n" +
"(-100) (N) (-100.00)\n" +
"(-100) (N0) (-100)\n" +
"(-100) (N1) (-100.0)\n" +
"(-100) (N2) (-100.00)\n" +
"(-100) (N3) (-100.000)\n" +
"(-100) (N4) (-100.0000)\n" +
"(-100) (N5) (-100.00000)\n" +
"(-100) (N6) (-100.000000)\n" +
"(-100) (N7) (-100.0000000)\n" +
"(-100) (N8) (-100.00000000)\n" +
"(-100) (N9) (-100.000000000)\n" +
"(-100) (N10) (-100.0000000000)\n" +
"(-100) (N99) (-100.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(-100) (P) (-10,000.00 %)\n" +
"(-100) (P0) (-10,000 %)\n" +
"(-100) (P1) (-10,000.0 %)\n" +
"(-100) (P2) (-10,000.00 %)\n" +
"(-100) (P3) (-10,000.000 %)\n" +
"(-100) (P4) (-10,000.0000 %)\n" +
"(-100) (P5) (-10,000.00000 %)\n" +
"(-100) (P6) (-10,000.000000 %)\n" +
"(-100) (P7) (-10,000.0000000 %)\n" +
"(-100) (P8) (-10,000.00000000 %)\n" +
"(-100) (P9) (-10,000.000000000 %)\n" +
"(-100) (P10) (-10,000.0000000000 %)\n" +
"(-100) (P99) (-10,000.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000 %)\n" +
"(-100) (X) (FFFFFFFFFFFFFF9C)\n" +
"(-100) (X0) (FFFFFFFFFFFFFF9C)\n" +
"(-100) (X1) (FFFFFFFFFFFFFF9C)\n" +
"(-100) (X2) (FFFFFFFFFFFFFF9C)\n" +
"(-100) (X3) (FFFFFFFFFFFFFF9C)\n" +
"(-100) (X4) (FFFFFFFFFFFFFF9C)\n" +
"(-100) (X5) (FFFFFFFFFFFFFF9C)\n" +
"(-100) (X6) (FFFFFFFFFFFFFF9C)\n" +
"(-100) (X7) (FFFFFFFFFFFFFF9C)\n" +
"(-100) (X8) (FFFFFFFFFFFFFF9C)\n" +
"(-100) (X9) (FFFFFFFFFFFFFF9C)\n" +
"(-100) (X10) (FFFFFFFFFFFFFF9C)\n" +
"(-100) (X99) (00000000000000000000000000000000000000000000000000000000000000000000000000000000000FFFFFFFFFFFFFF9C)\n" +
"(100) (C) ($100.00)\n" +
"(100) (C0) ($100)\n" +
"(100) (C1) ($100.0)\n" +
"(100) (C2) ($100.00)\n" +
"(100) (C3) ($100.000)\n" +
"(100) (C4) ($100.0000)\n" +
"(100) (C5) ($100.00000)\n" +
"(100) (C6) ($100.000000)\n" +
"(100) (C7) ($100.0000000)\n" +
"(100) (C8) ($100.00000000)\n" +
"(100) (C9) ($100.000000000)\n" +
"(100) (C10) ($100.0000000000)\n" +
"(100) (C99) ($100.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(100) (D) (100)\n" +
"(100) (D0) (100)\n" +
"(100) (D1) (100)\n" +
"(100) (D2) (100)\n" +
"(100) (D3) (100)\n" +
"(100) (D4) (0100)\n" +
"(100) (D5) (00100)\n" +
"(100) (D6) (000100)\n" +
"(100) (D7) (0000100)\n" +
"(100) (D8) (00000100)\n" +
"(100) (D9) (000000100)\n" +
"(100) (D10) (0000000100)\n" +
"(100) (D99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000100)\n" +
"(100) (E) (1.000000E+002)\n" +
"(100) (E0) (1E+002)\n" +
"(100) (E1) (1.0E+002)\n" +
"(100) (E2) (1.00E+002)\n" +
"(100) (E3) (1.000E+002)\n" +
"(100) (E4) (1.0000E+002)\n" +
"(100) (E5) (1.00000E+002)\n" +
"(100) (E6) (1.000000E+002)\n" +
"(100) (E7) (1.0000000E+002)\n" +
"(100) (E8) (1.00000000E+002)\n" +
"(100) (E9) (1.000000000E+002)\n" +
"(100) (E10) (1.0000000000E+002)\n" +
"(100) (E99) (1.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000E+002)\n" +
"(100) (F) (100.00)\n" +
"(100) (F0) (100)\n" +
"(100) (F1) (100.0)\n" +
"(100) (F2) (100.00)\n" +
"(100) (F3) (100.000)\n" +
"(100) (F4) (100.0000)\n" +
"(100) (F5) (100.00000)\n" +
"(100) (F6) (100.000000)\n" +
"(100) (F7) (100.0000000)\n" +
"(100) (F8) (100.00000000)\n" +
"(100) (F9) (100.000000000)\n" +
"(100) (F10) (100.0000000000)\n" +
"(100) (F99) (100.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(100) (G) (100)\n" +
"(100) (G0) (100)\n" +
"(100) (G1) (1E+02)\n" +
"(100) (G2) (1E+02)\n" +
"(100) (G3) (100)\n" +
"(100) (G4) (100)\n" +
"(100) (G5) (100)\n" +
"(100) (G6) (100)\n" +
"(100) (G7) (100)\n" +
"(100) (G8) (100)\n" +
"(100) (G9) (100)\n" +
"(100) (G10) (100)\n" +
"(100) (G99) (100)\n" +
"(100) (N) (100.00)\n" +
"(100) (N0) (100)\n" +
"(100) (N1) (100.0)\n" +
"(100) (N2) (100.00)\n" +
"(100) (N3) (100.000)\n" +
"(100) (N4) (100.0000)\n" +
"(100) (N5) (100.00000)\n" +
"(100) (N6) (100.000000)\n" +
"(100) (N7) (100.0000000)\n" +
"(100) (N8) (100.00000000)\n" +
"(100) (N9) (100.000000000)\n" +
"(100) (N10) (100.0000000000)\n" +
"(100) (N99) (100.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(100) (P) (10,000.00 %)\n" +
"(100) (P0) (10,000 %)\n" +
"(100) (P1) (10,000.0 %)\n" +
"(100) (P2) (10,000.00 %)\n" +
"(100) (P3) (10,000.000 %)\n" +
"(100) (P4) (10,000.0000 %)\n" +
"(100) (P5) (10,000.00000 %)\n" +
"(100) (P6) (10,000.000000 %)\n" +
"(100) (P7) (10,000.0000000 %)\n" +
"(100) (P8) (10,000.00000000 %)\n" +
"(100) (P9) (10,000.000000000 %)\n" +
"(100) (P10) (10,000.0000000000 %)\n" +
"(100) (P99) (10,000.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000 %)\n" +
"(100) (X) (64)\n" +
"(100) (X0) (64)\n" +
"(100) (X1) (64)\n" +
"(100) (X2) (64)\n" +
"(100) (X3) (064)\n" +
"(100) (X4) (0064)\n" +
"(100) (X5) (00064)\n" +
"(100) (X6) (000064)\n" +
"(100) (X7) (0000064)\n" +
"(100) (X8) (00000064)\n" +
"(100) (X9) (000000064)\n" +
"(100) (X10) (0000000064)\n" +
"(100) (X99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000064)\n" +
"UInt32\n" +
"(0) (C) ($0.00)\n" +
"(0) (C0) ($0)\n" +
"(0) (C1) ($0.0)\n" +
"(0) (C2) ($0.00)\n" +
"(0) (C3) ($0.000)\n" +
"(0) (C4) ($0.0000)\n" +
"(0) (C5) ($0.00000)\n" +
"(0) (C6) ($0.000000)\n" +
"(0) (C7) ($0.0000000)\n" +
"(0) (C8) ($0.00000000)\n" +
"(0) (C9) ($0.000000000)\n" +
"(0) (C10) ($0.0000000000)\n" +
"(0) (C99) ($0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (D) (0)\n" +
"(0) (D0) (0)\n" +
"(0) (D1) (0)\n" +
"(0) (D2) (00)\n" +
"(0) (D3) (000)\n" +
"(0) (D4) (0000)\n" +
"(0) (D5) (00000)\n" +
"(0) (D6) (000000)\n" +
"(0) (D7) (0000000)\n" +
"(0) (D8) (00000000)\n" +
"(0) (D9) (000000000)\n" +
"(0) (D10) (0000000000)\n" +
"(0) (D99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (E) (0.000000E+000)\n" +
"(0) (E0) (0E+000)\n" +
"(0) (E1) (0.0E+000)\n" +
"(0) (E2) (0.00E+000)\n" +
"(0) (E3) (0.000E+000)\n" +
"(0) (E4) (0.0000E+000)\n" +
"(0) (E5) (0.00000E+000)\n" +
"(0) (E6) (0.000000E+000)\n" +
"(0) (E7) (0.0000000E+000)\n" +
"(0) (E8) (0.00000000E+000)\n" +
"(0) (E9) (0.000000000E+000)\n" +
"(0) (E10) (0.0000000000E+000)\n" +
"(0) (E99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000E+000)\n" +
"(0) (F) (0.00)\n" +
"(0) (F0) (0)\n" +
"(0) (F1) (0.0)\n" +
"(0) (F2) (0.00)\n" +
"(0) (F3) (0.000)\n" +
"(0) (F4) (0.0000)\n" +
"(0) (F5) (0.00000)\n" +
"(0) (F6) (0.000000)\n" +
"(0) (F7) (0.0000000)\n" +
"(0) (F8) (0.00000000)\n" +
"(0) (F9) (0.000000000)\n" +
"(0) (F10) (0.0000000000)\n" +
"(0) (F99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (G) (0)\n" +
"(0) (G0) (0)\n" +
"(0) (G1) (0)\n" +
"(0) (G2) (0)\n" +
"(0) (G3) (0)\n" +
"(0) (G4) (0)\n" +
"(0) (G5) (0)\n" +
"(0) (G6) (0)\n" +
"(0) (G7) (0)\n" +
"(0) (G8) (0)\n" +
"(0) (G9) (0)\n" +
"(0) (G10) (0)\n" +
"(0) (G99) (0)\n" +
"(0) (N) (0.00)\n" +
"(0) (N0) (0)\n" +
"(0) (N1) (0.0)\n" +
"(0) (N2) (0.00)\n" +
"(0) (N3) (0.000)\n" +
"(0) (N4) (0.0000)\n" +
"(0) (N5) (0.00000)\n" +
"(0) (N6) (0.000000)\n" +
"(0) (N7) (0.0000000)\n" +
"(0) (N8) (0.00000000)\n" +
"(0) (N9) (0.000000000)\n" +
"(0) (N10) (0.0000000000)\n" +
"(0) (N99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (P) (0.00 %)\n" +
"(0) (P0) (0 %)\n" +
"(0) (P1) (0.0 %)\n" +
"(0) (P2) (0.00 %)\n" +
"(0) (P3) (0.000 %)\n" +
"(0) (P4) (0.0000 %)\n" +
"(0) (P5) (0.00000 %)\n" +
"(0) (P6) (0.000000 %)\n" +
"(0) (P7) (0.0000000 %)\n" +
"(0) (P8) (0.00000000 %)\n" +
"(0) (P9) (0.000000000 %)\n" +
"(0) (P10) (0.0000000000 %)\n" +
"(0) (P99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000 %)\n" +
"(0) (X) (0)\n" +
"(0) (X0) (0)\n" +
"(0) (X1) (0)\n" +
"(0) (X2) (00)\n" +
"(0) (X3) (000)\n" +
"(0) (X4) (0000)\n" +
"(0) (X5) (00000)\n" +
"(0) (X6) (000000)\n" +
"(0) (X7) (0000000)\n" +
"(0) (X8) (00000000)\n" +
"(0) (X9) (000000000)\n" +
"(0) (X10) (0000000000)\n" +
"(0) (X99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(4294967295) (C) ($4,294,967,295.00)\n" +
"(4294967295) (C0) ($4,294,967,295)\n" +
"(4294967295) (C1) ($4,294,967,295.0)\n" +
"(4294967295) (C2) ($4,294,967,295.00)\n" +
"(4294967295) (C3) ($4,294,967,295.000)\n" +
"(4294967295) (C4) ($4,294,967,295.0000)\n" +
"(4294967295) (C5) ($4,294,967,295.00000)\n" +
"(4294967295) (C6) ($4,294,967,295.000000)\n" +
"(4294967295) (C7) ($4,294,967,295.0000000)\n" +
"(4294967295) (C8) ($4,294,967,295.00000000)\n" +
"(4294967295) (C9) ($4,294,967,295.000000000)\n" +
"(4294967295) (C10) ($4,294,967,295.0000000000)\n" +
"(4294967295) (C99) ($4,294,967,295.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(4294967295) (D) (4294967295)\n" +
"(4294967295) (D0) (4294967295)\n" +
"(4294967295) (D1) (4294967295)\n" +
"(4294967295) (D2) (4294967295)\n" +
"(4294967295) (D3) (4294967295)\n" +
"(4294967295) (D4) (4294967295)\n" +
"(4294967295) (D5) (4294967295)\n" +
"(4294967295) (D6) (4294967295)\n" +
"(4294967295) (D7) (4294967295)\n" +
"(4294967295) (D8) (4294967295)\n" +
"(4294967295) (D9) (4294967295)\n" +
"(4294967295) (D10) (4294967295)\n" +
"(4294967295) (D99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000000000004294967295)\n" +
"(4294967295) (E) (4.294967E+009)\n" +
"(4294967295) (E0) (4E+009)\n" +
"(4294967295) (E1) (4.3E+009)\n" +
"(4294967295) (E2) (4.29E+009)\n" +
"(4294967295) (E3) (4.295E+009)\n" +
"(4294967295) (E4) (4.2950E+009)\n" +
"(4294967295) (E5) (4.29497E+009)\n" +
"(4294967295) (E6) (4.294967E+009)\n" +
"(4294967295) (E7) (4.2949673E+009)\n" +
"(4294967295) (E8) (4.29496730E+009)\n" +
"(4294967295) (E9) (4.294967295E+009)\n" +
"(4294967295) (E10) (4.2949672950E+009)\n" +
"(4294967295) (E99) (4.294967295000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000E+009)\n" +
"(4294967295) (F) (4294967295.00)\n" +
"(4294967295) (F0) (4294967295)\n" +
"(4294967295) (F1) (4294967295.0)\n" +
"(4294967295) (F2) (4294967295.00)\n" +
"(4294967295) (F3) (4294967295.000)\n" +
"(4294967295) (F4) (4294967295.0000)\n" +
"(4294967295) (F5) (4294967295.00000)\n" +
"(4294967295) (F6) (4294967295.000000)\n" +
"(4294967295) (F7) (4294967295.0000000)\n" +
"(4294967295) (F8) (4294967295.00000000)\n" +
"(4294967295) (F9) (4294967295.000000000)\n" +
"(4294967295) (F10) (4294967295.0000000000)\n" +
"(4294967295) (F99) (4294967295.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(4294967295) (G) (4294967295)\n" +
"(4294967295) (G0) (4294967295)\n" +
"(4294967295) (G1) (4E+09)\n" +
"(4294967295) (G2) (4.3E+09)\n" +
"(4294967295) (G3) (4.29E+09)\n" +
"(4294967295) (G4) (4.295E+09)\n" +
"(4294967295) (G5) (4.295E+09)\n" +
"(4294967295) (G6) (4.29497E+09)\n" +
"(4294967295) (G7) (4.294967E+09)\n" +
"(4294967295) (G8) (4.2949673E+09)\n" +
"(4294967295) (G9) (4.2949673E+09)\n" +
"(4294967295) (G10) (4294967295)\n" +
"(4294967295) (G99) (4294967295)\n" +
"(4294967295) (N) (4,294,967,295.00)\n" +
"(4294967295) (N0) (4,294,967,295)\n" +
"(4294967295) (N1) (4,294,967,295.0)\n" +
"(4294967295) (N2) (4,294,967,295.00)\n" +
"(4294967295) (N3) (4,294,967,295.000)\n" +
"(4294967295) (N4) (4,294,967,295.0000)\n" +
"(4294967295) (N5) (4,294,967,295.00000)\n" +
"(4294967295) (N6) (4,294,967,295.000000)\n" +
"(4294967295) (N7) (4,294,967,295.0000000)\n" +
"(4294967295) (N8) (4,294,967,295.00000000)\n" +
"(4294967295) (N9) (4,294,967,295.000000000)\n" +
"(4294967295) (N10) (4,294,967,295.0000000000)\n" +
"(4294967295) (N99) (4,294,967,295.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(4294967295) (P) (429,496,729,500.00 %)\n" +
"(4294967295) (P0) (429,496,729,500 %)\n" +
"(4294967295) (P1) (429,496,729,500.0 %)\n" +
"(4294967295) (P2) (429,496,729,500.00 %)\n" +
"(4294967295) (P3) (429,496,729,500.000 %)\n" +
"(4294967295) (P4) (429,496,729,500.0000 %)\n" +
"(4294967295) (P5) (429,496,729,500.00000 %)\n" +
"(4294967295) (P6) (429,496,729,500.000000 %)\n" +
"(4294967295) (P7) (429,496,729,500.0000000 %)\n" +
"(4294967295) (P8) (429,496,729,500.00000000 %)\n" +
"(4294967295) (P9) (429,496,729,500.000000000 %)\n" +
"(4294967295) (P10) (429,496,729,500.0000000000 %)\n" +
"(4294967295) (P99) (429,496,729,500.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000 %)\n" +
"(4294967295) (X) (FFFFFFFF)\n" +
"(4294967295) (X0) (FFFFFFFF)\n" +
"(4294967295) (X1) (FFFFFFFF)\n" +
"(4294967295) (X2) (FFFFFFFF)\n" +
"(4294967295) (X3) (FFFFFFFF)\n" +
"(4294967295) (X4) (FFFFFFFF)\n" +
"(4294967295) (X5) (FFFFFFFF)\n" +
"(4294967295) (X6) (FFFFFFFF)\n" +
"(4294967295) (X7) (FFFFFFFF)\n" +
"(4294967295) (X8) (FFFFFFFF)\n" +
"(4294967295) (X9) (0FFFFFFFF)\n" +
"(4294967295) (X10) (00FFFFFFFF)\n" +
"(4294967295) (X99) (0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000FFFFFFFF)\n" +
"(0) (C) ($0.00)\n" +
"(0) (C0) ($0)\n" +
"(0) (C1) ($0.0)\n" +
"(0) (C2) ($0.00)\n" +
"(0) (C3) ($0.000)\n" +
"(0) (C4) ($0.0000)\n" +
"(0) (C5) ($0.00000)\n" +
"(0) (C6) ($0.000000)\n" +
"(0) (C7) ($0.0000000)\n" +
"(0) (C8) ($0.00000000)\n" +
"(0) (C9) ($0.000000000)\n" +
"(0) (C10) ($0.0000000000)\n" +
"(0) (C99) ($0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (D) (0)\n" +
"(0) (D0) (0)\n" +
"(0) (D1) (0)\n" +
"(0) (D2) (00)\n" +
"(0) (D3) (000)\n" +
"(0) (D4) (0000)\n" +
"(0) (D5) (00000)\n" +
"(0) (D6) (000000)\n" +
"(0) (D7) (0000000)\n" +
"(0) (D8) (00000000)\n" +
"(0) (D9) (000000000)\n" +
"(0) (D10) (0000000000)\n" +
"(0) (D99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (E) (0.000000E+000)\n" +
"(0) (E0) (0E+000)\n" +
"(0) (E1) (0.0E+000)\n" +
"(0) (E2) (0.00E+000)\n" +
"(0) (E3) (0.000E+000)\n" +
"(0) (E4) (0.0000E+000)\n" +
"(0) (E5) (0.00000E+000)\n" +
"(0) (E6) (0.000000E+000)\n" +
"(0) (E7) (0.0000000E+000)\n" +
"(0) (E8) (0.00000000E+000)\n" +
"(0) (E9) (0.000000000E+000)\n" +
"(0) (E10) (0.0000000000E+000)\n" +
"(0) (E99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000E+000)\n" +
"(0) (F) (0.00)\n" +
"(0) (F0) (0)\n" +
"(0) (F1) (0.0)\n" +
"(0) (F2) (0.00)\n" +
"(0) (F3) (0.000)\n" +
"(0) (F4) (0.0000)\n" +
"(0) (F5) (0.00000)\n" +
"(0) (F6) (0.000000)\n" +
"(0) (F7) (0.0000000)\n" +
"(0) (F8) (0.00000000)\n" +
"(0) (F9) (0.000000000)\n" +
"(0) (F10) (0.0000000000)\n" +
"(0) (F99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (G) (0)\n" +
"(0) (G0) (0)\n" +
"(0) (G1) (0)\n" +
"(0) (G2) (0)\n" +
"(0) (G3) (0)\n" +
"(0) (G4) (0)\n" +
"(0) (G5) (0)\n" +
"(0) (G6) (0)\n" +
"(0) (G7) (0)\n" +
"(0) (G8) (0)\n" +
"(0) (G9) (0)\n" +
"(0) (G10) (0)\n" +
"(0) (G99) (0)\n" +
"(0) (N) (0.00)\n" +
"(0) (N0) (0)\n" +
"(0) (N1) (0.0)\n" +
"(0) (N2) (0.00)\n" +
"(0) (N3) (0.000)\n" +
"(0) (N4) (0.0000)\n" +
"(0) (N5) (0.00000)\n" +
"(0) (N6) (0.000000)\n" +
"(0) (N7) (0.0000000)\n" +
"(0) (N8) (0.00000000)\n" +
"(0) (N9) (0.000000000)\n" +
"(0) (N10) (0.0000000000)\n" +
"(0) (N99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (P) (0.00 %)\n" +
"(0) (P0) (0 %)\n" +
"(0) (P1) (0.0 %)\n" +
"(0) (P2) (0.00 %)\n" +
"(0) (P3) (0.000 %)\n" +
"(0) (P4) (0.0000 %)\n" +
"(0) (P5) (0.00000 %)\n" +
"(0) (P6) (0.000000 %)\n" +
"(0) (P7) (0.0000000 %)\n" +
"(0) (P8) (0.00000000 %)\n" +
"(0) (P9) (0.000000000 %)\n" +
"(0) (P10) (0.0000000000 %)\n" +
"(0) (P99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000 %)\n" +
"(0) (X) (0)\n" +
"(0) (X0) (0)\n" +
"(0) (X1) (0)\n" +
"(0) (X2) (00)\n" +
"(0) (X3) (000)\n" +
"(0) (X4) (0000)\n" +
"(0) (X5) (00000)\n" +
"(0) (X6) (000000)\n" +
"(0) (X7) (0000000)\n" +
"(0) (X8) (00000000)\n" +
"(0) (X9) (000000000)\n" +
"(0) (X10) (0000000000)\n" +
"(0) (X99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(100) (C) ($100.00)\n" +
"(100) (C0) ($100)\n" +
"(100) (C1) ($100.0)\n" +
"(100) (C2) ($100.00)\n" +
"(100) (C3) ($100.000)\n" +
"(100) (C4) ($100.0000)\n" +
"(100) (C5) ($100.00000)\n" +
"(100) (C6) ($100.000000)\n" +
"(100) (C7) ($100.0000000)\n" +
"(100) (C8) ($100.00000000)\n" +
"(100) (C9) ($100.000000000)\n" +
"(100) (C10) ($100.0000000000)\n" +
"(100) (C99) ($100.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(100) (D) (100)\n" +
"(100) (D0) (100)\n" +
"(100) (D1) (100)\n" +
"(100) (D2) (100)\n" +
"(100) (D3) (100)\n" +
"(100) (D4) (0100)\n" +
"(100) (D5) (00100)\n" +
"(100) (D6) (000100)\n" +
"(100) (D7) (0000100)\n" +
"(100) (D8) (00000100)\n" +
"(100) (D9) (000000100)\n" +
"(100) (D10) (0000000100)\n" +
"(100) (D99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000100)\n" +
"(100) (E) (1.000000E+002)\n" +
"(100) (E0) (1E+002)\n" +
"(100) (E1) (1.0E+002)\n" +
"(100) (E2) (1.00E+002)\n" +
"(100) (E3) (1.000E+002)\n" +
"(100) (E4) (1.0000E+002)\n" +
"(100) (E5) (1.00000E+002)\n" +
"(100) (E6) (1.000000E+002)\n" +
"(100) (E7) (1.0000000E+002)\n" +
"(100) (E8) (1.00000000E+002)\n" +
"(100) (E9) (1.000000000E+002)\n" +
"(100) (E10) (1.0000000000E+002)\n" +
"(100) (E99) (1.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000E+002)\n" +
"(100) (F) (100.00)\n" +
"(100) (F0) (100)\n" +
"(100) (F1) (100.0)\n" +
"(100) (F2) (100.00)\n" +
"(100) (F3) (100.000)\n" +
"(100) (F4) (100.0000)\n" +
"(100) (F5) (100.00000)\n" +
"(100) (F6) (100.000000)\n" +
"(100) (F7) (100.0000000)\n" +
"(100) (F8) (100.00000000)\n" +
"(100) (F9) (100.000000000)\n" +
"(100) (F10) (100.0000000000)\n" +
"(100) (F99) (100.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(100) (G) (100)\n" +
"(100) (G0) (100)\n" +
"(100) (G1) (1E+02)\n" +
"(100) (G2) (1E+02)\n" +
"(100) (G3) (100)\n" +
"(100) (G4) (100)\n" +
"(100) (G5) (100)\n" +
"(100) (G6) (100)\n" +
"(100) (G7) (100)\n" +
"(100) (G8) (100)\n" +
"(100) (G9) (100)\n" +
"(100) (G10) (100)\n" +
"(100) (G99) (100)\n" +
"(100) (N) (100.00)\n" +
"(100) (N0) (100)\n" +
"(100) (N1) (100.0)\n" +
"(100) (N2) (100.00)\n" +
"(100) (N3) (100.000)\n" +
"(100) (N4) (100.0000)\n" +
"(100) (N5) (100.00000)\n" +
"(100) (N6) (100.000000)\n" +
"(100) (N7) (100.0000000)\n" +
"(100) (N8) (100.00000000)\n" +
"(100) (N9) (100.000000000)\n" +
"(100) (N10) (100.0000000000)\n" +
"(100) (N99) (100.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(100) (P) (10,000.00 %)\n" +
"(100) (P0) (10,000 %)\n" +
"(100) (P1) (10,000.0 %)\n" +
"(100) (P2) (10,000.00 %)\n" +
"(100) (P3) (10,000.000 %)\n" +
"(100) (P4) (10,000.0000 %)\n" +
"(100) (P5) (10,000.00000 %)\n" +
"(100) (P6) (10,000.000000 %)\n" +
"(100) (P7) (10,000.0000000 %)\n" +
"(100) (P8) (10,000.00000000 %)\n" +
"(100) (P9) (10,000.000000000 %)\n" +
"(100) (P10) (10,000.0000000000 %)\n" +
"(100) (P99) (10,000.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000 %)\n" +
"(100) (X) (64)\n" +
"(100) (X0) (64)\n" +
"(100) (X1) (64)\n" +
"(100) (X2) (64)\n" +
"(100) (X3) (064)\n" +
"(100) (X4) (0064)\n" +
"(100) (X5) (00064)\n" +
"(100) (X6) (000064)\n" +
"(100) (X7) (0000064)\n" +
"(100) (X8) (00000064)\n" +
"(100) (X9) (000000064)\n" +
"(100) (X10) (0000000064)\n" +
"(100) (X99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000064)\n" +
"UInt64\n" +
"(0) (C) ($0.00)\n" +
"(0) (C0) ($0)\n" +
"(0) (C1) ($0.0)\n" +
"(0) (C2) ($0.00)\n" +
"(0) (C3) ($0.000)\n" +
"(0) (C4) ($0.0000)\n" +
"(0) (C5) ($0.00000)\n" +
"(0) (C6) ($0.000000)\n" +
"(0) (C7) ($0.0000000)\n" +
"(0) (C8) ($0.00000000)\n" +
"(0) (C9) ($0.000000000)\n" +
"(0) (C10) ($0.0000000000)\n" +
"(0) (C99) ($0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (D) (0)\n" +
"(0) (D0) (0)\n" +
"(0) (D1) (0)\n" +
"(0) (D2) (00)\n" +
"(0) (D3) (000)\n" +
"(0) (D4) (0000)\n" +
"(0) (D5) (00000)\n" +
"(0) (D6) (000000)\n" +
"(0) (D7) (0000000)\n" +
"(0) (D8) (00000000)\n" +
"(0) (D9) (000000000)\n" +
"(0) (D10) (0000000000)\n" +
"(0) (D99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (E) (0.000000E+000)\n" +
"(0) (E0) (0E+000)\n" +
"(0) (E1) (0.0E+000)\n" +
"(0) (E2) (0.00E+000)\n" +
"(0) (E3) (0.000E+000)\n" +
"(0) (E4) (0.0000E+000)\n" +
"(0) (E5) (0.00000E+000)\n" +
"(0) (E6) (0.000000E+000)\n" +
"(0) (E7) (0.0000000E+000)\n" +
"(0) (E8) (0.00000000E+000)\n" +
"(0) (E9) (0.000000000E+000)\n" +
"(0) (E10) (0.0000000000E+000)\n" +
"(0) (E99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000E+000)\n" +
"(0) (F) (0.00)\n" +
"(0) (F0) (0)\n" +
"(0) (F1) (0.0)\n" +
"(0) (F2) (0.00)\n" +
"(0) (F3) (0.000)\n" +
"(0) (F4) (0.0000)\n" +
"(0) (F5) (0.00000)\n" +
"(0) (F6) (0.000000)\n" +
"(0) (F7) (0.0000000)\n" +
"(0) (F8) (0.00000000)\n" +
"(0) (F9) (0.000000000)\n" +
"(0) (F10) (0.0000000000)\n" +
"(0) (F99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (G) (0)\n" +
"(0) (G0) (0)\n" +
"(0) (G1) (0)\n" +
"(0) (G2) (0)\n" +
"(0) (G3) (0)\n" +
"(0) (G4) (0)\n" +
"(0) (G5) (0)\n" +
"(0) (G6) (0)\n" +
"(0) (G7) (0)\n" +
"(0) (G8) (0)\n" +
"(0) (G9) (0)\n" +
"(0) (G10) (0)\n" +
"(0) (G99) (0)\n" +
"(0) (N) (0.00)\n" +
"(0) (N0) (0)\n" +
"(0) (N1) (0.0)\n" +
"(0) (N2) (0.00)\n" +
"(0) (N3) (0.000)\n" +
"(0) (N4) (0.0000)\n" +
"(0) (N5) (0.00000)\n" +
"(0) (N6) (0.000000)\n" +
"(0) (N7) (0.0000000)\n" +
"(0) (N8) (0.00000000)\n" +
"(0) (N9) (0.000000000)\n" +
"(0) (N10) (0.0000000000)\n" +
"(0) (N99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (P) (0.00 %)\n" +
"(0) (P0) (0 %)\n" +
"(0) (P1) (0.0 %)\n" +
"(0) (P2) (0.00 %)\n" +
"(0) (P3) (0.000 %)\n" +
"(0) (P4) (0.0000 %)\n" +
"(0) (P5) (0.00000 %)\n" +
"(0) (P6) (0.000000 %)\n" +
"(0) (P7) (0.0000000 %)\n" +
"(0) (P8) (0.00000000 %)\n" +
"(0) (P9) (0.000000000 %)\n" +
"(0) (P10) (0.0000000000 %)\n" +
"(0) (P99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000 %)\n" +
"(0) (X) (0)\n" +
"(0) (X0) (0)\n" +
"(0) (X1) (0)\n" +
"(0) (X2) (00)\n" +
"(0) (X3) (000)\n" +
"(0) (X4) (0000)\n" +
"(0) (X5) (00000)\n" +
"(0) (X6) (000000)\n" +
"(0) (X7) (0000000)\n" +
"(0) (X8) (00000000)\n" +
"(0) (X9) (000000000)\n" +
"(0) (X10) (0000000000)\n" +
"(0) (X99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(18446744073709551615) (C) ($18,446,744,073,709,551,615.00)\n" +
"(18446744073709551615) (C0) ($18,446,744,073,709,551,615)\n" +
"(18446744073709551615) (C1) ($18,446,744,073,709,551,615.0)\n" +
"(18446744073709551615) (C2) ($18,446,744,073,709,551,615.00)\n" +
"(18446744073709551615) (C3) ($18,446,744,073,709,551,615.000)\n" +
"(18446744073709551615) (C4) ($18,446,744,073,709,551,615.0000)\n" +
"(18446744073709551615) (C5) ($18,446,744,073,709,551,615.00000)\n" +
"(18446744073709551615) (C6) ($18,446,744,073,709,551,615.000000)\n" +
"(18446744073709551615) (C7) ($18,446,744,073,709,551,615.0000000)\n" +
"(18446744073709551615) (C8) ($18,446,744,073,709,551,615.00000000)\n" +
"(18446744073709551615) (C9) ($18,446,744,073,709,551,615.000000000)\n" +
"(18446744073709551615) (C10) ($18,446,744,073,709,551,615.0000000000)\n" +
"(18446744073709551615) (C99) ($18,446,744,073,709,551,615.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(18446744073709551615) (D) (18446744073709551615)\n" +
"(18446744073709551615) (D0) (18446744073709551615)\n" +
"(18446744073709551615) (D1) (18446744073709551615)\n" +
"(18446744073709551615) (D2) (18446744073709551615)\n" +
"(18446744073709551615) (D3) (18446744073709551615)\n" +
"(18446744073709551615) (D4) (18446744073709551615)\n" +
"(18446744073709551615) (D5) (18446744073709551615)\n" +
"(18446744073709551615) (D6) (18446744073709551615)\n" +
"(18446744073709551615) (D7) (18446744073709551615)\n" +
"(18446744073709551615) (D8) (18446744073709551615)\n" +
"(18446744073709551615) (D9) (18446744073709551615)\n" +
"(18446744073709551615) (D10) (18446744073709551615)\n" +
"(18446744073709551615) (D99) (000000000000000000000000000000000000000000000000000000000000000000000000000000018446744073709551615)\n" +
"(18446744073709551615) (E) (1.844674E+019)\n" +
"(18446744073709551615) (E0) (2E+019)\n" +
"(18446744073709551615) (E1) (1.8E+019)\n" +
"(18446744073709551615) (E2) (1.84E+019)\n" +
"(18446744073709551615) (E3) (1.845E+019)\n" +
"(18446744073709551615) (E4) (1.8447E+019)\n" +
"(18446744073709551615) (E5) (1.84467E+019)\n" +
"(18446744073709551615) (E6) (1.844674E+019)\n" +
"(18446744073709551615) (E7) (1.8446744E+019)\n" +
"(18446744073709551615) (E8) (1.84467441E+019)\n" +
"(18446744073709551615) (E9) (1.844674407E+019)\n" +
"(18446744073709551615) (E10) (1.8446744074E+019)\n" +
"(18446744073709551615) (E99) (1.844674407370955161500000000000000000000000000000000000000000000000000000000000000000000000000000000E+019)\n" +
"(18446744073709551615) (F) (18446744073709551615.00)\n" +
"(18446744073709551615) (F0) (18446744073709551615)\n" +
"(18446744073709551615) (F1) (18446744073709551615.0)\n" +
"(18446744073709551615) (F2) (18446744073709551615.00)\n" +
"(18446744073709551615) (F3) (18446744073709551615.000)\n" +
"(18446744073709551615) (F4) (18446744073709551615.0000)\n" +
"(18446744073709551615) (F5) (18446744073709551615.00000)\n" +
"(18446744073709551615) (F6) (18446744073709551615.000000)\n" +
"(18446744073709551615) (F7) (18446744073709551615.0000000)\n" +
"(18446744073709551615) (F8) (18446744073709551615.00000000)\n" +
"(18446744073709551615) (F9) (18446744073709551615.000000000)\n" +
"(18446744073709551615) (F10) (18446744073709551615.0000000000)\n" +
"(18446744073709551615) (F99) (18446744073709551615.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(18446744073709551615) (G) (18446744073709551615)\n" +
"(18446744073709551615) (G0) (18446744073709551615)\n" +
"(18446744073709551615) (G1) (2E+19)\n" +
"(18446744073709551615) (G2) (1.8E+19)\n" +
"(18446744073709551615) (G3) (1.84E+19)\n" +
"(18446744073709551615) (G4) (1.845E+19)\n" +
"(18446744073709551615) (G5) (1.8447E+19)\n" +
"(18446744073709551615) (G6) (1.84467E+19)\n" +
"(18446744073709551615) (G7) (1.844674E+19)\n" +
"(18446744073709551615) (G8) (1.8446744E+19)\n" +
"(18446744073709551615) (G9) (1.84467441E+19)\n" +
"(18446744073709551615) (G10) (1.844674407E+19)\n" +
"(18446744073709551615) (G99) (18446744073709551615)\n" +
"(18446744073709551615) (N) (18,446,744,073,709,551,615.00)\n" +
"(18446744073709551615) (N0) (18,446,744,073,709,551,615)\n" +
"(18446744073709551615) (N1) (18,446,744,073,709,551,615.0)\n" +
"(18446744073709551615) (N2) (18,446,744,073,709,551,615.00)\n" +
"(18446744073709551615) (N3) (18,446,744,073,709,551,615.000)\n" +
"(18446744073709551615) (N4) (18,446,744,073,709,551,615.0000)\n" +
"(18446744073709551615) (N5) (18,446,744,073,709,551,615.00000)\n" +
"(18446744073709551615) (N6) (18,446,744,073,709,551,615.000000)\n" +
"(18446744073709551615) (N7) (18,446,744,073,709,551,615.0000000)\n" +
"(18446744073709551615) (N8) (18,446,744,073,709,551,615.00000000)\n" +
"(18446744073709551615) (N9) (18,446,744,073,709,551,615.000000000)\n" +
"(18446744073709551615) (N10) (18,446,744,073,709,551,615.0000000000)\n" +
"(18446744073709551615) (N99) (18,446,744,073,709,551,615.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(18446744073709551615) (P) (1,844,674,407,370,955,161,500.00 %)\n" +
"(18446744073709551615) (P0) (1,844,674,407,370,955,161,500 %)\n" +
"(18446744073709551615) (P1) (1,844,674,407,370,955,161,500.0 %)\n" +
"(18446744073709551615) (P2) (1,844,674,407,370,955,161,500.00 %)\n" +
"(18446744073709551615) (P3) (1,844,674,407,370,955,161,500.000 %)\n" +
"(18446744073709551615) (P4) (1,844,674,407,370,955,161,500.0000 %)\n" +
"(18446744073709551615) (P5) (1,844,674,407,370,955,161,500.00000 %)\n" +
"(18446744073709551615) (P6) (1,844,674,407,370,955,161,500.000000 %)\n" +
"(18446744073709551615) (P7) (1,844,674,407,370,955,161,500.0000000 %)\n" +
"(18446744073709551615) (P8) (1,844,674,407,370,955,161,500.00000000 %)\n" +
"(18446744073709551615) (P9) (1,844,674,407,370,955,161,500.000000000 %)\n" +
"(18446744073709551615) (P10) (1,844,674,407,370,955,161,500.0000000000 %)\n" +
"(18446744073709551615) (P99) (1,844,674,407,370,955,161,500.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000 %)\n" +
"(18446744073709551615) (X) (FFFFFFFFFFFFFFFF)\n" +
"(18446744073709551615) (X0) (FFFFFFFFFFFFFFFF)\n" +
"(18446744073709551615) (X1) (FFFFFFFFFFFFFFFF)\n" +
"(18446744073709551615) (X2) (FFFFFFFFFFFFFFFF)\n" +
"(18446744073709551615) (X3) (FFFFFFFFFFFFFFFF)\n" +
"(18446744073709551615) (X4) (FFFFFFFFFFFFFFFF)\n" +
"(18446744073709551615) (X5) (FFFFFFFFFFFFFFFF)\n" +
"(18446744073709551615) (X6) (FFFFFFFFFFFFFFFF)\n" +
"(18446744073709551615) (X7) (FFFFFFFFFFFFFFFF)\n" +
"(18446744073709551615) (X8) (FFFFFFFFFFFFFFFF)\n" +
"(18446744073709551615) (X9) (FFFFFFFFFFFFFFFF)\n" +
"(18446744073709551615) (X10) (FFFFFFFFFFFFFFFF)\n" +
"(18446744073709551615) (X99) (00000000000000000000000000000000000000000000000000000000000000000000000000000000000FFFFFFFFFFFFFFFF)\n" +
"(0) (C) ($0.00)\n" +
"(0) (C0) ($0)\n" +
"(0) (C1) ($0.0)\n" +
"(0) (C2) ($0.00)\n" +
"(0) (C3) ($0.000)\n" +
"(0) (C4) ($0.0000)\n" +
"(0) (C5) ($0.00000)\n" +
"(0) (C6) ($0.000000)\n" +
"(0) (C7) ($0.0000000)\n" +
"(0) (C8) ($0.00000000)\n" +
"(0) (C9) ($0.000000000)\n" +
"(0) (C10) ($0.0000000000)\n" +
"(0) (C99) ($0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (D) (0)\n" +
"(0) (D0) (0)\n" +
"(0) (D1) (0)\n" +
"(0) (D2) (00)\n" +
"(0) (D3) (000)\n" +
"(0) (D4) (0000)\n" +
"(0) (D5) (00000)\n" +
"(0) (D6) (000000)\n" +
"(0) (D7) (0000000)\n" +
"(0) (D8) (00000000)\n" +
"(0) (D9) (000000000)\n" +
"(0) (D10) (0000000000)\n" +
"(0) (D99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (E) (0.000000E+000)\n" +
"(0) (E0) (0E+000)\n" +
"(0) (E1) (0.0E+000)\n" +
"(0) (E2) (0.00E+000)\n" +
"(0) (E3) (0.000E+000)\n" +
"(0) (E4) (0.0000E+000)\n" +
"(0) (E5) (0.00000E+000)\n" +
"(0) (E6) (0.000000E+000)\n" +
"(0) (E7) (0.0000000E+000)\n" +
"(0) (E8) (0.00000000E+000)\n" +
"(0) (E9) (0.000000000E+000)\n" +
"(0) (E10) (0.0000000000E+000)\n" +
"(0) (E99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000E+000)\n" +
"(0) (F) (0.00)\n" +
"(0) (F0) (0)\n" +
"(0) (F1) (0.0)\n" +
"(0) (F2) (0.00)\n" +
"(0) (F3) (0.000)\n" +
"(0) (F4) (0.0000)\n" +
"(0) (F5) (0.00000)\n" +
"(0) (F6) (0.000000)\n" +
"(0) (F7) (0.0000000)\n" +
"(0) (F8) (0.00000000)\n" +
"(0) (F9) (0.000000000)\n" +
"(0) (F10) (0.0000000000)\n" +
"(0) (F99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (G) (0)\n" +
"(0) (G0) (0)\n" +
"(0) (G1) (0)\n" +
"(0) (G2) (0)\n" +
"(0) (G3) (0)\n" +
"(0) (G4) (0)\n" +
"(0) (G5) (0)\n" +
"(0) (G6) (0)\n" +
"(0) (G7) (0)\n" +
"(0) (G8) (0)\n" +
"(0) (G9) (0)\n" +
"(0) (G10) (0)\n" +
"(0) (G99) (0)\n" +
"(0) (N) (0.00)\n" +
"(0) (N0) (0)\n" +
"(0) (N1) (0.0)\n" +
"(0) (N2) (0.00)\n" +
"(0) (N3) (0.000)\n" +
"(0) (N4) (0.0000)\n" +
"(0) (N5) (0.00000)\n" +
"(0) (N6) (0.000000)\n" +
"(0) (N7) (0.0000000)\n" +
"(0) (N8) (0.00000000)\n" +
"(0) (N9) (0.000000000)\n" +
"(0) (N10) (0.0000000000)\n" +
"(0) (N99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(0) (P) (0.00 %)\n" +
"(0) (P0) (0 %)\n" +
"(0) (P1) (0.0 %)\n" +
"(0) (P2) (0.00 %)\n" +
"(0) (P3) (0.000 %)\n" +
"(0) (P4) (0.0000 %)\n" +
"(0) (P5) (0.00000 %)\n" +
"(0) (P6) (0.000000 %)\n" +
"(0) (P7) (0.0000000 %)\n" +
"(0) (P8) (0.00000000 %)\n" +
"(0) (P9) (0.000000000 %)\n" +
"(0) (P10) (0.0000000000 %)\n" +
"(0) (P99) (0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000 %)\n" +
"(0) (X) (0)\n" +
"(0) (X0) (0)\n" +
"(0) (X1) (0)\n" +
"(0) (X2) (00)\n" +
"(0) (X3) (000)\n" +
"(0) (X4) (0000)\n" +
"(0) (X5) (00000)\n" +
"(0) (X6) (000000)\n" +
"(0) (X7) (0000000)\n" +
"(0) (X8) (00000000)\n" +
"(0) (X9) (000000000)\n" +
"(0) (X10) (0000000000)\n" +
"(0) (X99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(100) (C) ($100.00)\n" +
"(100) (C0) ($100)\n" +
"(100) (C1) ($100.0)\n" +
"(100) (C2) ($100.00)\n" +
"(100) (C3) ($100.000)\n" +
"(100) (C4) ($100.0000)\n" +
"(100) (C5) ($100.00000)\n" +
"(100) (C6) ($100.000000)\n" +
"(100) (C7) ($100.0000000)\n" +
"(100) (C8) ($100.00000000)\n" +
"(100) (C9) ($100.000000000)\n" +
"(100) (C10) ($100.0000000000)\n" +
"(100) (C99) ($100.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(100) (D) (100)\n" +
"(100) (D0) (100)\n" +
"(100) (D1) (100)\n" +
"(100) (D2) (100)\n" +
"(100) (D3) (100)\n" +
"(100) (D4) (0100)\n" +
"(100) (D5) (00100)\n" +
"(100) (D6) (000100)\n" +
"(100) (D7) (0000100)\n" +
"(100) (D8) (00000100)\n" +
"(100) (D9) (000000100)\n" +
"(100) (D10) (0000000100)\n" +
"(100) (D99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000100)\n" +
"(100) (E) (1.000000E+002)\n" +
"(100) (E0) (1E+002)\n" +
"(100) (E1) (1.0E+002)\n" +
"(100) (E2) (1.00E+002)\n" +
"(100) (E3) (1.000E+002)\n" +
"(100) (E4) (1.0000E+002)\n" +
"(100) (E5) (1.00000E+002)\n" +
"(100) (E6) (1.000000E+002)\n" +
"(100) (E7) (1.0000000E+002)\n" +
"(100) (E8) (1.00000000E+002)\n" +
"(100) (E9) (1.000000000E+002)\n" +
"(100) (E10) (1.0000000000E+002)\n" +
"(100) (E99) (1.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000E+002)\n" +
"(100) (F) (100.00)\n" +
"(100) (F0) (100)\n" +
"(100) (F1) (100.0)\n" +
"(100) (F2) (100.00)\n" +
"(100) (F3) (100.000)\n" +
"(100) (F4) (100.0000)\n" +
"(100) (F5) (100.00000)\n" +
"(100) (F6) (100.000000)\n" +
"(100) (F7) (100.0000000)\n" +
"(100) (F8) (100.00000000)\n" +
"(100) (F9) (100.000000000)\n" +
"(100) (F10) (100.0000000000)\n" +
"(100) (F99) (100.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(100) (G) (100)\n" +
"(100) (G0) (100)\n" +
"(100) (G1) (1E+02)\n" +
"(100) (G2) (1E+02)\n" +
"(100) (G3) (100)\n" +
"(100) (G4) (100)\n" +
"(100) (G5) (100)\n" +
"(100) (G6) (100)\n" +
"(100) (G7) (100)\n" +
"(100) (G8) (100)\n" +
"(100) (G9) (100)\n" +
"(100) (G10) (100)\n" +
"(100) (G99) (100)\n" +
"(100) (N) (100.00)\n" +
"(100) (N0) (100)\n" +
"(100) (N1) (100.0)\n" +
"(100) (N2) (100.00)\n" +
"(100) (N3) (100.000)\n" +
"(100) (N4) (100.0000)\n" +
"(100) (N5) (100.00000)\n" +
"(100) (N6) (100.000000)\n" +
"(100) (N7) (100.0000000)\n" +
"(100) (N8) (100.00000000)\n" +
"(100) (N9) (100.000000000)\n" +
"(100) (N10) (100.0000000000)\n" +
"(100) (N99) (100.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000)\n" +
"(100) (P) (10,000.00 %)\n" +
"(100) (P0) (10,000 %)\n" +
"(100) (P1) (10,000.0 %)\n" +
"(100) (P2) (10,000.00 %)\n" +
"(100) (P3) (10,000.000 %)\n" +
"(100) (P4) (10,000.0000 %)\n" +
"(100) (P5) (10,000.00000 %)\n" +
"(100) (P6) (10,000.000000 %)\n" +
"(100) (P7) (10,000.0000000 %)\n" +
"(100) (P8) (10,000.00000000 %)\n" +
"(100) (P9) (10,000.000000000 %)\n" +
"(100) (P10) (10,000.0000000000 %)\n" +
"(100) (P99) (10,000.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000 %)\n" +
"(100) (X) (64)\n" +
"(100) (X0) (64)\n" +
"(100) (X1) (64)\n" +
"(100) (X2) (64)\n" +
"(100) (X3) (064)\n" +
"(100) (X4) (0064)\n" +
"(100) (X5) (00064)\n" +
"(100) (X6) (000064)\n" +
"(100) (X7) (0000064)\n" +
"(100) (X8) (00000064)\n" +
"(100) (X9) (000000064)\n" +
"(100) (X10) (0000000064)\n" +
"(100) (X99) (000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000064)\n";
}

}

