/*
 * Copyright (C) 2012 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Pawel Stolowski <pawel.stolowski@canonical.com>
 */

[CCode (gir_namespace = "UnityExtras", gir_version = "1.0")]
namespace Unity.Extras
{
  /**
   * Helper class for controlling PreviewPlayer and keeping MusicPreview up-to-date
   * with playback progress. It maintans an instance of PreviewPlayer, connects it
   * to active_preview signals, subscribes to progress signal and updates preview
   * state accordingly during playback. Playback is controlled by play/pause
   * signals from active_preview.
   */
  public class PreviewPlayerController: GLib.Object
  {
    private PreviewPlayer preview_player;
    private Unity.MusicPreview active_preview_;
    private ulong play_sig_id_;
    private ulong pause_sig_id_;
    private ulong closed_sig_id_;

    public Unity.MusicPreview active_preview
    {
      get
      {
        return active_preview_;
      }
      set
      {
        disconnect_signals ();

        active_preview_ = value;
        if (value != null)
        {
          play_sig_id_ = active_preview_.play.connect (play);
          pause_sig_id_ = active_preview_.pause.connect (pause);
          closed_sig_id_ = active_preview_.closed.connect (closed);
        }
      }
    }

    public PreviewPlayerController ()
    {
    }

    ~PreviewPlayerController ()
    {
      disconnect_signals ();
    }

    private void disconnect_signals ()
    {
      if (active_preview_ != null)
      {
        if (play_sig_id_ > 0)
        {
          active_preview_.disconnect (play_sig_id_);
          play_sig_id_ = 0;
        }
        if (pause_sig_id_ > 0)
        {
          active_preview_.disconnect (pause_sig_id_);
          pause_sig_id_ = 0;
        }
        if (closed_sig_id_ > 0)
        {
          active_preview_.disconnect (closed_sig_id_);
          closed_sig_id_ = 0;
        }
      }
    }

    private void on_progress_changed (string uri, Unity.MusicPreview.TrackState state, double progress)
    {
      if (active_preview_ != null)
      {
          active_preview_.current_track_uri = uri;
          active_preview_.current_track_state = state;
          active_preview_.current_progress = (float)progress;
        }
      }

    private void closed (Unity.Preview preview)
    {
      if (preview_player != null)
      {
        try
        {
          preview_player.close ();
        }
        catch (Error e)
        {
          warning ("Failed to close preview player: %s", e.message);
        }
      }
    }

    private void play (Unity.Preview preview, string uri)
    {
      debug ("play request: '%s'", uri);

      try
      {
        if (preview_player == null)
        {
          preview_player = new PreviewPlayer ();
          preview_player.progress.connect (on_progress_changed);
        }

        // we will receive state back in on_progress_changed, but set it now so that it's immediately reflected in the dash
        active_preview_.current_track_uri = uri;
        active_preview_.current_progress = 0.0f;
        active_preview_.current_track_state = Unity.MusicPreview.TrackState.PLAYING;

        preview_player.play (uri);
      }
      catch (Error e)
      {
        warning ("Failed to play '%s': %s", uri, e.message);
      }
    }

    private void pause (Unity.Preview preview, string uri)
    {
      debug ("pause request: '%s'", uri);

      try
      {
        if (preview_player != null)
        {
          // we will receive state back in on_progress_changed, but set it now so that it's immediately reflected in the dash
          active_preview_.current_track_uri = uri;
          active_preview_.current_track_state = Unity.MusicPreview.TrackState.PAUSED;

          preview_player.pause ();
        }
      }
      catch (Error e)
      {
        warning ("Failed to pause '%s': %s", uri, e.message);
      }
    }
  }
}
