/*
 * Copyright (C) 2011 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

using GLib;

namespace Unity {

public class Scope : GLib.Object
{
  public bool active { get; set; default = false; }
  public bool exported { get; private set; default = false; }
  public bool search_in_global { get; set; default = true; }
  public string dbus_path { get; construct; }
  public string[] sources { get; set; }
  public List<Filter> filters { get { return _filters; } }

  public Dee.SharedModel results_model {
    get { return _pimpl._results_model; }
  }

  public Dee.SharedModel global_results_model {
    get { return _pimpl._global_results_model; }
  }

  public LensSearch active_search { get; set; }
  public LensSearch active_global_search { get; set; }
  public string[] active_sources { get; private set; }

  public signal ActivationResponse? activate_uri (string uri);
  public signal Preview? preview_uri (string uri);
  public signal void filters_changed ();
  
  internal signal void search_finished (string search_string,
                                        HashTable<string, Variant> hints);
  internal signal void global_search_finished (string search_string,
                                               HashTable<string, Variant> hints);

  private ScopeImpl _pimpl;
  private bool is_local = false;
  internal List<Filter> _filters = new List<Filter>();
   
  public Scope (string dbus_path_)
  {
    Object (dbus_path:dbus_path_);
  }

  construct
  {
    _pimpl = new ScopeImpl (this);
    _pimpl.search_finished.connect ((s, h) =>
    {
      search_finished (s, h);
    });
    _pimpl.global_search_finished.connect ((s, h) =>
    {
      global_search_finished (s, h);
    });
  }

  public void export () throws IOError
  {
    if (!exported && !is_local)
    {
      _pimpl.export();
      exported = true;
    }
  }

  public Filter? get_filter (string id)
  {
    foreach (Filter filter in _filters)
    {
      if (filter.id == id)
        return filter;
    }
    return null;
  }

  /*
   * For our private implmentation
   */
  internal void set_active_internal (bool is_active)
  {
    active = is_active;
  }

  internal void set_active_sources_internal (string[] active_sources_)
  {
    active_sources = active_sources_;
  }

  internal void set_local (bool is_local_)
  {
    is_local = is_local_;
  }

  /*
   * For local Scope implementations, only used internally
   */
  internal unowned Dee.SharedModel get_filter_model ()
  {
    return _pimpl._filters_model;
  }

  internal async ActivationReplyRaw activate (string uri, uint action_type)
  {
    try {
      var reply = yield _pimpl.activate (uri, action_type);
      return reply;
    } catch (Error e) {
      warning (@"Unable to activate scope: $(e.message)");
    }
    var res = ActivationReplyRaw ();
    res.uri = uri;
    res.handled = HandledType.NOT_HANDLED;
    res.hints = new HashTable<string, Variant> (null, null);
    return res;
  }

  internal async void search (string search_string,
                        HashTable<string, Variant> hints)
  {
    try {
      yield _pimpl.search (search_string, hints);
    } catch (Error e) {
      warning (@"Unable to search scope: $(e.message)");
    }
  }

  internal async void global_search (string search_string,
                               HashTable<string, Variant> hints)
  {
    try {
      yield _pimpl.global_search (search_string, hints);
    } catch (Error e) {
      warning (@"Unable to global search scope: $(e.message)");
    }
  }

  internal async PreviewReplyRaw preview (string uri)
  {
    try {
      var reply = yield _pimpl.preview (uri);
      return reply;
    } catch (Error e) {
      warning (@"Unable to preview scope: $(e.message)");
    }
    return PreviewReplyRaw ();
  }
}

} /* namespace */
