/*
 * Copyright (C) 2011 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

using GLib;
using Dee;

namespace Unity {

/*
 * The private implementation of the Lens. This makes sure that none of the 
 * implementation details leak out into the public interface.
 */
private class LensImpl : GLib.Object, LensService
{
  private unowned Lens _owner;
  private uint _dbus_id;
  private uint _info_changed_id;

  private Dee.SharedModel _results_model;
  private Dee.SharedModel _global_results_model;
  private Dee.SharedModel _categories_model;
  private Dee.SharedModel _filters_model;

  private ScopeFactory _scope_factory;

  private ResultsSynchronizer _results_sync;
  private ResultsSynchronizer _global_results_sync;
  private FiltersSynchronizer _filters_sync;

  public LensImpl (Lens owner)
  {
    /* NOTE: Vala isn't allowing me to make Owner a construct variable so our
     * construction happens here instead of in construct {}
     */
    _owner = owner;
    _owner.notify["search-in-global"].connect (queue_info_changed);
    _owner.notify["visible"].connect (queue_info_changed);
    _owner.notify["search-hint"].connect (queue_info_changed);
    
    create_models (create_dbus_name ());
    create_synchronizers ();

    /* ScopeFactory handles finding Scopes and creating their proxies */
    _scope_factory = new ScopeFactory (_owner.id);
    _scope_factory.scope_added.connect (on_scope_added);
  }
  
  /* Create usable name prefix for the models */
  private string create_dbus_name ()
  {
    /* We make sure the name is unique to this instance so there is no chance of
     * old instances, or Unity itself, from stopping the models we create be the
     * swarm leaders
     */
    uint t = (uint)time_t ();
    return "com.canonical.Unity.Lens" + "." + _owner.id + ".T%u".printf (t);
  }

  private void create_models (string dbus_name)
  {
    /* Schema definitions come from the Lens specification */
    _results_model = new Dee.SharedModel (dbus_name + ".Results");
    _results_model.set_schema ("s", "s", "u", "s", "s", "s", "s");

    _global_results_model = new Dee.SharedModel (dbus_name + ".GlobalResults");
    _global_results_model.set_schema ("s", "s", "u", "s", "s", "s", "s");

    _categories_model = new Dee.SharedModel (dbus_name + ".Categories"); 
    _categories_model.set_schema ("s", "s", "s", "a{sv}");

    _filters_model = new Dee.SharedModel (dbus_name + ".Filters");
    _filters_model.set_schema ("s", "s", "s", "s", "a{sv}", "b", "b", "b");
  }

  private void create_synchronizers ()
  {
    _results_sync = new ResultsSynchronizer (_results_model);
    _global_results_sync = new ResultsSynchronizer (_global_results_model);
    _filters_sync = new FiltersSynchronizer (_filters_model);
  }

  public void export () throws IOError
  {
    var conn = Bus.get_sync (BusType.SESSION);
    _dbus_id = conn.register_object (_owner.dbus_path, this as LensService);

    queue_info_changed ();
  }

  public void load_categories (List<Category> categories)
  {
    foreach (Category category in categories)
    {
      string icon_hint_s = category.icon_hint != null ?
                                         category.icon_hint.to_string() : "";
      _categories_model.append (category.name,
                                icon_hint_s,
                                category.renderer,
                                Tools.hash_table_to_asv (category.hints));
    }
  }

  public void load_filters (List<Filter> filters)
  {
    foreach (Filter filter in filters)
    {
      string icon_hint_s = filter.icon_hint != null ?
                                         filter.icon_hint.to_string() : "";
      _filters_model.append (filter.id,
                             filter.display_name,
                             icon_hint_s,
                             filter.renderer_name,
                             Tools.hash_table_to_asv (filter.get_hints()),
                             filter.visible,
                             filter.collapsed,
                             filter.filtering);
    }
  }

  /* Queue up info-changed requests as we don't want to be spamming Unity with
   * them.
   */
  private void queue_info_changed ()
  {
    if (_info_changed_id == 0)
    {
      _info_changed_id = Idle.add (emit_info_changed);
    }
  }

  private bool emit_info_changed ()
  {
    var info = LensInfo();
    info.dbus_path = _owner.dbus_path;
    info.search_in_global = _owner.search_in_global;
    info.search_hint = _owner.search_hint;
    info.visible = _owner.visible;
    info.private_connection_name = "<not implemented>";
    info.results_model_name = _results_model.get_swarm_name ();
    info.global_results_model_name = _global_results_model.get_swarm_name ();
    info.categories_model_name = _categories_model.get_swarm_name ();
    info.filters_model_name = _filters_model.get_swarm_name ();
    info.hints = new HashTable<string, Variant> (null, null);
    
    changed (info);

    _info_changed_id = 0;
    return false;
  }

  private void on_scope_added (ScopeFactory factory, ScopeProxy scope)
  {
    scope.notify["results-model"].connect (on_scope_results_model_changed);
    scope.notify["global-results-model"].connect (on_scope_global_results_model_changed);
    scope.notify["filters-model"].connect (on_scope_filters_model_changed);
  }

  private void on_scope_results_model_changed (Object obj, ParamSpec pspec)
  {
    ScopeProxy? scope = obj as ScopeProxy;
    _results_sync.add_provider (scope.results_model, uid_for_scope (scope));
  }

  private void on_scope_global_results_model_changed (Object obj, ParamSpec pspec)
  {
    ScopeProxy? scope = obj as ScopeProxy;
    _global_results_sync.add_provider (scope.global_results_model, "%p".printf(scope));
  }

  private void on_scope_filters_model_changed (Object obj, ParamSpec pspec)
  {
    ScopeProxy? scope = obj as ScopeProxy;

    /* Because we need to manipulate it straight away, we have to wait until it's
     * synchronized
     */
    if (scope is ScopeProxyRemote && scope.filters_model.synchronized == false)
    {
      scope.filters_model.notify["synchronized"].connect (() =>
      {
        _filters_sync.add_receiver (scope.filters_model);
      });
    }
    else
      _filters_sync.add_receiver (scope.filters_model);
  }


  public void add_local_scope (Scope scope)
  {
    _scope_factory.add_local_scope (scope);
  }

  /*
   * DBus Interface Implementation
   */
  public async void info_request ()
  {
    queue_info_changed ();
  }

  public async ActivationReplyRaw activate (string uri,
                                            uint action_type) throws IOError
  {
    string[] tokens = uri.split(":", 2);
    ScopeProxy? scope = get_scope_for_uid (tokens[0]);
    var raw = ActivationReplyRaw();
    raw.handled = HandledType.NOT_HANDLED;
    raw.hints = new HashTable<string, Variant> (null, null);

    if (scope is ScopeProxy)
      raw = yield scope.activate(tokens[1], action_type);

    raw.uri = uri;
    return raw;
  }

  public async void search (string search_string,
                            HashTable<string, Variant> hints) throws IOError
  {
    foreach (ScopeProxy scope in _scope_factory.scopes)
      scope.search.begin (search_string, hints);
  }

  public async void global_search (string search_string,
                                   HashTable<string, Variant> hints) throws IOError
  {
    foreach (ScopeProxy scope in _scope_factory.scopes)
    {
      if (scope.search_in_global)
        scope.global_search.begin (search_string, hints);
    }
  }

  public async PreviewReplyRaw preview (string uri) throws IOError
  {
    string[] tokens = uri.split(":", 2);
    ScopeProxy? scope = get_scope_for_uid (tokens[0]);
    var raw = PreviewReplyRaw ();
    raw.renderer_name = "preview-none";
    raw.properties = new HashTable<string, Variant> (null, null);

    if (scope is ScopeProxy)
      raw = yield scope.preview (tokens[1]);

    raw.uri = uri;
    return raw;
  }

  public async void update_filter (string filter_name,
                                   HashTable<string, Variant> properties) throws IOError
  {
    debug ("%s", filter_name);
  }

  public async void set_active (bool is_active) throws IOError
  {
    foreach (ScopeProxy scope in _scope_factory.scopes)
      scope.set_active (is_active);

    _owner.set_active_internal (is_active);
  }

  private ScopeProxy? get_scope_for_uid (string uid)
  {
    foreach(ScopeProxy scope in _scope_factory.scopes)
    {
      if (uid == uid_for_scope (scope))
        return scope;
    }
    return null;
  }

  private string uid_for_scope (ScopeProxy scope)
  {
    return "%p".printf (scope);
  }
}

} /* namespace */
