
package org.jibx.schema.codegen;

import java.util.HashMap;
import java.util.Map;

import org.jibx.schema.SchemaTestBase;
import org.jibx.schema.codegen.custom.SchemasetCustom;
import org.jibx.schema.elements.SchemaElement;

/**
 * Test schema simplification and normalization.
 */
public class TypeReplacementTest extends SchemaTestBase
{
    // test data, also used by CodeGenerationTest
    // any changes here also need to be matched by changes there
    static final String MAIN_SCHEMA1 =
        "<schema targetNamespace='urn:something'" +
        "    xmlns='http://www.w3.org/2001/XMLSchema'" +
        "    xmlns:ins='urn:anything'" +
        "    elementFormDefault='qualified'>\n" +
        "  <import namespace='urn:anything' schemaLocation='INCLUDED_SCHEMA1'/>\n" +
        "  <element name='element' type='ins:simple3'/>\n" +
        "</schema>";
    static final String INCLUDED_SCHEMA1 =
        "<schema targetNamespace='urn:anything'" +
        "    xmlns='http://www.w3.org/2001/XMLSchema'" +
        "    xmlns:tns='urn:anything'" +
        "    elementFormDefault='qualified'>\n" +
        "  <complexType name='simple1'>\n" +
        "    <sequence>\n" +
        "      <element name='name' type='string'/>\n" +
        "      <element name='rated' type='tns:rating'/>\n" +
        "    </sequence>\n" +
        "    <attribute name='male' use='required' type='boolean'/>\n" +
        "  </complexType>\n" +
        "  <element name='simple1' type='tns:simple1'/>\n" +
        "  <simpleType name='simple2'>\n" +
        "    <list itemType='string'/>\n" +
        "  </simpleType>\n" +
        "  <complexType name='simple3'>\n" +
        "    <sequence>\n" +
        "      <element ref='tns:simple1'/>\n" +
        "      <element name='mixed' type='tns:mixedUnion'/>\n" +
        "    </sequence>\n" +
        "    <attribute name='rated' type='tns:rating'/>\n" +
        "  </complexType>\n" +
        "  <simpleType name='rating'>\n" +
        "    <restriction base='int'>\n" +
        "      <minExclusive value='1'/>\n" +
        "      <maxInclusive value='10' fixed='true'/>\n" +
        "    </restriction>\n" +
        "  </simpleType>\n" +
        "  <simpleType name='mixedUnion'>\n" +
        "    <union memberTypes='date dateTime tns:rating'/>\n" +
        "  </simpleType>\n" +
        "</schema>";
    static final String RESULT_SCHEMA1_A =
        "<schema targetNamespace='urn:anything'" +
        "    xmlns='http://www.w3.org/2001/XMLSchema'" +
        "    xmlns:tns='urn:anything'" +
        "    elementFormDefault='qualified'>\n" +
        "  <complexType name='simple1'>\n" +
        "    <sequence>\n" +
        "      <element name='name' type='string'/>\n" +
        "      <element name='rated' type='tns:rating'/>\n" +
        "    </sequence>\n" +
        "    <attribute name='male' use='required' type='boolean'/>\n" +
        "  </complexType>\n" +
        "  <element name='simple1' type='tns:simple1'/>\n" +
        "  <complexType name='simple3'>\n" +
        "    <sequence>\n" +
        "      <element ref='tns:simple1'/>\n" +
        "      <element name='mixed' type='tns:mixedUnion'/>\n" +
        "    </sequence>\n" +
        "    <attribute name='rated' type='tns:rating'/>\n" +
        "  </complexType>\n" +
        "  <simpleType name='rating'>\n" +
        "    <restriction base='int'/>\n" +
        "  </simpleType>\n" +
        "  <simpleType name='mixedUnion'>\n" +
        "    <union memberTypes='date dateTime tns:rating'/>\n" +
        "  </simpleType>\n" +
        "</schema>";
    static final String CUSTOMIZATION1_B =
        "<schema-set>\n" +
        "  <schema xmlns:tns='urn:anything' xmlns:xsd='http://www.w3.org/2001/XMLSchema' namespace='urn:anything'\n" +
        "      type-substitutions='tns:rating xsd:int'/>\n" +
        "</schema-set> ";
    static final String RESULT_SCHEMA1_B =
        "<schema targetNamespace='urn:anything'" +
        "    xmlns='http://www.w3.org/2001/XMLSchema'" +
        "    xmlns:xsd='http://www.w3.org/2001/XMLSchema'" +
        "    xmlns:tns='urn:anything'" +
        "    elementFormDefault='qualified'>\n" +
        "  <complexType name='simple1'>\n" +
        "    <sequence>\n" +
        "      <element name='name' type='string'/>\n" +
        "      <element name='rated' type='xsd:int'/>\n" +
        "    </sequence>\n" +
        "    <attribute name='male' use='required' type='boolean'/>\n" +
        "  </complexType>\n" +
        "  <element name='simple1' type='tns:simple1'/>\n" +
        "  <complexType name='simple3'>\n" +
        "    <sequence>\n" +
        "      <element ref='tns:simple1'/>\n" +
        "      <element name='mixed' type='tns:mixedUnion'/>\n" +
        "    </sequence>\n" +
        "    <attribute name='rated' type='xsd:int'/>\n" +
        "  </complexType>\n" +
        "  <simpleType name='mixedUnion'>\n" +
        "    <union memberTypes='date dateTime xsd:int'/>\n" +
        "  </simpleType>\n" +
        "</schema>";
    static final String CUSTOMIZATION1_C =
        "<schema-set>\n" +
        "  <schema xmlns:tns='urn:anything' namespace='urn:anything' excludes='rating'/>\n" +
        "</schema-set> ";
    static final String RESULT_SCHEMA1_C =
        "<schema targetNamespace='urn:anything'" +
        "    xmlns='http://www.w3.org/2001/XMLSchema'" +
        "    xmlns:tns='urn:anything'" +
        "    elementFormDefault='qualified'>\n" +
        "  <complexType name='simple1'>\n" +
        "    <sequence>\n" +
        "      <element name='name' type='string'/>\n" +
        "    </sequence>\n" +
        "    <attribute name='male' use='required' type='boolean'/>\n" +
        "  </complexType>\n" +
        "  <element name='simple1' type='tns:simple1'/>\n" +
        "  <complexType name='simple3'>\n" +
        "    <sequence>\n" +
        "      <element ref='tns:simple1'/>\n" +
        "      <element name='mixed' type='tns:mixedUnion'/>\n" +
        "    </sequence>\n" +
        "  </complexType>\n" +
        "  <simpleType name='mixedUnion'>\n" +
        "    <union memberTypes='date dateTime'/>\n" +
        "  </simpleType>\n" +
        "</schema>";
    static final String MAIN_SCHEMA2 =
        "<schema targetNamespace='urn:something'" +
        "    xmlns='http://www.w3.org/2001/XMLSchema'" +
        "    xmlns:ins='urn:anything'" +
        "    elementFormDefault='qualified'>\n" +
        "  <import namespace='urn:anything' schemaLocation='INCLUDED_SCHEMA2'/>\n" +
        "  <element name='element' type='ins:simple3'/>\n" +
        "</schema>";
    static final String INCLUDED_SCHEMA2 =
        "<schema targetNamespace='urn:anything'" +
        "    xmlns='http://www.w3.org/2001/XMLSchema'" +
        "    xmlns:tns='urn:anything'" +
        "    elementFormDefault='qualified'>\n" +
        "  <complexType name='simple1'>\n" +
        "    <sequence>\n" +
        "      <element name='name' type='string'/>\n" +
        "      <element name='rated' type='tns:rating'/>\n" +
        "      <element ref='tns:timestamp'/>\n" +
        "      <element name='altstamp' type='tns:DateTime_NoID'/>\n" +
        "    </sequence>\n" +
        "    <attribute name='male' use='required' type='boolean'/>\n" +
        "  </complexType>\n" +
        "  <element name='timestamp' type='tns:DateTime'/>\n" +
        "  <element name='simple1' type='tns:simple1'/>\n" +
        "  <simpleType name='simple2'>\n" +
        "    <list itemType='string'/>\n" +
        "  </simpleType>\n" +
        "  <complexType name='simple3'>\n" +
        "    <sequence>\n" +
        "      <element ref='tns:simple1'/>\n" +
        "      <element name='mixed' type='tns:mixedUnion'/>\n" +
        "    </sequence>\n" +
        "    <attribute name='rated' type='tns:rating'/>\n" +
        "  </complexType>\n" +
        "  <simpleType name='rating'>\n" +
        "    <restriction base='int'>\n" +
        "      <minExclusive value='1'/>\n" +
        "      <maxInclusive value='10' fixed='true'/>\n" +
        "    </restriction>\n" +
        "  </simpleType>\n" +
        "  <simpleType name='mixedUnion'>\n" +
        "    <union memberTypes='date dateTime tns:rating'/>\n" +
        "  </simpleType>\n" +
        "  <complexType name='DateTime'>\n" +
        "    <simpleContent>\n" +
        "      <extension base='tns:DateTime_NoID'>\n" +
        "        <attribute name='id' type='tns:ID'/>\n" +
        "      </extension>\n" +
        "    </simpleContent>\n" +
        "  </complexType>\n" +
        "  <simpleType name='ID'>\n" +
        "    <restriction base='ID'/>\n" +
        "  </simpleType>\n" +
        "  <simpleType name='DateTime_NoID'>\n" +
        "    <restriction base='dateTime'/>\n" +
        "  </simpleType>" +
        "</schema>";
    static final String CUSTOMIZATION2_A =
        "<schema-set>\n" +
        "  <schema xmlns:tns='urn:anything' xmlns:xsd='http://www.w3.org/2001/XMLSchema' namespace='urn:anything'\n" +
        "      type-substitutions='tns:rating xsd:int'/>\n" +
        "</schema-set> ";
    static final String RESULT_SCHEMA2_A =
        "<schema targetNamespace='urn:anything'" +
        "    xmlns='http://www.w3.org/2001/XMLSchema'" +
        "    xmlns:xsd='http://www.w3.org/2001/XMLSchema'" +
        "    xmlns:tns='urn:anything'" +
        "    elementFormDefault='qualified'>\n" +
        "  <complexType name='simple1'>\n" +
        "    <sequence>\n" +
        "      <element name='name' type='string'/>\n" +
        "      <element name='rated' type='xsd:int'/>\n" +
        "      <element ref='tns:timestamp'/>\n" +
        "      <element name='altstamp' type='tns:DateTime_NoID'/>\n" +
        "    </sequence>\n" +
        "    <attribute name='male' use='required' type='boolean'/>\n" +
        "  </complexType>\n" +
        "  <element name='timestamp' type='tns:DateTime'/>\n" +
        "  <element name='simple1' type='tns:simple1'/>\n" +
        "  <complexType name='simple3'>\n" +
        "    <sequence>\n" +
        "      <element ref='tns:simple1'/>\n" +
        "      <element name='mixed' type='tns:mixedUnion'/>\n" +
        "    </sequence>\n" +
        "    <attribute name='rated' type='xsd:int'/>\n" +
        "  </complexType>\n" +
        "  <simpleType name='mixedUnion'>\n" +
        "    <union memberTypes='date dateTime xsd:int'/>\n" +
        "  </simpleType>\n" +
        "  <complexType name='DateTime'>\n" +
        "    <simpleContent>\n" +
        "      <extension base='tns:DateTime_NoID'>\n" +
        "        <attribute name='id' type='tns:ID'/>\n" +
        "      </extension>\n" +
        "    </simpleContent>\n" +
        "  </complexType>\n" +
        "  <simpleType name='ID'>\n" +
        "    <restriction base='ID'/>\n" +
        "  </simpleType>\n" +
        "  <simpleType name='DateTime_NoID'>\n" +
        "    <restriction base='dateTime'/>\n" +
        "  </simpleType>" +
        "</schema>";
    static final String CUSTOMIZATION2_B =
        "<schema-set>\n" +
        "  <schema xmlns:tns='urn:anything' xmlns:xsd='http://www.w3.org/2001/XMLSchema' namespace='urn:anything'\n" +
        "      type-substitutions='tns:rating xsd:int tns:DateTime xsd:dateTime' excludes='ID'/>\n" +
        "</schema-set> ";
    static final String RESULT_SCHEMA2_B =
        "<schema targetNamespace='urn:anything'" +
        "    xmlns='http://www.w3.org/2001/XMLSchema'" +
        "    xmlns:xsd='http://www.w3.org/2001/XMLSchema'" +
        "    xmlns:tns='urn:anything'" +
        "    elementFormDefault='qualified'>\n" +
        "  <complexType name='simple1'>\n" +
        "    <sequence>\n" +
        "      <element name='name' type='string'/>\n" +
        "      <element name='rated' type='xsd:int'/>\n" +
        "      <element ref='tns:timestamp'/>\n" +
        "      <element name='altstamp' type='tns:DateTime_NoID'/>\n" +
        "    </sequence>\n" +
        "    <attribute name='male' use='required' type='boolean'/>\n" +
        "  </complexType>\n" +
        "  <element name='timestamp' type='xsd:dateTime'/>\n" +
        "  <element name='simple1' type='tns:simple1'/>\n" +
        "  <complexType name='simple3'>\n" +
        "    <sequence>\n" +
        "      <element ref='tns:simple1'/>\n" +
        "      <element name='mixed' type='tns:mixedUnion'/>\n" +
        "    </sequence>\n" +
        "    <attribute name='rated' type='xsd:int'/>\n" +
        "  </complexType>\n" +
        "  <simpleType name='mixedUnion'>\n" +
        "    <union memberTypes='date dateTime xsd:int'/>\n" +
        "  </simpleType>\n" +
        "  <simpleType name='DateTime_NoID'>\n" +
        "    <restriction base='dateTime'/>\n" +
        "  </simpleType>" +
        "</schema>";
    static final String MAIN_SCHEMA3 =
        "<schema targetNamespace='urn:something'" +
        "    xmlns='http://www.w3.org/2001/XMLSchema'" +
        "    xmlns:ins='urn:anything'" +
        "    elementFormDefault='qualified'>\n" +
        "  <import namespace='urn:anything' schemaLocation='INCLUDED_SCHEMA3'/>\n" +
        "  <element name='element' type='ins:simple3'/>\n" +
        "</schema>";
    static final String INCLUDED_SCHEMA3 =
        "<schema targetNamespace='urn:anything'" +
        "    xmlns='http://www.w3.org/2001/XMLSchema'" +
        "    xmlns:tns='urn:anything'" +
        "    elementFormDefault='qualified'>\n" +
        "  <simpleType name='shortString'>\n" +
        "    <restriction base='string'>\n" +
        "      <maxLength value='10'/>\n" +
        "    </restriction>\n" +
        "  </simpleType>\n" +
        "  <simpleType name='enum1'>\n" +
        "    <restriction base='string'>\n" +
        "      <enumeration value='A1'/>\n" +
        "      <enumeration value='B2'/>\n" +
        "    </restriction>\n" +
        "  </simpleType>\n" +
        "  <simpleType name='enum2'>\n" +
        "    <restriction base='tns:shortString'>\n" +
        "      <enumeration value='a1'/>\n" +
        "      <enumeration value='b2'/>\n" +
        "    </restriction>\n" +
        "  </simpleType>\n" +
        "  <complexType name='simple1'>\n" +
        "    <sequence>\n" +
        "      <element ref='tns:name'/>\n" +
        "      <element ref='tns:rated'/>\n" +
        "    </sequence>\n" +
        "    <attributeGroup ref='tns:enum1Group'/>\n" +
        "    <attributeGroup ref='tns:enum23Group'/>\n" +
        "    <attribute name='male' use='required' type='boolean'/>\n" +
        "  </complexType>\n" +
        "  <element name='simple1' type='tns:simple1'/>\n" +
        "  <simpleType name='simple2'>\n" +
        "    <list itemType='string'/>\n" +
        "  </simpleType>\n" +
        "  <attributeGroup name='enum1Group'>\n" +
        "    <attribute name='enum1' use='required' type='tns:enum1'/>\n" +
        "  </attributeGroup>\n" +
        "  <attributeGroup name='enum23Group'>\n" +
        "    <attribute name='enum2' use='required' type='tns:enum2'/>\n" +
        "    <attribute name='enum3' use='required'>\n" +
        "      <simpleType>\n" +
        "        <restriction base='string'>\n" +
        "          <enumeration value='x1'/>\n" +
        "          <enumeration value='y2'/>\n" +
        "          <enumeration value='z3'/>\n" +
        "        </restriction>\n" +
        "      </simpleType>\n" +
        "    </attribute>\n" +
        "  </attributeGroup>\n" +
        "  <element name='name' type='string'/>\n" +
        "  <element name='rated' type='tns:rating'/>\n" +
        "  <element name='mixed' type='tns:mixedUnion'/>" +
        "  <complexType name='simple3'>\n" +
        "    <sequence>\n" +
        "      <element ref='tns:simple1'/>\n" +
        "      <element ref='tns:mixed'/>\n" +
        "    </sequence>\n" +
        "    <attribute name='rated' type='tns:rating'/>\n" +
        "    <attributeGroup ref='tns:enum1Group'/>\n" +
        "    <attributeGroup ref='tns:enum23Group'/>\n" +
        "  </complexType>\n" +
        "  <simpleType name='rating'>\n" +
        "    <restriction base='int'>\n" +
        "      <minExclusive value='1'/>\n" +
        "      <maxInclusive value='10' fixed='true'/>\n" +
        "    </restriction>\n" +
        "  </simpleType>\n" +
        "  <simpleType name='mixedUnion'>\n" +
        "    <union memberTypes='date dateTime tns:rating'/>\n" +
        "  </simpleType>\n" +
        "</schema>";
    static final String CUSTOMIZATION3_A =
        "<schema-set>\n" +
        "  <schema xmlns:tns='urn:anything' xmlns:xsd='http://www.w3.org/2001/XMLSchema' namespace='urn:anything'\n" +
        "      type-substitutions='tns:rating xsd:int'/>\n" +
        "</schema-set> ";
    static final String RESULT_SCHEMA3_A =
        "<schema targetNamespace='urn:anything'" +
        "    xmlns='http://www.w3.org/2001/XMLSchema'" +
        "    xmlns:xsd='http://www.w3.org/2001/XMLSchema'" +
        "    xmlns:tns='urn:anything'" +
        "    elementFormDefault='qualified'>\n" +
        "  <xsd:simpleType name='shortString'>\n" +
        "    <xsd:restriction base='string'/>\n" +
        "  </xsd:simpleType>\n" +
        "  <xsd:simpleType name='enum1'>\n" +
        "    <xsd:restriction base='string'>\n" +
        "      <xsd:enumeration value='A1'/>\n" +
        "      <xsd:enumeration value='B2'/>\n" +
        "    </xsd:restriction>\n" +
        "  </xsd:simpleType>\n" +
        "  <xsd:simpleType name='enum2'>\n" +
        "    <xsd:restriction base='tns:shortString'>\n" +
        "      <xsd:enumeration value='a1'/>\n" +
        "      <xsd:enumeration value='b2'/>\n" +
        "    </xsd:restriction>\n" +
        "  </xsd:simpleType>\n" +
        "  <xsd:complexType name='simple1'>\n" +
        "    <xsd:sequence>\n" +
        "      <xsd:element ref='tns:name'/>\n" +
        "      <xsd:element ref='tns:rated'/>\n" +
        "    </xsd:sequence>\n" +
        "    <attributeGroup ref='tns:enum1Group'/>\n" +
        "    <attributeGroup ref='tns:enum23Group'/>\n" +
        "    <xsd:attribute type='boolean' use='required' name='male'/>\n" +
        "  </xsd:complexType>\n" +
        "  <element name='simple1' type='tns:simple1'/>\n" +
        "  <attributeGroup name='enum1Group'>\n" +
        "    <attribute name='enum1' use='required' type='tns:enum1'/>\n" +
        "  </attributeGroup>\n" +
        "  <attributeGroup name='enum23Group'>\n" +
        "    <attribute name='enum2' use='required' type='tns:enum2'/>\n" +
        "    <attribute name='enum3' use='required'>\n" +
        "      <simpleType>\n" +
        "        <restriction base='string'>\n" +
        "          <enumeration value='x1'/>\n" +
        "          <enumeration value='y2'/>\n" +
        "          <enumeration value='z3'/>\n" +
        "        </restriction>\n" +
        "      </simpleType>\n" +
        "    </attribute>\n" +
        "  </attributeGroup>\n" +
        "  <element name='name' type='string'/>\n" +
        "  <element name='rated' type='xsd:int'/>\n" +
        "  <element name='mixed' type='tns:mixedUnion'/>" +
        "  <complexType name='simple3'>\n" +
        "    <sequence>\n" +
        "      <element ref='tns:simple1'/>\n" +
        "      <element ref='tns:mixed'/>\n" +
        "    </sequence>\n" +
        "    <attribute name='rated' type='xsd:int'/>\n" +
        "    <attributeGroup ref='tns:enum1Group'/>\n" +
        "    <attributeGroup ref='tns:enum23Group'/>\n" +
        "  </complexType>\n" +
        "  <simpleType name='mixedUnion'>\n" +
        "    <union memberTypes='date dateTime xsd:int'/>\n" +
        "  </simpleType>\n" +
        "</schema>";
    static final String CUSTOMIZATION3_B =
        "<schema-set>\n" +
        "  <schema xmlns:tns='urn:anything' xmlns:xsd='http://www.w3.org/2001/XMLSchema' namespace='urn:anything'\n" +
        "      type-substitutions='tns:rating xsd:int'>\n" +
        "    <attributeGroup name='enum1Group' generation='inline'/>\n" +
        "    <attributeGroup name='enum23Group' generation='inline'/>\n" +
        "  </schema>\n" +
        "</schema-set>";
    private static final Map MAP = new HashMap();
    static final TestResolver RESOLVER1 = new TestResolver(MAIN_SCHEMA1, "MAIN_SCHEMA1", MAP);
    static final TestResolver RESOLVER2 = new TestResolver(MAIN_SCHEMA2, "MAIN_SCHEMA2", MAP);
    static final TestResolver RESOLVER3 = new TestResolver(MAIN_SCHEMA3, "MAIN_SCHEMA3", MAP);
    static {
        MAP.put(RESOLVER1.getName(), RESOLVER1);
        MAP.put("INCLUDED_SCHEMA1", new TestResolver(INCLUDED_SCHEMA1, "INCLUDED_SCHEMA1", MAP));
        MAP.put(RESOLVER2.getName(), RESOLVER2);
        MAP.put("INCLUDED_SCHEMA2", new TestResolver(INCLUDED_SCHEMA2, "INCLUDED_SCHEMA2", MAP));
        MAP.put(RESOLVER3.getName(), RESOLVER3);
        MAP.put("INCLUDED_SCHEMA3", new TestResolver(INCLUDED_SCHEMA3, "INCLUDED_SCHEMA3", MAP));
    }
    
    /**
     * Test the schema simplification handling. This first loads and validates the supplied main schema, which
     * automatically loads and validates any included/imported schemas. It then applies any customizations and finally
     * simplifies the (possibly modified) included/imported schemas and checks that the result matches expectations.
     *
     * @param resolver root schema resolver
     * @param custom customizations document text
     * @param inclname included schema name
     * @param rslttext simplified result schema expected
     * @throws Exception
     */
    private void testSimplification(TestResolver resolver, String custom, String inclname, String rslttext)
        throws Exception {
        SchemaElement[] schemas = new SchemaElement[] { loadSchema(resolver) };
        SchemasetCustom custroot;
        if (custom == null) {
            custroot = new SchemasetCustom(null);
        } else {
            custroot = loadCustomization(custom);
        }
        CodeGenerator generator = new CodeGenerator(custroot, m_validationContext, schemas);
        assertTrue("Schema customization failure", generator.customizeSchemas());
        generator.applyExtensions();
        generator.pruneDefinitions();
        verifySchema(resolver.getText(), writeSchema(schemas[0]));
        SchemaElement schema = m_validationContext.getSchema(inclname);
        verifySchema(rslttext, writeSchema(schema));
    }
    
    public void testSchemaSimplification() throws Exception {
        testSimplification(RESOLVER1, null, "INCLUDED_SCHEMA1", RESULT_SCHEMA1_A);
    }
    
    public void testTypeReplacement() throws Exception {
        testSimplification(RESOLVER1, CUSTOMIZATION1_B, "INCLUDED_SCHEMA1", RESULT_SCHEMA1_B);
    }
    
    public void testTypeRemoval() throws Exception {
        testSimplification(RESOLVER1, CUSTOMIZATION1_C, "INCLUDED_SCHEMA1", RESULT_SCHEMA1_C);
    }
    
    public void testTypeReplacement2() throws Exception {
        testSimplification(RESOLVER2, CUSTOMIZATION2_A, "INCLUDED_SCHEMA2", RESULT_SCHEMA2_A);
    }
    
    public void testCombinedRemovalReplacement() throws Exception {
        testSimplification(RESOLVER2, CUSTOMIZATION2_B, "INCLUDED_SCHEMA2", RESULT_SCHEMA2_B);
    }
    
    public void testSimplificationWithEnums() throws Exception {
        testSimplification(RESOLVER3, CUSTOMIZATION3_A, "INCLUDED_SCHEMA3", RESULT_SCHEMA3_A);
    }
}