/*
 * Copyright (c) Likewise Software.  All rights Reserved.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the license, or (at
 * your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
 * General Public License for more details.  You should have received a copy
 * of the GNU Lesser General Public License along with this program.  If
 * not, see <http://www.gnu.org/licenses/>.
 *
 * LIKEWISE SOFTWARE MAKES THIS SOFTWARE AVAILABLE UNDER OTHER LICENSING
 * TERMS AS WELL.  IF YOU HAVE ENTERED INTO A SEPARATE LICENSE AGREEMENT
 * WITH LIKEWISE SOFTWARE, THEN YOU MAY ELECT TO USE THE SOFTWARE UNDER THE
 * TERMS OF THAT SOFTWARE LICENSE AGREEMENT INSTEAD OF THE TERMS OF THE GNU
 * LESSER GENERAL PUBLIC LICENSE, NOTWITHSTANDING THE ABOVE NOTICE.  IF YOU
 * HAVE QUESTIONS, OR WISH TO REQUEST A COPY OF THE ALTERNATE LICENSING
 * TERMS OFFERED BY LIKEWISE SOFTWARE, PLEASE CONTACT LIKEWISE SOFTWARE AT
 * license@likewisesoftware.com
 */

/*
 * Module Name:
 *
 *        base.h
 *
 * Abstract:
 *
 *        NT status codes
 *
 * Authors: Brian Koropoff (bkoropoff@likewisesoftware.com)
 *
 */

#ifndef __LWBASE_NTSTATUS_H__
#define __LWBASE_NTSTATUS_H__

#include <lw/types.h>

//
// NTSTATUS
//
// NTSTATUS is similar to HRESULT in that it is a 32-bit integer with the format:
//
//     <SS|X|R|FFFFFFFFFFFF|CCCCCCCCCCCCCCC>
//              (12 bits)     (16 bits)
//
//   S - severity level (2 bits):
//
//       00 - success
//       01 - informational
//       10 - warning
//       11 - error
//
//   X - customer code bit (1 bit)
//
//   R - reserved bit (1 bit)
//
//   F - facility code (12 bits)
//
//   C - facility's status code (16 bits)
//
// Defined facility codes:
// 0 - FACILITY_SYSTEM
// 2 - FACILITY_RPC_RUNTIME
// 3 - FACILITY_RPC_STUBS
//
// In general, stay away from informational and success codes.
// That being said, some system facilities require the use of
// success codes.  The most important of the success codes is
// STATUS_PENDING.
//

#define LW_STATUS_SEVERITY_SUCCESS       0x0
#define LW_STATUS_SEVERITY_INFORMATIONAL 0x1
#define LW_STATUS_SEVERITY_WARNING       0x2
#define LW_STATUS_SEVERITY_ERROR         0x3

#define LW_FACILITY_SYSTEM      0x000
#define LW_FACILITY_RPC_RUNTIME 0x002
#define LW_FACILITY_RPC_STUBS   0x003
#define LW_FACILITY_LW_ERRNO    0xFFF

#define LW_MAKE_CUSTOM_NTSTATUS(Severity, Facility, Code) \
    ((NTSTATUS)(((ULONG)(Severity) << 30) | \
                (1 << 29) | \
                ((ULONG)(Facility) << 16) | \
                (ULONG) Code))

#define LW_NTSTATUS_GET_SEVERITY(status)    ((((ULONG)(status)) >> 30)
#define LW_NTSTATUS_GET_FACILITY(status)    ((((ULONG)(status)) >> 16) & 0xFFF)
#define LW_NTSTATUS_GET_CODE(status)        (((ULONG)(status)) & 0xFFFF)
#define LW_NTSTATUS_IS_CUSTOM(status)       ((((ULONG)(status)) >> 29) & 0x1)
#define LW_NTSTATUS_IS_RESERVED(status)     ((((ULONG)(status)) >> 28) & 0x1)

#define _LW_MAKE_ERRNO_NTSTATUS(Code) \
    LW_MAKE_CUSTOM_NTSTATUS(LW_STATUS_SEVERITY_ERROR, LW_FACILITY_LW_ERRNO, Code)
#define _LW_IS_ERRNO_NTSTATUS(status) \
    (((status) & _LW_MAKE_ERRNO_NTSTATUS(0)) == _LW_MAKE_ERRNO_NTSTATUS(0))

// Determine whether this is a failure code.  Only warning and
// error codes are considered failure.
#define LW_NT_SUCCESS(status) \
    (((LW_LONG)(status)) >= 0)

// Determines whether NTSTATUS is 0 or !NT_SUCCESS(status).
// This helps check for non-zero success code from APIs.
// TODO--Need a better name for this macro?
#define LW_NT_SUCCESS_OR_NOT(status) \
    (!LW_NT_SUCCESS(status) || (0 == (status)))

//
// NTSTATUS codes (sorted numerically)
//

// Success status codes
#define LW_STATUS_SUCCESS                             ((LW_NTSTATUS)0x00000000)
#define LW_STATUS_WAIT_1                              ((LW_NTSTATUS)0x00000001)
#define LW_STATUS_WAIT_2                              ((LW_NTSTATUS)0x00000002)
#define LW_STATUS_WAIT_3                              ((LW_NTSTATUS)0x00000003)
#define LW_STATUS_WAIT_63                             ((LW_NTSTATUS)0x0000003F)
#define LW_STATUS_ABANDONED_WAIT_0                    ((LW_NTSTATUS)0x00000080)
#define LW_STATUS_ABANDONED_WAIT_63                   ((LW_NTSTATUS)0x000000BF)
#define LW_STATUS_USER_APC                            ((LW_NTSTATUS)0x000000C0)
#define LW_STATUS_KERNEL_APC                          ((LW_NTSTATUS)0x00000100)
#define LW_STATUS_ALERTED                             ((LW_NTSTATUS)0x00000101)
#define LW_STATUS_TIMEOUT                             ((LW_NTSTATUS)0x00000102)
#define LW_STATUS_PENDING                             ((LW_NTSTATUS)0x00000103)
#define LW_STATUS_REPARSE                             ((LW_NTSTATUS)0x00000104)
#define LW_STATUS_MORE_ENTRIES                        ((LW_NTSTATUS)0x00000105)
#define LW_STATUS_NOT_ALL_ASSIGNED                    ((LW_NTSTATUS)0x00000106)
#define LW_STATUS_SOME_NOT_MAPPED                     ((LW_NTSTATUS)0x00000107)
#define LW_STATUS_OPLOCK_BREAK_IN_PROGRESS            ((LW_NTSTATUS)0x00000108)
#define LW_STATUS_VOLUME_MOUNTED                      ((LW_NTSTATUS)0x00000109)
#define LW_STATUS_RXACT_COMMITTED                     ((LW_NTSTATUS)0x0000010A)
#define LW_STATUS_NOTIFY_CLEANUP                      ((LW_NTSTATUS)0x0000010B)
#define LW_STATUS_NOTIFY_ENUM_DIR                     ((LW_NTSTATUS)0x0000010C)
#define LW_STATUS_NO_QUOTAS_FOR_ACCOUNT               ((LW_NTSTATUS)0x0000010D)
#define LW_STATUS_PRIMARY_TRANSPORT_CONNECT_FAILED    ((LW_NTSTATUS)0x0000010E)
#define LW_STATUS_PAGE_FAULT_TRANSITION               ((LW_NTSTATUS)0x00000110)
#define LW_STATUS_PAGE_FAULT_DEMAND_ZERO              ((LW_NTSTATUS)0x00000111)
#define LW_STATUS_PAGE_FAULT_COPY_ON_WRITE            ((LW_NTSTATUS)0x00000112)
#define LW_STATUS_PAGE_FAULT_GUARD_PAGE               ((LW_NTSTATUS)0x00000113)
#define LW_STATUS_PAGE_FAULT_PAGING_FILE              ((LW_NTSTATUS)0x00000114)
#define LW_STATUS_CACHE_PAGE_LOCKED                   ((LW_NTSTATUS)0x00000115)
#define LW_STATUS_CRASH_DUMP                          ((LW_NTSTATUS)0x00000116)
#define LW_STATUS_BUFFER_ALL_ZEROS                    ((LW_NTSTATUS)0x00000117)
#define LW_STATUS_REPARSE_OBJECT                      ((LW_NTSTATUS)0x00000118)
#define LW_STATUS_RESOURCE_REQUIREMENTS_CHANGED       ((LW_NTSTATUS)0x00000119)
#define LW_STATUS_TRANSLATION_COMPLETE                ((LW_NTSTATUS)0x00000120)
#define LW_STATUS_DS_MEMBERSHIP_EVALUATED_LOCALLY     ((LW_NTSTATUS)0x00000121)
#define LW_STATUS_NOTHING_TO_TERMINATE                ((LW_NTSTATUS)0x00000122)
#define LW_STATUS_PROCESS_NOT_IN_JOB                  ((LW_NTSTATUS)0x00000123)
#define LW_STATUS_PROCESS_IN_JOB                      ((LW_NTSTATUS)0x00000124)
#define LW_STATUS_VOLSNAP_HIBERNATE_READY             ((LW_NTSTATUS)0x00000125)
#define LW_STATUS_FSFILTER_OP_COMPLETED_SUCCESSFULLY  ((LW_NTSTATUS)0x00000126)
#define LW_STATUS_INTERRUPT_VECTOR_ALREADY_CONNECTED  ((LW_NTSTATUS)0x00000127)
#define LW_STATUS_INTERRUPT_STILL_CONNECTED           ((LW_NTSTATUS)0x00000128)
#define LW_STATUS_PROCESS_CLONED                      ((LW_NTSTATUS)0x00000129)
#define LW_STATUS_FILE_LOCKED_WITH_ONLY_READERS       ((LW_NTSTATUS)0x0000012A)
#define LW_STATUS_FILE_LOCKED_WITH_WRITERS            ((LW_NTSTATUS)0x0000012B)
#define LW_STATUS_RESOURCEMANAGER_READ_ONLY           ((LW_NTSTATUS)0x00000202)
#define LW_STATUS_WAIT_FOR_OPLOCK                     ((LW_NTSTATUS)0x00000367)
#define LW_DBG_EXCEPTION_HANDLED                      ((LW_NTSTATUS)0x00010001)
#define LW_DBG_CONTINUE                               ((LW_NTSTATUS)0x00010002)

// Informational status codes
#define LW_STATUS_OBJECT_NAME_EXISTS                  ((LW_NTSTATUS)0x40000000)
#define LW_STATUS_THREAD_WAS_SUSPENDED                ((LW_NTSTATUS)0x40000001)
#define LW_STATUS_WORKING_SET_LIMIT_RANGE             ((LW_NTSTATUS)0x40000002)
#define LW_STATUS_IMAGE_NOT_AT_BASE                   ((LW_NTSTATUS)0x40000003)
#define LW_STATUS_RXACT_STATE_CREATED                 ((LW_NTSTATUS)0x40000004)
#define LW_STATUS_SEGMENT_NOTIFICATION                ((LW_NTSTATUS)0x40000005)
#define LW_STATUS_LOCAL_USER_SESSION_KEY              ((LW_NTSTATUS)0x40000006)
#define LW_STATUS_BAD_CURRENT_DIRECTORY               ((LW_NTSTATUS)0x40000007)
#define LW_STATUS_SERIAL_MORE_WRITES                  ((LW_NTSTATUS)0x40000008)
#define LW_STATUS_REGISTRY_RECOVERED                  ((LW_NTSTATUS)0x40000009)
#define LW_STATUS_FT_READ_RECOVERY_FROM_BACKUP        ((LW_NTSTATUS)0x4000000A)
#define LW_STATUS_FT_WRITE_RECOVERY                   ((LW_NTSTATUS)0x4000000B)
#define LW_STATUS_SERIAL_COUNTER_TIMEOUT              ((LW_NTSTATUS)0x4000000C)
#define LW_STATUS_NULL_LM_PASSWORD                    ((LW_NTSTATUS)0x4000000D)
#define LW_STATUS_IMAGE_MACHINE_TYPE_MISMATCH         ((LW_NTSTATUS)0x4000000E)
#define LW_STATUS_RECEIVE_PARTIAL                     ((LW_NTSTATUS)0x4000000F)
#define LW_STATUS_RECEIVE_EXPEDITED                   ((LW_NTSTATUS)0x40000010)
#define LW_STATUS_RECEIVE_PARTIAL_EXPEDITED           ((LW_NTSTATUS)0x40000011)
#define LW_STATUS_EVENT_DONE                          ((LW_NTSTATUS)0x40000012)
#define LW_STATUS_EVENT_PENDING                       ((LW_NTSTATUS)0x40000013)
#define LW_STATUS_CHECKING_FILE_SYSTEM                ((LW_NTSTATUS)0x40000014)
#define LW_STATUS_FATAL_APP_EXIT                      ((LW_NTSTATUS)0x40000015)
#define LW_STATUS_PREDEFINED_HANDLE                   ((LW_NTSTATUS)0x40000016)
#define LW_STATUS_WAS_UNLOCKED                        ((LW_NTSTATUS)0x40000017)
#define LW_STATUS_SERVICE_NOTIFICATION                ((LW_NTSTATUS)0x40000018)
#define LW_STATUS_WAS_LOCKED                          ((LW_NTSTATUS)0x40000019)
#define LW_STATUS_LOG_HARD_ERROR                      ((LW_NTSTATUS)0x4000001A)
#define LW_STATUS_ALREADY_WIN32                       ((LW_NTSTATUS)0x4000001B)
#define LW_STATUS_WX86_UNSIMULATE                     ((LW_NTSTATUS)0x4000001C)
#define LW_STATUS_WX86_CONTINUE                       ((LW_NTSTATUS)0x4000001D)
#define LW_STATUS_WX86_SINGLE_STEP                    ((LW_NTSTATUS)0x4000001E)
#define LW_STATUS_WX86_BREAKPOINT                     ((LW_NTSTATUS)0x4000001F)
#define LW_STATUS_WX86_EXCEPTION_CONTINUE             ((LW_NTSTATUS)0x40000020)
#define LW_STATUS_WX86_EXCEPTION_LASTCHANCE           ((LW_NTSTATUS)0x40000021)
#define LW_STATUS_WX86_EXCEPTION_CHAIN                ((LW_NTSTATUS)0x40000022)
#define LW_STATUS_IMAGE_MACHINE_TYPE_MISMATCH_EXE     ((LW_NTSTATUS)0x40000023)
#define LW_STATUS_NO_YIELD_PERFORMED                  ((LW_NTSTATUS)0x40000024)
#define LW_STATUS_TIMER_RESUME_IGNORED                ((LW_NTSTATUS)0x40000025)
#define LW_STATUS_ARBITRATION_UNHANDLED               ((LW_NTSTATUS)0x40000026)
#define LW_STATUS_CARDBUS_NOT_SUPPORTED               ((LW_NTSTATUS)0x40000027)
#define LW_STATUS_WX86_CREATEWX86TIB                  ((LW_NTSTATUS)0x40000028)
#define LW_STATUS_MP_PROCESSOR_MISMATCH               ((LW_NTSTATUS)0x40000029)
#define LW_STATUS_HIBERNATED                          ((LW_NTSTATUS)0x4000002A)
#define LW_STATUS_RESUME_HIBERNATION                  ((LW_NTSTATUS)0x4000002B)
#define LW_STATUS_FIRMWARE_UPDATED                    ((LW_NTSTATUS)0x4000002C)
#define LW_STATUS_DRIVERS_LEAKING_LOCKED_PAGES        ((LW_NTSTATUS)0x4000002D)
#define LW_STATUS_MESSAGE_RETRIEVED                   ((LW_NTSTATUS)0x4000002E)
#define LW_STATUS_SYSTEM_POWERSTATE_TRANSITION        ((LW_NTSTATUS)0x4000002F)
#define LW_STATUS_ALPC_CHECK_COMPLETION_LIST          ((LW_NTSTATUS)0x40000030)
#define LW_STATUS_SYSTEM_POWERSTATE_COMPLEX_TRANSITION ((LW_NTSTATUS)0x40000031)
#define LW_STATUS_ACCESS_AUDIT_BY_POLICY              ((LW_NTSTATUS)0x40000032)
#define LW_STATUS_ABANDON_HIBERFILE                   ((LW_NTSTATUS)0x40000033)
#define LW_STATUS_BIZRULES_NOT_ENABLED                ((LW_NTSTATUS)0x40000034)
#define LW_STATUS_WAKE_SYSTEM                         ((LW_NTSTATUS)0x40000294)
#define LW_STATUS_DS_SHUTTING_DOWN                    ((LW_NTSTATUS)0x40000370)
#define LW_DBG_REPLY_LATER                            ((LW_NTSTATUS)0x40010001)
#define LW_DBG_UNABLE_TO_PROVIDE_HANDLE               ((LW_NTSTATUS)0x40010002)
#define LW_DBG_TERMINATE_THREAD                       ((LW_NTSTATUS)0x40010003)
#define LW_DBG_TERMINATE_PROCESS                      ((LW_NTSTATUS)0x40010004)
#define LW_DBG_CONTROL_C                              ((LW_NTSTATUS)0x40010005)
#define LW_DBG_PRINTEXCEPTION_C                       ((LW_NTSTATUS)0x40010006)
#define LW_DBG_RIPEXCEPTION                           ((LW_NTSTATUS)0x40010007)
#define LW_DBG_CONTROL_BREAK                          ((LW_NTSTATUS)0x40010008)
#define LW_DBG_COMMAND_EXCEPTION                      ((LW_NTSTATUS)0x40010009)
#define LW_RPC_NT_UUID_LOCAL_ONLY                     ((LW_NTSTATUS)0x40020056)
#define LW_RPC_NT_SEND_INCOMPLETE                     ((LW_NTSTATUS)0x400200AF)

// Warning status codes
#define LW_STATUS_GUARD_PAGE_VIOLATION                ((LW_NTSTATUS)0x80000001)
#define LW_STATUS_DATATYPE_MISALIGNMENT               ((LW_NTSTATUS)0x80000002)
#define LW_STATUS_BREAKPOINT                          ((LW_NTSTATUS)0x80000003)
#define LW_STATUS_SINGLE_STEP                         ((LW_NTSTATUS)0x80000004)
#define LW_STATUS_BUFFER_OVERFLOW                     ((LW_NTSTATUS)0x80000005)
#define LW_STATUS_NO_MORE_FILES                       ((LW_NTSTATUS)0x80000006)
#define LW_STATUS_WAKE_SYSTEM_DEBUGGER                ((LW_NTSTATUS)0x80000007)
#define LW_STATUS_HANDLES_CLOSED                      ((LW_NTSTATUS)0x8000000A)
#define LW_STATUS_NO_INHERITANCE                      ((LW_NTSTATUS)0x8000000B)
#define LW_STATUS_GUID_SUBSTITUTION_MADE              ((LW_NTSTATUS)0x8000000C)
#define LW_STATUS_PARTIAL_COPY                        ((LW_NTSTATUS)0x8000000D)
#define LW_STATUS_DEVICE_PAPER_EMPTY                  ((LW_NTSTATUS)0x8000000E)
#define LW_STATUS_DEVICE_POWERED_OFF                  ((LW_NTSTATUS)0x8000000F)
#define LW_STATUS_DEVICE_OFF_LINE                     ((LW_NTSTATUS)0x80000010)
#define LW_STATUS_DEVICE_BUSY                         ((LW_NTSTATUS)0x80000011)
#define LW_STATUS_NO_MORE_EAS                         ((LW_NTSTATUS)0x80000012)
#define LW_STATUS_INVALID_EA_NAME                     ((LW_NTSTATUS)0x80000013)
#define LW_STATUS_EA_LIST_INCONSISTENT                ((LW_NTSTATUS)0x80000014)
#define LW_STATUS_INVALID_EA_FLAG                     ((LW_NTSTATUS)0x80000015)
#define LW_STATUS_VERIFY_REQUIRED                     ((LW_NTSTATUS)0x80000016)
#define LW_STATUS_EXTRANEOUS_INFORMATION              ((LW_NTSTATUS)0x80000017)
#define LW_STATUS_RXACT_COMMIT_NECESSARY              ((LW_NTSTATUS)0x80000018)
#define LW_STATUS_NO_MORE_ENTRIES                     ((LW_NTSTATUS)0x8000001A)
#define LW_STATUS_FILEMARK_DETECTED                   ((LW_NTSTATUS)0x8000001B)
#define LW_STATUS_MEDIA_CHANGED                       ((LW_NTSTATUS)0x8000001C)
#define LW_STATUS_BUS_RESET                           ((LW_NTSTATUS)0x8000001D)
#define LW_STATUS_END_OF_MEDIA                        ((LW_NTSTATUS)0x8000001E)
#define LW_STATUS_BEGINNING_OF_MEDIA                  ((LW_NTSTATUS)0x8000001F)
#define LW_STATUS_MEDIA_CHECK                         ((LW_NTSTATUS)0x80000020)
#define LW_STATUS_SETMARK_DETECTED                    ((LW_NTSTATUS)0x80000021)
#define LW_STATUS_NO_DATA_DETECTED                    ((LW_NTSTATUS)0x80000022)
#define LW_STATUS_REDIRECTOR_HAS_OPEN_HANDLES         ((LW_NTSTATUS)0x80000023)
#define LW_STATUS_SERVER_HAS_OPEN_HANDLES             ((LW_NTSTATUS)0x80000024)
#define LW_STATUS_ALREADY_DISCONNECTED                ((LW_NTSTATUS)0x80000025)
#define LW_STATUS_LONGJUMP                            ((LW_NTSTATUS)0x80000026)
#define LW_STATUS_CLEANER_CARTRIDGE_INSTALLED         ((LW_NTSTATUS)0x80000027)
#define LW_STATUS_PLUGPLAY_QUERY_VETOED               ((LW_NTSTATUS)0x80000028)
#define LW_STATUS_UNWIND_CONSOLIDATE                  ((LW_NTSTATUS)0x80000029)
#define LW_STATUS_REGISTRY_HIVE_RECOVERED             ((LW_NTSTATUS)0x8000002A)
#define LW_STATUS_DLL_MIGHT_BE_INSECURE               ((LW_NTSTATUS)0x8000002B)
#define LW_STATUS_DLL_MIGHT_BE_INCOMPATIBLE           ((LW_NTSTATUS)0x8000002C)
#define LW_STATUS_STOPPED_ON_SYMLINK                  ((LW_NTSTATUS)0x8000002D)
#define LW_STATUS_DEVICE_REQUIRES_CLEANING            ((LW_NTSTATUS)0x80000288)
#define LW_STATUS_DEVICE_DOOR_OPEN                    ((LW_NTSTATUS)0x80000289)
#define LW_DBG_EXCEPTION_NOT_HANDLED                  ((LW_NTSTATUS)0x80010001)

// Error status codes
#define LW_STATUS_UNSUCCESSFUL                        ((LW_NTSTATUS)0xC0000001)
#define LW_STATUS_NOT_IMPLEMENTED                     ((LW_NTSTATUS)0xC0000002)
#define LW_STATUS_INVALID_INFO_CLASS                  ((LW_NTSTATUS)0xC0000003)
#define LW_STATUS_INFO_LENGTH_MISMATCH                ((LW_NTSTATUS)0xC0000004)
#define LW_STATUS_ACCESS_VIOLATION                    ((LW_NTSTATUS)0xC0000005)
#define LW_STATUS_IN_PAGE_ERROR                       ((LW_NTSTATUS)0xC0000006)
#define LW_STATUS_PAGEFILE_QUOTA                      ((LW_NTSTATUS)0xC0000007)
#define LW_STATUS_INVALID_HANDLE                      ((LW_NTSTATUS)0xC0000008)
#define LW_STATUS_BAD_INITIAL_STACK                   ((LW_NTSTATUS)0xC0000009)
#define LW_STATUS_BAD_INITIAL_PC                      ((LW_NTSTATUS)0xC000000A)
#define LW_STATUS_INVALID_CID                         ((LW_NTSTATUS)0xC000000B)
#define LW_STATUS_TIMER_NOT_CANCELED                  ((LW_NTSTATUS)0xC000000C)
#define LW_STATUS_INVALID_PARAMETER                   ((LW_NTSTATUS)0xC000000D)
#define LW_STATUS_NO_SUCH_DEVICE                      ((LW_NTSTATUS)0xC000000E)
#define LW_STATUS_NO_SUCH_FILE                        ((LW_NTSTATUS)0xC000000F)
#define LW_STATUS_INVALID_DEVICE_REQUEST              ((LW_NTSTATUS)0xC0000010)
#define LW_STATUS_END_OF_FILE                         ((LW_NTSTATUS)0xC0000011)
#define LW_STATUS_WRONG_VOLUME                        ((LW_NTSTATUS)0xC0000012)
#define LW_STATUS_NO_MEDIA_IN_DEVICE                  ((LW_NTSTATUS)0xC0000013)
#define LW_STATUS_UNRECOGNIZED_MEDIA                  ((LW_NTSTATUS)0xC0000014)
#define LW_STATUS_NONEXISTENT_SECTOR                  ((LW_NTSTATUS)0xC0000015)
#define LW_STATUS_MORE_PROCESSING_REQUIRED            ((LW_NTSTATUS)0xC0000016)
#define LW_STATUS_NO_MEMORY                           ((LW_NTSTATUS)0xC0000017)
#define LW_STATUS_CONFLICTING_ADDRESSES               ((LW_NTSTATUS)0xC0000018)
#define LW_STATUS_NOT_MAPPED_VIEW                     ((LW_NTSTATUS)0xC0000019)
#define LW_STATUS_UNABLE_TO_FREE_VM                   ((LW_NTSTATUS)0xC000001A)
#define LW_STATUS_UNABLE_TO_DELETE_SECTION            ((LW_NTSTATUS)0xC000001B)
#define LW_STATUS_INVALID_SYSTEM_SERVICE              ((LW_NTSTATUS)0xC000001C)
#define LW_STATUS_ILLEGAL_INSTRUCTION                 ((LW_NTSTATUS)0xC000001D)
#define LW_STATUS_INVALID_LOCK_SEQUENCE               ((LW_NTSTATUS)0xC000001E)
#define LW_STATUS_INVALID_VIEW_SIZE                   ((LW_NTSTATUS)0xC000001F)
#define LW_STATUS_INVALID_FILE_FOR_SECTION            ((LW_NTSTATUS)0xC0000020)
#define LW_STATUS_ALREADY_COMMITTED                   ((LW_NTSTATUS)0xC0000021)
#define LW_STATUS_ACCESS_DENIED                       ((LW_NTSTATUS)0xC0000022)
#define LW_STATUS_BUFFER_TOO_SMALL                    ((LW_NTSTATUS)0xC0000023)
#define LW_STATUS_OBJECT_TYPE_MISMATCH                ((LW_NTSTATUS)0xC0000024)
#define LW_STATUS_NONCONTINUABLE_EXCEPTION            ((LW_NTSTATUS)0xC0000025)
#define LW_STATUS_INVALID_DISPOSITION                 ((LW_NTSTATUS)0xC0000026)
#define LW_STATUS_UNWIND                              ((LW_NTSTATUS)0xC0000027)
#define LW_STATUS_BAD_STACK                           ((LW_NTSTATUS)0xC0000028)
#define LW_STATUS_INVALID_UNWIND_TARGET               ((LW_NTSTATUS)0xC0000029)
#define LW_STATUS_NOT_LOCKED                          ((LW_NTSTATUS)0xC000002A)
#define LW_STATUS_PARITY_ERROR                        ((LW_NTSTATUS)0xC000002B)
#define LW_STATUS_UNABLE_TO_DECOMMIT_VM               ((LW_NTSTATUS)0xC000002C)
#define LW_STATUS_NOT_COMMITTED                       ((LW_NTSTATUS)0xC000002D)
#define LW_STATUS_INVALID_PORT_ATTRIBUTES             ((LW_NTSTATUS)0xC000002E)
#define LW_STATUS_PORT_MESSAGE_TOO_LONG               ((LW_NTSTATUS)0xC000002F)
#define LW_STATUS_INVALID_PARAMETER_MIX               ((LW_NTSTATUS)0xC0000030)
#define LW_STATUS_INVALID_QUOTA_LOWER                 ((LW_NTSTATUS)0xC0000031)
#define LW_STATUS_DISK_CORRUPT_ERROR                  ((LW_NTSTATUS)0xC0000032)
#define LW_STATUS_OBJECT_NAME_INVALID                 ((LW_NTSTATUS)0xC0000033)
#define LW_STATUS_OBJECT_NAME_NOT_FOUND               ((LW_NTSTATUS)0xC0000034)
#define LW_STATUS_OBJECT_NAME_COLLISION               ((LW_NTSTATUS)0xC0000035)
#define LW_STATUS_PORT_DISCONNECTED                   ((LW_NTSTATUS)0xC0000037)
#define LW_STATUS_DEVICE_ALREADY_ATTACHED             ((LW_NTSTATUS)0xC0000038)
#define LW_STATUS_OBJECT_PATH_INVALID                 ((LW_NTSTATUS)0xC0000039)
#define LW_STATUS_OBJECT_PATH_NOT_FOUND               ((LW_NTSTATUS)0xC000003A)
#define LW_STATUS_OBJECT_PATH_SYNTAX_BAD              ((LW_NTSTATUS)0xC000003B)
#define LW_STATUS_DATA_OVERRUN                        ((LW_NTSTATUS)0xC000003C)
#define LW_STATUS_DATA_LATE_ERROR                     ((LW_NTSTATUS)0xC000003D)
#define LW_STATUS_DATA_ERROR                          ((LW_NTSTATUS)0xC000003E)
#define LW_STATUS_CRC_ERROR                           ((LW_NTSTATUS)0xC000003F)
#define LW_STATUS_SECTION_TOO_BIG                     ((LW_NTSTATUS)0xC0000040)
#define LW_STATUS_PORT_CONNECTION_REFUSED             ((LW_NTSTATUS)0xC0000041)
#define LW_STATUS_INVALID_PORT_HANDLE                 ((LW_NTSTATUS)0xC0000042)
#define LW_STATUS_SHARING_VIOLATION                   ((LW_NTSTATUS)0xC0000043)
#define LW_STATUS_QUOTA_EXCEEDED                      ((LW_NTSTATUS)0xC0000044)
#define LW_STATUS_INVALID_PAGE_PROTECTION             ((LW_NTSTATUS)0xC0000045)
#define LW_STATUS_MUTANT_NOT_OWNED                    ((LW_NTSTATUS)0xC0000046)
#define LW_STATUS_SEMAPHORE_LIMIT_EXCEEDED            ((LW_NTSTATUS)0xC0000047)
#define LW_STATUS_PORT_ALREADY_SET                    ((LW_NTSTATUS)0xC0000048)
#define LW_STATUS_SECTION_NOT_IMAGE                   ((LW_NTSTATUS)0xC0000049)
#define LW_STATUS_SUSPEND_COUNT_EXCEEDED              ((LW_NTSTATUS)0xC000004A)
#define LW_STATUS_THREAD_IS_TERMINATING               ((LW_NTSTATUS)0xC000004B)
#define LW_STATUS_BAD_WORKING_SET_LIMIT               ((LW_NTSTATUS)0xC000004C)
#define LW_STATUS_INCOMPATIBLE_FILE_MAP               ((LW_NTSTATUS)0xC000004D)
#define LW_STATUS_SECTION_PROTECTION                  ((LW_NTSTATUS)0xC000004E)
#define LW_STATUS_EAS_NOT_SUPPORTED                   ((LW_NTSTATUS)0xC000004F)
#define LW_STATUS_EA_TOO_LARGE                        ((LW_NTSTATUS)0xC0000050)
#define LW_STATUS_NONEXISTENT_EA_ENTRY                ((LW_NTSTATUS)0xC0000051)
#define LW_STATUS_NO_EAS_ON_FILE                      ((LW_NTSTATUS)0xC0000052)
#define LW_STATUS_EA_CORRUPT_ERROR                    ((LW_NTSTATUS)0xC0000053)
#define LW_STATUS_FILE_LOCK_CONFLICT                  ((LW_NTSTATUS)0xC0000054)
#define LW_STATUS_LOCK_NOT_GRANTED                    ((LW_NTSTATUS)0xC0000055)
#define LW_STATUS_DELETE_PENDING                      ((LW_NTSTATUS)0xC0000056)
#define LW_STATUS_CTL_FILE_NOT_SUPPORTED              ((LW_NTSTATUS)0xC0000057)
#define LW_STATUS_UNKNOWN_REVISION                    ((LW_NTSTATUS)0xC0000058)
#define LW_STATUS_REVISION_MISMATCH                   ((LW_NTSTATUS)0xC0000059)
#define LW_STATUS_INVALID_OWNER                       ((LW_NTSTATUS)0xC000005A)
#define LW_STATUS_INVALID_PRIMARY_GROUP               ((LW_NTSTATUS)0xC000005B)
#define LW_STATUS_NO_IMPERSONATION_TOKEN              ((LW_NTSTATUS)0xC000005C)
#define LW_STATUS_CANT_DISABLE_MANDATORY              ((LW_NTSTATUS)0xC000005D)
#define LW_STATUS_NO_LOGON_SERVERS                    ((LW_NTSTATUS)0xC000005E)
#define LW_STATUS_NO_SUCH_LOGON_SESSION               ((LW_NTSTATUS)0xC000005F)
#define LW_STATUS_NO_SUCH_PRIVILEGE                   ((LW_NTSTATUS)0xC0000060)
#define LW_STATUS_PRIVILEGE_NOT_HELD                  ((LW_NTSTATUS)0xC0000061)
#define LW_STATUS_INVALID_ACCOUNT_NAME                ((LW_NTSTATUS)0xC0000062)
#define LW_STATUS_USER_EXISTS                         ((LW_NTSTATUS)0xC0000063)
#define LW_STATUS_NO_SUCH_USER                        ((LW_NTSTATUS)0xC0000064)
#define LW_STATUS_GROUP_EXISTS                        ((LW_NTSTATUS)0xC0000065)
#define LW_STATUS_NO_SUCH_GROUP                       ((LW_NTSTATUS)0xC0000066)
#define LW_STATUS_MEMBER_IN_GROUP                     ((LW_NTSTATUS)0xC0000067)
#define LW_STATUS_MEMBER_NOT_IN_GROUP                 ((LW_NTSTATUS)0xC0000068)
#define LW_STATUS_LAST_ADMIN                          ((LW_NTSTATUS)0xC0000069)
#define LW_STATUS_WRONG_PASSWORD                      ((LW_NTSTATUS)0xC000006A)
#define LW_STATUS_ILL_FORMED_PASSWORD                 ((LW_NTSTATUS)0xC000006B)
#define LW_STATUS_PASSWORD_RESTRICTION                ((LW_NTSTATUS)0xC000006C)
#define LW_STATUS_LOGON_FAILURE                       ((LW_NTSTATUS)0xC000006D)
#define LW_STATUS_ACCOUNT_RESTRICTION                 ((LW_NTSTATUS)0xC000006E)
#define LW_STATUS_INVALID_LOGON_HOURS                 ((LW_NTSTATUS)0xC000006F)
#define LW_STATUS_INVALID_WORKSTATION                 ((LW_NTSTATUS)0xC0000070)
#define LW_STATUS_PASSWORD_EXPIRED                    ((LW_NTSTATUS)0xC0000071)
#define LW_STATUS_ACCOUNT_DISABLED                    ((LW_NTSTATUS)0xC0000072)
#define LW_STATUS_NONE_MAPPED                         ((LW_NTSTATUS)0xC0000073)
#define LW_STATUS_TOO_MANY_LUIDS_REQUESTED            ((LW_NTSTATUS)0xC0000074)
#define LW_STATUS_LUIDS_EXHAUSTED                     ((LW_NTSTATUS)0xC0000075)
#define LW_STATUS_INVALID_SUB_AUTHORITY               ((LW_NTSTATUS)0xC0000076)
#define LW_STATUS_INVALID_ACL                         ((LW_NTSTATUS)0xC0000077)
#define LW_STATUS_INVALID_SID                         ((LW_NTSTATUS)0xC0000078)
#define LW_STATUS_INVALID_SECURITY_DESCR              ((LW_NTSTATUS)0xC0000079)
#define LW_STATUS_PROCEDURE_NOT_FOUND                 ((LW_NTSTATUS)0xC000007A)
#define LW_STATUS_INVALID_IMAGE_FORMAT                ((LW_NTSTATUS)0xC000007B)
#define LW_STATUS_NO_TOKEN                            ((LW_NTSTATUS)0xC000007C)
#define LW_STATUS_BAD_INHERITANCE_ACL                 ((LW_NTSTATUS)0xC000007D)
#define LW_STATUS_RANGE_NOT_LOCKED                    ((LW_NTSTATUS)0xC000007E)
#define LW_STATUS_DISK_FULL                           ((LW_NTSTATUS)0xC000007F)
#define LW_STATUS_SERVER_DISABLED                     ((LW_NTSTATUS)0xC0000080)
#define LW_STATUS_SERVER_NOT_DISABLED                 ((LW_NTSTATUS)0xC0000081)
#define LW_STATUS_TOO_MANY_GUIDS_REQUESTED            ((LW_NTSTATUS)0xC0000082)
#define LW_STATUS_GUIDS_EXHAUSTED                     ((LW_NTSTATUS)0xC0000083)
#define LW_STATUS_INVALID_ID_AUTHORITY                ((LW_NTSTATUS)0xC0000084)
#define LW_STATUS_AGENTS_EXHAUSTED                    ((LW_NTSTATUS)0xC0000085)
#define LW_STATUS_INVALID_VOLUME_LABEL                ((LW_NTSTATUS)0xC0000086)
#define LW_STATUS_SECTION_NOT_EXTENDED                ((LW_NTSTATUS)0xC0000087)
#define LW_STATUS_NOT_MAPPED_DATA                     ((LW_NTSTATUS)0xC0000088)
#define LW_STATUS_RESOURCE_DATA_NOT_FOUND             ((LW_NTSTATUS)0xC0000089)
#define LW_STATUS_RESOURCE_TYPE_NOT_FOUND             ((LW_NTSTATUS)0xC000008A)
#define LW_STATUS_RESOURCE_NAME_NOT_FOUND             ((LW_NTSTATUS)0xC000008B)
#define LW_STATUS_ARRAY_BOUNDS_EXCEEDED               ((LW_NTSTATUS)0xC000008C)
#define LW_STATUS_FLOAT_DENORMAL_OPERAND              ((LW_NTSTATUS)0xC000008D)
#define LW_STATUS_FLOAT_DIVIDE_BY_ZERO                ((LW_NTSTATUS)0xC000008E)
#define LW_STATUS_FLOAT_INEXACT_RESULT                ((LW_NTSTATUS)0xC000008F)
#define LW_STATUS_FLOAT_INVALID_OPERATION             ((LW_NTSTATUS)0xC0000090)
#define LW_STATUS_FLOAT_OVERFLOW                      ((LW_NTSTATUS)0xC0000091)
#define LW_STATUS_FLOAT_STACK_CHECK                   ((LW_NTSTATUS)0xC0000092)
#define LW_STATUS_FLOAT_UNDERFLOW                     ((LW_NTSTATUS)0xC0000093)
#define LW_STATUS_INTEGER_DIVIDE_BY_ZERO              ((LW_NTSTATUS)0xC0000094)
#define LW_STATUS_INTEGER_OVERFLOW                    ((LW_NTSTATUS)0xC0000095)
#define LW_STATUS_PRIVILEGED_INSTRUCTION              ((LW_NTSTATUS)0xC0000096)
#define LW_STATUS_TOO_MANY_PAGING_FILES               ((LW_NTSTATUS)0xC0000097)
#define LW_STATUS_FILE_INVALID                        ((LW_NTSTATUS)0xC0000098)
#define LW_STATUS_ALLOTTED_SPACE_EXCEEDED             ((LW_NTSTATUS)0xC0000099)
#define LW_STATUS_INSUFFICIENT_RESOURCES              ((LW_NTSTATUS)0xC000009A)
#define LW_STATUS_DFS_EXIT_PATH_FOUND                 ((LW_NTSTATUS)0xC000009B)
#define LW_STATUS_DEVICE_DATA_ERROR                   ((LW_NTSTATUS)0xC000009C)
#define LW_STATUS_DEVICE_NOT_CONNECTED                ((LW_NTSTATUS)0xC000009D)
#define LW_STATUS_DEVICE_POWER_FAILURE                ((LW_NTSTATUS)0xC000009E)
#define LW_STATUS_FREE_VM_NOT_AT_BASE                 ((LW_NTSTATUS)0xC000009F)
#define LW_STATUS_MEMORY_NOT_ALLOCATED                ((LW_NTSTATUS)0xC00000A0)
#define LW_STATUS_WORKING_SET_QUOTA                   ((LW_NTSTATUS)0xC00000A1)
#define LW_STATUS_MEDIA_WRITE_PROTECTED               ((LW_NTSTATUS)0xC00000A2)
#define LW_STATUS_DEVICE_NOT_READY                    ((LW_NTSTATUS)0xC00000A3)
#define LW_STATUS_INVALID_GROUP_ATTRIBUTES            ((LW_NTSTATUS)0xC00000A4)
#define LW_STATUS_BAD_IMPERSONATION_LEVEL             ((LW_NTSTATUS)0xC00000A5)
#define LW_STATUS_CANT_OPEN_ANONYMOUS                 ((LW_NTSTATUS)0xC00000A6)
#define LW_STATUS_BAD_VALIDATION_CLASS                ((LW_NTSTATUS)0xC00000A7)
#define LW_STATUS_BAD_TOKEN_TYPE                      ((LW_NTSTATUS)0xC00000A8)
#define LW_STATUS_BAD_MASTER_BOOT_RECORD              ((LW_NTSTATUS)0xC00000A9)
#define LW_STATUS_INSTRUCTION_MISALIGNMENT            ((LW_NTSTATUS)0xC00000AA)
#define LW_STATUS_INSTANCE_NOT_AVAILABLE              ((LW_NTSTATUS)0xC00000AB)
#define LW_STATUS_PIPE_NOT_AVAILABLE                  ((LW_NTSTATUS)0xC00000AC)
#define LW_STATUS_INVALID_PIPE_STATE                  ((LW_NTSTATUS)0xC00000AD)
#define LW_STATUS_PIPE_BUSY                           ((LW_NTSTATUS)0xC00000AE)
#define LW_STATUS_ILLEGAL_FUNCTION                    ((LW_NTSTATUS)0xC00000AF)
#define LW_STATUS_PIPE_DISCONNECTED                   ((LW_NTSTATUS)0xC00000B0)
#define LW_STATUS_PIPE_CLOSING                        ((LW_NTSTATUS)0xC00000B1)
#define LW_STATUS_PIPE_CONNECTED                      ((LW_NTSTATUS)0xC00000B2)
#define LW_STATUS_PIPE_LISTENING                      ((LW_NTSTATUS)0xC00000B3)
#define LW_STATUS_INVALID_READ_MODE                   ((LW_NTSTATUS)0xC00000B4)
#define LW_STATUS_IO_TIMEOUT                          ((LW_NTSTATUS)0xC00000B5)
#define LW_STATUS_FILE_FORCED_CLOSED                  ((LW_NTSTATUS)0xC00000B6)
#define LW_STATUS_PROFILING_NOT_STARTED               ((LW_NTSTATUS)0xC00000B7)
#define LW_STATUS_PROFILING_NOT_STOPPED               ((LW_NTSTATUS)0xC00000B8)
#define LW_STATUS_COULD_NOT_INTERPRET                 ((LW_NTSTATUS)0xC00000B9)
#define LW_STATUS_FILE_IS_A_DIRECTORY                 ((LW_NTSTATUS)0xC00000BA)
#define LW_STATUS_NOT_SUPPORTED                       ((LW_NTSTATUS)0xC00000BB)
#define LW_STATUS_REMOTE_NOT_LISTENING                ((LW_NTSTATUS)0xC00000BC)
#define LW_STATUS_DUPLICATE_NAME                      ((LW_NTSTATUS)0xC00000BD)
#define LW_STATUS_BAD_NETWORK_PATH                    ((LW_NTSTATUS)0xC00000BE)
#define LW_STATUS_NETWORK_BUSY                        ((LW_NTSTATUS)0xC00000BF)
#define LW_STATUS_DEVICE_DOES_NOT_EXIST               ((LW_NTSTATUS)0xC00000C0)
#define LW_STATUS_TOO_MANY_COMMANDS                   ((LW_NTSTATUS)0xC00000C1)
#define LW_STATUS_ADAPTER_HARDWARE_ERROR              ((LW_NTSTATUS)0xC00000C2)
#define LW_STATUS_INVALID_NETWORK_RESPONSE            ((LW_NTSTATUS)0xC00000C3)
#define LW_STATUS_UNEXPECTED_NETWORK_ERROR            ((LW_NTSTATUS)0xC00000C4)
#define LW_STATUS_BAD_REMOTE_ADAPTER                  ((LW_NTSTATUS)0xC00000C5)
#define LW_STATUS_PRINT_QUEUE_FULL                    ((LW_NTSTATUS)0xC00000C6)
#define LW_STATUS_NO_SPOOL_SPACE                      ((LW_NTSTATUS)0xC00000C7)
#define LW_STATUS_PRINT_CANCELLED                     ((LW_NTSTATUS)0xC00000C8)
#define LW_STATUS_NETWORK_NAME_DELETED                ((LW_NTSTATUS)0xC00000C9)
#define LW_STATUS_NETWORK_ACCESS_DENIED               ((LW_NTSTATUS)0xC00000CA)
#define LW_STATUS_BAD_DEVICE_TYPE                     ((LW_NTSTATUS)0xC00000CB)
#define LW_STATUS_BAD_NETWORK_NAME                    ((LW_NTSTATUS)0xC00000CC)
#define LW_STATUS_TOO_MANY_NAMES                      ((LW_NTSTATUS)0xC00000CD)
#define LW_STATUS_TOO_MANY_SESSIONS                   ((LW_NTSTATUS)0xC00000CE)
#define LW_STATUS_SHARING_PAUSED                      ((LW_NTSTATUS)0xC00000CF)
#define LW_STATUS_REQUEST_NOT_ACCEPTED                ((LW_NTSTATUS)0xC00000D0)
#define LW_STATUS_REDIRECTOR_PAUSED                   ((LW_NTSTATUS)0xC00000D1)
#define LW_STATUS_NET_WRITE_FAULT                     ((LW_NTSTATUS)0xC00000D2)
#define LW_STATUS_PROFILING_AT_LIMIT                  ((LW_NTSTATUS)0xC00000D3)
#define LW_STATUS_NOT_SAME_DEVICE                     ((LW_NTSTATUS)0xC00000D4)
#define LW_STATUS_FILE_RENAMED                        ((LW_NTSTATUS)0xC00000D5)
#define LW_STATUS_VIRTUAL_CIRCUIT_CLOSED              ((LW_NTSTATUS)0xC00000D6)
#define LW_STATUS_NO_SECURITY_ON_OBJECT               ((LW_NTSTATUS)0xC00000D7)
#define LW_STATUS_CANT_WAIT                           ((LW_NTSTATUS)0xC00000D8)
#define LW_STATUS_PIPE_EMPTY                          ((LW_NTSTATUS)0xC00000D9)
#define LW_STATUS_CANT_ACCESS_DOMAIN_INFO             ((LW_NTSTATUS)0xC00000DA)
#define LW_STATUS_CANT_TERMINATE_SELF                 ((LW_NTSTATUS)0xC00000DB)
#define LW_STATUS_INVALID_SERVER_STATE                ((LW_NTSTATUS)0xC00000DC)
#define LW_STATUS_INVALID_DOMAIN_STATE                ((LW_NTSTATUS)0xC00000DD)
#define LW_STATUS_INVALID_DOMAIN_ROLE                 ((LW_NTSTATUS)0xC00000DE)
#define LW_STATUS_NO_SUCH_DOMAIN                      ((LW_NTSTATUS)0xC00000DF)
#define LW_STATUS_DOMAIN_EXISTS                       ((LW_NTSTATUS)0xC00000E0)
#define LW_STATUS_DOMAIN_LIMIT_EXCEEDED               ((LW_NTSTATUS)0xC00000E1)
#define LW_STATUS_OPLOCK_NOT_GRANTED                  ((LW_NTSTATUS)0xC00000E2)
#define LW_STATUS_INVALID_OPLOCK_PROTOCOL             ((LW_NTSTATUS)0xC00000E3)
#define LW_STATUS_INTERNAL_DB_CORRUPTION              ((LW_NTSTATUS)0xC00000E4)
#define LW_STATUS_INTERNAL_ERROR                      ((LW_NTSTATUS)0xC00000E5)
#define LW_STATUS_GENERIC_NOT_MAPPED                  ((LW_NTSTATUS)0xC00000E6)
#define LW_STATUS_BAD_DESCRIPTOR_FORMAT               ((LW_NTSTATUS)0xC00000E7)
#define LW_STATUS_INVALID_USER_BUFFER                 ((LW_NTSTATUS)0xC00000E8)
#define LW_STATUS_UNEXPECTED_IO_ERROR                 ((LW_NTSTATUS)0xC00000E9)
#define LW_STATUS_UNEXPECTED_MM_CREATE_ERR            ((LW_NTSTATUS)0xC00000EA)
#define LW_STATUS_UNEXPECTED_MM_MAP_ERROR             ((LW_NTSTATUS)0xC00000EB)
#define LW_STATUS_UNEXPECTED_MM_EXTEND_ERR            ((LW_NTSTATUS)0xC00000EC)
#define LW_STATUS_NOT_LOGON_PROCESS                   ((LW_NTSTATUS)0xC00000ED)
#define LW_STATUS_LOGON_SESSION_EXISTS                ((LW_NTSTATUS)0xC00000EE)
#define LW_STATUS_INVALID_PARAMETER_1                 ((LW_NTSTATUS)0xC00000EF)
#define LW_STATUS_INVALID_PARAMETER_2                 ((LW_NTSTATUS)0xC00000F0)
#define LW_STATUS_INVALID_PARAMETER_3                 ((LW_NTSTATUS)0xC00000F1)
#define LW_STATUS_INVALID_PARAMETER_4                 ((LW_NTSTATUS)0xC00000F2)
#define LW_STATUS_INVALID_PARAMETER_5                 ((LW_NTSTATUS)0xC00000F3)
#define LW_STATUS_INVALID_PARAMETER_6                 ((LW_NTSTATUS)0xC00000F4)
#define LW_STATUS_INVALID_PARAMETER_7                 ((LW_NTSTATUS)0xC00000F5)
#define LW_STATUS_INVALID_PARAMETER_8                 ((LW_NTSTATUS)0xC00000F6)
#define LW_STATUS_INVALID_PARAMETER_9                 ((LW_NTSTATUS)0xC00000F7)
#define LW_STATUS_INVALID_PARAMETER_10                ((LW_NTSTATUS)0xC00000F8)
#define LW_STATUS_INVALID_PARAMETER_11                ((LW_NTSTATUS)0xC00000F9)
#define LW_STATUS_INVALID_PARAMETER_12                ((LW_NTSTATUS)0xC00000FA)
#define LW_STATUS_REDIRECTOR_NOT_STARTED              ((LW_NTSTATUS)0xC00000FB)
#define LW_STATUS_REDIRECTOR_STARTED                  ((LW_NTSTATUS)0xC00000FC)
#define LW_STATUS_STACK_OVERFLOW                      ((LW_NTSTATUS)0xC00000FD)
#define LW_STATUS_NO_SUCH_PACKAGE                     ((LW_NTSTATUS)0xC00000FE)
#define LW_STATUS_BAD_FUNCTION_TABLE                  ((LW_NTSTATUS)0xC00000FF)
#define LW_STATUS_VARIABLE_NOT_FOUND                  ((LW_NTSTATUS)0xC0000100)
#define LW_STATUS_DIRECTORY_NOT_EMPTY                 ((LW_NTSTATUS)0xC0000101)
#define LW_STATUS_FILE_CORRUPT_ERROR                  ((LW_NTSTATUS)0xC0000102)
#define LW_STATUS_NOT_A_DIRECTORY                     ((LW_NTSTATUS)0xC0000103)
#define LW_STATUS_BAD_LOGON_SESSION_STATE             ((LW_NTSTATUS)0xC0000104)
#define LW_STATUS_LOGON_SESSION_COLLISION             ((LW_NTSTATUS)0xC0000105)
#define LW_STATUS_NAME_TOO_LONG                       ((LW_NTSTATUS)0xC0000106)
#define LW_STATUS_FILES_OPEN                          ((LW_NTSTATUS)0xC0000107)
#define LW_STATUS_CONNECTION_IN_USE                   ((LW_NTSTATUS)0xC0000108)
#define LW_STATUS_MESSAGE_NOT_FOUND                   ((LW_NTSTATUS)0xC0000109)
#define LW_STATUS_PROCESS_IS_TERMINATING              ((LW_NTSTATUS)0xC000010A)
#define LW_STATUS_INVALID_LOGON_TYPE                  ((LW_NTSTATUS)0xC000010B)
#define LW_STATUS_NO_GUID_TRANSLATION                 ((LW_NTSTATUS)0xC000010C)
#define LW_STATUS_CANNOT_IMPERSONATE                  ((LW_NTSTATUS)0xC000010D)
#define LW_STATUS_IMAGE_ALREADY_LOADED                ((LW_NTSTATUS)0xC000010E)
#define LW_STATUS_ABIOS_NOT_PRESENT                   ((LW_NTSTATUS)0xC000010F)
#define LW_STATUS_ABIOS_LID_NOT_EXIST                 ((LW_NTSTATUS)0xC0000110)
#define LW_STATUS_ABIOS_LID_ALREADY_OWNED             ((LW_NTSTATUS)0xC0000111)
#define LW_STATUS_ABIOS_NOT_LID_OWNER                 ((LW_NTSTATUS)0xC0000112)
#define LW_STATUS_ABIOS_INVALID_COMMAND               ((LW_NTSTATUS)0xC0000113)
#define LW_STATUS_ABIOS_INVALID_LID                   ((LW_NTSTATUS)0xC0000114)
#define LW_STATUS_ABIOS_SELECTOR_NOT_AVAILABLE        ((LW_NTSTATUS)0xC0000115)
#define LW_STATUS_ABIOS_INVALID_SELECTOR              ((LW_NTSTATUS)0xC0000116)
#define LW_STATUS_NO_LDT                              ((LW_NTSTATUS)0xC0000117)
#define LW_STATUS_INVALID_LDT_SIZE                    ((LW_NTSTATUS)0xC0000118)
#define LW_STATUS_INVALID_LDT_OFFSET                  ((LW_NTSTATUS)0xC0000119)
#define LW_STATUS_INVALID_LDT_DESCRIPTOR              ((LW_NTSTATUS)0xC000011A)
#define LW_STATUS_INVALID_IMAGE_NE_FORMAT             ((LW_NTSTATUS)0xC000011B)
#define LW_STATUS_RXACT_INVALID_STATE                 ((LW_NTSTATUS)0xC000011C)
#define LW_STATUS_RXACT_COMMIT_FAILURE                ((LW_NTSTATUS)0xC000011D)
#define LW_STATUS_MAPPED_FILE_SIZE_ZERO               ((LW_NTSTATUS)0xC000011E)
#define LW_STATUS_TOO_MANY_OPENED_FILES               ((LW_NTSTATUS)0xC000011F)
#define LW_STATUS_CANCELLED                           ((LW_NTSTATUS)0xC0000120)
#define LW_STATUS_CANNOT_DELETE                       ((LW_NTSTATUS)0xC0000121)
#define LW_STATUS_INVALID_COMPUTER_NAME               ((LW_NTSTATUS)0xC0000122)
#define LW_STATUS_FILE_DELETED                        ((LW_NTSTATUS)0xC0000123)
#define LW_STATUS_SPECIAL_ACCOUNT                     ((LW_NTSTATUS)0xC0000124)
#define LW_STATUS_SPECIAL_GROUP                       ((LW_NTSTATUS)0xC0000125)
#define LW_STATUS_SPECIAL_USER                        ((LW_NTSTATUS)0xC0000126)
#define LW_STATUS_MEMBERS_PRIMARY_GROUP               ((LW_NTSTATUS)0xC0000127)
#define LW_STATUS_FILE_CLOSED                         ((LW_NTSTATUS)0xC0000128)
#define LW_STATUS_TOO_MANY_THREADS                    ((LW_NTSTATUS)0xC0000129)
#define LW_STATUS_THREAD_NOT_IN_PROCESS               ((LW_NTSTATUS)0xC000012A)
#define LW_STATUS_TOKEN_ALREADY_IN_USE                ((LW_NTSTATUS)0xC000012B)
#define LW_STATUS_PAGEFILE_QUOTA_EXCEEDED             ((LW_NTSTATUS)0xC000012C)
#define LW_STATUS_COMMITMENT_LIMIT                    ((LW_NTSTATUS)0xC000012D)
#define LW_STATUS_INVALID_IMAGE_LE_FORMAT             ((LW_NTSTATUS)0xC000012E)
#define LW_STATUS_INVALID_IMAGE_NOT_MZ                ((LW_NTSTATUS)0xC000012F)
#define LW_STATUS_INVALID_IMAGE_PROTECT               ((LW_NTSTATUS)0xC0000130)
#define LW_STATUS_INVALID_IMAGE_WIN_16                ((LW_NTSTATUS)0xC0000131)
#define LW_STATUS_LOGON_SERVER_CONFLICT               ((LW_NTSTATUS)0xC0000132)
#define LW_STATUS_TIME_DIFFERENCE_AT_DC               ((LW_NTSTATUS)0xC0000133)
#define LW_STATUS_SYNCHRONIZATION_REQUIRED            ((LW_NTSTATUS)0xC0000134)
#define LW_STATUS_DLL_NOT_FOUND                       ((LW_NTSTATUS)0xC0000135)
#define LW_STATUS_OPEN_FAILED                         ((LW_NTSTATUS)0xC0000136)
#define LW_STATUS_IO_PRIVILEGE_FAILED                 ((LW_NTSTATUS)0xC0000137)
#define LW_STATUS_ORDINAL_NOT_FOUND                   ((LW_NTSTATUS)0xC0000138)
#define LW_STATUS_ENTRYPOINT_NOT_FOUND                ((LW_NTSTATUS)0xC0000139)
#define LW_STATUS_CONTROL_C_EXIT                      ((LW_NTSTATUS)0xC000013A)
#define LW_STATUS_LOCAL_DISCONNECT                    ((LW_NTSTATUS)0xC000013B)
#define LW_STATUS_REMOTE_DISCONNECT                   ((LW_NTSTATUS)0xC000013C)
#define LW_STATUS_REMOTE_RESOURCES                    ((LW_NTSTATUS)0xC000013D)
#define LW_STATUS_LINK_FAILED                         ((LW_NTSTATUS)0xC000013E)
#define LW_STATUS_LINK_TIMEOUT                        ((LW_NTSTATUS)0xC000013F)
#define LW_STATUS_INVALID_CONNECTION                  ((LW_NTSTATUS)0xC0000140)
#define LW_STATUS_INVALID_ADDRESS                     ((LW_NTSTATUS)0xC0000141)
#define LW_STATUS_DLL_INIT_FAILED                     ((LW_NTSTATUS)0xC0000142)
#define LW_STATUS_MISSING_SYSTEMFILE                  ((LW_NTSTATUS)0xC0000143)
#define LW_STATUS_UNHANDLED_EXCEPTION                 ((LW_NTSTATUS)0xC0000144)
#define LW_STATUS_APP_INIT_FAILURE                    ((LW_NTSTATUS)0xC0000145)
#define LW_STATUS_PAGEFILE_CREATE_FAILED              ((LW_NTSTATUS)0xC0000146)
#define LW_STATUS_NO_PAGEFILE                         ((LW_NTSTATUS)0xC0000147)
#define LW_STATUS_INVALID_LEVEL                       ((LW_NTSTATUS)0xC0000148)
#define LW_STATUS_WRONG_PASSWORD_CORE                 ((LW_NTSTATUS)0xC0000149)
#define LW_STATUS_ILLEGAL_FLOAT_CONTEXT               ((LW_NTSTATUS)0xC000014A)
#define LW_STATUS_PIPE_BROKEN                         ((LW_NTSTATUS)0xC000014B)
#define LW_STATUS_REGISTRY_CORRUPT                    ((LW_NTSTATUS)0xC000014C)
#define LW_STATUS_REGISTRY_IO_FAILED                  ((LW_NTSTATUS)0xC000014D)
#define LW_STATUS_NO_EVENT_PAIR                       ((LW_NTSTATUS)0xC000014E)
#define LW_STATUS_UNRECOGNIZED_VOLUME                 ((LW_NTSTATUS)0xC000014F)
#define LW_STATUS_SERIAL_NO_DEVICE_INITED             ((LW_NTSTATUS)0xC0000150)
#define LW_STATUS_NO_SUCH_ALIAS                       ((LW_NTSTATUS)0xC0000151)
#define LW_STATUS_MEMBER_NOT_IN_ALIAS                 ((LW_NTSTATUS)0xC0000152)
#define LW_STATUS_MEMBER_IN_ALIAS                     ((LW_NTSTATUS)0xC0000153)
#define LW_STATUS_ALIAS_EXISTS                        ((LW_NTSTATUS)0xC0000154)
#define LW_STATUS_LOGON_NOT_GRANTED                   ((LW_NTSTATUS)0xC0000155)
#define LW_STATUS_TOO_MANY_SECRETS                    ((LW_NTSTATUS)0xC0000156)
#define LW_STATUS_SECRET_TOO_LONG                     ((LW_NTSTATUS)0xC0000157)
#define LW_STATUS_INTERNAL_DB_ERROR                   ((LW_NTSTATUS)0xC0000158)
#define LW_STATUS_FULLSCREEN_MODE                     ((LW_NTSTATUS)0xC0000159)
#define LW_STATUS_TOO_MANY_CONTEXT_IDS                ((LW_NTSTATUS)0xC000015A)
#define LW_STATUS_LOGON_TYPE_NOT_GRANTED              ((LW_NTSTATUS)0xC000015B)
#define LW_STATUS_NOT_REGISTRY_FILE                   ((LW_NTSTATUS)0xC000015C)
#define LW_STATUS_NT_CROSS_ENCRYPTION_REQUIRED        ((LW_NTSTATUS)0xC000015D)
#define LW_STATUS_DOMAIN_CTRLR_CONFIG_ERROR           ((LW_NTSTATUS)0xC000015E)
#define LW_STATUS_FT_MISSING_MEMBER                   ((LW_NTSTATUS)0xC000015F)
#define LW_STATUS_ILL_FORMED_SERVICE_ENTRY            ((LW_NTSTATUS)0xC0000160)
#define LW_STATUS_ILLEGAL_CHARACTER                   ((LW_NTSTATUS)0xC0000161)
#define LW_STATUS_UNMAPPABLE_CHARACTER                ((LW_NTSTATUS)0xC0000162)
#define LW_STATUS_UNDEFINED_CHARACTER                 ((LW_NTSTATUS)0xC0000163)
#define LW_STATUS_FLOPPY_VOLUME                       ((LW_NTSTATUS)0xC0000164)
#define LW_STATUS_FLOPPY_ID_MARK_NOT_FOUND            ((LW_NTSTATUS)0xC0000165)
#define LW_STATUS_FLOPPY_WRONG_CYLINDER               ((LW_NTSTATUS)0xC0000166)
#define LW_STATUS_FLOPPY_UNKNOWN_ERROR                ((LW_NTSTATUS)0xC0000167)
#define LW_STATUS_FLOPPY_BAD_REGISTERS                ((LW_NTSTATUS)0xC0000168)
#define LW_STATUS_DISK_RECALIBRATE_FAILED             ((LW_NTSTATUS)0xC0000169)
#define LW_STATUS_DISK_OPERATION_FAILED               ((LW_NTSTATUS)0xC000016A)
#define LW_STATUS_DISK_RESET_FAILED                   ((LW_NTSTATUS)0xC000016B)
#define LW_STATUS_SHARED_IRQ_BUSY                     ((LW_NTSTATUS)0xC000016C)
#define LW_STATUS_FT_ORPHANING                        ((LW_NTSTATUS)0xC000016D)
#define LW_STATUS_BIOS_FAILED_TO_CONNECT_INTERRUPT    ((LW_NTSTATUS)0xC000016E)
#define LW_STATUS_PARTITION_FAILURE                   ((LW_NTSTATUS)0xC0000172)
#define LW_STATUS_INVALID_BLOCK_LENGTH                ((LW_NTSTATUS)0xC0000173)
#define LW_STATUS_DEVICE_NOT_PARTITIONED              ((LW_NTSTATUS)0xC0000174)
#define LW_STATUS_UNABLE_TO_LOCK_MEDIA                ((LW_NTSTATUS)0xC0000175)
#define LW_STATUS_UNABLE_TO_UNLOAD_MEDIA              ((LW_NTSTATUS)0xC0000176)
#define LW_STATUS_EOM_OVERFLOW                        ((LW_NTSTATUS)0xC0000177)
#define LW_STATUS_NO_MEDIA                            ((LW_NTSTATUS)0xC0000178)
#define LW_STATUS_NO_SUCH_MEMBER                      ((LW_NTSTATUS)0xC000017A)
#define LW_STATUS_INVALID_MEMBER                      ((LW_NTSTATUS)0xC000017B)
#define LW_STATUS_KEY_DELETED                         ((LW_NTSTATUS)0xC000017C)
#define LW_STATUS_NO_LOG_SPACE                        ((LW_NTSTATUS)0xC000017D)
#define LW_STATUS_TOO_MANY_SIDS                       ((LW_NTSTATUS)0xC000017E)
#define LW_STATUS_LM_CROSS_ENCRYPTION_REQUIRED        ((LW_NTSTATUS)0xC000017F)
#define LW_STATUS_KEY_HAS_CHILDREN                    ((LW_NTSTATUS)0xC0000180)
#define LW_STATUS_CHILD_MUST_BE_VOLATILE              ((LW_NTSTATUS)0xC0000181)
#define LW_STATUS_DEVICE_CONFIGURATION_ERROR          ((LW_NTSTATUS)0xC0000182)
#define LW_STATUS_DRIVER_INTERNAL_ERROR               ((LW_NTSTATUS)0xC0000183)
#define LW_STATUS_INVALID_DEVICE_STATE                ((LW_NTSTATUS)0xC0000184)
#define LW_STATUS_IO_DEVICE_ERROR                     ((LW_NTSTATUS)0xC0000185)
#define LW_STATUS_DEVICE_PROTOCOL_ERROR               ((LW_NTSTATUS)0xC0000186)
#define LW_STATUS_BACKUP_CONTROLLER                   ((LW_NTSTATUS)0xC0000187)
#define LW_STATUS_LOG_FILE_FULL                       ((LW_NTSTATUS)0xC0000188)
#define LW_STATUS_TOO_LATE                            ((LW_NTSTATUS)0xC0000189)
#define LW_STATUS_NO_TRUST_LSA_SECRET                 ((LW_NTSTATUS)0xC000018A)
#define LW_STATUS_NO_TRUST_SAM_ACCOUNT                ((LW_NTSTATUS)0xC000018B)
#define LW_STATUS_TRUSTED_DOMAIN_FAILURE              ((LW_NTSTATUS)0xC000018C)
#define LW_STATUS_TRUSTED_RELATIONSHIP_FAILURE        ((LW_NTSTATUS)0xC000018D)
#define LW_STATUS_EVENTLOG_FILE_CORRUPT               ((LW_NTSTATUS)0xC000018E)
#define LW_STATUS_EVENTLOG_CANT_START                 ((LW_NTSTATUS)0xC000018F)
#define LW_STATUS_TRUST_FAILURE                       ((LW_NTSTATUS)0xC0000190)
#define LW_STATUS_MUTANT_LIMIT_EXCEEDED               ((LW_NTSTATUS)0xC0000191)
#define LW_STATUS_NETLOGON_NOT_STARTED                ((LW_NTSTATUS)0xC0000192)
#define LW_STATUS_ACCOUNT_EXPIRED                     ((LW_NTSTATUS)0xC0000193)
#define LW_STATUS_POSSIBLE_DEADLOCK                   ((LW_NTSTATUS)0xC0000194)
#define LW_STATUS_NETWORK_CREDENTIAL_CONFLICT         ((LW_NTSTATUS)0xC0000195)
#define LW_STATUS_REMOTE_SESSION_LIMIT                ((LW_NTSTATUS)0xC0000196)
#define LW_STATUS_EVENTLOG_FILE_CHANGED               ((LW_NTSTATUS)0xC0000197)
#define LW_STATUS_NOLOGON_INTERDOMAIN_TRUST_ACCOUNT   ((LW_NTSTATUS)0xC0000198)
#define LW_STATUS_NOLOGON_WORKSTATION_TRUST_ACCOUNT   ((LW_NTSTATUS)0xC0000199)
#define LW_STATUS_NOLOGON_SERVER_TRUST_ACCOUNT        ((LW_NTSTATUS)0xC000019A)
#define LW_STATUS_DOMAIN_TRUST_INCONSISTENT           ((LW_NTSTATUS)0xC000019B)
#define LW_STATUS_FS_DRIVER_REQUIRED                  ((LW_NTSTATUS)0xC000019C)
#define LW_STATUS_IMAGE_ALREADY_LOADED_AS_DLL         ((LW_NTSTATUS)0xC000019D)
#define LW_STATUS_INVALID_LOCK_RANGE                  ((LW_NTSTATUS)0xC00001a1)
#define LW_STATUS_NETWORK_OPEN_RESTRICTION            ((LW_NTSTATUS)0xC0000201)
#define LW_STATUS_NO_USER_SESSION_KEY                 ((LW_NTSTATUS)0xC0000202)
#define LW_STATUS_USER_SESSION_DELETED                ((LW_NTSTATUS)0xC0000203)
#define LW_STATUS_RESOURCE_LANG_NOT_FOUND             ((LW_NTSTATUS)0xC0000204)
#define LW_STATUS_INSUFF_SERVER_RESOURCES             ((LW_NTSTATUS)0xC0000205)
#define LW_STATUS_INVALID_BUFFER_SIZE                 ((LW_NTSTATUS)0xC0000206)
#define LW_STATUS_INVALID_ADDRESS_COMPONENT           ((LW_NTSTATUS)0xC0000207)
#define LW_STATUS_INVALID_ADDRESS_WILDCARD            ((LW_NTSTATUS)0xC0000208)
#define LW_STATUS_TOO_MANY_ADDRESSES                  ((LW_NTSTATUS)0xC0000209)
#define LW_STATUS_ADDRESS_ALREADY_EXISTS              ((LW_NTSTATUS)0xC000020A)
#define LW_STATUS_ADDRESS_CLOSED                      ((LW_NTSTATUS)0xC000020B)
#define LW_STATUS_CONNECTION_DISCONNECTED             ((LW_NTSTATUS)0xC000020C)
#define LW_STATUS_CONNECTION_RESET                    ((LW_NTSTATUS)0xC000020D)
#define LW_STATUS_TOO_MANY_NODES                      ((LW_NTSTATUS)0xC000020E)
#define LW_STATUS_TRANSACTION_ABORTED                 ((LW_NTSTATUS)0xC000020F)
#define LW_STATUS_TRANSACTION_TIMED_OUT               ((LW_NTSTATUS)0xC0000210)
#define LW_STATUS_TRANSACTION_NO_RELEASE              ((LW_NTSTATUS)0xC0000211)
#define LW_STATUS_TRANSACTION_NO_MATCH                ((LW_NTSTATUS)0xC0000212)
#define LW_STATUS_TRANSACTION_RESPONDED               ((LW_NTSTATUS)0xC0000213)
#define LW_STATUS_TRANSACTION_INVALID_ID              ((LW_NTSTATUS)0xC0000214)
#define LW_STATUS_TRANSACTION_INVALID_TYPE            ((LW_NTSTATUS)0xC0000215)
#define LW_STATUS_NOT_SERVER_SESSION                  ((LW_NTSTATUS)0xC0000216)
#define LW_STATUS_NOT_CLIENT_SESSION                  ((LW_NTSTATUS)0xC0000217)
#define LW_STATUS_CANNOT_LOAD_REGISTRY_FILE           ((LW_NTSTATUS)0xC0000218)
#define LW_STATUS_DEBUG_ATTACH_FAILED                 ((LW_NTSTATUS)0xC0000219)
#define LW_STATUS_SYSTEM_PROCESS_TERMINATED           ((LW_NTSTATUS)0xC000021A)
#define LW_STATUS_DATA_NOT_ACCEPTED                   ((LW_NTSTATUS)0xC000021B)
#define LW_STATUS_NO_BROWSER_SERVERS_FOUND            ((LW_NTSTATUS)0xC000021C)
#define LW_STATUS_VDM_HARD_ERROR                      ((LW_NTSTATUS)0xC000021D)
#define LW_STATUS_DRIVER_CANCEL_TIMEOUT               ((LW_NTSTATUS)0xC000021E)
#define LW_STATUS_REPLY_MESSAGE_MISMATCH              ((LW_NTSTATUS)0xC000021F)
#define LW_STATUS_MAPPED_ALIGNMENT                    ((LW_NTSTATUS)0xC0000220)
#define LW_STATUS_IMAGE_CHECKSUM_MISMATCH             ((LW_NTSTATUS)0xC0000221)
#define LW_STATUS_LOST_WRITEBEHIND_DATA               ((LW_NTSTATUS)0xC0000222)
#define LW_STATUS_CLIENT_SERVER_PARAMETERS_INVALID    ((LW_NTSTATUS)0xC0000223)
#define LW_STATUS_PASSWORD_MUST_CHANGE                ((LW_NTSTATUS)0xC0000224)
#define LW_STATUS_NOT_FOUND                           ((LW_NTSTATUS)0xC0000225)
#define LW_STATUS_NOT_TINY_STREAM                     ((LW_NTSTATUS)0xC0000226)
#define LW_STATUS_RECOVERY_FAILURE                    ((LW_NTSTATUS)0xC0000227)
#define LW_STATUS_STACK_OVERFLOW_READ                 ((LW_NTSTATUS)0xC0000228)
#define LW_STATUS_FAIL_CHECK                          ((LW_NTSTATUS)0xC0000229)
#define LW_STATUS_DUPLICATE_OBJECTID                  ((LW_NTSTATUS)0xC000022A)
#define LW_STATUS_OBJECTID_EXISTS                     ((LW_NTSTATUS)0xC000022B)
#define LW_STATUS_CONVERT_TO_LARGE                    ((LW_NTSTATUS)0xC000022C)
#define LW_STATUS_RETRY                               ((LW_NTSTATUS)0xC000022D)
#define LW_STATUS_FOUND_OUT_OF_SCOPE                  ((LW_NTSTATUS)0xC000022E)
#define LW_STATUS_ALLOCATE_BUCKET                     ((LW_NTSTATUS)0xC000022F)
#define LW_STATUS_PROPSET_NOT_FOUND                   ((LW_NTSTATUS)0xC0000230)
#define LW_STATUS_MARSHALL_OVERFLOW                   ((LW_NTSTATUS)0xC0000231)
#define LW_STATUS_INVALID_VARIANT                     ((LW_NTSTATUS)0xC0000232)
#define LW_STATUS_DOMAIN_CONTROLLER_NOT_FOUND         ((LW_NTSTATUS)0xC0000233)
#define LW_STATUS_ACCOUNT_LOCKED_OUT                  ((LW_NTSTATUS)0xC0000234)
#define LW_STATUS_HANDLE_NOT_CLOSABLE                 ((LW_NTSTATUS)0xC0000235)
#define LW_STATUS_CONNECTION_REFUSED                  ((LW_NTSTATUS)0xC0000236)
#define LW_STATUS_GRACEFUL_DISCONNECT                 ((LW_NTSTATUS)0xC0000237)
#define LW_STATUS_ADDRESS_ALREADY_ASSOCIATED          ((LW_NTSTATUS)0xC0000238)
#define LW_STATUS_ADDRESS_NOT_ASSOCIATED              ((LW_NTSTATUS)0xC0000239)
#define LW_STATUS_CONNECTION_INVALID                  ((LW_NTSTATUS)0xC000023A)
#define LW_STATUS_CONNECTION_ACTIVE                   ((LW_NTSTATUS)0xC000023B)
#define LW_STATUS_NETWORK_UNREACHABLE                 ((LW_NTSTATUS)0xC000023C)
#define LW_STATUS_HOST_UNREACHABLE                    ((LW_NTSTATUS)0xC000023D)
#define LW_STATUS_PROTOCOL_UNREACHABLE                ((LW_NTSTATUS)0xC000023E)
#define LW_STATUS_PORT_UNREACHABLE                    ((LW_NTSTATUS)0xC000023F)
#define LW_STATUS_REQUEST_ABORTED                     ((LW_NTSTATUS)0xC0000240)
#define LW_STATUS_CONNECTION_ABORTED                  ((LW_NTSTATUS)0xC0000241)
#define LW_STATUS_BAD_COMPRESSION_BUFFER              ((LW_NTSTATUS)0xC0000242)
#define LW_STATUS_USER_MAPPED_FILE                    ((LW_NTSTATUS)0xC0000243)
#define LW_STATUS_AUDIT_FAILED                        ((LW_NTSTATUS)0xC0000244)
#define LW_STATUS_TIMER_RESOLUTION_NOT_SET            ((LW_NTSTATUS)0xC0000245)
#define LW_STATUS_CONNECTION_COUNT_LIMIT              ((LW_NTSTATUS)0xC0000246)
#define LW_STATUS_LOGIN_TIME_RESTRICTION              ((LW_NTSTATUS)0xC0000247)
#define LW_STATUS_LOGIN_WKSTA_RESTRICTION             ((LW_NTSTATUS)0xC0000248)
#define LW_STATUS_IMAGE_MP_UP_MISMATCH                ((LW_NTSTATUS)0xC0000249)
#define LW_STATUS_INSUFFICIENT_LOGON_INFO             ((LW_NTSTATUS)0xC0000250)
#define LW_STATUS_BAD_DLL_ENTRYPOINT                  ((LW_NTSTATUS)0xC0000251)
#define LW_STATUS_BAD_SERVICE_ENTRYPOINT              ((LW_NTSTATUS)0xC0000252)
#define LW_STATUS_LPC_REPLY_LOST                      ((LW_NTSTATUS)0xC0000253)
#define LW_STATUS_IP_ADDRESS_CONFLICT1                ((LW_NTSTATUS)0xC0000254)
#define LW_STATUS_IP_ADDRESS_CONFLICT2                ((LW_NTSTATUS)0xC0000255)
#define LW_STATUS_REGISTRY_QUOTA_LIMIT                ((LW_NTSTATUS)0xC0000256)
#define LW_STATUS_PATH_NOT_COVERED                    ((LW_NTSTATUS)0xC0000257)
#define LW_STATUS_NO_CALLBACK_ACTIVE                  ((LW_NTSTATUS)0xC0000258)
#define LW_STATUS_LICENSE_QUOTA_EXCEEDED              ((LW_NTSTATUS)0xC0000259)
#define LW_STATUS_PWD_TOO_SHORT                       ((LW_NTSTATUS)0xC000025A)
#define LW_STATUS_PWD_TOO_RECENT                      ((LW_NTSTATUS)0xC000025B)
#define LW_STATUS_PWD_HISTORY_CONFLICT                ((LW_NTSTATUS)0xC000025C)
#define LW_STATUS_PLUGPLAY_NO_DEVICE                  ((LW_NTSTATUS)0xC000025E)
#define LW_STATUS_UNSUPPORTED_COMPRESSION             ((LW_NTSTATUS)0xC000025F)
#define LW_STATUS_INVALID_HW_PROFILE                  ((LW_NTSTATUS)0xC0000260)
#define LW_STATUS_INVALID_PLUGPLAY_DEVICE_PATH        ((LW_NTSTATUS)0xC0000261)
#define LW_STATUS_DRIVER_ORDINAL_NOT_FOUND            ((LW_NTSTATUS)0xC0000262)
#define LW_STATUS_DRIVER_ENTRYPOINT_NOT_FOUND         ((LW_NTSTATUS)0xC0000263)
#define LW_STATUS_RESOURCE_NOT_OWNED                  ((LW_NTSTATUS)0xC0000264)
#define LW_STATUS_TOO_MANY_LINKS                      ((LW_NTSTATUS)0xC0000265)
#define LW_STATUS_QUOTA_LIST_INCONSISTENT             ((LW_NTSTATUS)0xC0000266)
#define LW_STATUS_FILE_IS_OFFLINE                     ((LW_NTSTATUS)0xC0000267)
#define LW_STATUS_EVALUATION_EXPIRATION               ((LW_NTSTATUS)0xC0000268)
#define LW_STATUS_ILLEGAL_DLL_RELOCATION              ((LW_NTSTATUS)0xC0000269)
#define LW_STATUS_LICENSE_VIOLATION                   ((LW_NTSTATUS)0xC000026A)
#define LW_STATUS_DLL_INIT_FAILED_LOGOFF              ((LW_NTSTATUS)0xC000026B)
#define LW_STATUS_DRIVER_UNABLE_TO_LOAD               ((LW_NTSTATUS)0xC000026C)
#define LW_STATUS_DFS_UNAVAILABLE                     ((LW_NTSTATUS)0xC000026D)
#define LW_STATUS_VOLUME_DISMOUNTED                   ((LW_NTSTATUS)0xC000026E)
#define LW_STATUS_WX86_INTERNAL_ERROR                 ((LW_NTSTATUS)0xC000026F)
#define LW_STATUS_WX86_FLOAT_STACK_CHECK              ((LW_NTSTATUS)0xC0000270)
#define LW_STATUS_VALIDATE_CONTINUE                   ((LW_NTSTATUS)0xC0000271)
#define LW_STATUS_NO_MATCH                            ((LW_NTSTATUS)0xC0000272)
#define LW_STATUS_NO_MORE_MATCHES                     ((LW_NTSTATUS)0xC0000273)
#define LW_STATUS_NOT_A_REPARSE_POINT                 ((LW_NTSTATUS)0xC0000275)
#define LW_STATUS_IO_REPARSE_TAG_INVALID              ((LW_NTSTATUS)0xC0000276)
#define LW_STATUS_IO_REPARSE_TAG_MISMATCH             ((LW_NTSTATUS)0xC0000277)
#define LW_STATUS_IO_REPARSE_DATA_INVALID             ((LW_NTSTATUS)0xC0000278)
#define LW_STATUS_IO_REPARSE_TAG_NOT_HANDLED          ((LW_NTSTATUS)0xC0000279)
#define LW_STATUS_REPARSE_POINT_NOT_RESOLVED          ((LW_NTSTATUS)0xC0000280)
#define LW_STATUS_DIRECTORY_IS_A_REPARSE_POINT        ((LW_NTSTATUS)0xC0000281)
#define LW_STATUS_RANGE_LIST_CONFLICT                 ((LW_NTSTATUS)0xC0000282)
#define LW_STATUS_SOURCE_ELEMENT_EMPTY                ((LW_NTSTATUS)0xC0000283)
#define LW_STATUS_DESTINATION_ELEMENT_FULL            ((LW_NTSTATUS)0xC0000284)
#define LW_STATUS_ILLEGAL_ELEMENT_ADDRESS             ((LW_NTSTATUS)0xC0000285)
#define LW_STATUS_MAGAZINE_NOT_PRESENT                ((LW_NTSTATUS)0xC0000286)
#define LW_STATUS_REINITIALIZATION_NEEDED             ((LW_NTSTATUS)0xC0000287)
#define LW_STATUS_ENCRYPTION_FAILED                   ((LW_NTSTATUS)0xC000028A)
#define LW_STATUS_DECRYPTION_FAILED                   ((LW_NTSTATUS)0xC000028B)
#define LW_STATUS_RANGE_NOT_FOUND                     ((LW_NTSTATUS)0xC000028C)
#define LW_STATUS_NO_RECOVERY_POLICY                  ((LW_NTSTATUS)0xC000028D)
#define LW_STATUS_NO_EFS                              ((LW_NTSTATUS)0xC000028E)
#define LW_STATUS_WRONG_EFS                           ((LW_NTSTATUS)0xC000028F)
#define LW_STATUS_NO_USER_KEYS                        ((LW_NTSTATUS)0xC0000290)
#define LW_STATUS_FILE_NOT_ENCRYPTED                  ((LW_NTSTATUS)0xC0000291)
#define LW_STATUS_NOT_EXPORT_FORMAT                   ((LW_NTSTATUS)0xC0000292)
#define LW_STATUS_FILE_ENCRYPTED                      ((LW_NTSTATUS)0xC0000293)
#define LW_STATUS_WMI_GUID_NOT_FOUND                  ((LW_NTSTATUS)0xC0000295)
#define LW_STATUS_WMI_INSTANCE_NOT_FOUND              ((LW_NTSTATUS)0xC0000296)
#define LW_STATUS_WMI_ITEMID_NOT_FOUND                ((LW_NTSTATUS)0xC0000297)
#define LW_STATUS_WMI_TRY_AGAIN                       ((LW_NTSTATUS)0xC0000298)
#define LW_STATUS_SHARED_POLICY                       ((LW_NTSTATUS)0xC0000299)
#define LW_STATUS_POLICY_OBJECT_NOT_FOUND             ((LW_NTSTATUS)0xC000029A)
#define LW_STATUS_POLICY_ONLY_IN_DS                   ((LW_NTSTATUS)0xC000029B)
#define LW_STATUS_VOLUME_NOT_UPGRADED                 ((LW_NTSTATUS)0xC000029C)
#define LW_STATUS_REMOTE_STORAGE_NOT_ACTIVE           ((LW_NTSTATUS)0xC000029D)
#define LW_STATUS_REMOTE_STORAGE_MEDIA_ERROR          ((LW_NTSTATUS)0xC000029E)
#define LW_STATUS_NO_TRACKING_SERVICE                 ((LW_NTSTATUS)0xC000029F)
#define LW_STATUS_SERVER_SID_MISMATCH                 ((LW_NTSTATUS)0xC00002A0)
#define LW_STATUS_DS_NO_ATTRIBUTE_OR_VALUE            ((LW_NTSTATUS)0xC00002A1)
#define LW_STATUS_DS_INVALID_ATTRIBUTE_SYNTAX         ((LW_NTSTATUS)0xC00002A2)
#define LW_STATUS_DS_ATTRIBUTE_TYPE_UNDEFINED         ((LW_NTSTATUS)0xC00002A3)
#define LW_STATUS_DS_ATTRIBUTE_OR_VALUE_EXISTS        ((LW_NTSTATUS)0xC00002A4)
#define LW_STATUS_DS_BUSY                             ((LW_NTSTATUS)0xC00002A5)
#define LW_STATUS_DS_UNAVAILABLE                      ((LW_NTSTATUS)0xC00002A6)
#define LW_STATUS_DS_NO_RIDS_ALLOCATED                ((LW_NTSTATUS)0xC00002A7)
#define LW_STATUS_DS_NO_MORE_RIDS                     ((LW_NTSTATUS)0xC00002A8)
#define LW_STATUS_DS_INCORRECT_ROLE_OWNER             ((LW_NTSTATUS)0xC00002A9)
#define LW_STATUS_DS_RIDMGR_INIT_ERROR                ((LW_NTSTATUS)0xC00002AA)
#define LW_STATUS_DS_OBJ_CLASS_VIOLATION              ((LW_NTSTATUS)0xC00002AB)
#define LW_STATUS_DS_CANT_ON_NON_LEAF                 ((LW_NTSTATUS)0xC00002AC)
#define LW_STATUS_DS_CANT_ON_RDN                      ((LW_NTSTATUS)0xC00002AD)
#define LW_STATUS_DS_CANT_MOD_OBJ_CLASS               ((LW_NTSTATUS)0xC00002AE)
#define LW_STATUS_DS_CROSS_DOM_MOVE_FAILED            ((LW_NTSTATUS)0xC00002AF)
#define LW_STATUS_DS_GC_NOT_AVAILABLE                 ((LW_NTSTATUS)0xC00002B0)
#define LW_STATUS_DIRECTORY_SERVICE_REQUIRED          ((LW_NTSTATUS)0xC00002B1)
#define LW_STATUS_REPARSE_ATTRIBUTE_CONFLICT          ((LW_NTSTATUS)0xC00002B2)
#define LW_STATUS_CANT_ENABLE_DENY_ONLY               ((LW_NTSTATUS)0xC00002B3)
#define LW_STATUS_FLOAT_MULTIPLE_FAULTS               ((LW_NTSTATUS)0xC00002B4)
#define LW_STATUS_FLOAT_MULTIPLE_TRAPS                ((LW_NTSTATUS)0xC00002B5)
#define LW_STATUS_DEVICE_REMOVED                      ((LW_NTSTATUS)0xC00002B6)
#define LW_STATUS_JOURNAL_DELETE_IN_PROGRESS          ((LW_NTSTATUS)0xC00002B7)
#define LW_STATUS_JOURNAL_NOT_ACTIVE                  ((LW_NTSTATUS)0xC00002B8)
#define LW_STATUS_NOINTERFACE                         ((LW_NTSTATUS)0xC00002B9)
#define LW_STATUS_DS_ADMIN_LIMIT_EXCEEDED             ((LW_NTSTATUS)0xC00002C1)
#define LW_STATUS_DRIVER_FAILED_SLEEP                 ((LW_NTSTATUS)0xC00002C2)
#define LW_STATUS_MUTUAL_AUTHENTICATION_FAILED        ((LW_NTSTATUS)0xC00002C3)
#define LW_STATUS_CORRUPT_SYSTEM_FILE                 ((LW_NTSTATUS)0xC00002C4)
#define LW_STATUS_DATATYPE_MISALIGNMENT_ERROR         ((LW_NTSTATUS)0xC00002C5)
#define LW_STATUS_WMI_READ_ONLY                       ((LW_NTSTATUS)0xC00002C6)
#define LW_STATUS_WMI_SET_FAILURE                     ((LW_NTSTATUS)0xC00002C7)
#define LW_STATUS_COMMITMENT_MINIMUM                  ((LW_NTSTATUS)0xC00002C8)
#define LW_STATUS_REG_NAT_CONSUMPTION                 ((LW_NTSTATUS)0xC00002C9)
#define LW_STATUS_TRANSPORT_FULL                      ((LW_NTSTATUS)0xC00002CA)
#define LW_STATUS_DS_SAM_INIT_FAILURE                 ((LW_NTSTATUS)0xC00002CB)
#define LW_STATUS_ONLY_IF_CONNECTED                   ((LW_NTSTATUS)0xC00002CC)
#define LW_STATUS_DS_SENSITIVE_GROUP_VIOLATION        ((LW_NTSTATUS)0xC00002CD)
#define LW_STATUS_PNP_RESTART_ENUMERATION             ((LW_NTSTATUS)0xC00002CE)
#define LW_STATUS_JOURNAL_ENTRY_DELETED               ((LW_NTSTATUS)0xC00002CF)
#define LW_STATUS_DS_CANT_MOD_PRIMARYGROUPID          ((LW_NTSTATUS)0xC00002D0)
#define LW_STATUS_SYSTEM_IMAGE_BAD_SIGNATURE          ((LW_NTSTATUS)0xC00002D1)
#define LW_STATUS_PNP_REBOOT_REQUIRED                 ((LW_NTSTATUS)0xC00002D2)
#define LW_STATUS_POWER_STATE_INVALID                 ((LW_NTSTATUS)0xC00002D3)
#define LW_STATUS_DS_INVALID_GROUP_TYPE               ((LW_NTSTATUS)0xC00002D4)
#define LW_STATUS_DS_NO_NEST_GLOBALGROUP_IN_MIXEDDOMAIN ((LW_NTSTATUS)0xC00002D5)
#define LW_STATUS_DS_NO_NEST_LOCALGROUP_IN_MIXEDDOMAIN ((LW_NTSTATUS)0xC00002D6)
#define LW_STATUS_DS_GLOBAL_CANT_HAVE_LOCAL_MEMBER    ((LW_NTSTATUS)0xC00002D7)
#define LW_STATUS_DS_GLOBAL_CANT_HAVE_UNIVERSAL_MEMBER ((LW_NTSTATUS)0xC00002D8)
#define LW_STATUS_DS_UNIVERSAL_CANT_HAVE_LOCAL_MEMBER ((LW_NTSTATUS)0xC00002D9)
#define LW_STATUS_DS_GLOBAL_CANT_HAVE_CROSSDOMAIN_MEMBER ((LW_NTSTATUS)0xC00002DA)
#define LW_STATUS_DS_LOCAL_CANT_HAVE_CROSSDOMAIN_LOCAL_MEMBER ((LW_NTSTATUS)0xC00002DB)
#define LW_STATUS_DS_HAVE_PRIMARY_MEMBERS             ((LW_NTSTATUS)0xC00002DC)
#define LW_STATUS_WMI_NOT_SUPPORTED                   ((LW_NTSTATUS)0xC00002DD)
#define LW_STATUS_INSUFFICIENT_POWER                  ((LW_NTSTATUS)0xC00002DE)
#define LW_STATUS_SAM_NEED_BOOTKEY_PASSWORD           ((LW_NTSTATUS)0xC00002DF)
#define LW_STATUS_SAM_NEED_BOOTKEY_FLOPPY             ((LW_NTSTATUS)0xC00002E0)
#define LW_STATUS_DS_CANT_START                       ((LW_NTSTATUS)0xC00002E1)
#define LW_STATUS_DS_INIT_FAILURE                     ((LW_NTSTATUS)0xC00002E2)
#define LW_STATUS_SAM_INIT_FAILURE                    ((LW_NTSTATUS)0xC00002E3)
#define LW_STATUS_DS_GC_REQUIRED                      ((LW_NTSTATUS)0xC00002E4)
#define LW_STATUS_DS_LOCAL_MEMBER_OF_LOCAL_ONLY       ((LW_NTSTATUS)0xC00002E5)
#define LW_STATUS_DS_NO_FPO_IN_UNIVERSAL_GROUPS       ((LW_NTSTATUS)0xC00002E6)
#define LW_STATUS_DS_MACHINE_ACCOUNT_QUOTA_EXCEEDED   ((LW_NTSTATUS)0xC00002E7)
#define LW_STATUS_MULTIPLE_FAULT_VIOLATION            ((LW_NTSTATUS)0xC00002E8)
#define LW_STATUS_CURRENT_DOMAIN_NOT_ALLOWED          ((LW_NTSTATUS)0xC00002E9)
#define LW_STATUS_CANNOT_MAKE                         ((LW_NTSTATUS)0xC00002EA)
#define LW_STATUS_SYSTEM_SHUTDOWN                     ((LW_NTSTATUS)0xC00002EB)
#define LW_STATUS_DS_INIT_FAILURE_CONSOLE             ((LW_NTSTATUS)0xC00002EC)
#define LW_STATUS_DS_SAM_INIT_FAILURE_CONSOLE         ((LW_NTSTATUS)0xC00002ED)
#define LW_STATUS_UNFINISHED_CONTEXT_DELETED          ((LW_NTSTATUS)0xC00002EE)
#define LW_STATUS_NO_TGT_REPLY                        ((LW_NTSTATUS)0xC00002EF)
#define LW_STATUS_OBJECTID_NOT_FOUND                  ((LW_NTSTATUS)0xC00002F0)
#define LW_STATUS_NO_IP_ADDRESSES                     ((LW_NTSTATUS)0xC00002F1)
#define LW_STATUS_WRONG_CREDENTIAL_HANDLE             ((LW_NTSTATUS)0xC00002F2)
#define LW_STATUS_CRYPTO_SYSTEM_INVALID               ((LW_NTSTATUS)0xC00002F3)
#define LW_STATUS_MAX_REFERRALS_EXCEEDED              ((LW_NTSTATUS)0xC00002F4)
#define LW_STATUS_MUST_BE_KDC                         ((LW_NTSTATUS)0xC00002F5)
#define LW_STATUS_STRONG_CRYPTO_NOT_SUPPORTED         ((LW_NTSTATUS)0xC00002F6)
#define LW_STATUS_TOO_MANY_PRINCIPALS                 ((LW_NTSTATUS)0xC00002F7)
#define LW_STATUS_NO_PA_DATA                          ((LW_NTSTATUS)0xC00002F8)
#define LW_STATUS_PKINIT_NAME_MISMATCH                ((LW_NTSTATUS)0xC00002F9)
#define LW_STATUS_SMARTCARD_LOGON_REQUIRED            ((LW_NTSTATUS)0xC00002FA)
#define LW_STATUS_KDC_INVALID_REQUEST                 ((LW_NTSTATUS)0xC00002FB)
#define LW_STATUS_KDC_UNABLE_TO_REFER                 ((LW_NTSTATUS)0xC00002FC)
#define LW_STATUS_KDC_UNKNOWN_ETYPE                   ((LW_NTSTATUS)0xC00002FD)
#define LW_STATUS_SHUTDOWN_IN_PROGRESS                ((LW_NTSTATUS)0xC00002FE)
#define LW_STATUS_SERVER_SHUTDOWN_IN_PROGRESS         ((LW_NTSTATUS)0xC00002FF)
#define LW_STATUS_NOT_SUPPORTED_ON_SBS                ((LW_NTSTATUS)0xC0000300)
#define LW_STATUS_WMI_GUID_DISCONNECTED               ((LW_NTSTATUS)0xC0000301)
#define LW_STATUS_WMI_ALREADY_DISABLED                ((LW_NTSTATUS)0xC0000302)
#define LW_STATUS_WMI_ALREADY_ENABLED                 ((LW_NTSTATUS)0xC0000303)
#define LW_STATUS_MFT_TOO_FRAGMENTED                  ((LW_NTSTATUS)0xC0000304)
#define LW_STATUS_COPY_PROTECTION_FAILURE             ((LW_NTSTATUS)0xC0000305)
#define LW_STATUS_CSS_AUTHENTICATION_FAILURE          ((LW_NTSTATUS)0xC0000306)
#define LW_STATUS_CSS_KEY_NOT_PRESENT                 ((LW_NTSTATUS)0xC0000307)
#define LW_STATUS_CSS_KEY_NOT_ESTABLISHED             ((LW_NTSTATUS)0xC0000308)
#define LW_STATUS_CSS_SCRAMBLED_SECTOR                ((LW_NTSTATUS)0xC0000309)
#define LW_STATUS_CSS_REGION_MISMATCH                 ((LW_NTSTATUS)0xC000030A)
#define LW_STATUS_CSS_RESETS_EXHAUSTED                ((LW_NTSTATUS)0xC000030B)
#define LW_STATUS_PKINIT_FAILURE                      ((LW_NTSTATUS)0xC0000320)
#define LW_STATUS_SMARTCARD_SUBSYSTEM_FAILURE         ((LW_NTSTATUS)0xC0000321)
#define LW_STATUS_NO_KERB_KEY                         ((LW_NTSTATUS)0xC0000322)
#define LW_STATUS_HOST_DOWN                           ((LW_NTSTATUS)0xC0000350)
#define LW_STATUS_UNSUPPORTED_PREAUTH                 ((LW_NTSTATUS)0xC0000351)
#define LW_STATUS_EFS_ALG_BLOB_TOO_BIG                ((LW_NTSTATUS)0xC0000352)
#define LW_STATUS_PORT_NOT_SET                        ((LW_NTSTATUS)0xC0000353)
#define LW_STATUS_DEBUGGER_INACTIVE                   ((LW_NTSTATUS)0xC0000354)
#define LW_STATUS_DS_VERSION_CHECK_FAILURE            ((LW_NTSTATUS)0xC0000355)
#define LW_STATUS_AUDITING_DISABLED                   ((LW_NTSTATUS)0xC0000356)
#define LW_STATUS_PRENT4_MACHINE_ACCOUNT              ((LW_NTSTATUS)0xC0000357)
#define LW_STATUS_DS_AG_CANT_HAVE_UNIVERSAL_MEMBER    ((LW_NTSTATUS)0xC0000358)
#define LW_STATUS_INVALID_IMAGE_WIN_32                ((LW_NTSTATUS)0xC0000359)
#define LW_STATUS_INVALID_IMAGE_WIN_64                ((LW_NTSTATUS)0xC000035A)
#define LW_STATUS_BAD_BINDINGS                        ((LW_NTSTATUS)0xC000035B)
#define LW_STATUS_NETWORK_SESSION_EXPIRED             ((LW_NTSTATUS)0xC000035C)
#define LW_STATUS_APPHELP_BLOCK                       ((LW_NTSTATUS)0xC000035D)
#define LW_STATUS_ALL_SIDS_FILTERED                   ((LW_NTSTATUS)0xC000035E)
#define LW_STATUS_NOT_SAFE_MODE_DRIVER                ((LW_NTSTATUS)0xC000035F)
#define LW_STATUS_ACCESS_DISABLED_BY_POLICY_DEFAULT   ((LW_NTSTATUS)0xC0000361)
#define LW_STATUS_ACCESS_DISABLED_BY_POLICY_PATH      ((LW_NTSTATUS)0xC0000362)
#define LW_STATUS_ACCESS_DISABLED_BY_POLICY_PUBLISHER ((LW_NTSTATUS)0xC0000363)
#define LW_STATUS_ACCESS_DISABLED_BY_POLICY_OTHER     ((LW_NTSTATUS)0xC0000364)
#define LW_STATUS_FAILED_DRIVER_ENTRY                 ((LW_NTSTATUS)0xC0000365)
#define LW_STATUS_DEVICE_ENUMERATION_ERROR            ((LW_NTSTATUS)0xC0000366)
#define LW_STATUS_MOUNT_POINT_NOT_RESOLVED            ((LW_NTSTATUS)0xC0000368)
#define LW_STATUS_INVALID_DEVICE_OBJECT_PARAMETER     ((LW_NTSTATUS)0xC0000369)
#define LW_STATUS_MCA_OCCURED                         ((LW_NTSTATUS)0xC000036A)
#define LW_STATUS_DRIVER_BLOCKED_CRITICAL             ((LW_NTSTATUS)0xC000036B)
#define LW_STATUS_DRIVER_BLOCKED                      ((LW_NTSTATUS)0xC000036C)
#define LW_STATUS_DRIVER_DATABASE_ERROR               ((LW_NTSTATUS)0xC000036D)
#define LW_STATUS_SYSTEM_HIVE_TOO_LARGE               ((LW_NTSTATUS)0xC000036E)
#define LW_STATUS_INVALID_IMPORT_OF_NON_DLL           ((LW_NTSTATUS)0xC000036F)
#define LW_STATUS_NO_SECRETS                          ((LW_NTSTATUS)0xC0000371)
#define LW_STATUS_ACCESS_DISABLED_NO_SAFER_UI_BY_POLICY ((LW_NTSTATUS)0xC0000372)
#define LW_STATUS_FAILED_STACK_SWITCH                 ((LW_NTSTATUS)0xC0000373)
#define LW_STATUS_HEAP_CORRUPTION                     ((LW_NTSTATUS)0xC0000374)
#define LW_STATUS_SMARTCARD_WRONG_PIN                 ((LW_NTSTATUS)0xC0000380)
#define LW_STATUS_SMARTCARD_CARD_BLOCKED              ((LW_NTSTATUS)0xC0000381)
#define LW_STATUS_SMARTCARD_CARD_NOT_AUTHENTICATED    ((LW_NTSTATUS)0xC0000382)
#define LW_STATUS_SMARTCARD_NO_CARD                   ((LW_NTSTATUS)0xC0000383)
#define LW_STATUS_SMARTCARD_NO_KEY_CONTAINER          ((LW_NTSTATUS)0xC0000384)
#define LW_STATUS_SMARTCARD_NO_CERTIFICATE            ((LW_NTSTATUS)0xC0000385)
#define LW_STATUS_SMARTCARD_NO_KEYSET                 ((LW_NTSTATUS)0xC0000386)
#define LW_STATUS_SMARTCARD_IO_ERROR                  ((LW_NTSTATUS)0xC0000387)
#define LW_STATUS_DOWNGRADE_DETECTED                  ((LW_NTSTATUS)0xC0000388)
#define LW_STATUS_SMARTCARD_CERT_REVOKED              ((LW_NTSTATUS)0xC0000389)
#define LW_STATUS_ISSUING_CA_UNTRUSTED                ((LW_NTSTATUS)0xC000038A)
#define LW_STATUS_REVOCATION_OFFLINE_C                ((LW_NTSTATUS)0xC000038B)
#define LW_STATUS_PKINIT_CLIENT_FAILURE               ((LW_NTSTATUS)0xC000038C)
#define LW_STATUS_SMARTCARD_CERT_EXPIRED              ((LW_NTSTATUS)0xC000038D)
#define LW_STATUS_DRIVER_FAILED_PRIOR_UNLOAD          ((LW_NTSTATUS)0xC000038E)
#define LW_STATUS_SMARTCARD_SILENT_CONTEXT            ((LW_NTSTATUS)0xC000038F)
#define LW_STATUS_PER_USER_TRUST_QUOTA_EXCEEDED       ((LW_NTSTATUS)0xC0000401)
#define LW_STATUS_ALL_USER_TRUST_QUOTA_EXCEEDED       ((LW_NTSTATUS)0xC0000402)
#define LW_STATUS_USER_DELETE_TRUST_QUOTA_EXCEEDED    ((LW_NTSTATUS)0xC0000403)
#define LW_STATUS_DS_NAME_NOT_UNIQUE                  ((LW_NTSTATUS)0xC0000404)
#define LW_STATUS_DS_DUPLICATE_ID_FOUND               ((LW_NTSTATUS)0xC0000405)
#define LW_STATUS_DS_GROUP_CONVERSION_ERROR           ((LW_NTSTATUS)0xC0000406)
#define LW_STATUS_VOLSNAP_PREPARE_HIBERNATE           ((LW_NTSTATUS)0xC0000407)
#define LW_STATUS_USER2USER_REQUIRED                  ((LW_NTSTATUS)0xC0000408)
#define LW_STATUS_STACK_BUFFER_OVERRUN                ((LW_NTSTATUS)0xC0000409)
#define LW_STATUS_NO_S4U_PROT_SUPPORT                 ((LW_NTSTATUS)0xC000040A)
#define LW_STATUS_CROSSREALM_DELEGATION_FAILURE       ((LW_NTSTATUS)0xC000040B)
#define LW_STATUS_REVOCATION_OFFLINE_KDC              ((LW_NTSTATUS)0xC000040C)
#define LW_STATUS_ISSUING_CA_UNTRUSTED_KDC            ((LW_NTSTATUS)0xC000040D)
#define LW_STATUS_KDC_CERT_EXPIRED                    ((LW_NTSTATUS)0xC000040E)
#define LW_STATUS_KDC_CERT_REVOKED                    ((LW_NTSTATUS)0xC000040F)
#define LW_STATUS_PARAMETER_QUOTA_EXCEEDED            ((LW_NTSTATUS)0xC0000410)
#define LW_STATUS_HIBERNATION_FAILURE                 ((LW_NTSTATUS)0xC0000411)
#define LW_STATUS_DELAY_LOAD_FAILED                   ((LW_NTSTATUS)0xC0000412)
#define LW_STATUS_AUTHENTICATION_FIREWALL_FAILED      ((LW_NTSTATUS)0xC0000413)
#define LW_STATUS_VDM_DISALLOWED                      ((LW_NTSTATUS)0xC0000414)
#define LW_STATUS_HUNG_DISPLAY_DRIVER_THREAD          ((LW_NTSTATUS)0xC0000415)
#define LW_STATUS_INSUFFICIENT_RESOURCE_FOR_SPECIFIED_SHARED_SECTION_SIZE ((LW_NTSTATUS)0xC0000416)
#define LW_STATUS_INVALID_CRUNTIME_PARAMETER          ((LW_NTSTATUS)0xC0000417)
#define LW_STATUS_NTLM_BLOCKED                        ((LW_NTSTATUS)0xC0000418)
#define LW_STATUS_ASSERTION_FAILURE                   ((LW_NTSTATUS)0xC0000420)
#define LW_STATUS_VERIFIER_STOP                       ((LW_NTSTATUS)0xC0000421)
#define LW_STATUS_CALLBACK_POP_STACK                  ((LW_NTSTATUS)0xC0000423)
#define LW_STATUS_INCOMPATIBLE_DRIVER_BLOCKED         ((LW_NTSTATUS)0xC0000424)
#define LW_STATUS_HIVE_UNLOADED                       ((LW_NTSTATUS)0xC0000425)
#define LW_STATUS_COMPRESSION_DISABLED                ((LW_NTSTATUS)0xC0000426)
#define LW_STATUS_FILE_SYSTEM_LIMITATION              ((LW_NTSTATUS)0xC0000427)
#define LW_STATUS_INVALID_IMAGE_HASH                  ((LW_NTSTATUS)0xC0000428)
#define LW_STATUS_NOT_CAPABLE                         ((LW_NTSTATUS)0xC0000429)
#define LW_STATUS_REQUEST_OUT_OF_SEQUENCE             ((LW_NTSTATUS)0xC000042A)
#define LW_STATUS_IMPLEMENTATION_LIMIT                ((LW_NTSTATUS)0xC000042B)
#define LW_STATUS_ELEVATION_REQUIRED                  ((LW_NTSTATUS)0xC000042C)
#define LW_STATUS_BEYOND_VDL                          ((LW_NTSTATUS)0xC0000432)
#define LW_STATUS_ENCOUNTERED_WRITE_IN_PROGRESS       ((LW_NTSTATUS)0xC0000433)
#define LW_STATUS_PTE_CHANGED                         ((LW_NTSTATUS)0xC0000434)
#define LW_STATUS_PURGE_FAILED                        ((LW_NTSTATUS)0xC0000435)
#define LW_STATUS_CRED_REQUIRES_CONFIRMATION          ((LW_NTSTATUS)0xC0000440)
#define LW_STATUS_CS_ENCRYPTION_INVALID_SERVER_RESPONSE ((LW_NTSTATUS)0xC0000441)
#define LW_STATUS_CS_ENCRYPTION_UNSUPPORTED_SERVER    ((LW_NTSTATUS)0xC0000442)
#define LW_STATUS_CS_ENCRYPTION_EXISTING_ENCRYPTED_FILE ((LW_NTSTATUS)0xC0000443)
#define LW_STATUS_CS_ENCRYPTION_NEW_ENCRYPTED_FILE    ((LW_NTSTATUS)0xC0000444)
#define LW_STATUS_CS_ENCRYPTION_FILE_NOT_CSE          ((LW_NTSTATUS)0xC0000445)
#define LW_STATUS_INVALID_LABEL                       ((LW_NTSTATUS)0xC0000446)
#define LW_STATUS_DRIVER_PROCESS_TERMINATED           ((LW_NTSTATUS)0xC0000450)
#define LW_STATUS_AMBIGUOUS_SYSTEM_DEVICE             ((LW_NTSTATUS)0xC0000451)
#define LW_STATUS_SYSTEM_DEVICE_NOT_FOUND             ((LW_NTSTATUS)0xC0000452)
#define LW_STATUS_RESTART_BOOT_APPLICATION            ((LW_NTSTATUS)0xC0000453)
#define LW_STATUS_INVALID_TASK_NAME                   ((LW_NTSTATUS)0xC0000500)
#define LW_STATUS_INVALID_TASK_INDEX                  ((LW_NTSTATUS)0xC0000501)
#define LW_STATUS_THREAD_ALREADY_IN_TASK              ((LW_NTSTATUS)0xC0000502)
#define LW_STATUS_CALLBACK_BYPASS                     ((LW_NTSTATUS)0xC0000503)
#define LW_STATUS_PORT_CLOSED                         ((LW_NTSTATUS)0xC0000700)
#define LW_STATUS_MESSAGE_LOST                        ((LW_NTSTATUS)0xC0000701)
#define LW_STATUS_INVALID_MESSAGE                     ((LW_NTSTATUS)0xC0000702)
#define LW_STATUS_REQUEST_CANCELED                    ((LW_NTSTATUS)0xC0000703)
#define LW_STATUS_RECURSIVE_DISPATCH                  ((LW_NTSTATUS)0xC0000704)
#define LW_STATUS_LPC_RECEIVE_BUFFER_EXPECTED         ((LW_NTSTATUS)0xC0000705)
#define LW_STATUS_LPC_INVALID_CONNECTION_USAGE        ((LW_NTSTATUS)0xC0000706)
#define LW_STATUS_LPC_REQUESTS_NOT_ALLOWED            ((LW_NTSTATUS)0xC0000707)
#define LW_STATUS_RESOURCE_IN_USE                     ((LW_NTSTATUS)0xC0000708)
#define LW_STATUS_HARDWARE_MEMORY_ERROR               ((LW_NTSTATUS)0xC0000709)
#define LW_STATUS_THREADPOOL_HANDLE_EXCEPTION         ((LW_NTSTATUS)0xC000070A)
#define LW_STATUS_THREADPOOL_SET_EVENT_ON_COMPLETION_FAILED ((LW_NTSTATUS)0xC000070B)
#define LW_STATUS_THREADPOOL_RELEASE_SEMAPHORE_ON_COMPLETION_FAILED ((LW_NTSTATUS)0xC000070C)
#define LW_STATUS_THREADPOOL_RELEASE_MUTEX_ON_COMPLETION_FAILED ((LW_NTSTATUS)0xC000070D)
#define LW_STATUS_THREADPOOL_FREE_LIBRARY_ON_COMPLETION_FAILED ((LW_NTSTATUS)0xC000070E)
#define LW_STATUS_THREADPOOL_RELEASED_DURING_OPERATION ((LW_NTSTATUS)0xC000070F)
#define LW_STATUS_CALLBACK_RETURNED_WHILE_IMPERSONATING ((LW_NTSTATUS)0xC0000710)
#define LW_STATUS_APC_RETURNED_WHILE_IMPERSONATING    ((LW_NTSTATUS)0xC0000711)
#define LW_STATUS_PROCESS_IS_PROTECTED                ((LW_NTSTATUS)0xC0000712)
#define LW_STATUS_MCA_EXCEPTION                       ((LW_NTSTATUS)0xC0000713)
#define LW_STATUS_CERTIFICATE_MAPPING_NOT_UNIQUE      ((LW_NTSTATUS)0xC0000714)
#define LW_STATUS_SYMLINK_CLASS_DISABLED              ((LW_NTSTATUS)0xC0000715)
#define LW_STATUS_INVALID_IDN_NORMALIZATION           ((LW_NTSTATUS)0xC0000716)
#define LW_STATUS_NO_UNICODE_TRANSLATION              ((LW_NTSTATUS)0xC0000717)
#define LW_STATUS_ALREADY_REGISTERED                  ((LW_NTSTATUS)0xC0000718)
#define LW_STATUS_CONTEXT_MISMATCH                    ((LW_NTSTATUS)0xC0000719)
#define LW_STATUS_PORT_ALREADY_HAS_COMPLETION_LIST    ((LW_NTSTATUS)0xC000071A)
#define LW_STATUS_CALLBACK_RETURNED_THREAD_PRIORITY   ((LW_NTSTATUS)0xC000071B)
#define LW_STATUS_INVALID_THREAD                      ((LW_NTSTATUS)0xC000071C)
#define LW_STATUS_CALLBACK_RETURNED_TRANSACTION       ((LW_NTSTATUS)0xC000071D)
#define LW_STATUS_CALLBACK_RETURNED_LDR_LOCK          ((LW_NTSTATUS)0xC000071E)
#define LW_STATUS_CALLBACK_RETURNED_LANG              ((LW_NTSTATUS)0xC000071F)
#define LW_STATUS_CALLBACK_RETURNED_PRI_BACK          ((LW_NTSTATUS)0xC0000720)
#define LW_STATUS_CALLBACK_RETURNED_THREAD_AFFINITY   ((LW_NTSTATUS)0xC0000721)
#define LW_DBG_NO_STATE_CHANGE                        ((LW_NTSTATUS)0xC0010001)
#define LW_DBG_APP_NOT_IDLE                           ((LW_NTSTATUS)0xC0010002)
#define LW_RPC_NT_INVALID_STRING_BINDING              ((LW_NTSTATUS)0xC0020001)
#define LW_RPC_NT_WRONG_KIND_OF_BINDING               ((LW_NTSTATUS)0xC0020002)
#define LW_RPC_NT_INVALID_BINDING                     ((LW_NTSTATUS)0xC0020003)
#define LW_RPC_NT_PROTSEQ_NOT_SUPPORTED               ((LW_NTSTATUS)0xC0020004)
#define LW_RPC_NT_INVALID_RPC_PROTSEQ                 ((LW_NTSTATUS)0xC0020005)
#define LW_RPC_NT_INVALID_STRING_UUID                 ((LW_NTSTATUS)0xC0020006)
#define LW_RPC_NT_INVALID_ENDPOINT_FORMAT             ((LW_NTSTATUS)0xC0020007)
#define LW_RPC_NT_INVALID_NET_ADDR                    ((LW_NTSTATUS)0xC0020008)
#define LW_RPC_NT_NO_ENDPOINT_FOUND                   ((LW_NTSTATUS)0xC0020009)
#define LW_RPC_NT_INVALID_TIMEOUT                     ((LW_NTSTATUS)0xC002000A)
#define LW_RPC_NT_OBJECT_NOT_FOUND                    ((LW_NTSTATUS)0xC002000B)
#define LW_RPC_NT_ALREADY_REGISTERED                  ((LW_NTSTATUS)0xC002000C)
#define LW_RPC_NT_TYPE_ALREADY_REGISTERED             ((LW_NTSTATUS)0xC002000D)
#define LW_RPC_NT_ALREADY_LISTENING                   ((LW_NTSTATUS)0xC002000E)
#define LW_RPC_NT_NO_PROTSEQS_REGISTERED              ((LW_NTSTATUS)0xC002000F)
#define LW_RPC_NT_NOT_LISTENING                       ((LW_NTSTATUS)0xC0020010)
#define LW_RPC_NT_UNKNOWN_MGR_TYPE                    ((LW_NTSTATUS)0xC0020011)
#define LW_RPC_NT_UNKNOWN_IF                          ((LW_NTSTATUS)0xC0020012)
#define LW_RPC_NT_NO_BINDINGS                         ((LW_NTSTATUS)0xC0020013)
#define LW_RPC_NT_NO_PROTSEQS                         ((LW_NTSTATUS)0xC0020014)
#define LW_RPC_NT_CANT_CREATE_ENDPOINT                ((LW_NTSTATUS)0xC0020015)
#define LW_RPC_NT_OUT_OF_RESOURCES                    ((LW_NTSTATUS)0xC0020016)
#define LW_RPC_NT_SERVER_UNAVAILABLE                  ((LW_NTSTATUS)0xC0020017)
#define LW_RPC_NT_SERVER_TOO_BUSY                     ((LW_NTSTATUS)0xC0020018)
#define LW_RPC_NT_INVALID_NETWORK_OPTIONS             ((LW_NTSTATUS)0xC0020019)
#define LW_RPC_NT_NO_CALL_ACTIVE                      ((LW_NTSTATUS)0xC002001A)
#define LW_RPC_NT_CALL_FAILED                         ((LW_NTSTATUS)0xC002001B)
#define LW_RPC_NT_CALL_FAILED_DNE                     ((LW_NTSTATUS)0xC002001C)
#define LW_RPC_NT_PROTOCOL_ERROR                      ((LW_NTSTATUS)0xC002001D)
#define LW_RPC_NT_UNSUPPORTED_TRANS_SYN               ((LW_NTSTATUS)0xC002001F)
#define LW_RPC_NT_UNSUPPORTED_TYPE                    ((LW_NTSTATUS)0xC0020021)
#define LW_RPC_NT_INVALID_TAG                         ((LW_NTSTATUS)0xC0020022)
#define LW_RPC_NT_INVALID_BOUND                       ((LW_NTSTATUS)0xC0020023)
#define LW_RPC_NT_NO_ENTRY_NAME                       ((LW_NTSTATUS)0xC0020024)
#define LW_RPC_NT_INVALID_NAME_SYNTAX                 ((LW_NTSTATUS)0xC0020025)
#define LW_RPC_NT_UNSUPPORTED_NAME_SYNTAX             ((LW_NTSTATUS)0xC0020026)
#define LW_RPC_NT_UUID_NO_ADDRESS                     ((LW_NTSTATUS)0xC0020028)
#define LW_RPC_NT_DUPLICATE_ENDPOINT                  ((LW_NTSTATUS)0xC0020029)
#define LW_RPC_NT_UNKNOWN_AUTHN_TYPE                  ((LW_NTSTATUS)0xC002002A)
#define LW_RPC_NT_MAX_CALLS_TOO_SMALL                 ((LW_NTSTATUS)0xC002002B)
#define LW_RPC_NT_STRING_TOO_LONG                     ((LW_NTSTATUS)0xC002002C)
#define LW_RPC_NT_PROTSEQ_NOT_FOUND                   ((LW_NTSTATUS)0xC002002D)
#define LW_RPC_NT_PROCNUM_OUT_OF_RANGE                ((LW_NTSTATUS)0xC002002E)
#define LW_RPC_NT_BINDING_HAS_NO_AUTH                 ((LW_NTSTATUS)0xC002002F)
#define LW_RPC_NT_UNKNOWN_AUTHN_SERVICE               ((LW_NTSTATUS)0xC0020030)
#define LW_RPC_NT_UNKNOWN_AUTHN_LEVEL                 ((LW_NTSTATUS)0xC0020031)
#define LW_RPC_NT_INVALID_AUTH_IDENTITY               ((LW_NTSTATUS)0xC0020032)
#define LW_RPC_NT_UNKNOWN_AUTHZ_SERVICE               ((LW_NTSTATUS)0xC0020033)
#define LW_EPT_NT_INVALID_ENTRY                       ((LW_NTSTATUS)0xC0020034)
#define LW_EPT_NT_CANT_PERFORM_OP                     ((LW_NTSTATUS)0xC0020035)
#define LW_EPT_NT_NOT_REGISTERED                      ((LW_NTSTATUS)0xC0020036)
#define LW_RPC_NT_NOTHING_TO_EXPORT                   ((LW_NTSTATUS)0xC0020037)
#define LW_RPC_NT_INCOMPLETE_NAME                     ((LW_NTSTATUS)0xC0020038)
#define LW_RPC_NT_INVALID_VERS_OPTION                 ((LW_NTSTATUS)0xC0020039)
#define LW_RPC_NT_NO_MORE_MEMBERS                     ((LW_NTSTATUS)0xC002003A)
#define LW_RPC_NT_NOT_ALL_OBJS_UNEXPORTED             ((LW_NTSTATUS)0xC002003B)
#define LW_RPC_NT_INTERFACE_NOT_FOUND                 ((LW_NTSTATUS)0xC002003C)
#define LW_RPC_NT_ENTRY_ALREADY_EXISTS                ((LW_NTSTATUS)0xC002003D)
#define LW_RPC_NT_ENTRY_NOT_FOUND                     ((LW_NTSTATUS)0xC002003E)
#define LW_RPC_NT_NAME_SERVICE_UNAVAILABLE            ((LW_NTSTATUS)0xC002003F)
#define LW_RPC_NT_INVALID_NAF_ID                      ((LW_NTSTATUS)0xC0020040)
#define LW_RPC_NT_CANNOT_SUPPORT                      ((LW_NTSTATUS)0xC0020041)
#define LW_RPC_NT_NO_CONTEXT_AVAILABLE                ((LW_NTSTATUS)0xC0020042)
#define LW_RPC_NT_INTERNAL_ERROR                      ((LW_NTSTATUS)0xC0020043)
#define LW_RPC_NT_ZERO_DIVIDE                         ((LW_NTSTATUS)0xC0020044)
#define LW_RPC_NT_ADDRESS_ERROR                       ((LW_NTSTATUS)0xC0020045)
#define LW_RPC_NT_FP_DIV_ZERO                         ((LW_NTSTATUS)0xC0020046)
#define LW_RPC_NT_FP_UNDERFLOW                        ((LW_NTSTATUS)0xC0020047)
#define LW_RPC_NT_FP_OVERFLOW                         ((LW_NTSTATUS)0xC0020048)
#define LW_RPC_NT_CALL_IN_PROGRESS                    ((LW_NTSTATUS)0xC0020049)
#define LW_RPC_NT_NO_MORE_BINDINGS                    ((LW_NTSTATUS)0xC002004A)
#define LW_RPC_NT_GROUP_MEMBER_NOT_FOUND              ((LW_NTSTATUS)0xC002004B)
#define LW_EPT_NT_CANT_CREATE                         ((LW_NTSTATUS)0xC002004C)
#define LW_RPC_NT_INVALID_OBJECT                      ((LW_NTSTATUS)0xC002004D)
#define LW_RPC_NT_NO_INTERFACES                       ((LW_NTSTATUS)0xC002004F)
#define LW_RPC_NT_CALL_CANCELLED                      ((LW_NTSTATUS)0xC0020050)
#define LW_RPC_NT_BINDING_INCOMPLETE                  ((LW_NTSTATUS)0xC0020051)
#define LW_RPC_NT_COMM_FAILURE                        ((LW_NTSTATUS)0xC0020052)
#define LW_RPC_NT_UNSUPPORTED_AUTHN_LEVEL             ((LW_NTSTATUS)0xC0020053)
#define LW_RPC_NT_NO_PRINC_NAME                       ((LW_NTSTATUS)0xC0020054)
#define LW_RPC_NT_NOT_RPC_ERROR                       ((LW_NTSTATUS)0xC0020055)
#define LW_RPC_NT_SEC_PKG_ERROR                       ((LW_NTSTATUS)0xC0020057)
#define LW_RPC_NT_NOT_CANCELLED                       ((LW_NTSTATUS)0xC0020058)
#define LW_RPC_NT_INVALID_ASYNC_HANDLE                ((LW_NTSTATUS)0xC0020062)
#define LW_RPC_NT_INVALID_ASYNC_CALL                  ((LW_NTSTATUS)0xC0020063)
#define LW_RPC_NT_PROXY_ACCESS_DENIED                 ((LW_NTSTATUS)0xC0020064)
#define LW_RPC_NT_NO_MORE_ENTRIES                     ((LW_NTSTATUS)0xC0030001)
#define LW_RPC_NT_SS_CHAR_TRANS_OPEN_FAIL             ((LW_NTSTATUS)0xC0030002)
#define LW_RPC_NT_SS_CHAR_TRANS_SHORT_FILE            ((LW_NTSTATUS)0xC0030003)
#define LW_RPC_NT_SS_IN_NULL_CONTEXT                  ((LW_NTSTATUS)0xC0030004)
#define LW_RPC_NT_SS_CONTEXT_MISMATCH                 ((LW_NTSTATUS)0xC0030005)
#define LW_RPC_NT_SS_CONTEXT_DAMAGED                  ((LW_NTSTATUS)0xC0030006)
#define LW_RPC_NT_SS_HANDLES_MISMATCH                 ((LW_NTSTATUS)0xC0030007)
#define LW_RPC_NT_SS_CANNOT_GET_CALL_HANDLE           ((LW_NTSTATUS)0xC0030008)
#define LW_RPC_NT_NULL_REF_POINTER                    ((LW_NTSTATUS)0xC0030009)
#define LW_RPC_NT_ENUM_VALUE_OUT_OF_RANGE             ((LW_NTSTATUS)0xC003000A)
#define LW_RPC_NT_BYTE_COUNT_TOO_SMALL                ((LW_NTSTATUS)0xC003000B)
#define LW_RPC_NT_BAD_STUB_DATA                       ((LW_NTSTATUS)0xC003000C)
#define LW_RPC_NT_INVALID_ES_ACTION                   ((LW_NTSTATUS)0xC0030059)
#define LW_RPC_NT_WRONG_ES_VERSION                    ((LW_NTSTATUS)0xC003005A)
#define LW_RPC_NT_WRONG_STUB_VERSION                  ((LW_NTSTATUS)0xC003005B)
#define LW_RPC_NT_INVALID_PIPE_OBJECT                 ((LW_NTSTATUS)0xC003005C)
#define LW_RPC_NT_INVALID_PIPE_OPERATION              ((LW_NTSTATUS)0xC003005D)
#define LW_RPC_NT_WRONG_PIPE_VERSION                  ((LW_NTSTATUS)0xC003005E)
#define LW_RPC_NT_PIPE_CLOSED                         ((LW_NTSTATUS)0xC003005F)
#define LW_RPC_NT_PIPE_DISCIPLINE_ERROR               ((LW_NTSTATUS)0xC0030060)
#define LW_RPC_NT_PIPE_EMPTY                          ((LW_NTSTATUS)0xC0030061)

#ifndef LW_STRICT_NAMESPACE

#define NT_SUCCESS(status) LW_NT_SUCCESS(status)
#define NT_SUCCESS_OR_NOT(status) LW_NT_SUCCESS_OR_NOT(status)

// NOTE: STATUS_<XXX> definitions generated using process-ntstatus.sh
#define STATUS_SUCCESS                                     LW_STATUS_SUCCESS
#define STATUS_WAIT_1                                      LW_STATUS_WAIT_1
#define STATUS_WAIT_2                                      LW_STATUS_WAIT_2
#define STATUS_WAIT_3                                      LW_STATUS_WAIT_3
#define STATUS_WAIT_63                                     LW_STATUS_WAIT_63
#define STATUS_ABANDONED_WAIT_0                            LW_STATUS_ABANDONED_WAIT_0
#define STATUS_ABANDONED_WAIT_63                           LW_STATUS_ABANDONED_WAIT_63
#define STATUS_USER_APC                                    LW_STATUS_USER_APC
#define STATUS_KERNEL_APC                                  LW_STATUS_KERNEL_APC
#define STATUS_ALERTED                                     LW_STATUS_ALERTED
#define STATUS_TIMEOUT                                     LW_STATUS_TIMEOUT
#define STATUS_PENDING                                     LW_STATUS_PENDING
#define STATUS_REPARSE                                     LW_STATUS_REPARSE
#define STATUS_MORE_ENTRIES                                LW_STATUS_MORE_ENTRIES
#define STATUS_NOT_ALL_ASSIGNED                            LW_STATUS_NOT_ALL_ASSIGNED
#define STATUS_SOME_NOT_MAPPED                             LW_STATUS_SOME_NOT_MAPPED
#define STATUS_OPLOCK_BREAK_IN_PROGRESS                    LW_STATUS_OPLOCK_BREAK_IN_PROGRESS
#define STATUS_VOLUME_MOUNTED                              LW_STATUS_VOLUME_MOUNTED
#define STATUS_RXACT_COMMITTED                             LW_STATUS_RXACT_COMMITTED
#define STATUS_NOTIFY_CLEANUP                              LW_STATUS_NOTIFY_CLEANUP
#define STATUS_NOTIFY_ENUM_DIR                             LW_STATUS_NOTIFY_ENUM_DIR
#define STATUS_NO_QUOTAS_FOR_ACCOUNT                       LW_STATUS_NO_QUOTAS_FOR_ACCOUNT
#define STATUS_PRIMARY_TRANSPORT_CONNECT_FAILED            LW_STATUS_PRIMARY_TRANSPORT_CONNECT_FAILED
#define STATUS_PAGE_FAULT_TRANSITION                       LW_STATUS_PAGE_FAULT_TRANSITION
#define STATUS_PAGE_FAULT_DEMAND_ZERO                      LW_STATUS_PAGE_FAULT_DEMAND_ZERO
#define STATUS_PAGE_FAULT_COPY_ON_WRITE                    LW_STATUS_PAGE_FAULT_COPY_ON_WRITE
#define STATUS_PAGE_FAULT_GUARD_PAGE                       LW_STATUS_PAGE_FAULT_GUARD_PAGE
#define STATUS_PAGE_FAULT_PAGING_FILE                      LW_STATUS_PAGE_FAULT_PAGING_FILE
#define STATUS_CACHE_PAGE_LOCKED                           LW_STATUS_CACHE_PAGE_LOCKED
#define STATUS_CRASH_DUMP                                  LW_STATUS_CRASH_DUMP
#define STATUS_BUFFER_ALL_ZEROS                            LW_STATUS_BUFFER_ALL_ZEROS
#define STATUS_REPARSE_OBJECT                              LW_STATUS_REPARSE_OBJECT
#define STATUS_RESOURCE_REQUIREMENTS_CHANGED               LW_STATUS_RESOURCE_REQUIREMENTS_CHANGED
#define STATUS_TRANSLATION_COMPLETE                        LW_STATUS_TRANSLATION_COMPLETE
#define STATUS_DS_MEMBERSHIP_EVALUATED_LOCALLY             LW_STATUS_DS_MEMBERSHIP_EVALUATED_LOCALLY
#define STATUS_NOTHING_TO_TERMINATE                        LW_STATUS_NOTHING_TO_TERMINATE
#define STATUS_PROCESS_NOT_IN_JOB                          LW_STATUS_PROCESS_NOT_IN_JOB
#define STATUS_PROCESS_IN_JOB                              LW_STATUS_PROCESS_IN_JOB
#define STATUS_VOLSNAP_HIBERNATE_READY                     LW_STATUS_VOLSNAP_HIBERNATE_READY
#define STATUS_FSFILTER_OP_COMPLETED_SUCCESSFULLY          LW_STATUS_FSFILTER_OP_COMPLETED_SUCCESSFULLY
#define STATUS_INTERRUPT_VECTOR_ALREADY_CONNECTED          LW_STATUS_INTERRUPT_VECTOR_ALREADY_CONNECTED
#define STATUS_INTERRUPT_STILL_CONNECTED                   LW_STATUS_INTERRUPT_STILL_CONNECTED
#define STATUS_PROCESS_CLONED                              LW_STATUS_PROCESS_CLONED
#define STATUS_FILE_LOCKED_WITH_ONLY_READERS               LW_STATUS_FILE_LOCKED_WITH_ONLY_READERS
#define STATUS_FILE_LOCKED_WITH_WRITERS                    LW_STATUS_FILE_LOCKED_WITH_WRITERS
#define STATUS_RESOURCEMANAGER_READ_ONLY                   LW_STATUS_RESOURCEMANAGER_READ_ONLY
#define STATUS_WAIT_FOR_OPLOCK                             LW_STATUS_WAIT_FOR_OPLOCK
#define STATUS_OBJECT_NAME_EXISTS                          LW_STATUS_OBJECT_NAME_EXISTS
#define STATUS_THREAD_WAS_SUSPENDED                        LW_STATUS_THREAD_WAS_SUSPENDED
#define STATUS_WORKING_SET_LIMIT_RANGE                     LW_STATUS_WORKING_SET_LIMIT_RANGE
#define STATUS_IMAGE_NOT_AT_BASE                           LW_STATUS_IMAGE_NOT_AT_BASE
#define STATUS_RXACT_STATE_CREATED                         LW_STATUS_RXACT_STATE_CREATED
#define STATUS_SEGMENT_NOTIFICATION                        LW_STATUS_SEGMENT_NOTIFICATION
#define STATUS_LOCAL_USER_SESSION_KEY                      LW_STATUS_LOCAL_USER_SESSION_KEY
#define STATUS_BAD_CURRENT_DIRECTORY                       LW_STATUS_BAD_CURRENT_DIRECTORY
#define STATUS_SERIAL_MORE_WRITES                          LW_STATUS_SERIAL_MORE_WRITES
#define STATUS_REGISTRY_RECOVERED                          LW_STATUS_REGISTRY_RECOVERED
#define STATUS_FT_READ_RECOVERY_FROM_BACKUP                LW_STATUS_FT_READ_RECOVERY_FROM_BACKUP
#define STATUS_FT_WRITE_RECOVERY                           LW_STATUS_FT_WRITE_RECOVERY
#define STATUS_SERIAL_COUNTER_TIMEOUT                      LW_STATUS_SERIAL_COUNTER_TIMEOUT
#define STATUS_NULL_LM_PASSWORD                            LW_STATUS_NULL_LM_PASSWORD
#define STATUS_IMAGE_MACHINE_TYPE_MISMATCH                 LW_STATUS_IMAGE_MACHINE_TYPE_MISMATCH
#define STATUS_RECEIVE_PARTIAL                             LW_STATUS_RECEIVE_PARTIAL
#define STATUS_RECEIVE_EXPEDITED                           LW_STATUS_RECEIVE_EXPEDITED
#define STATUS_RECEIVE_PARTIAL_EXPEDITED                   LW_STATUS_RECEIVE_PARTIAL_EXPEDITED
#define STATUS_EVENT_DONE                                  LW_STATUS_EVENT_DONE
#define STATUS_EVENT_PENDING                               LW_STATUS_EVENT_PENDING
#define STATUS_CHECKING_FILE_SYSTEM                        LW_STATUS_CHECKING_FILE_SYSTEM
#define STATUS_FATAL_APP_EXIT                              LW_STATUS_FATAL_APP_EXIT
#define STATUS_PREDEFINED_HANDLE                           LW_STATUS_PREDEFINED_HANDLE
#define STATUS_WAS_UNLOCKED                                LW_STATUS_WAS_UNLOCKED
#define STATUS_SERVICE_NOTIFICATION                        LW_STATUS_SERVICE_NOTIFICATION
#define STATUS_WAS_LOCKED                                  LW_STATUS_WAS_LOCKED
#define STATUS_LOG_HARD_ERROR                              LW_STATUS_LOG_HARD_ERROR
#define STATUS_ALREADY_WIN32                               LW_STATUS_ALREADY_WIN32
#define STATUS_WX86_UNSIMULATE                             LW_STATUS_WX86_UNSIMULATE
#define STATUS_WX86_CONTINUE                               LW_STATUS_WX86_CONTINUE
#define STATUS_WX86_SINGLE_STEP                            LW_STATUS_WX86_SINGLE_STEP
#define STATUS_WX86_BREAKPOINT                             LW_STATUS_WX86_BREAKPOINT
#define STATUS_WX86_EXCEPTION_CONTINUE                     LW_STATUS_WX86_EXCEPTION_CONTINUE
#define STATUS_WX86_EXCEPTION_LASTCHANCE                   LW_STATUS_WX86_EXCEPTION_LASTCHANCE
#define STATUS_WX86_EXCEPTION_CHAIN                        LW_STATUS_WX86_EXCEPTION_CHAIN
#define STATUS_IMAGE_MACHINE_TYPE_MISMATCH_EXE             LW_STATUS_IMAGE_MACHINE_TYPE_MISMATCH_EXE
#define STATUS_NO_YIELD_PERFORMED                          LW_STATUS_NO_YIELD_PERFORMED
#define STATUS_TIMER_RESUME_IGNORED                        LW_STATUS_TIMER_RESUME_IGNORED
#define STATUS_ARBITRATION_UNHANDLED                       LW_STATUS_ARBITRATION_UNHANDLED
#define STATUS_CARDBUS_NOT_SUPPORTED                       LW_STATUS_CARDBUS_NOT_SUPPORTED
#define STATUS_WX86_CREATEWX86TIB                          LW_STATUS_WX86_CREATEWX86TIB
#define STATUS_MP_PROCESSOR_MISMATCH                       LW_STATUS_MP_PROCESSOR_MISMATCH
#define STATUS_HIBERNATED                                  LW_STATUS_HIBERNATED
#define STATUS_RESUME_HIBERNATION                          LW_STATUS_RESUME_HIBERNATION
#define STATUS_FIRMWARE_UPDATED                            LW_STATUS_FIRMWARE_UPDATED
#define STATUS_DRIVERS_LEAKING_LOCKED_PAGES                LW_STATUS_DRIVERS_LEAKING_LOCKED_PAGES
#define STATUS_MESSAGE_RETRIEVED                           LW_STATUS_MESSAGE_RETRIEVED
#define STATUS_SYSTEM_POWERSTATE_TRANSITION                LW_STATUS_SYSTEM_POWERSTATE_TRANSITION
#define STATUS_ALPC_CHECK_COMPLETION_LIST                  LW_STATUS_ALPC_CHECK_COMPLETION_LIST
#define STATUS_SYSTEM_POWERSTATE_COMPLEX_TRANSITION        LW_STATUS_SYSTEM_POWERSTATE_COMPLEX_TRANSITION
#define STATUS_ACCESS_AUDIT_BY_POLICY                      LW_STATUS_ACCESS_AUDIT_BY_POLICY
#define STATUS_ABANDON_HIBERFILE                           LW_STATUS_ABANDON_HIBERFILE
#define STATUS_BIZRULES_NOT_ENABLED                        LW_STATUS_BIZRULES_NOT_ENABLED
#define STATUS_WAKE_SYSTEM                                 LW_STATUS_WAKE_SYSTEM
#define STATUS_DS_SHUTTING_DOWN                            LW_STATUS_DS_SHUTTING_DOWN
#define STATUS_GUARD_PAGE_VIOLATION                        LW_STATUS_GUARD_PAGE_VIOLATION
#define STATUS_DATATYPE_MISALIGNMENT                       LW_STATUS_DATATYPE_MISALIGNMENT
#define STATUS_BREAKPOINT                                  LW_STATUS_BREAKPOINT
#define STATUS_SINGLE_STEP                                 LW_STATUS_SINGLE_STEP
#define STATUS_BUFFER_OVERFLOW                             LW_STATUS_BUFFER_OVERFLOW
#define STATUS_NO_MORE_FILES                               LW_STATUS_NO_MORE_FILES
#define STATUS_WAKE_SYSTEM_DEBUGGER                        LW_STATUS_WAKE_SYSTEM_DEBUGGER
#define STATUS_HANDLES_CLOSED                              LW_STATUS_HANDLES_CLOSED
#define STATUS_NO_INHERITANCE                              LW_STATUS_NO_INHERITANCE
#define STATUS_GUID_SUBSTITUTION_MADE                      LW_STATUS_GUID_SUBSTITUTION_MADE
#define STATUS_PARTIAL_COPY                                LW_STATUS_PARTIAL_COPY
#define STATUS_DEVICE_PAPER_EMPTY                          LW_STATUS_DEVICE_PAPER_EMPTY
#define STATUS_DEVICE_POWERED_OFF                          LW_STATUS_DEVICE_POWERED_OFF
#define STATUS_DEVICE_OFF_LINE                             LW_STATUS_DEVICE_OFF_LINE
#define STATUS_DEVICE_BUSY                                 LW_STATUS_DEVICE_BUSY
#define STATUS_NO_MORE_EAS                                 LW_STATUS_NO_MORE_EAS
#define STATUS_INVALID_EA_NAME                             LW_STATUS_INVALID_EA_NAME
#define STATUS_EA_LIST_INCONSISTENT                        LW_STATUS_EA_LIST_INCONSISTENT
#define STATUS_INVALID_EA_FLAG                             LW_STATUS_INVALID_EA_FLAG
#define STATUS_VERIFY_REQUIRED                             LW_STATUS_VERIFY_REQUIRED
#define STATUS_EXTRANEOUS_INFORMATION                      LW_STATUS_EXTRANEOUS_INFORMATION
#define STATUS_RXACT_COMMIT_NECESSARY                      LW_STATUS_RXACT_COMMIT_NECESSARY
#define STATUS_NO_MORE_ENTRIES                             LW_STATUS_NO_MORE_ENTRIES
#define STATUS_FILEMARK_DETECTED                           LW_STATUS_FILEMARK_DETECTED
#define STATUS_MEDIA_CHANGED                               LW_STATUS_MEDIA_CHANGED
#define STATUS_BUS_RESET                                   LW_STATUS_BUS_RESET
#define STATUS_END_OF_MEDIA                                LW_STATUS_END_OF_MEDIA
#define STATUS_BEGINNING_OF_MEDIA                          LW_STATUS_BEGINNING_OF_MEDIA
#define STATUS_MEDIA_CHECK                                 LW_STATUS_MEDIA_CHECK
#define STATUS_SETMARK_DETECTED                            LW_STATUS_SETMARK_DETECTED
#define STATUS_NO_DATA_DETECTED                            LW_STATUS_NO_DATA_DETECTED
#define STATUS_REDIRECTOR_HAS_OPEN_HANDLES                 LW_STATUS_REDIRECTOR_HAS_OPEN_HANDLES
#define STATUS_SERVER_HAS_OPEN_HANDLES                     LW_STATUS_SERVER_HAS_OPEN_HANDLES
#define STATUS_ALREADY_DISCONNECTED                        LW_STATUS_ALREADY_DISCONNECTED
#define STATUS_LONGJUMP                                    LW_STATUS_LONGJUMP
#define STATUS_CLEANER_CARTRIDGE_INSTALLED                 LW_STATUS_CLEANER_CARTRIDGE_INSTALLED
#define STATUS_PLUGPLAY_QUERY_VETOED                       LW_STATUS_PLUGPLAY_QUERY_VETOED
#define STATUS_UNWIND_CONSOLIDATE                          LW_STATUS_UNWIND_CONSOLIDATE
#define STATUS_REGISTRY_HIVE_RECOVERED                     LW_STATUS_REGISTRY_HIVE_RECOVERED
#define STATUS_DLL_MIGHT_BE_INSECURE                       LW_STATUS_DLL_MIGHT_BE_INSECURE
#define STATUS_DLL_MIGHT_BE_INCOMPATIBLE                   LW_STATUS_DLL_MIGHT_BE_INCOMPATIBLE
#define STATUS_STOPPED_ON_SYMLINK                          LW_STATUS_STOPPED_ON_SYMLINK
#define STATUS_DEVICE_REQUIRES_CLEANING                    LW_STATUS_DEVICE_REQUIRES_CLEANING
#define STATUS_DEVICE_DOOR_OPEN                            LW_STATUS_DEVICE_DOOR_OPEN
#define STATUS_UNSUCCESSFUL                                LW_STATUS_UNSUCCESSFUL
#define STATUS_NOT_IMPLEMENTED                             LW_STATUS_NOT_IMPLEMENTED
#define STATUS_INVALID_INFO_CLASS                          LW_STATUS_INVALID_INFO_CLASS
#define STATUS_INFO_LENGTH_MISMATCH                        LW_STATUS_INFO_LENGTH_MISMATCH
#define STATUS_ACCESS_VIOLATION                            LW_STATUS_ACCESS_VIOLATION
#define STATUS_IN_PAGE_ERROR                               LW_STATUS_IN_PAGE_ERROR
#define STATUS_PAGEFILE_QUOTA                              LW_STATUS_PAGEFILE_QUOTA
#define STATUS_INVALID_HANDLE                              LW_STATUS_INVALID_HANDLE
#define STATUS_BAD_INITIAL_STACK                           LW_STATUS_BAD_INITIAL_STACK
#define STATUS_BAD_INITIAL_PC                              LW_STATUS_BAD_INITIAL_PC
#define STATUS_INVALID_CID                                 LW_STATUS_INVALID_CID
#define STATUS_TIMER_NOT_CANCELED                          LW_STATUS_TIMER_NOT_CANCELED
#define STATUS_INVALID_PARAMETER                           LW_STATUS_INVALID_PARAMETER
#define STATUS_NO_SUCH_DEVICE                              LW_STATUS_NO_SUCH_DEVICE
#define STATUS_NO_SUCH_FILE                                LW_STATUS_NO_SUCH_FILE
#define STATUS_INVALID_DEVICE_REQUEST                      LW_STATUS_INVALID_DEVICE_REQUEST
#define STATUS_END_OF_FILE                                 LW_STATUS_END_OF_FILE
#define STATUS_WRONG_VOLUME                                LW_STATUS_WRONG_VOLUME
#define STATUS_NO_MEDIA_IN_DEVICE                          LW_STATUS_NO_MEDIA_IN_DEVICE
#define STATUS_UNRECOGNIZED_MEDIA                          LW_STATUS_UNRECOGNIZED_MEDIA
#define STATUS_NONEXISTENT_SECTOR                          LW_STATUS_NONEXISTENT_SECTOR
#define STATUS_MORE_PROCESSING_REQUIRED                    LW_STATUS_MORE_PROCESSING_REQUIRED
#define STATUS_NO_MEMORY                                   LW_STATUS_NO_MEMORY
#define STATUS_CONFLICTING_ADDRESSES                       LW_STATUS_CONFLICTING_ADDRESSES
#define STATUS_NOT_MAPPED_VIEW                             LW_STATUS_NOT_MAPPED_VIEW
#define STATUS_UNABLE_TO_FREE_VM                           LW_STATUS_UNABLE_TO_FREE_VM
#define STATUS_UNABLE_TO_DELETE_SECTION                    LW_STATUS_UNABLE_TO_DELETE_SECTION
#define STATUS_INVALID_SYSTEM_SERVICE                      LW_STATUS_INVALID_SYSTEM_SERVICE
#define STATUS_ILLEGAL_INSTRUCTION                         LW_STATUS_ILLEGAL_INSTRUCTION
#define STATUS_INVALID_LOCK_SEQUENCE                       LW_STATUS_INVALID_LOCK_SEQUENCE
#define STATUS_INVALID_VIEW_SIZE                           LW_STATUS_INVALID_VIEW_SIZE
#define STATUS_INVALID_FILE_FOR_SECTION                    LW_STATUS_INVALID_FILE_FOR_SECTION
#define STATUS_ALREADY_COMMITTED                           LW_STATUS_ALREADY_COMMITTED
#define STATUS_ACCESS_DENIED                               LW_STATUS_ACCESS_DENIED
#define STATUS_BUFFER_TOO_SMALL                            LW_STATUS_BUFFER_TOO_SMALL
#define STATUS_OBJECT_TYPE_MISMATCH                        LW_STATUS_OBJECT_TYPE_MISMATCH
#define STATUS_NONCONTINUABLE_EXCEPTION                    LW_STATUS_NONCONTINUABLE_EXCEPTION
#define STATUS_INVALID_DISPOSITION                         LW_STATUS_INVALID_DISPOSITION
#define STATUS_UNWIND                                      LW_STATUS_UNWIND
#define STATUS_BAD_STACK                                   LW_STATUS_BAD_STACK
#define STATUS_INVALID_UNWIND_TARGET                       LW_STATUS_INVALID_UNWIND_TARGET
#define STATUS_NOT_LOCKED                                  LW_STATUS_NOT_LOCKED
#define STATUS_PARITY_ERROR                                LW_STATUS_PARITY_ERROR
#define STATUS_UNABLE_TO_DECOMMIT_VM                       LW_STATUS_UNABLE_TO_DECOMMIT_VM
#define STATUS_NOT_COMMITTED                               LW_STATUS_NOT_COMMITTED
#define STATUS_INVALID_PORT_ATTRIBUTES                     LW_STATUS_INVALID_PORT_ATTRIBUTES
#define STATUS_PORT_MESSAGE_TOO_LONG                       LW_STATUS_PORT_MESSAGE_TOO_LONG
#define STATUS_INVALID_PARAMETER_MIX                       LW_STATUS_INVALID_PARAMETER_MIX
#define STATUS_INVALID_QUOTA_LOWER                         LW_STATUS_INVALID_QUOTA_LOWER
#define STATUS_DISK_CORRUPT_ERROR                          LW_STATUS_DISK_CORRUPT_ERROR
#define STATUS_OBJECT_NAME_INVALID                         LW_STATUS_OBJECT_NAME_INVALID
#define STATUS_OBJECT_NAME_NOT_FOUND                       LW_STATUS_OBJECT_NAME_NOT_FOUND
#define STATUS_OBJECT_NAME_COLLISION                       LW_STATUS_OBJECT_NAME_COLLISION
#define STATUS_PORT_DISCONNECTED                           LW_STATUS_PORT_DISCONNECTED
#define STATUS_DEVICE_ALREADY_ATTACHED                     LW_STATUS_DEVICE_ALREADY_ATTACHED
#define STATUS_OBJECT_PATH_INVALID                         LW_STATUS_OBJECT_PATH_INVALID
#define STATUS_OBJECT_PATH_NOT_FOUND                       LW_STATUS_OBJECT_PATH_NOT_FOUND
#define STATUS_OBJECT_PATH_SYNTAX_BAD                      LW_STATUS_OBJECT_PATH_SYNTAX_BAD
#define STATUS_DATA_OVERRUN                                LW_STATUS_DATA_OVERRUN
#define STATUS_DATA_LATE_ERROR                             LW_STATUS_DATA_LATE_ERROR
#define STATUS_DATA_ERROR                                  LW_STATUS_DATA_ERROR
#define STATUS_CRC_ERROR                                   LW_STATUS_CRC_ERROR
#define STATUS_SECTION_TOO_BIG                             LW_STATUS_SECTION_TOO_BIG
#define STATUS_PORT_CONNECTION_REFUSED                     LW_STATUS_PORT_CONNECTION_REFUSED
#define STATUS_INVALID_PORT_HANDLE                         LW_STATUS_INVALID_PORT_HANDLE
#define STATUS_SHARING_VIOLATION                           LW_STATUS_SHARING_VIOLATION
#define STATUS_QUOTA_EXCEEDED                              LW_STATUS_QUOTA_EXCEEDED
#define STATUS_INVALID_PAGE_PROTECTION                     LW_STATUS_INVALID_PAGE_PROTECTION
#define STATUS_MUTANT_NOT_OWNED                            LW_STATUS_MUTANT_NOT_OWNED
#define STATUS_SEMAPHORE_LIMIT_EXCEEDED                    LW_STATUS_SEMAPHORE_LIMIT_EXCEEDED
#define STATUS_PORT_ALREADY_SET                            LW_STATUS_PORT_ALREADY_SET
#define STATUS_SECTION_NOT_IMAGE                           LW_STATUS_SECTION_NOT_IMAGE
#define STATUS_SUSPEND_COUNT_EXCEEDED                      LW_STATUS_SUSPEND_COUNT_EXCEEDED
#define STATUS_THREAD_IS_TERMINATING                       LW_STATUS_THREAD_IS_TERMINATING
#define STATUS_BAD_WORKING_SET_LIMIT                       LW_STATUS_BAD_WORKING_SET_LIMIT
#define STATUS_INCOMPATIBLE_FILE_MAP                       LW_STATUS_INCOMPATIBLE_FILE_MAP
#define STATUS_SECTION_PROTECTION                          LW_STATUS_SECTION_PROTECTION
#define STATUS_EAS_NOT_SUPPORTED                           LW_STATUS_EAS_NOT_SUPPORTED
#define STATUS_EA_TOO_LARGE                                LW_STATUS_EA_TOO_LARGE
#define STATUS_NONEXISTENT_EA_ENTRY                        LW_STATUS_NONEXISTENT_EA_ENTRY
#define STATUS_NO_EAS_ON_FILE                              LW_STATUS_NO_EAS_ON_FILE
#define STATUS_EA_CORRUPT_ERROR                            LW_STATUS_EA_CORRUPT_ERROR
#define STATUS_FILE_LOCK_CONFLICT                          LW_STATUS_FILE_LOCK_CONFLICT
#define STATUS_LOCK_NOT_GRANTED                            LW_STATUS_LOCK_NOT_GRANTED
#define STATUS_DELETE_PENDING                              LW_STATUS_DELETE_PENDING
#define STATUS_CTL_FILE_NOT_SUPPORTED                      LW_STATUS_CTL_FILE_NOT_SUPPORTED
#define STATUS_UNKNOWN_REVISION                            LW_STATUS_UNKNOWN_REVISION
#define STATUS_REVISION_MISMATCH                           LW_STATUS_REVISION_MISMATCH
#define STATUS_INVALID_OWNER                               LW_STATUS_INVALID_OWNER
#define STATUS_INVALID_PRIMARY_GROUP                       LW_STATUS_INVALID_PRIMARY_GROUP
#define STATUS_NO_IMPERSONATION_TOKEN                      LW_STATUS_NO_IMPERSONATION_TOKEN
#define STATUS_CANT_DISABLE_MANDATORY                      LW_STATUS_CANT_DISABLE_MANDATORY
#define STATUS_NO_LOGON_SERVERS                            LW_STATUS_NO_LOGON_SERVERS
#define STATUS_NO_SUCH_LOGON_SESSION                       LW_STATUS_NO_SUCH_LOGON_SESSION
#define STATUS_NO_SUCH_PRIVILEGE                           LW_STATUS_NO_SUCH_PRIVILEGE
#define STATUS_PRIVILEGE_NOT_HELD                          LW_STATUS_PRIVILEGE_NOT_HELD
#define STATUS_INVALID_ACCOUNT_NAME                        LW_STATUS_INVALID_ACCOUNT_NAME
#define STATUS_USER_EXISTS                                 LW_STATUS_USER_EXISTS
#define STATUS_NO_SUCH_USER                                LW_STATUS_NO_SUCH_USER
#define STATUS_GROUP_EXISTS                                LW_STATUS_GROUP_EXISTS
#define STATUS_NO_SUCH_GROUP                               LW_STATUS_NO_SUCH_GROUP
#define STATUS_MEMBER_IN_GROUP                             LW_STATUS_MEMBER_IN_GROUP
#define STATUS_MEMBER_NOT_IN_GROUP                         LW_STATUS_MEMBER_NOT_IN_GROUP
#define STATUS_LAST_ADMIN                                  LW_STATUS_LAST_ADMIN
#define STATUS_WRONG_PASSWORD                              LW_STATUS_WRONG_PASSWORD
#define STATUS_ILL_FORMED_PASSWORD                         LW_STATUS_ILL_FORMED_PASSWORD
#define STATUS_PASSWORD_RESTRICTION                        LW_STATUS_PASSWORD_RESTRICTION
#define STATUS_LOGON_FAILURE                               LW_STATUS_LOGON_FAILURE
#define STATUS_ACCOUNT_RESTRICTION                         LW_STATUS_ACCOUNT_RESTRICTION
#define STATUS_INVALID_LOGON_HOURS                         LW_STATUS_INVALID_LOGON_HOURS
#define STATUS_INVALID_WORKSTATION                         LW_STATUS_INVALID_WORKSTATION
#define STATUS_PASSWORD_EXPIRED                            LW_STATUS_PASSWORD_EXPIRED
#define STATUS_ACCOUNT_DISABLED                            LW_STATUS_ACCOUNT_DISABLED
#define STATUS_NONE_MAPPED                                 LW_STATUS_NONE_MAPPED
#define STATUS_TOO_MANY_LUIDS_REQUESTED                    LW_STATUS_TOO_MANY_LUIDS_REQUESTED
#define STATUS_LUIDS_EXHAUSTED                             LW_STATUS_LUIDS_EXHAUSTED
#define STATUS_INVALID_SUB_AUTHORITY                       LW_STATUS_INVALID_SUB_AUTHORITY
#define STATUS_INVALID_ACL                                 LW_STATUS_INVALID_ACL
#define STATUS_INVALID_SID                                 LW_STATUS_INVALID_SID
#define STATUS_INVALID_SECURITY_DESCR                      LW_STATUS_INVALID_SECURITY_DESCR
#define STATUS_PROCEDURE_NOT_FOUND                         LW_STATUS_PROCEDURE_NOT_FOUND
#define STATUS_INVALID_IMAGE_FORMAT                        LW_STATUS_INVALID_IMAGE_FORMAT
#define STATUS_NO_TOKEN                                    LW_STATUS_NO_TOKEN
#define STATUS_BAD_INHERITANCE_ACL                         LW_STATUS_BAD_INHERITANCE_ACL
#define STATUS_RANGE_NOT_LOCKED                            LW_STATUS_RANGE_NOT_LOCKED
#define STATUS_DISK_FULL                                   LW_STATUS_DISK_FULL
#define STATUS_SERVER_DISABLED                             LW_STATUS_SERVER_DISABLED
#define STATUS_SERVER_NOT_DISABLED                         LW_STATUS_SERVER_NOT_DISABLED
#define STATUS_TOO_MANY_GUIDS_REQUESTED                    LW_STATUS_TOO_MANY_GUIDS_REQUESTED
#define STATUS_GUIDS_EXHAUSTED                             LW_STATUS_GUIDS_EXHAUSTED
#define STATUS_INVALID_ID_AUTHORITY                        LW_STATUS_INVALID_ID_AUTHORITY
#define STATUS_AGENTS_EXHAUSTED                            LW_STATUS_AGENTS_EXHAUSTED
#define STATUS_INVALID_VOLUME_LABEL                        LW_STATUS_INVALID_VOLUME_LABEL
#define STATUS_SECTION_NOT_EXTENDED                        LW_STATUS_SECTION_NOT_EXTENDED
#define STATUS_NOT_MAPPED_DATA                             LW_STATUS_NOT_MAPPED_DATA
#define STATUS_RESOURCE_DATA_NOT_FOUND                     LW_STATUS_RESOURCE_DATA_NOT_FOUND
#define STATUS_RESOURCE_TYPE_NOT_FOUND                     LW_STATUS_RESOURCE_TYPE_NOT_FOUND
#define STATUS_RESOURCE_NAME_NOT_FOUND                     LW_STATUS_RESOURCE_NAME_NOT_FOUND
#define STATUS_ARRAY_BOUNDS_EXCEEDED                       LW_STATUS_ARRAY_BOUNDS_EXCEEDED
#define STATUS_FLOAT_DENORMAL_OPERAND                      LW_STATUS_FLOAT_DENORMAL_OPERAND
#define STATUS_FLOAT_DIVIDE_BY_ZERO                        LW_STATUS_FLOAT_DIVIDE_BY_ZERO
#define STATUS_FLOAT_INEXACT_RESULT                        LW_STATUS_FLOAT_INEXACT_RESULT
#define STATUS_FLOAT_INVALID_OPERATION                     LW_STATUS_FLOAT_INVALID_OPERATION
#define STATUS_FLOAT_OVERFLOW                              LW_STATUS_FLOAT_OVERFLOW
#define STATUS_FLOAT_STACK_CHECK                           LW_STATUS_FLOAT_STACK_CHECK
#define STATUS_FLOAT_UNDERFLOW                             LW_STATUS_FLOAT_UNDERFLOW
#define STATUS_INTEGER_DIVIDE_BY_ZERO                      LW_STATUS_INTEGER_DIVIDE_BY_ZERO
#define STATUS_INTEGER_OVERFLOW                            LW_STATUS_INTEGER_OVERFLOW
#define STATUS_PRIVILEGED_INSTRUCTION                      LW_STATUS_PRIVILEGED_INSTRUCTION
#define STATUS_TOO_MANY_PAGING_FILES                       LW_STATUS_TOO_MANY_PAGING_FILES
#define STATUS_FILE_INVALID                                LW_STATUS_FILE_INVALID
#define STATUS_ALLOTTED_SPACE_EXCEEDED                     LW_STATUS_ALLOTTED_SPACE_EXCEEDED
#define STATUS_INSUFFICIENT_RESOURCES                      LW_STATUS_INSUFFICIENT_RESOURCES
#define STATUS_DFS_EXIT_PATH_FOUND                         LW_STATUS_DFS_EXIT_PATH_FOUND
#define STATUS_DEVICE_DATA_ERROR                           LW_STATUS_DEVICE_DATA_ERROR
#define STATUS_DEVICE_NOT_CONNECTED                        LW_STATUS_DEVICE_NOT_CONNECTED
#define STATUS_DEVICE_POWER_FAILURE                        LW_STATUS_DEVICE_POWER_FAILURE
#define STATUS_FREE_VM_NOT_AT_BASE                         LW_STATUS_FREE_VM_NOT_AT_BASE
#define STATUS_MEMORY_NOT_ALLOCATED                        LW_STATUS_MEMORY_NOT_ALLOCATED
#define STATUS_WORKING_SET_QUOTA                           LW_STATUS_WORKING_SET_QUOTA
#define STATUS_MEDIA_WRITE_PROTECTED                       LW_STATUS_MEDIA_WRITE_PROTECTED
#define STATUS_DEVICE_NOT_READY                            LW_STATUS_DEVICE_NOT_READY
#define STATUS_INVALID_GROUP_ATTRIBUTES                    LW_STATUS_INVALID_GROUP_ATTRIBUTES
#define STATUS_BAD_IMPERSONATION_LEVEL                     LW_STATUS_BAD_IMPERSONATION_LEVEL
#define STATUS_CANT_OPEN_ANONYMOUS                         LW_STATUS_CANT_OPEN_ANONYMOUS
#define STATUS_BAD_VALIDATION_CLASS                        LW_STATUS_BAD_VALIDATION_CLASS
#define STATUS_BAD_TOKEN_TYPE                              LW_STATUS_BAD_TOKEN_TYPE
#define STATUS_BAD_MASTER_BOOT_RECORD                      LW_STATUS_BAD_MASTER_BOOT_RECORD
#define STATUS_INSTRUCTION_MISALIGNMENT                    LW_STATUS_INSTRUCTION_MISALIGNMENT
#define STATUS_INSTANCE_NOT_AVAILABLE                      LW_STATUS_INSTANCE_NOT_AVAILABLE
#define STATUS_PIPE_NOT_AVAILABLE                          LW_STATUS_PIPE_NOT_AVAILABLE
#define STATUS_INVALID_PIPE_STATE                          LW_STATUS_INVALID_PIPE_STATE
#define STATUS_PIPE_BUSY                                   LW_STATUS_PIPE_BUSY
#define STATUS_ILLEGAL_FUNCTION                            LW_STATUS_ILLEGAL_FUNCTION
#define STATUS_PIPE_DISCONNECTED                           LW_STATUS_PIPE_DISCONNECTED
#define STATUS_PIPE_CLOSING                                LW_STATUS_PIPE_CLOSING
#define STATUS_PIPE_CONNECTED                              LW_STATUS_PIPE_CONNECTED
#define STATUS_PIPE_LISTENING                              LW_STATUS_PIPE_LISTENING
#define STATUS_INVALID_READ_MODE                           LW_STATUS_INVALID_READ_MODE
#define STATUS_IO_TIMEOUT                                  LW_STATUS_IO_TIMEOUT
#define STATUS_FILE_FORCED_CLOSED                          LW_STATUS_FILE_FORCED_CLOSED
#define STATUS_PROFILING_NOT_STARTED                       LW_STATUS_PROFILING_NOT_STARTED
#define STATUS_PROFILING_NOT_STOPPED                       LW_STATUS_PROFILING_NOT_STOPPED
#define STATUS_COULD_NOT_INTERPRET                         LW_STATUS_COULD_NOT_INTERPRET
#define STATUS_FILE_IS_A_DIRECTORY                         LW_STATUS_FILE_IS_A_DIRECTORY
#define STATUS_NOT_SUPPORTED                               LW_STATUS_NOT_SUPPORTED
#define STATUS_REMOTE_NOT_LISTENING                        LW_STATUS_REMOTE_NOT_LISTENING
#define STATUS_DUPLICATE_NAME                              LW_STATUS_DUPLICATE_NAME
#define STATUS_BAD_NETWORK_PATH                            LW_STATUS_BAD_NETWORK_PATH
#define STATUS_NETWORK_BUSY                                LW_STATUS_NETWORK_BUSY
#define STATUS_DEVICE_DOES_NOT_EXIST                       LW_STATUS_DEVICE_DOES_NOT_EXIST
#define STATUS_TOO_MANY_COMMANDS                           LW_STATUS_TOO_MANY_COMMANDS
#define STATUS_ADAPTER_HARDWARE_ERROR                      LW_STATUS_ADAPTER_HARDWARE_ERROR
#define STATUS_INVALID_NETWORK_RESPONSE                    LW_STATUS_INVALID_NETWORK_RESPONSE
#define STATUS_UNEXPECTED_NETWORK_ERROR                    LW_STATUS_UNEXPECTED_NETWORK_ERROR
#define STATUS_BAD_REMOTE_ADAPTER                          LW_STATUS_BAD_REMOTE_ADAPTER
#define STATUS_PRINT_QUEUE_FULL                            LW_STATUS_PRINT_QUEUE_FULL
#define STATUS_NO_SPOOL_SPACE                              LW_STATUS_NO_SPOOL_SPACE
#define STATUS_PRINT_CANCELLED                             LW_STATUS_PRINT_CANCELLED
#define STATUS_NETWORK_NAME_DELETED                        LW_STATUS_NETWORK_NAME_DELETED
#define STATUS_NETWORK_ACCESS_DENIED                       LW_STATUS_NETWORK_ACCESS_DENIED
#define STATUS_BAD_DEVICE_TYPE                             LW_STATUS_BAD_DEVICE_TYPE
#define STATUS_BAD_NETWORK_NAME                            LW_STATUS_BAD_NETWORK_NAME
#define STATUS_TOO_MANY_NAMES                              LW_STATUS_TOO_MANY_NAMES
#define STATUS_TOO_MANY_SESSIONS                           LW_STATUS_TOO_MANY_SESSIONS
#define STATUS_SHARING_PAUSED                              LW_STATUS_SHARING_PAUSED
#define STATUS_REQUEST_NOT_ACCEPTED                        LW_STATUS_REQUEST_NOT_ACCEPTED
#define STATUS_REDIRECTOR_PAUSED                           LW_STATUS_REDIRECTOR_PAUSED
#define STATUS_NET_WRITE_FAULT                             LW_STATUS_NET_WRITE_FAULT
#define STATUS_PROFILING_AT_LIMIT                          LW_STATUS_PROFILING_AT_LIMIT
#define STATUS_NOT_SAME_DEVICE                             LW_STATUS_NOT_SAME_DEVICE
#define STATUS_FILE_RENAMED                                LW_STATUS_FILE_RENAMED
#define STATUS_VIRTUAL_CIRCUIT_CLOSED                      LW_STATUS_VIRTUAL_CIRCUIT_CLOSED
#define STATUS_NO_SECURITY_ON_OBJECT                       LW_STATUS_NO_SECURITY_ON_OBJECT
#define STATUS_CANT_WAIT                                   LW_STATUS_CANT_WAIT
#define STATUS_PIPE_EMPTY                                  LW_STATUS_PIPE_EMPTY
#define STATUS_CANT_ACCESS_DOMAIN_INFO                     LW_STATUS_CANT_ACCESS_DOMAIN_INFO
#define STATUS_CANT_TERMINATE_SELF                         LW_STATUS_CANT_TERMINATE_SELF
#define STATUS_INVALID_SERVER_STATE                        LW_STATUS_INVALID_SERVER_STATE
#define STATUS_INVALID_DOMAIN_STATE                        LW_STATUS_INVALID_DOMAIN_STATE
#define STATUS_INVALID_DOMAIN_ROLE                         LW_STATUS_INVALID_DOMAIN_ROLE
#define STATUS_NO_SUCH_DOMAIN                              LW_STATUS_NO_SUCH_DOMAIN
#define STATUS_DOMAIN_EXISTS                               LW_STATUS_DOMAIN_EXISTS
#define STATUS_DOMAIN_LIMIT_EXCEEDED                       LW_STATUS_DOMAIN_LIMIT_EXCEEDED
#define STATUS_OPLOCK_NOT_GRANTED                          LW_STATUS_OPLOCK_NOT_GRANTED
#define STATUS_INVALID_OPLOCK_PROTOCOL                     LW_STATUS_INVALID_OPLOCK_PROTOCOL
#define STATUS_INTERNAL_DB_CORRUPTION                      LW_STATUS_INTERNAL_DB_CORRUPTION
#define STATUS_INTERNAL_ERROR                              LW_STATUS_INTERNAL_ERROR
#define STATUS_GENERIC_NOT_MAPPED                          LW_STATUS_GENERIC_NOT_MAPPED
#define STATUS_BAD_DESCRIPTOR_FORMAT                       LW_STATUS_BAD_DESCRIPTOR_FORMAT
#define STATUS_INVALID_USER_BUFFER                         LW_STATUS_INVALID_USER_BUFFER
#define STATUS_UNEXPECTED_IO_ERROR                         LW_STATUS_UNEXPECTED_IO_ERROR
#define STATUS_UNEXPECTED_MM_CREATE_ERR                    LW_STATUS_UNEXPECTED_MM_CREATE_ERR
#define STATUS_UNEXPECTED_MM_MAP_ERROR                     LW_STATUS_UNEXPECTED_MM_MAP_ERROR
#define STATUS_UNEXPECTED_MM_EXTEND_ERR                    LW_STATUS_UNEXPECTED_MM_EXTEND_ERR
#define STATUS_NOT_LOGON_PROCESS                           LW_STATUS_NOT_LOGON_PROCESS
#define STATUS_LOGON_SESSION_EXISTS                        LW_STATUS_LOGON_SESSION_EXISTS
#define STATUS_INVALID_PARAMETER_1                         LW_STATUS_INVALID_PARAMETER_1
#define STATUS_INVALID_PARAMETER_2                         LW_STATUS_INVALID_PARAMETER_2
#define STATUS_INVALID_PARAMETER_3                         LW_STATUS_INVALID_PARAMETER_3
#define STATUS_INVALID_PARAMETER_4                         LW_STATUS_INVALID_PARAMETER_4
#define STATUS_INVALID_PARAMETER_5                         LW_STATUS_INVALID_PARAMETER_5
#define STATUS_INVALID_PARAMETER_6                         LW_STATUS_INVALID_PARAMETER_6
#define STATUS_INVALID_PARAMETER_7                         LW_STATUS_INVALID_PARAMETER_7
#define STATUS_INVALID_PARAMETER_8                         LW_STATUS_INVALID_PARAMETER_8
#define STATUS_INVALID_PARAMETER_9                         LW_STATUS_INVALID_PARAMETER_9
#define STATUS_INVALID_PARAMETER_10                        LW_STATUS_INVALID_PARAMETER_10
#define STATUS_INVALID_PARAMETER_11                        LW_STATUS_INVALID_PARAMETER_11
#define STATUS_INVALID_PARAMETER_12                        LW_STATUS_INVALID_PARAMETER_12
#define STATUS_REDIRECTOR_NOT_STARTED                      LW_STATUS_REDIRECTOR_NOT_STARTED
#define STATUS_REDIRECTOR_STARTED                          LW_STATUS_REDIRECTOR_STARTED
#define STATUS_STACK_OVERFLOW                              LW_STATUS_STACK_OVERFLOW
#define STATUS_NO_SUCH_PACKAGE                             LW_STATUS_NO_SUCH_PACKAGE
#define STATUS_BAD_FUNCTION_TABLE                          LW_STATUS_BAD_FUNCTION_TABLE
#define STATUS_VARIABLE_NOT_FOUND                          LW_STATUS_VARIABLE_NOT_FOUND
#define STATUS_DIRECTORY_NOT_EMPTY                         LW_STATUS_DIRECTORY_NOT_EMPTY
#define STATUS_FILE_CORRUPT_ERROR                          LW_STATUS_FILE_CORRUPT_ERROR
#define STATUS_NOT_A_DIRECTORY                             LW_STATUS_NOT_A_DIRECTORY
#define STATUS_BAD_LOGON_SESSION_STATE                     LW_STATUS_BAD_LOGON_SESSION_STATE
#define STATUS_LOGON_SESSION_COLLISION                     LW_STATUS_LOGON_SESSION_COLLISION
#define STATUS_NAME_TOO_LONG                               LW_STATUS_NAME_TOO_LONG
#define STATUS_FILES_OPEN                                  LW_STATUS_FILES_OPEN
#define STATUS_CONNECTION_IN_USE                           LW_STATUS_CONNECTION_IN_USE
#define STATUS_MESSAGE_NOT_FOUND                           LW_STATUS_MESSAGE_NOT_FOUND
#define STATUS_PROCESS_IS_TERMINATING                      LW_STATUS_PROCESS_IS_TERMINATING
#define STATUS_INVALID_LOGON_TYPE                          LW_STATUS_INVALID_LOGON_TYPE
#define STATUS_NO_GUID_TRANSLATION                         LW_STATUS_NO_GUID_TRANSLATION
#define STATUS_CANNOT_IMPERSONATE                          LW_STATUS_CANNOT_IMPERSONATE
#define STATUS_IMAGE_ALREADY_LOADED                        LW_STATUS_IMAGE_ALREADY_LOADED
#define STATUS_INVALID_LOCK_RANGE                          LW_STATUS_INVALID_LOCK_RANGE
#define STATUS_ABIOS_NOT_PRESENT                           LW_STATUS_ABIOS_NOT_PRESENT
#define STATUS_ABIOS_LID_NOT_EXIST                         LW_STATUS_ABIOS_LID_NOT_EXIST
#define STATUS_ABIOS_LID_ALREADY_OWNED                     LW_STATUS_ABIOS_LID_ALREADY_OWNED
#define STATUS_ABIOS_NOT_LID_OWNER                         LW_STATUS_ABIOS_NOT_LID_OWNER
#define STATUS_ABIOS_INVALID_COMMAND                       LW_STATUS_ABIOS_INVALID_COMMAND
#define STATUS_ABIOS_INVALID_LID                           LW_STATUS_ABIOS_INVALID_LID
#define STATUS_ABIOS_SELECTOR_NOT_AVAILABLE                LW_STATUS_ABIOS_SELECTOR_NOT_AVAILABLE
#define STATUS_ABIOS_INVALID_SELECTOR                      LW_STATUS_ABIOS_INVALID_SELECTOR
#define STATUS_NO_LDT                                      LW_STATUS_NO_LDT
#define STATUS_INVALID_LDT_SIZE                            LW_STATUS_INVALID_LDT_SIZE
#define STATUS_INVALID_LDT_OFFSET                          LW_STATUS_INVALID_LDT_OFFSET
#define STATUS_INVALID_LDT_DESCRIPTOR                      LW_STATUS_INVALID_LDT_DESCRIPTOR
#define STATUS_INVALID_IMAGE_NE_FORMAT                     LW_STATUS_INVALID_IMAGE_NE_FORMAT
#define STATUS_RXACT_INVALID_STATE                         LW_STATUS_RXACT_INVALID_STATE
#define STATUS_RXACT_COMMIT_FAILURE                        LW_STATUS_RXACT_COMMIT_FAILURE
#define STATUS_MAPPED_FILE_SIZE_ZERO                       LW_STATUS_MAPPED_FILE_SIZE_ZERO
#define STATUS_TOO_MANY_OPENED_FILES                       LW_STATUS_TOO_MANY_OPENED_FILES
#define STATUS_CANCELLED                                   LW_STATUS_CANCELLED
#define STATUS_CANNOT_DELETE                               LW_STATUS_CANNOT_DELETE
#define STATUS_INVALID_COMPUTER_NAME                       LW_STATUS_INVALID_COMPUTER_NAME
#define STATUS_FILE_DELETED                                LW_STATUS_FILE_DELETED
#define STATUS_SPECIAL_ACCOUNT                             LW_STATUS_SPECIAL_ACCOUNT
#define STATUS_SPECIAL_GROUP                               LW_STATUS_SPECIAL_GROUP
#define STATUS_SPECIAL_USER                                LW_STATUS_SPECIAL_USER
#define STATUS_MEMBERS_PRIMARY_GROUP                       LW_STATUS_MEMBERS_PRIMARY_GROUP
#define STATUS_FILE_CLOSED                                 LW_STATUS_FILE_CLOSED
#define STATUS_TOO_MANY_THREADS                            LW_STATUS_TOO_MANY_THREADS
#define STATUS_THREAD_NOT_IN_PROCESS                       LW_STATUS_THREAD_NOT_IN_PROCESS
#define STATUS_TOKEN_ALREADY_IN_USE                        LW_STATUS_TOKEN_ALREADY_IN_USE
#define STATUS_PAGEFILE_QUOTA_EXCEEDED                     LW_STATUS_PAGEFILE_QUOTA_EXCEEDED
#define STATUS_COMMITMENT_LIMIT                            LW_STATUS_COMMITMENT_LIMIT
#define STATUS_INVALID_IMAGE_LE_FORMAT                     LW_STATUS_INVALID_IMAGE_LE_FORMAT
#define STATUS_INVALID_IMAGE_NOT_MZ                        LW_STATUS_INVALID_IMAGE_NOT_MZ
#define STATUS_INVALID_IMAGE_PROTECT                       LW_STATUS_INVALID_IMAGE_PROTECT
#define STATUS_INVALID_IMAGE_WIN_16                        LW_STATUS_INVALID_IMAGE_WIN_16
#define STATUS_LOGON_SERVER_CONFLICT                       LW_STATUS_LOGON_SERVER_CONFLICT
#define STATUS_TIME_DIFFERENCE_AT_DC                       LW_STATUS_TIME_DIFFERENCE_AT_DC
#define STATUS_SYNCHRONIZATION_REQUIRED                    LW_STATUS_SYNCHRONIZATION_REQUIRED
#define STATUS_DLL_NOT_FOUND                               LW_STATUS_DLL_NOT_FOUND
#define STATUS_OPEN_FAILED                                 LW_STATUS_OPEN_FAILED
#define STATUS_IO_PRIVILEGE_FAILED                         LW_STATUS_IO_PRIVILEGE_FAILED
#define STATUS_ORDINAL_NOT_FOUND                           LW_STATUS_ORDINAL_NOT_FOUND
#define STATUS_ENTRYPOINT_NOT_FOUND                        LW_STATUS_ENTRYPOINT_NOT_FOUND
#define STATUS_CONTROL_C_EXIT                              LW_STATUS_CONTROL_C_EXIT
#define STATUS_LOCAL_DISCONNECT                            LW_STATUS_LOCAL_DISCONNECT
#define STATUS_REMOTE_DISCONNECT                           LW_STATUS_REMOTE_DISCONNECT
#define STATUS_REMOTE_RESOURCES                            LW_STATUS_REMOTE_RESOURCES
#define STATUS_LINK_FAILED                                 LW_STATUS_LINK_FAILED
#define STATUS_LINK_TIMEOUT                                LW_STATUS_LINK_TIMEOUT
#define STATUS_INVALID_CONNECTION                          LW_STATUS_INVALID_CONNECTION
#define STATUS_INVALID_ADDRESS                             LW_STATUS_INVALID_ADDRESS
#define STATUS_DLL_INIT_FAILED                             LW_STATUS_DLL_INIT_FAILED
#define STATUS_MISSING_SYSTEMFILE                          LW_STATUS_MISSING_SYSTEMFILE
#define STATUS_UNHANDLED_EXCEPTION                         LW_STATUS_UNHANDLED_EXCEPTION
#define STATUS_APP_INIT_FAILURE                            LW_STATUS_APP_INIT_FAILURE
#define STATUS_PAGEFILE_CREATE_FAILED                      LW_STATUS_PAGEFILE_CREATE_FAILED
#define STATUS_NO_PAGEFILE                                 LW_STATUS_NO_PAGEFILE
#define STATUS_INVALID_LEVEL                               LW_STATUS_INVALID_LEVEL
#define STATUS_WRONG_PASSWORD_CORE                         LW_STATUS_WRONG_PASSWORD_CORE
#define STATUS_ILLEGAL_FLOAT_CONTEXT                       LW_STATUS_ILLEGAL_FLOAT_CONTEXT
#define STATUS_PIPE_BROKEN                                 LW_STATUS_PIPE_BROKEN
#define STATUS_REGISTRY_CORRUPT                            LW_STATUS_REGISTRY_CORRUPT
#define STATUS_REGISTRY_IO_FAILED                          LW_STATUS_REGISTRY_IO_FAILED
#define STATUS_NO_EVENT_PAIR                               LW_STATUS_NO_EVENT_PAIR
#define STATUS_UNRECOGNIZED_VOLUME                         LW_STATUS_UNRECOGNIZED_VOLUME
#define STATUS_SERIAL_NO_DEVICE_INITED                     LW_STATUS_SERIAL_NO_DEVICE_INITED
#define STATUS_NO_SUCH_ALIAS                               LW_STATUS_NO_SUCH_ALIAS
#define STATUS_MEMBER_NOT_IN_ALIAS                         LW_STATUS_MEMBER_NOT_IN_ALIAS
#define STATUS_MEMBER_IN_ALIAS                             LW_STATUS_MEMBER_IN_ALIAS
#define STATUS_ALIAS_EXISTS                                LW_STATUS_ALIAS_EXISTS
#define STATUS_LOGON_NOT_GRANTED                           LW_STATUS_LOGON_NOT_GRANTED
#define STATUS_TOO_MANY_SECRETS                            LW_STATUS_TOO_MANY_SECRETS
#define STATUS_SECRET_TOO_LONG                             LW_STATUS_SECRET_TOO_LONG
#define STATUS_INTERNAL_DB_ERROR                           LW_STATUS_INTERNAL_DB_ERROR
#define STATUS_FULLSCREEN_MODE                             LW_STATUS_FULLSCREEN_MODE
#define STATUS_TOO_MANY_CONTEXT_IDS                        LW_STATUS_TOO_MANY_CONTEXT_IDS
#define STATUS_LOGON_TYPE_NOT_GRANTED                      LW_STATUS_LOGON_TYPE_NOT_GRANTED
#define STATUS_NOT_REGISTRY_FILE                           LW_STATUS_NOT_REGISTRY_FILE
#define STATUS_NT_CROSS_ENCRYPTION_REQUIRED                LW_STATUS_NT_CROSS_ENCRYPTION_REQUIRED
#define STATUS_DOMAIN_CTRLR_CONFIG_ERROR                   LW_STATUS_DOMAIN_CTRLR_CONFIG_ERROR
#define STATUS_FT_MISSING_MEMBER                           LW_STATUS_FT_MISSING_MEMBER
#define STATUS_ILL_FORMED_SERVICE_ENTRY                    LW_STATUS_ILL_FORMED_SERVICE_ENTRY
#define STATUS_ILLEGAL_CHARACTER                           LW_STATUS_ILLEGAL_CHARACTER
#define STATUS_UNMAPPABLE_CHARACTER                        LW_STATUS_UNMAPPABLE_CHARACTER
#define STATUS_UNDEFINED_CHARACTER                         LW_STATUS_UNDEFINED_CHARACTER
#define STATUS_FLOPPY_VOLUME                               LW_STATUS_FLOPPY_VOLUME
#define STATUS_FLOPPY_ID_MARK_NOT_FOUND                    LW_STATUS_FLOPPY_ID_MARK_NOT_FOUND
#define STATUS_FLOPPY_WRONG_CYLINDER                       LW_STATUS_FLOPPY_WRONG_CYLINDER
#define STATUS_FLOPPY_UNKNOWN_ERROR                        LW_STATUS_FLOPPY_UNKNOWN_ERROR
#define STATUS_FLOPPY_BAD_REGISTERS                        LW_STATUS_FLOPPY_BAD_REGISTERS
#define STATUS_DISK_RECALIBRATE_FAILED                     LW_STATUS_DISK_RECALIBRATE_FAILED
#define STATUS_DISK_OPERATION_FAILED                       LW_STATUS_DISK_OPERATION_FAILED
#define STATUS_DISK_RESET_FAILED                           LW_STATUS_DISK_RESET_FAILED
#define STATUS_SHARED_IRQ_BUSY                             LW_STATUS_SHARED_IRQ_BUSY
#define STATUS_FT_ORPHANING                                LW_STATUS_FT_ORPHANING
#define STATUS_BIOS_FAILED_TO_CONNECT_INTERRUPT            LW_STATUS_BIOS_FAILED_TO_CONNECT_INTERRUPT
#define STATUS_PARTITION_FAILURE                           LW_STATUS_PARTITION_FAILURE
#define STATUS_INVALID_BLOCK_LENGTH                        LW_STATUS_INVALID_BLOCK_LENGTH
#define STATUS_DEVICE_NOT_PARTITIONED                      LW_STATUS_DEVICE_NOT_PARTITIONED
#define STATUS_UNABLE_TO_LOCK_MEDIA                        LW_STATUS_UNABLE_TO_LOCK_MEDIA
#define STATUS_UNABLE_TO_UNLOAD_MEDIA                      LW_STATUS_UNABLE_TO_UNLOAD_MEDIA
#define STATUS_EOM_OVERFLOW                                LW_STATUS_EOM_OVERFLOW
#define STATUS_NO_MEDIA                                    LW_STATUS_NO_MEDIA
#define STATUS_NO_SUCH_MEMBER                              LW_STATUS_NO_SUCH_MEMBER
#define STATUS_INVALID_MEMBER                              LW_STATUS_INVALID_MEMBER
#define STATUS_KEY_DELETED                                 LW_STATUS_KEY_DELETED
#define STATUS_NO_LOG_SPACE                                LW_STATUS_NO_LOG_SPACE
#define STATUS_TOO_MANY_SIDS                               LW_STATUS_TOO_MANY_SIDS
#define STATUS_LM_CROSS_ENCRYPTION_REQUIRED                LW_STATUS_LM_CROSS_ENCRYPTION_REQUIRED
#define STATUS_KEY_HAS_CHILDREN                            LW_STATUS_KEY_HAS_CHILDREN
#define STATUS_CHILD_MUST_BE_VOLATILE                      LW_STATUS_CHILD_MUST_BE_VOLATILE
#define STATUS_DEVICE_CONFIGURATION_ERROR                  LW_STATUS_DEVICE_CONFIGURATION_ERROR
#define STATUS_DRIVER_INTERNAL_ERROR                       LW_STATUS_DRIVER_INTERNAL_ERROR
#define STATUS_INVALID_DEVICE_STATE                        LW_STATUS_INVALID_DEVICE_STATE
#define STATUS_IO_DEVICE_ERROR                             LW_STATUS_IO_DEVICE_ERROR
#define STATUS_DEVICE_PROTOCOL_ERROR                       LW_STATUS_DEVICE_PROTOCOL_ERROR
#define STATUS_BACKUP_CONTROLLER                           LW_STATUS_BACKUP_CONTROLLER
#define STATUS_LOG_FILE_FULL                               LW_STATUS_LOG_FILE_FULL
#define STATUS_TOO_LATE                                    LW_STATUS_TOO_LATE
#define STATUS_NO_TRUST_LSA_SECRET                         LW_STATUS_NO_TRUST_LSA_SECRET
#define STATUS_NO_TRUST_SAM_ACCOUNT                        LW_STATUS_NO_TRUST_SAM_ACCOUNT
#define STATUS_TRUSTED_DOMAIN_FAILURE                      LW_STATUS_TRUSTED_DOMAIN_FAILURE
#define STATUS_TRUSTED_RELATIONSHIP_FAILURE                LW_STATUS_TRUSTED_RELATIONSHIP_FAILURE
#define STATUS_EVENTLOG_FILE_CORRUPT                       LW_STATUS_EVENTLOG_FILE_CORRUPT
#define STATUS_EVENTLOG_CANT_START                         LW_STATUS_EVENTLOG_CANT_START
#define STATUS_TRUST_FAILURE                               LW_STATUS_TRUST_FAILURE
#define STATUS_MUTANT_LIMIT_EXCEEDED                       LW_STATUS_MUTANT_LIMIT_EXCEEDED
#define STATUS_NETLOGON_NOT_STARTED                        LW_STATUS_NETLOGON_NOT_STARTED
#define STATUS_ACCOUNT_EXPIRED                             LW_STATUS_ACCOUNT_EXPIRED
#define STATUS_POSSIBLE_DEADLOCK                           LW_STATUS_POSSIBLE_DEADLOCK
#define STATUS_NETWORK_CREDENTIAL_CONFLICT                 LW_STATUS_NETWORK_CREDENTIAL_CONFLICT
#define STATUS_REMOTE_SESSION_LIMIT                        LW_STATUS_REMOTE_SESSION_LIMIT
#define STATUS_EVENTLOG_FILE_CHANGED                       LW_STATUS_EVENTLOG_FILE_CHANGED
#define STATUS_NOLOGON_INTERDOMAIN_TRUST_ACCOUNT           LW_STATUS_NOLOGON_INTERDOMAIN_TRUST_ACCOUNT
#define STATUS_NOLOGON_WORKSTATION_TRUST_ACCOUNT           LW_STATUS_NOLOGON_WORKSTATION_TRUST_ACCOUNT
#define STATUS_NOLOGON_SERVER_TRUST_ACCOUNT                LW_STATUS_NOLOGON_SERVER_TRUST_ACCOUNT
#define STATUS_DOMAIN_TRUST_INCONSISTENT                   LW_STATUS_DOMAIN_TRUST_INCONSISTENT
#define STATUS_FS_DRIVER_REQUIRED                          LW_STATUS_FS_DRIVER_REQUIRED
#define STATUS_IMAGE_ALREADY_LOADED_AS_DLL                 LW_STATUS_IMAGE_ALREADY_LOADED_AS_DLL
#define STATUS_NETWORK_OPEN_RESTRICTION                    LW_STATUS_NETWORK_OPEN_RESTRICTION
#define STATUS_NO_USER_SESSION_KEY                         LW_STATUS_NO_USER_SESSION_KEY
#define STATUS_USER_SESSION_DELETED                        LW_STATUS_USER_SESSION_DELETED
#define STATUS_RESOURCE_LANG_NOT_FOUND                     LW_STATUS_RESOURCE_LANG_NOT_FOUND
#define STATUS_INSUFF_SERVER_RESOURCES                     LW_STATUS_INSUFF_SERVER_RESOURCES
#define STATUS_INVALID_BUFFER_SIZE                         LW_STATUS_INVALID_BUFFER_SIZE
#define STATUS_INVALID_ADDRESS_COMPONENT                   LW_STATUS_INVALID_ADDRESS_COMPONENT
#define STATUS_INVALID_ADDRESS_WILDCARD                    LW_STATUS_INVALID_ADDRESS_WILDCARD
#define STATUS_TOO_MANY_ADDRESSES                          LW_STATUS_TOO_MANY_ADDRESSES
#define STATUS_ADDRESS_ALREADY_EXISTS                      LW_STATUS_ADDRESS_ALREADY_EXISTS
#define STATUS_ADDRESS_CLOSED                              LW_STATUS_ADDRESS_CLOSED
#define STATUS_CONNECTION_DISCONNECTED                     LW_STATUS_CONNECTION_DISCONNECTED
#define STATUS_CONNECTION_RESET                            LW_STATUS_CONNECTION_RESET
#define STATUS_TOO_MANY_NODES                              LW_STATUS_TOO_MANY_NODES
#define STATUS_TRANSACTION_ABORTED                         LW_STATUS_TRANSACTION_ABORTED
#define STATUS_TRANSACTION_TIMED_OUT                       LW_STATUS_TRANSACTION_TIMED_OUT
#define STATUS_TRANSACTION_NO_RELEASE                      LW_STATUS_TRANSACTION_NO_RELEASE
#define STATUS_TRANSACTION_NO_MATCH                        LW_STATUS_TRANSACTION_NO_MATCH
#define STATUS_TRANSACTION_RESPONDED                       LW_STATUS_TRANSACTION_RESPONDED
#define STATUS_TRANSACTION_INVALID_ID                      LW_STATUS_TRANSACTION_INVALID_ID
#define STATUS_TRANSACTION_INVALID_TYPE                    LW_STATUS_TRANSACTION_INVALID_TYPE
#define STATUS_NOT_SERVER_SESSION                          LW_STATUS_NOT_SERVER_SESSION
#define STATUS_NOT_CLIENT_SESSION                          LW_STATUS_NOT_CLIENT_SESSION
#define STATUS_CANNOT_LOAD_REGISTRY_FILE                   LW_STATUS_CANNOT_LOAD_REGISTRY_FILE
#define STATUS_DEBUG_ATTACH_FAILED                         LW_STATUS_DEBUG_ATTACH_FAILED
#define STATUS_SYSTEM_PROCESS_TERMINATED                   LW_STATUS_SYSTEM_PROCESS_TERMINATED
#define STATUS_DATA_NOT_ACCEPTED                           LW_STATUS_DATA_NOT_ACCEPTED
#define STATUS_NO_BROWSER_SERVERS_FOUND                    LW_STATUS_NO_BROWSER_SERVERS_FOUND
#define STATUS_VDM_HARD_ERROR                              LW_STATUS_VDM_HARD_ERROR
#define STATUS_DRIVER_CANCEL_TIMEOUT                       LW_STATUS_DRIVER_CANCEL_TIMEOUT
#define STATUS_REPLY_MESSAGE_MISMATCH                      LW_STATUS_REPLY_MESSAGE_MISMATCH
#define STATUS_MAPPED_ALIGNMENT                            LW_STATUS_MAPPED_ALIGNMENT
#define STATUS_IMAGE_CHECKSUM_MISMATCH                     LW_STATUS_IMAGE_CHECKSUM_MISMATCH
#define STATUS_LOST_WRITEBEHIND_DATA                       LW_STATUS_LOST_WRITEBEHIND_DATA
#define STATUS_CLIENT_SERVER_PARAMETERS_INVALID            LW_STATUS_CLIENT_SERVER_PARAMETERS_INVALID
#define STATUS_PASSWORD_MUST_CHANGE                        LW_STATUS_PASSWORD_MUST_CHANGE
#define STATUS_NOT_FOUND                                   LW_STATUS_NOT_FOUND
#define STATUS_NOT_TINY_STREAM                             LW_STATUS_NOT_TINY_STREAM
#define STATUS_RECOVERY_FAILURE                            LW_STATUS_RECOVERY_FAILURE
#define STATUS_STACK_OVERFLOW_READ                         LW_STATUS_STACK_OVERFLOW_READ
#define STATUS_FAIL_CHECK                                  LW_STATUS_FAIL_CHECK
#define STATUS_DUPLICATE_OBJECTID                          LW_STATUS_DUPLICATE_OBJECTID
#define STATUS_OBJECTID_EXISTS                             LW_STATUS_OBJECTID_EXISTS
#define STATUS_CONVERT_TO_LARGE                            LW_STATUS_CONVERT_TO_LARGE
#define STATUS_RETRY                                       LW_STATUS_RETRY
#define STATUS_FOUND_OUT_OF_SCOPE                          LW_STATUS_FOUND_OUT_OF_SCOPE
#define STATUS_ALLOCATE_BUCKET                             LW_STATUS_ALLOCATE_BUCKET
#define STATUS_PROPSET_NOT_FOUND                           LW_STATUS_PROPSET_NOT_FOUND
#define STATUS_MARSHALL_OVERFLOW                           LW_STATUS_MARSHALL_OVERFLOW
#define STATUS_INVALID_VARIANT                             LW_STATUS_INVALID_VARIANT
#define STATUS_DOMAIN_CONTROLLER_NOT_FOUND                 LW_STATUS_DOMAIN_CONTROLLER_NOT_FOUND
#define STATUS_ACCOUNT_LOCKED_OUT                          LW_STATUS_ACCOUNT_LOCKED_OUT
#define STATUS_HANDLE_NOT_CLOSABLE                         LW_STATUS_HANDLE_NOT_CLOSABLE
#define STATUS_CONNECTION_REFUSED                          LW_STATUS_CONNECTION_REFUSED
#define STATUS_GRACEFUL_DISCONNECT                         LW_STATUS_GRACEFUL_DISCONNECT
#define STATUS_ADDRESS_ALREADY_ASSOCIATED                  LW_STATUS_ADDRESS_ALREADY_ASSOCIATED
#define STATUS_ADDRESS_NOT_ASSOCIATED                      LW_STATUS_ADDRESS_NOT_ASSOCIATED
#define STATUS_CONNECTION_INVALID                          LW_STATUS_CONNECTION_INVALID
#define STATUS_CONNECTION_ACTIVE                           LW_STATUS_CONNECTION_ACTIVE
#define STATUS_NETWORK_UNREACHABLE                         LW_STATUS_NETWORK_UNREACHABLE
#define STATUS_HOST_UNREACHABLE                            LW_STATUS_HOST_UNREACHABLE
#define STATUS_PROTOCOL_UNREACHABLE                        LW_STATUS_PROTOCOL_UNREACHABLE
#define STATUS_PORT_UNREACHABLE                            LW_STATUS_PORT_UNREACHABLE
#define STATUS_REQUEST_ABORTED                             LW_STATUS_REQUEST_ABORTED
#define STATUS_CONNECTION_ABORTED                          LW_STATUS_CONNECTION_ABORTED
#define STATUS_BAD_COMPRESSION_BUFFER                      LW_STATUS_BAD_COMPRESSION_BUFFER
#define STATUS_USER_MAPPED_FILE                            LW_STATUS_USER_MAPPED_FILE
#define STATUS_AUDIT_FAILED                                LW_STATUS_AUDIT_FAILED
#define STATUS_TIMER_RESOLUTION_NOT_SET                    LW_STATUS_TIMER_RESOLUTION_NOT_SET
#define STATUS_CONNECTION_COUNT_LIMIT                      LW_STATUS_CONNECTION_COUNT_LIMIT
#define STATUS_LOGIN_TIME_RESTRICTION                      LW_STATUS_LOGIN_TIME_RESTRICTION
#define STATUS_LOGIN_WKSTA_RESTRICTION                     LW_STATUS_LOGIN_WKSTA_RESTRICTION
#define STATUS_IMAGE_MP_UP_MISMATCH                        LW_STATUS_IMAGE_MP_UP_MISMATCH
#define STATUS_INSUFFICIENT_LOGON_INFO                     LW_STATUS_INSUFFICIENT_LOGON_INFO
#define STATUS_BAD_DLL_ENTRYPOINT                          LW_STATUS_BAD_DLL_ENTRYPOINT
#define STATUS_BAD_SERVICE_ENTRYPOINT                      LW_STATUS_BAD_SERVICE_ENTRYPOINT
#define STATUS_LPC_REPLY_LOST                              LW_STATUS_LPC_REPLY_LOST
#define STATUS_IP_ADDRESS_CONFLICT1                        LW_STATUS_IP_ADDRESS_CONFLICT1
#define STATUS_IP_ADDRESS_CONFLICT2                        LW_STATUS_IP_ADDRESS_CONFLICT2
#define STATUS_REGISTRY_QUOTA_LIMIT                        LW_STATUS_REGISTRY_QUOTA_LIMIT
#define STATUS_PATH_NOT_COVERED                            LW_STATUS_PATH_NOT_COVERED
#define STATUS_NO_CALLBACK_ACTIVE                          LW_STATUS_NO_CALLBACK_ACTIVE
#define STATUS_LICENSE_QUOTA_EXCEEDED                      LW_STATUS_LICENSE_QUOTA_EXCEEDED
#define STATUS_PWD_TOO_SHORT                               LW_STATUS_PWD_TOO_SHORT
#define STATUS_PWD_TOO_RECENT                              LW_STATUS_PWD_TOO_RECENT
#define STATUS_PWD_HISTORY_CONFLICT                        LW_STATUS_PWD_HISTORY_CONFLICT
#define STATUS_PLUGPLAY_NO_DEVICE                          LW_STATUS_PLUGPLAY_NO_DEVICE
#define STATUS_UNSUPPORTED_COMPRESSION                     LW_STATUS_UNSUPPORTED_COMPRESSION
#define STATUS_INVALID_HW_PROFILE                          LW_STATUS_INVALID_HW_PROFILE
#define STATUS_INVALID_PLUGPLAY_DEVICE_PATH                LW_STATUS_INVALID_PLUGPLAY_DEVICE_PATH
#define STATUS_DRIVER_ORDINAL_NOT_FOUND                    LW_STATUS_DRIVER_ORDINAL_NOT_FOUND
#define STATUS_DRIVER_ENTRYPOINT_NOT_FOUND                 LW_STATUS_DRIVER_ENTRYPOINT_NOT_FOUND
#define STATUS_RESOURCE_NOT_OWNED                          LW_STATUS_RESOURCE_NOT_OWNED
#define STATUS_TOO_MANY_LINKS                              LW_STATUS_TOO_MANY_LINKS
#define STATUS_QUOTA_LIST_INCONSISTENT                     LW_STATUS_QUOTA_LIST_INCONSISTENT
#define STATUS_FILE_IS_OFFLINE                             LW_STATUS_FILE_IS_OFFLINE
#define STATUS_EVALUATION_EXPIRATION                       LW_STATUS_EVALUATION_EXPIRATION
#define STATUS_ILLEGAL_DLL_RELOCATION                      LW_STATUS_ILLEGAL_DLL_RELOCATION
#define STATUS_LICENSE_VIOLATION                           LW_STATUS_LICENSE_VIOLATION
#define STATUS_DLL_INIT_FAILED_LOGOFF                      LW_STATUS_DLL_INIT_FAILED_LOGOFF
#define STATUS_DRIVER_UNABLE_TO_LOAD                       LW_STATUS_DRIVER_UNABLE_TO_LOAD
#define STATUS_DFS_UNAVAILABLE                             LW_STATUS_DFS_UNAVAILABLE
#define STATUS_VOLUME_DISMOUNTED                           LW_STATUS_VOLUME_DISMOUNTED
#define STATUS_WX86_INTERNAL_ERROR                         LW_STATUS_WX86_INTERNAL_ERROR
#define STATUS_WX86_FLOAT_STACK_CHECK                      LW_STATUS_WX86_FLOAT_STACK_CHECK
#define STATUS_VALIDATE_CONTINUE                           LW_STATUS_VALIDATE_CONTINUE
#define STATUS_NO_MATCH                                    LW_STATUS_NO_MATCH
#define STATUS_NO_MORE_MATCHES                             LW_STATUS_NO_MORE_MATCHES
#define STATUS_NOT_A_REPARSE_POINT                         LW_STATUS_NOT_A_REPARSE_POINT
#define STATUS_IO_REPARSE_TAG_INVALID                      LW_STATUS_IO_REPARSE_TAG_INVALID
#define STATUS_IO_REPARSE_TAG_MISMATCH                     LW_STATUS_IO_REPARSE_TAG_MISMATCH
#define STATUS_IO_REPARSE_DATA_INVALID                     LW_STATUS_IO_REPARSE_DATA_INVALID
#define STATUS_IO_REPARSE_TAG_NOT_HANDLED                  LW_STATUS_IO_REPARSE_TAG_NOT_HANDLED
#define STATUS_REPARSE_POINT_NOT_RESOLVED                  LW_STATUS_REPARSE_POINT_NOT_RESOLVED
#define STATUS_DIRECTORY_IS_A_REPARSE_POINT                LW_STATUS_DIRECTORY_IS_A_REPARSE_POINT
#define STATUS_RANGE_LIST_CONFLICT                         LW_STATUS_RANGE_LIST_CONFLICT
#define STATUS_SOURCE_ELEMENT_EMPTY                        LW_STATUS_SOURCE_ELEMENT_EMPTY
#define STATUS_DESTINATION_ELEMENT_FULL                    LW_STATUS_DESTINATION_ELEMENT_FULL
#define STATUS_ILLEGAL_ELEMENT_ADDRESS                     LW_STATUS_ILLEGAL_ELEMENT_ADDRESS
#define STATUS_MAGAZINE_NOT_PRESENT                        LW_STATUS_MAGAZINE_NOT_PRESENT
#define STATUS_REINITIALIZATION_NEEDED                     LW_STATUS_REINITIALIZATION_NEEDED
#define STATUS_ENCRYPTION_FAILED                           LW_STATUS_ENCRYPTION_FAILED
#define STATUS_DECRYPTION_FAILED                           LW_STATUS_DECRYPTION_FAILED
#define STATUS_RANGE_NOT_FOUND                             LW_STATUS_RANGE_NOT_FOUND
#define STATUS_NO_RECOVERY_POLICY                          LW_STATUS_NO_RECOVERY_POLICY
#define STATUS_NO_EFS                                      LW_STATUS_NO_EFS
#define STATUS_WRONG_EFS                                   LW_STATUS_WRONG_EFS
#define STATUS_NO_USER_KEYS                                LW_STATUS_NO_USER_KEYS
#define STATUS_FILE_NOT_ENCRYPTED                          LW_STATUS_FILE_NOT_ENCRYPTED
#define STATUS_NOT_EXPORT_FORMAT                           LW_STATUS_NOT_EXPORT_FORMAT
#define STATUS_FILE_ENCRYPTED                              LW_STATUS_FILE_ENCRYPTED
#define STATUS_WMI_GUID_NOT_FOUND                          LW_STATUS_WMI_GUID_NOT_FOUND
#define STATUS_WMI_INSTANCE_NOT_FOUND                      LW_STATUS_WMI_INSTANCE_NOT_FOUND
#define STATUS_WMI_ITEMID_NOT_FOUND                        LW_STATUS_WMI_ITEMID_NOT_FOUND
#define STATUS_WMI_TRY_AGAIN                               LW_STATUS_WMI_TRY_AGAIN
#define STATUS_SHARED_POLICY                               LW_STATUS_SHARED_POLICY
#define STATUS_POLICY_OBJECT_NOT_FOUND                     LW_STATUS_POLICY_OBJECT_NOT_FOUND
#define STATUS_POLICY_ONLY_IN_DS                           LW_STATUS_POLICY_ONLY_IN_DS
#define STATUS_VOLUME_NOT_UPGRADED                         LW_STATUS_VOLUME_NOT_UPGRADED
#define STATUS_REMOTE_STORAGE_NOT_ACTIVE                   LW_STATUS_REMOTE_STORAGE_NOT_ACTIVE
#define STATUS_REMOTE_STORAGE_MEDIA_ERROR                  LW_STATUS_REMOTE_STORAGE_MEDIA_ERROR
#define STATUS_NO_TRACKING_SERVICE                         LW_STATUS_NO_TRACKING_SERVICE
#define STATUS_SERVER_SID_MISMATCH                         LW_STATUS_SERVER_SID_MISMATCH
#define STATUS_DS_NO_ATTRIBUTE_OR_VALUE                    LW_STATUS_DS_NO_ATTRIBUTE_OR_VALUE
#define STATUS_DS_INVALID_ATTRIBUTE_SYNTAX                 LW_STATUS_DS_INVALID_ATTRIBUTE_SYNTAX
#define STATUS_DS_ATTRIBUTE_TYPE_UNDEFINED                 LW_STATUS_DS_ATTRIBUTE_TYPE_UNDEFINED
#define STATUS_DS_ATTRIBUTE_OR_VALUE_EXISTS                LW_STATUS_DS_ATTRIBUTE_OR_VALUE_EXISTS
#define STATUS_DS_BUSY                                     LW_STATUS_DS_BUSY
#define STATUS_DS_UNAVAILABLE                              LW_STATUS_DS_UNAVAILABLE
#define STATUS_DS_NO_RIDS_ALLOCATED                        LW_STATUS_DS_NO_RIDS_ALLOCATED
#define STATUS_DS_NO_MORE_RIDS                             LW_STATUS_DS_NO_MORE_RIDS
#define STATUS_DS_INCORRECT_ROLE_OWNER                     LW_STATUS_DS_INCORRECT_ROLE_OWNER
#define STATUS_DS_RIDMGR_INIT_ERROR                        LW_STATUS_DS_RIDMGR_INIT_ERROR
#define STATUS_DS_OBJ_CLASS_VIOLATION                      LW_STATUS_DS_OBJ_CLASS_VIOLATION
#define STATUS_DS_CANT_ON_NON_LEAF                         LW_STATUS_DS_CANT_ON_NON_LEAF
#define STATUS_DS_CANT_ON_RDN                              LW_STATUS_DS_CANT_ON_RDN
#define STATUS_DS_CANT_MOD_OBJ_CLASS                       LW_STATUS_DS_CANT_MOD_OBJ_CLASS
#define STATUS_DS_CROSS_DOM_MOVE_FAILED                    LW_STATUS_DS_CROSS_DOM_MOVE_FAILED
#define STATUS_DS_GC_NOT_AVAILABLE                         LW_STATUS_DS_GC_NOT_AVAILABLE
#define STATUS_DIRECTORY_SERVICE_REQUIRED                  LW_STATUS_DIRECTORY_SERVICE_REQUIRED
#define STATUS_REPARSE_ATTRIBUTE_CONFLICT                  LW_STATUS_REPARSE_ATTRIBUTE_CONFLICT
#define STATUS_CANT_ENABLE_DENY_ONLY                       LW_STATUS_CANT_ENABLE_DENY_ONLY
#define STATUS_FLOAT_MULTIPLE_FAULTS                       LW_STATUS_FLOAT_MULTIPLE_FAULTS
#define STATUS_FLOAT_MULTIPLE_TRAPS                        LW_STATUS_FLOAT_MULTIPLE_TRAPS
#define STATUS_DEVICE_REMOVED                              LW_STATUS_DEVICE_REMOVED
#define STATUS_JOURNAL_DELETE_IN_PROGRESS                  LW_STATUS_JOURNAL_DELETE_IN_PROGRESS
#define STATUS_JOURNAL_NOT_ACTIVE                          LW_STATUS_JOURNAL_NOT_ACTIVE
#define STATUS_NOINTERFACE                                 LW_STATUS_NOINTERFACE
#define STATUS_DS_ADMIN_LIMIT_EXCEEDED                     LW_STATUS_DS_ADMIN_LIMIT_EXCEEDED
#define STATUS_DRIVER_FAILED_SLEEP                         LW_STATUS_DRIVER_FAILED_SLEEP
#define STATUS_MUTUAL_AUTHENTICATION_FAILED                LW_STATUS_MUTUAL_AUTHENTICATION_FAILED
#define STATUS_CORRUPT_SYSTEM_FILE                         LW_STATUS_CORRUPT_SYSTEM_FILE
#define STATUS_DATATYPE_MISALIGNMENT_ERROR                 LW_STATUS_DATATYPE_MISALIGNMENT_ERROR
#define STATUS_WMI_READ_ONLY                               LW_STATUS_WMI_READ_ONLY
#define STATUS_WMI_SET_FAILURE                             LW_STATUS_WMI_SET_FAILURE
#define STATUS_COMMITMENT_MINIMUM                          LW_STATUS_COMMITMENT_MINIMUM
#define STATUS_REG_NAT_CONSUMPTION                         LW_STATUS_REG_NAT_CONSUMPTION
#define STATUS_TRANSPORT_FULL                              LW_STATUS_TRANSPORT_FULL
#define STATUS_DS_SAM_INIT_FAILURE                         LW_STATUS_DS_SAM_INIT_FAILURE
#define STATUS_ONLY_IF_CONNECTED                           LW_STATUS_ONLY_IF_CONNECTED
#define STATUS_DS_SENSITIVE_GROUP_VIOLATION                LW_STATUS_DS_SENSITIVE_GROUP_VIOLATION
#define STATUS_PNP_RESTART_ENUMERATION                     LW_STATUS_PNP_RESTART_ENUMERATION
#define STATUS_JOURNAL_ENTRY_DELETED                       LW_STATUS_JOURNAL_ENTRY_DELETED
#define STATUS_DS_CANT_MOD_PRIMARYGROUPID                  LW_STATUS_DS_CANT_MOD_PRIMARYGROUPID
#define STATUS_SYSTEM_IMAGE_BAD_SIGNATURE                  LW_STATUS_SYSTEM_IMAGE_BAD_SIGNATURE
#define STATUS_PNP_REBOOT_REQUIRED                         LW_STATUS_PNP_REBOOT_REQUIRED
#define STATUS_POWER_STATE_INVALID                         LW_STATUS_POWER_STATE_INVALID
#define STATUS_DS_INVALID_GROUP_TYPE                       LW_STATUS_DS_INVALID_GROUP_TYPE
#define STATUS_DS_NO_NEST_GLOBALGROUP_IN_MIXEDDOMAIN       LW_STATUS_DS_NO_NEST_GLOBALGROUP_IN_MIXEDDOMAIN
#define STATUS_DS_NO_NEST_LOCALGROUP_IN_MIXEDDOMAIN        LW_STATUS_DS_NO_NEST_LOCALGROUP_IN_MIXEDDOMAIN
#define STATUS_DS_GLOBAL_CANT_HAVE_LOCAL_MEMBER            LW_STATUS_DS_GLOBAL_CANT_HAVE_LOCAL_MEMBER
#define STATUS_DS_GLOBAL_CANT_HAVE_UNIVERSAL_MEMBER        LW_STATUS_DS_GLOBAL_CANT_HAVE_UNIVERSAL_MEMBER
#define STATUS_DS_UNIVERSAL_CANT_HAVE_LOCAL_MEMBER         LW_STATUS_DS_UNIVERSAL_CANT_HAVE_LOCAL_MEMBER
#define STATUS_DS_GLOBAL_CANT_HAVE_CROSSDOMAIN_MEMBER      LW_STATUS_DS_GLOBAL_CANT_HAVE_CROSSDOMAIN_MEMBER
#define STATUS_DS_LOCAL_CANT_HAVE_CROSSDOMAIN_LOCAL_MEMBER LW_STATUS_DS_LOCAL_CANT_HAVE_CROSSDOMAIN_LOCAL_MEMBER
#define STATUS_DS_HAVE_PRIMARY_MEMBERS                     LW_STATUS_DS_HAVE_PRIMARY_MEMBERS
#define STATUS_WMI_NOT_SUPPORTED                           LW_STATUS_WMI_NOT_SUPPORTED
#define STATUS_INSUFFICIENT_POWER                          LW_STATUS_INSUFFICIENT_POWER
#define STATUS_SAM_NEED_BOOTKEY_PASSWORD                   LW_STATUS_SAM_NEED_BOOTKEY_PASSWORD
#define STATUS_SAM_NEED_BOOTKEY_FLOPPY                     LW_STATUS_SAM_NEED_BOOTKEY_FLOPPY
#define STATUS_DS_CANT_START                               LW_STATUS_DS_CANT_START
#define STATUS_DS_INIT_FAILURE                             LW_STATUS_DS_INIT_FAILURE
#define STATUS_SAM_INIT_FAILURE                            LW_STATUS_SAM_INIT_FAILURE
#define STATUS_DS_GC_REQUIRED                              LW_STATUS_DS_GC_REQUIRED
#define STATUS_DS_LOCAL_MEMBER_OF_LOCAL_ONLY               LW_STATUS_DS_LOCAL_MEMBER_OF_LOCAL_ONLY
#define STATUS_DS_NO_FPO_IN_UNIVERSAL_GROUPS               LW_STATUS_DS_NO_FPO_IN_UNIVERSAL_GROUPS
#define STATUS_DS_MACHINE_ACCOUNT_QUOTA_EXCEEDED           LW_STATUS_DS_MACHINE_ACCOUNT_QUOTA_EXCEEDED
#define STATUS_MULTIPLE_FAULT_VIOLATION                    LW_STATUS_MULTIPLE_FAULT_VIOLATION
#define STATUS_CURRENT_DOMAIN_NOT_ALLOWED                  LW_STATUS_CURRENT_DOMAIN_NOT_ALLOWED
#define STATUS_CANNOT_MAKE                                 LW_STATUS_CANNOT_MAKE
#define STATUS_SYSTEM_SHUTDOWN                             LW_STATUS_SYSTEM_SHUTDOWN
#define STATUS_DS_INIT_FAILURE_CONSOLE                     LW_STATUS_DS_INIT_FAILURE_CONSOLE
#define STATUS_DS_SAM_INIT_FAILURE_CONSOLE                 LW_STATUS_DS_SAM_INIT_FAILURE_CONSOLE
#define STATUS_UNFINISHED_CONTEXT_DELETED                  LW_STATUS_UNFINISHED_CONTEXT_DELETED
#define STATUS_NO_TGT_REPLY                                LW_STATUS_NO_TGT_REPLY
#define STATUS_OBJECTID_NOT_FOUND                          LW_STATUS_OBJECTID_NOT_FOUND
#define STATUS_NO_IP_ADDRESSES                             LW_STATUS_NO_IP_ADDRESSES
#define STATUS_WRONG_CREDENTIAL_HANDLE                     LW_STATUS_WRONG_CREDENTIAL_HANDLE
#define STATUS_CRYPTO_SYSTEM_INVALID                       LW_STATUS_CRYPTO_SYSTEM_INVALID
#define STATUS_MAX_REFERRALS_EXCEEDED                      LW_STATUS_MAX_REFERRALS_EXCEEDED
#define STATUS_MUST_BE_KDC                                 LW_STATUS_MUST_BE_KDC
#define STATUS_STRONG_CRYPTO_NOT_SUPPORTED                 LW_STATUS_STRONG_CRYPTO_NOT_SUPPORTED
#define STATUS_TOO_MANY_PRINCIPALS                         LW_STATUS_TOO_MANY_PRINCIPALS
#define STATUS_NO_PA_DATA                                  LW_STATUS_NO_PA_DATA
#define STATUS_PKINIT_NAME_MISMATCH                        LW_STATUS_PKINIT_NAME_MISMATCH
#define STATUS_SMARTCARD_LOGON_REQUIRED                    LW_STATUS_SMARTCARD_LOGON_REQUIRED
#define STATUS_KDC_INVALID_REQUEST                         LW_STATUS_KDC_INVALID_REQUEST
#define STATUS_KDC_UNABLE_TO_REFER                         LW_STATUS_KDC_UNABLE_TO_REFER
#define STATUS_KDC_UNKNOWN_ETYPE                           LW_STATUS_KDC_UNKNOWN_ETYPE
#define STATUS_SHUTDOWN_IN_PROGRESS                        LW_STATUS_SHUTDOWN_IN_PROGRESS
#define STATUS_SERVER_SHUTDOWN_IN_PROGRESS                 LW_STATUS_SERVER_SHUTDOWN_IN_PROGRESS
#define STATUS_NOT_SUPPORTED_ON_SBS                        LW_STATUS_NOT_SUPPORTED_ON_SBS
#define STATUS_WMI_GUID_DISCONNECTED                       LW_STATUS_WMI_GUID_DISCONNECTED
#define STATUS_WMI_ALREADY_DISABLED                        LW_STATUS_WMI_ALREADY_DISABLED
#define STATUS_WMI_ALREADY_ENABLED                         LW_STATUS_WMI_ALREADY_ENABLED
#define STATUS_MFT_TOO_FRAGMENTED                          LW_STATUS_MFT_TOO_FRAGMENTED
#define STATUS_COPY_PROTECTION_FAILURE                     LW_STATUS_COPY_PROTECTION_FAILURE
#define STATUS_CSS_AUTHENTICATION_FAILURE                  LW_STATUS_CSS_AUTHENTICATION_FAILURE
#define STATUS_CSS_KEY_NOT_PRESENT                         LW_STATUS_CSS_KEY_NOT_PRESENT
#define STATUS_CSS_KEY_NOT_ESTABLISHED                     LW_STATUS_CSS_KEY_NOT_ESTABLISHED
#define STATUS_CSS_SCRAMBLED_SECTOR                        LW_STATUS_CSS_SCRAMBLED_SECTOR
#define STATUS_CSS_REGION_MISMATCH                         LW_STATUS_CSS_REGION_MISMATCH
#define STATUS_CSS_RESETS_EXHAUSTED                        LW_STATUS_CSS_RESETS_EXHAUSTED
#define STATUS_PKINIT_FAILURE                              LW_STATUS_PKINIT_FAILURE
#define STATUS_SMARTCARD_SUBSYSTEM_FAILURE                 LW_STATUS_SMARTCARD_SUBSYSTEM_FAILURE
#define STATUS_NO_KERB_KEY                                 LW_STATUS_NO_KERB_KEY
#define STATUS_HOST_DOWN                                   LW_STATUS_HOST_DOWN
#define STATUS_UNSUPPORTED_PREAUTH                         LW_STATUS_UNSUPPORTED_PREAUTH
#define STATUS_EFS_ALG_BLOB_TOO_BIG                        LW_STATUS_EFS_ALG_BLOB_TOO_BIG
#define STATUS_PORT_NOT_SET                                LW_STATUS_PORT_NOT_SET
#define STATUS_DEBUGGER_INACTIVE                           LW_STATUS_DEBUGGER_INACTIVE
#define STATUS_DS_VERSION_CHECK_FAILURE                    LW_STATUS_DS_VERSION_CHECK_FAILURE
#define STATUS_AUDITING_DISABLED                           LW_STATUS_AUDITING_DISABLED
#define STATUS_PRENT4_MACHINE_ACCOUNT                      LW_STATUS_PRENT4_MACHINE_ACCOUNT
#define STATUS_DS_AG_CANT_HAVE_UNIVERSAL_MEMBER            LW_STATUS_DS_AG_CANT_HAVE_UNIVERSAL_MEMBER
#define STATUS_INVALID_IMAGE_WIN_32                        LW_STATUS_INVALID_IMAGE_WIN_32
#define STATUS_INVALID_IMAGE_WIN_64                        LW_STATUS_INVALID_IMAGE_WIN_64
#define STATUS_BAD_BINDINGS                                LW_STATUS_BAD_BINDINGS
#define STATUS_NETWORK_SESSION_EXPIRED                     LW_STATUS_NETWORK_SESSION_EXPIRED
#define STATUS_APPHELP_BLOCK                               LW_STATUS_APPHELP_BLOCK
#define STATUS_ALL_SIDS_FILTERED                           LW_STATUS_ALL_SIDS_FILTERED
#define STATUS_NOT_SAFE_MODE_DRIVER                        LW_STATUS_NOT_SAFE_MODE_DRIVER
#define STATUS_ACCESS_DISABLED_BY_POLICY_DEFAULT           LW_STATUS_ACCESS_DISABLED_BY_POLICY_DEFAULT
#define STATUS_ACCESS_DISABLED_BY_POLICY_PATH              LW_STATUS_ACCESS_DISABLED_BY_POLICY_PATH
#define STATUS_ACCESS_DISABLED_BY_POLICY_PUBLISHER         LW_STATUS_ACCESS_DISABLED_BY_POLICY_PUBLISHER
#define STATUS_ACCESS_DISABLED_BY_POLICY_OTHER             LW_STATUS_ACCESS_DISABLED_BY_POLICY_OTHER
#define STATUS_FAILED_DRIVER_ENTRY                         LW_STATUS_FAILED_DRIVER_ENTRY
#define STATUS_DEVICE_ENUMERATION_ERROR                    LW_STATUS_DEVICE_ENUMERATION_ERROR
#define STATUS_MOUNT_POINT_NOT_RESOLVED                    LW_STATUS_MOUNT_POINT_NOT_RESOLVED
#define STATUS_INVALID_DEVICE_OBJECT_PARAMETER             LW_STATUS_INVALID_DEVICE_OBJECT_PARAMETER
#define STATUS_MCA_OCCURED                                 LW_STATUS_MCA_OCCURED
#define STATUS_DRIVER_BLOCKED_CRITICAL                     LW_STATUS_DRIVER_BLOCKED_CRITICAL
#define STATUS_DRIVER_BLOCKED                              LW_STATUS_DRIVER_BLOCKED
#define STATUS_DRIVER_DATABASE_ERROR                       LW_STATUS_DRIVER_DATABASE_ERROR
#define STATUS_SYSTEM_HIVE_TOO_LARGE                       LW_STATUS_SYSTEM_HIVE_TOO_LARGE
#define STATUS_INVALID_IMPORT_OF_NON_DLL                   LW_STATUS_INVALID_IMPORT_OF_NON_DLL
#define STATUS_NO_SECRETS                                  LW_STATUS_NO_SECRETS
#define STATUS_ACCESS_DISABLED_NO_SAFER_UI_BY_POLICY       LW_STATUS_ACCESS_DISABLED_NO_SAFER_UI_BY_POLICY
#define STATUS_FAILED_STACK_SWITCH                         LW_STATUS_FAILED_STACK_SWITCH
#define STATUS_HEAP_CORRUPTION                             LW_STATUS_HEAP_CORRUPTION
#define STATUS_SMARTCARD_WRONG_PIN                         LW_STATUS_SMARTCARD_WRONG_PIN
#define STATUS_SMARTCARD_CARD_BLOCKED                      LW_STATUS_SMARTCARD_CARD_BLOCKED
#define STATUS_SMARTCARD_CARD_NOT_AUTHENTICATED            LW_STATUS_SMARTCARD_CARD_NOT_AUTHENTICATED
#define STATUS_SMARTCARD_NO_CARD                           LW_STATUS_SMARTCARD_NO_CARD
#define STATUS_SMARTCARD_NO_KEY_CONTAINER                  LW_STATUS_SMARTCARD_NO_KEY_CONTAINER
#define STATUS_SMARTCARD_NO_CERTIFICATE                    LW_STATUS_SMARTCARD_NO_CERTIFICATE
#define STATUS_SMARTCARD_NO_KEYSET                         LW_STATUS_SMARTCARD_NO_KEYSET
#define STATUS_SMARTCARD_IO_ERROR                          LW_STATUS_SMARTCARD_IO_ERROR
#define STATUS_DOWNGRADE_DETECTED                          LW_STATUS_DOWNGRADE_DETECTED
#define STATUS_SMARTCARD_CERT_REVOKED                      LW_STATUS_SMARTCARD_CERT_REVOKED
#define STATUS_ISSUING_CA_UNTRUSTED                        LW_STATUS_ISSUING_CA_UNTRUSTED
#define STATUS_REVOCATION_OFFLINE_C                        LW_STATUS_REVOCATION_OFFLINE_C
#define STATUS_PKINIT_CLIENT_FAILURE                       LW_STATUS_PKINIT_CLIENT_FAILURE
#define STATUS_SMARTCARD_CERT_EXPIRED                      LW_STATUS_SMARTCARD_CERT_EXPIRED
#define STATUS_DRIVER_FAILED_PRIOR_UNLOAD                  LW_STATUS_DRIVER_FAILED_PRIOR_UNLOAD
#define STATUS_SMARTCARD_SILENT_CONTEXT                    LW_STATUS_SMARTCARD_SILENT_CONTEXT
#define STATUS_PER_USER_TRUST_QUOTA_EXCEEDED               LW_STATUS_PER_USER_TRUST_QUOTA_EXCEEDED
#define STATUS_ALL_USER_TRUST_QUOTA_EXCEEDED               LW_STATUS_ALL_USER_TRUST_QUOTA_EXCEEDED
#define STATUS_USER_DELETE_TRUST_QUOTA_EXCEEDED            LW_STATUS_USER_DELETE_TRUST_QUOTA_EXCEEDED
#define STATUS_DS_NAME_NOT_UNIQUE                          LW_STATUS_DS_NAME_NOT_UNIQUE
#define STATUS_DS_DUPLICATE_ID_FOUND                       LW_STATUS_DS_DUPLICATE_ID_FOUND
#define STATUS_DS_GROUP_CONVERSION_ERROR                   LW_STATUS_DS_GROUP_CONVERSION_ERROR
#define STATUS_VOLSNAP_PREPARE_HIBERNATE                   LW_STATUS_VOLSNAP_PREPARE_HIBERNATE
#define STATUS_USER2USER_REQUIRED                          LW_STATUS_USER2USER_REQUIRED
#define STATUS_STACK_BUFFER_OVERRUN                        LW_STATUS_STACK_BUFFER_OVERRUN
#define STATUS_NO_S4U_PROT_SUPPORT                         LW_STATUS_NO_S4U_PROT_SUPPORT
#define STATUS_CROSSREALM_DELEGATION_FAILURE               LW_STATUS_CROSSREALM_DELEGATION_FAILURE
#define STATUS_REVOCATION_OFFLINE_KDC                      LW_STATUS_REVOCATION_OFFLINE_KDC
#define STATUS_ISSUING_CA_UNTRUSTED_KDC                    LW_STATUS_ISSUING_CA_UNTRUSTED_KDC
#define STATUS_KDC_CERT_EXPIRED                            LW_STATUS_KDC_CERT_EXPIRED
#define STATUS_KDC_CERT_REVOKED                            LW_STATUS_KDC_CERT_REVOKED
#define STATUS_PARAMETER_QUOTA_EXCEEDED                    LW_STATUS_PARAMETER_QUOTA_EXCEEDED
#define STATUS_HIBERNATION_FAILURE                         LW_STATUS_HIBERNATION_FAILURE
#define STATUS_DELAY_LOAD_FAILED                           LW_STATUS_DELAY_LOAD_FAILED
#define STATUS_AUTHENTICATION_FIREWALL_FAILED              LW_STATUS_AUTHENTICATION_FIREWALL_FAILED
#define STATUS_VDM_DISALLOWED                              LW_STATUS_VDM_DISALLOWED
#define STATUS_HUNG_DISPLAY_DRIVER_THREAD                  LW_STATUS_HUNG_DISPLAY_DRIVER_THREAD
#define STATUS_INSUFFICIENT_RESOURCE_FOR_SPECIFIED_SHARED_SECTION_SIZE LW_STATUS_INSUFFICIENT_RESOURCE_FOR_SPECIFIED_SHARED_SECTION_SIZE
#define STATUS_INVALID_CRUNTIME_PARAMETER                  LW_STATUS_INVALID_CRUNTIME_PARAMETER
#define STATUS_NTLM_BLOCKED                                LW_STATUS_NTLM_BLOCKED
#define STATUS_ASSERTION_FAILURE                           LW_STATUS_ASSERTION_FAILURE
#define STATUS_VERIFIER_STOP                               LW_STATUS_VERIFIER_STOP
#define STATUS_CALLBACK_POP_STACK                          LW_STATUS_CALLBACK_POP_STACK
#define STATUS_INCOMPATIBLE_DRIVER_BLOCKED                 LW_STATUS_INCOMPATIBLE_DRIVER_BLOCKED
#define STATUS_HIVE_UNLOADED                               LW_STATUS_HIVE_UNLOADED
#define STATUS_COMPRESSION_DISABLED                        LW_STATUS_COMPRESSION_DISABLED
#define STATUS_FILE_SYSTEM_LIMITATION                      LW_STATUS_FILE_SYSTEM_LIMITATION
#define STATUS_INVALID_IMAGE_HASH                          LW_STATUS_INVALID_IMAGE_HASH
#define STATUS_NOT_CAPABLE                                 LW_STATUS_NOT_CAPABLE
#define STATUS_REQUEST_OUT_OF_SEQUENCE                     LW_STATUS_REQUEST_OUT_OF_SEQUENCE
#define STATUS_IMPLEMENTATION_LIMIT                        LW_STATUS_IMPLEMENTATION_LIMIT
#define STATUS_ELEVATION_REQUIRED                          LW_STATUS_ELEVATION_REQUIRED
#define STATUS_BEYOND_VDL                                  LW_STATUS_BEYOND_VDL
#define STATUS_ENCOUNTERED_WRITE_IN_PROGRESS               LW_STATUS_ENCOUNTERED_WRITE_IN_PROGRESS
#define STATUS_PTE_CHANGED                                 LW_STATUS_PTE_CHANGED
#define STATUS_PURGE_FAILED                                LW_STATUS_PURGE_FAILED
#define STATUS_CRED_REQUIRES_CONFIRMATION                  LW_STATUS_CRED_REQUIRES_CONFIRMATION
#define STATUS_CS_ENCRYPTION_INVALID_SERVER_RESPONSE       LW_STATUS_CS_ENCRYPTION_INVALID_SERVER_RESPONSE
#define STATUS_CS_ENCRYPTION_UNSUPPORTED_SERVER            LW_STATUS_CS_ENCRYPTION_UNSUPPORTED_SERVER
#define STATUS_CS_ENCRYPTION_EXISTING_ENCRYPTED_FILE       LW_STATUS_CS_ENCRYPTION_EXISTING_ENCRYPTED_FILE
#define STATUS_CS_ENCRYPTION_NEW_ENCRYPTED_FILE            LW_STATUS_CS_ENCRYPTION_NEW_ENCRYPTED_FILE
#define STATUS_CS_ENCRYPTION_FILE_NOT_CSE                  LW_STATUS_CS_ENCRYPTION_FILE_NOT_CSE
#define STATUS_INVALID_LABEL                               LW_STATUS_INVALID_LABEL
#define STATUS_DRIVER_PROCESS_TERMINATED                   LW_STATUS_DRIVER_PROCESS_TERMINATED
#define STATUS_AMBIGUOUS_SYSTEM_DEVICE                     LW_STATUS_AMBIGUOUS_SYSTEM_DEVICE
#define STATUS_SYSTEM_DEVICE_NOT_FOUND                     LW_STATUS_SYSTEM_DEVICE_NOT_FOUND
#define STATUS_RESTART_BOOT_APPLICATION                    LW_STATUS_RESTART_BOOT_APPLICATION
#define STATUS_INVALID_TASK_NAME                           LW_STATUS_INVALID_TASK_NAME
#define STATUS_INVALID_TASK_INDEX                          LW_STATUS_INVALID_TASK_INDEX
#define STATUS_THREAD_ALREADY_IN_TASK                      LW_STATUS_THREAD_ALREADY_IN_TASK
#define STATUS_CALLBACK_BYPASS                             LW_STATUS_CALLBACK_BYPASS
#define STATUS_PORT_CLOSED                                 LW_STATUS_PORT_CLOSED
#define STATUS_MESSAGE_LOST                                LW_STATUS_MESSAGE_LOST
#define STATUS_INVALID_MESSAGE                             LW_STATUS_INVALID_MESSAGE
#define STATUS_REQUEST_CANCELED                            LW_STATUS_REQUEST_CANCELED
#define STATUS_RECURSIVE_DISPATCH                          LW_STATUS_RECURSIVE_DISPATCH
#define STATUS_LPC_RECEIVE_BUFFER_EXPECTED                 LW_STATUS_LPC_RECEIVE_BUFFER_EXPECTED
#define STATUS_LPC_INVALID_CONNECTION_USAGE                LW_STATUS_LPC_INVALID_CONNECTION_USAGE
#define STATUS_LPC_REQUESTS_NOT_ALLOWED                    LW_STATUS_LPC_REQUESTS_NOT_ALLOWED
#define STATUS_RESOURCE_IN_USE                             LW_STATUS_RESOURCE_IN_USE
#define STATUS_HARDWARE_MEMORY_ERROR                       LW_STATUS_HARDWARE_MEMORY_ERROR
#define STATUS_THREADPOOL_HANDLE_EXCEPTION                 LW_STATUS_THREADPOOL_HANDLE_EXCEPTION
#define STATUS_THREADPOOL_SET_EVENT_ON_COMPLETION_FAILED   LW_STATUS_THREADPOOL_SET_EVENT_ON_COMPLETION_FAILED
#define STATUS_THREADPOOL_RELEASE_SEMAPHORE_ON_COMPLETION_FAILED LW_STATUS_THREADPOOL_RELEASE_SEMAPHORE_ON_COMPLETION_FAILED
#define STATUS_THREADPOOL_RELEASE_MUTEX_ON_COMPLETION_FAILED LW_STATUS_THREADPOOL_RELEASE_MUTEX_ON_COMPLETION_FAILED
#define STATUS_THREADPOOL_FREE_LIBRARY_ON_COMPLETION_FAILED LW_STATUS_THREADPOOL_FREE_LIBRARY_ON_COMPLETION_FAILED
#define STATUS_THREADPOOL_RELEASED_DURING_OPERATION        LW_STATUS_THREADPOOL_RELEASED_DURING_OPERATION
#define STATUS_CALLBACK_RETURNED_WHILE_IMPERSONATING       LW_STATUS_CALLBACK_RETURNED_WHILE_IMPERSONATING
#define STATUS_APC_RETURNED_WHILE_IMPERSONATING            LW_STATUS_APC_RETURNED_WHILE_IMPERSONATING
#define STATUS_PROCESS_IS_PROTECTED                        LW_STATUS_PROCESS_IS_PROTECTED
#define STATUS_MCA_EXCEPTION                               LW_STATUS_MCA_EXCEPTION
#define STATUS_CERTIFICATE_MAPPING_NOT_UNIQUE              LW_STATUS_CERTIFICATE_MAPPING_NOT_UNIQUE
#define STATUS_SYMLINK_CLASS_DISABLED                      LW_STATUS_SYMLINK_CLASS_DISABLED
#define STATUS_INVALID_IDN_NORMALIZATION                   LW_STATUS_INVALID_IDN_NORMALIZATION
#define STATUS_NO_UNICODE_TRANSLATION                      LW_STATUS_NO_UNICODE_TRANSLATION
#define STATUS_ALREADY_REGISTERED                          LW_STATUS_ALREADY_REGISTERED
#define STATUS_CONTEXT_MISMATCH                            LW_STATUS_CONTEXT_MISMATCH
#define STATUS_PORT_ALREADY_HAS_COMPLETION_LIST            LW_STATUS_PORT_ALREADY_HAS_COMPLETION_LIST
#define STATUS_CALLBACK_RETURNED_THREAD_PRIORITY           LW_STATUS_CALLBACK_RETURNED_THREAD_PRIORITY
#define STATUS_INVALID_THREAD                              LW_STATUS_INVALID_THREAD
#define STATUS_CALLBACK_RETURNED_TRANSACTION               LW_STATUS_CALLBACK_RETURNED_TRANSACTION
#define STATUS_CALLBACK_RETURNED_LDR_LOCK                  LW_STATUS_CALLBACK_RETURNED_LDR_LOCK
#define STATUS_CALLBACK_RETURNED_LANG                      LW_STATUS_CALLBACK_RETURNED_LANG
#define STATUS_CALLBACK_RETURNED_PRI_BACK                  LW_STATUS_CALLBACK_RETURNED_PRI_BACK
#define STATUS_CALLBACK_RETURNED_THREAD_AFFINITY           LW_STATUS_CALLBACK_RETURNED_THREAD_AFFINITY

#endif /* LW_STRICT_NAMESPACE */

#ifndef _DCE_IDL_
#include <lw/attrs.h>

LW_BEGIN_EXTERN_C

int
LwNtStatusToErrno(
    LW_IN LW_NTSTATUS NtStatus
    );

LW_WINERROR
LwNtStatusToWin32Error(
    LW_IN LW_NTSTATUS NtStatus
    );

LW_PCSTR
LwNtStatusToName(
    LW_IN LW_NTSTATUS NtStatus
    );

LW_PCSTR
LwNtStatusToDescription(
    LW_IN LW_NTSTATUS NtStatus
    );

#ifndef LW_STRICT_NAMESPACE
#define NtStatusToErrno(NtStatus)        LwNtStatusToErrno(NtStatus)
#define NtStatusToWin32Error(NtStatus)   LwNtStatusToWin32Error(NtStatus)
#define NtStatusToName(NtStatus)         LwNtStatusToName(NtStatus)
#define NtStatusToDescription(NtStatus)  LwNtStatusToDescription(NtStatus)
#endif /* LW_STRICT_NAMESPACE */

LW_END_EXTERN_C

#endif /* _DCE_IDL_ */

#endif
