/*

GObject wrapper for the gwibber service

Copyright 2010 Canonical Ltd.

Authors:
    David Barth <david.barth@canonical.com>

This program is free software: you can redistribute it and/or modify it 
under the terms of the GNU General Public License version 3, as published 
by the Free Software Foundation.

This program is distributed in the hope that it will be useful, but 
WITHOUT ANY WARRANTY; without even the implied warranties of 
MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along 
with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "me-service-gwibber.h"

#include <glib.h>

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>

#define DBUS_SERVICE_ADDRESS            "com.Gwibber.Service"
#define DBUS_SERVICE_SERVICE_OBJECT     "/com/gwibber/Service"
#define DBUS_SERVICE_SERVICE_INTERFACE  "com.Gwibber.Service"

#define DBUS_SERVICE_ACCOUNTS_OBJECT    "/com/gwibber/Accounts"
#define DBUS_SERVICE_ACCOUNTS_INTERFACE "com.Gwibber.Accounts"

/* gobject infrastructure */

typedef struct _GwibberServicePrivate GwibberServicePrivate;
struct _GwibberServicePrivate {
	DBusGProxy * dbus_proxy;
	DBusGProxy * service_proxy;
	DBusGProxy * accounts_proxy;
	int status;
	gboolean has_configured_accounts;
};

/* Signals */
enum {
	STATUS_CHANGED,
	LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0 };

/* Prototypes */
static void gwibber_service_class_init (GwibberServiceClass *klass);
static void gwibber_service_init       (GwibberService *self);
static void gwibber_service_dispose    (GObject *object);
static void gwibber_service_finalize   (GObject *object);

static void dbus_namechange (DBusGProxy * proxy, const gchar * name, const gchar * prev, const gchar * new, GwibberService * self);
static void gwibber_exists_cb (DBusGProxy * proxy, gboolean exists, GError * error, gpointer userdata);
static void setup_service_proxies (GwibberService *self);
static void query_account_manager (GwibberService *self);

G_DEFINE_TYPE (GwibberService, gwibber_service, G_TYPE_OBJECT);

#define GWIBBER_SERVICE_GET_PRIVATE(o) \
(G_TYPE_INSTANCE_GET_PRIVATE ((o), GWIBBER_SERVICE_TYPE, GwibberServicePrivate))

static void
gwibber_service_class_init (GwibberServiceClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	g_type_class_add_private (klass, sizeof (GwibberServicePrivate));

	object_class->dispose = gwibber_service_dispose;
	object_class->finalize = gwibber_service_finalize;

	signals[STATUS_CHANGED]  = g_signal_new(GWIBBER_SERVICE_SIGNAL_STATUS_CHANGED,
											G_TYPE_FROM_CLASS(klass),
											G_SIGNAL_RUN_LAST,
											G_STRUCT_OFFSET(GwibberServiceClass, status_changed),
											NULL, NULL,
											g_cclosure_marshal_VOID__UINT,
											G_TYPE_NONE, 1, G_TYPE_UINT);
	return;
}

static void
gwibber_service_init (GwibberService *self)
{
	GwibberServicePrivate * priv = GWIBBER_SERVICE_GET_PRIVATE(self);

	priv->dbus_proxy = NULL;
	priv->service_proxy = NULL;
	priv->accounts_proxy = NULL;
	priv->status = GWIBBER_SERVICE_STATUS_NOT_RUNNING;
	priv->has_configured_accounts = FALSE;

	DBusGConnection * bus = dbus_g_bus_get(DBUS_BUS_SESSION, NULL);
	g_return_if_fail(bus != NULL); /* Can't do anymore DBus stuff without this,
	                                  all non-DBus stuff should be done */

	GError * error = NULL;

	/* Set up the dbus Proxy */
	priv->dbus_proxy = dbus_g_proxy_new_for_name_owner (bus,
	                                                    DBUS_SERVICE_DBUS,
	                                                    DBUS_PATH_DBUS,
	                                                    DBUS_INTERFACE_DBUS,
	                                                    &error);
	if (error != NULL) {
		g_warning("Unable to connect to DBus events: %s", error->message);
		g_error_free(error);
		return;
	}

	/* Configure the name owner changing */
	dbus_g_proxy_add_signal(priv->dbus_proxy, "NameOwnerChanged",
	                        G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING,
							G_TYPE_INVALID);
	dbus_g_proxy_connect_signal(priv->dbus_proxy, "NameOwnerChanged",
	                        G_CALLBACK(dbus_namechange),
	                        self, NULL);

	org_freedesktop_DBus_name_has_owner_async(priv->dbus_proxy, DBUS_SERVICE_ADDRESS, gwibber_exists_cb, self);

	return;
}

static void
gwibber_service_dispose (GObject *object)
{
	GwibberServicePrivate * priv = GWIBBER_SERVICE_GET_PRIVATE(object);

	if (priv->dbus_proxy != NULL) {
		g_object_unref(priv->dbus_proxy);
		priv->dbus_proxy = NULL;
	}

	if (priv->service_proxy != NULL) {
		g_object_unref(priv->service_proxy);
		priv->service_proxy = NULL;
	}

	if (priv->accounts_proxy != NULL) {
		g_object_unref(priv->accounts_proxy);
		priv->accounts_proxy = NULL;
	}

	G_OBJECT_CLASS (gwibber_service_parent_class)->dispose (object);
	return;
}

static void
gwibber_service_finalize (GObject *object)
{
	G_OBJECT_CLASS (gwibber_service_parent_class)->finalize (object);
	return;
}


/* Watch for Gwibber coming on and off the bus. */
static void
dbus_namechange (DBusGProxy * proxy, const gchar * name, const gchar * prev, const gchar * new, GwibberService * self)
{
	g_return_if_fail (IS_GWIBBER_SERVICE (self));

	GwibberServicePrivate * priv = GWIBBER_SERVICE_GET_PRIVATE(self);

	if (prev[0] == '\0' && g_strcmp0(name, DBUS_SERVICE_ADDRESS) == 0) {
		g_debug("Gwibber Service coming online");
		priv->status = GWIBBER_SERVICE_STATUS_RUNNING;
		if (priv->service_proxy == NULL ||
			priv->accounts_proxy == NULL)
			setup_service_proxies (self);
		query_account_manager (self);
	}

	if (new[0] == '\0' && g_strcmp0(name, DBUS_SERVICE_ADDRESS) == 0) {
		g_debug("Gwibber Service going offline");
		priv->status = GWIBBER_SERVICE_STATUS_NOT_RUNNING;
		g_signal_emit (G_OBJECT (self), GWIBBER_SERVICE_SIGNAL_STATUS_CHANGED_ID, 0, priv->status, TRUE);
		if (priv->service_proxy != NULL) {
			g_object_unref(priv->service_proxy);
			priv->service_proxy = NULL;
		}
		if (priv->accounts_proxy != NULL) {
			g_object_unref(priv->accounts_proxy);
			priv->accounts_proxy = NULL;
		}
	}

	return;
}

static void
get_accounts_response (DBusGProxy * proxy, DBusGProxyCall * call, gpointer data)
{
	GError *error = NULL;
	gchar  *accounts_string = NULL;

	dbus_g_proxy_end_call (proxy, call, &error,
						   G_TYPE_STRING, &accounts_string,
						   G_TYPE_INVALID);

	if (error != NULL) {
		g_warning ("GetAccounts: %s", error->message);
		g_error_free (error);
		g_free (accounts_string);
		return;
	}

	/* don't print the accounts string, it contains passwords */
	/* g_debug ("GetAccounts: %s", accounts_string); */

	g_return_if_fail (IS_GWIBBER_SERVICE (data));
	GwibberServicePrivate * priv = GWIBBER_SERVICE_GET_PRIVATE(data);
	g_return_if_fail (priv != NULL);
	/* Note: not free'ing accounts_string here, but g_return_if_fail are
	   meant to catch design errors, not runtime ones */

	priv->status = GWIBBER_SERVICE_STATUS_RUNNING;
	priv->has_configured_accounts = g_strrstr (accounts_string, " \"send_enabled\": true,") ? TRUE : FALSE;

	/* trigger a status update */
	g_signal_emit (G_OBJECT (data),
				   GWIBBER_SERVICE_SIGNAL_STATUS_CHANGED_ID,
				   0, priv->status, TRUE);

	g_free (accounts_string);

	return;
}

static void
query_account_manager (GwibberService *self)
{
	g_return_if_fail (IS_GWIBBER_SERVICE (self));

	GwibberServicePrivate * priv = GWIBBER_SERVICE_GET_PRIVATE(self);

	if (priv->service_proxy == NULL) {
		g_warning ("no service_proxy, can't query for accounts");
		return;
	}

	dbus_g_proxy_begin_call(priv->service_proxy,
	                        "GetAccounts",
	                        get_accounts_response,
	                        self,
	                        NULL,
							G_TYPE_INVALID);
}

static void
accounts_changed (DBusGProxy *proxy, const gchar *id, GwibberService *self)
{
	g_return_if_fail (IS_GWIBBER_SERVICE (self));

	g_debug ("accounts_changed");

	query_account_manager (self);

	return;
}

static void
setup_service_proxies (GwibberService *self)
{
	g_return_if_fail (IS_GWIBBER_SERVICE (self));

	GwibberServicePrivate * priv = GWIBBER_SERVICE_GET_PRIVATE(self);

	DBusGConnection * bus = dbus_g_bus_get(DBUS_BUS_SESSION, NULL);
	g_return_if_fail(bus != NULL); /* Can't do anymore DBus stuff without this,
	                                  all non-DBus stuff should be done */

	if (priv->service_proxy == NULL) {
		priv->service_proxy = dbus_g_proxy_new_for_name(
			bus,
			DBUS_SERVICE_ADDRESS,
			DBUS_SERVICE_SERVICE_OBJECT,
			DBUS_SERVICE_SERVICE_INTERFACE);
		if (priv->service_proxy == NULL) {
			g_warning ("can't setup service_proxy");
			return;
		}
	}

	if (priv->accounts_proxy == NULL) {
		priv->accounts_proxy =
			dbus_g_proxy_new_for_name(bus,
									  DBUS_SERVICE_ADDRESS,
									  DBUS_SERVICE_ACCOUNTS_OBJECT,
									  DBUS_SERVICE_ACCOUNTS_INTERFACE);
		if (priv->accounts_proxy == NULL) {
			g_warning ("can't setup accounts_proxy");
			return;
		}
	}

	/* Monitor signals about account changes */
	dbus_g_proxy_add_signal(priv->accounts_proxy, "AccountChanged",
	                        G_TYPE_STRING,
							G_TYPE_INVALID);
	dbus_g_proxy_connect_signal(priv->accounts_proxy, "AccountChanged",
	                        G_CALLBACK(accounts_changed),
	                        self, NULL);
	dbus_g_proxy_add_signal(priv->accounts_proxy, "AccountDeleted",
	                        G_TYPE_STRING,
							G_TYPE_INVALID);
	dbus_g_proxy_connect_signal(priv->accounts_proxy, "AccountDeleted",
	                        G_CALLBACK(accounts_changed),
	                        self, NULL);

}

static void
gwibber_exists_cb (DBusGProxy * proxy, gboolean exists, GError * error, gpointer userdata)
{
	if (error) {
		g_warning ("Unable to check if Gwibber is running: %s", error->message);
		return;
	}

	if (exists) {
		setup_service_proxies (GWIBBER_SERVICE (userdata));
		query_account_manager (GWIBBER_SERVICE (userdata));
	}

	return;
}

GwibberService * gwibber_service_new (void){
	return GWIBBER_SERVICE (g_object_new (GWIBBER_SERVICE_TYPE, NULL));
}

static void
send_message_response (DBusGProxy * proxy, DBusGProxyCall * call, gpointer data)
{
	g_debug ("send_message_response");

	return;
}

void
gwibber_service_send (GwibberService *self, const gchar *msg)
{
	g_return_if_fail (IS_GWIBBER_SERVICE (self));

	GwibberServicePrivate * priv = GWIBBER_SERVICE_GET_PRIVATE(self);

	GValue value = {0};
	g_value_init(&value, G_TYPE_STRING);
	g_value_set_string(&value, msg);

	g_debug ("gwibber_send: %s\n", msg);

	if (priv->service_proxy == NULL) {
		setup_service_proxies (self);
		if (priv->service_proxy == NULL) {
			g_warning ("can't setup service_proxy");
			return;
		}
	}

	dbus_g_proxy_begin_call(priv->service_proxy,
	                        "SendMessage",
	                        send_message_response,
	                        NULL,
	                        NULL,
	                        G_TYPE_VALUE,
							&value,
							G_TYPE_INVALID);

	g_value_unset(&value);

	return;
}

GwibberService *
gwibber_service_get (void)
{
  static GwibberService *singleton = NULL;

  if (! singleton) {
	  singleton = gwibber_service_new ();
  }	

  return singleton;
}

gboolean
gwibber_service_has_configured_accounts (GwibberService *self)
{
	g_return_val_if_fail (IS_GWIBBER_SERVICE (self), FALSE);

	GwibberServicePrivate * priv = GWIBBER_SERVICE_GET_PRIVATE(self);

	return priv->has_configured_accounts;
}
