/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 2; tab-width: 2 -*- */
/*
  A small wrapper utility to load indicators and put them as menu items
  into the gnome-panel using it's applet interface.

  Copyright 2009 Canonical Ltd.

  Authors:
  Ted Gould <ted@canonical.com>

  This program is free software: you can redistribute it and/or modify it
  under the terms of the GNU General Public License version 3, as published
  by the Free Software Foundation.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranties of
  MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
  PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along
  with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <config.h>

#include <sys/types.h>
#include <pwd.h>
#include <unistd.h>

#include <glib/gi18n.h>

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>

#include <gio/gio.h>

#include <gconf/gconf-client.h>

#include <libdbusmenu-glib/client.h>
#include <libdbusmenu-glib/server.h>
#include <libdbusmenu-glib/menuitem.h>

#include <libindicator/indicator-service.h>

#include "dbus-shared-names.h"

#include "me-service-dbus.h"

#include "status-provider.h"
#include "status-provider-mc5.h"
#include "status-provider-pidgin.h"
#include "status-provider-emesene.h"
#include "status-provider-telepathy.h"

#include "me-service-gwibber.h"
#include "entry-menu-item.h"

typedef StatusProvider * (*newfunc) (void);
#define STATUS_PROVIDER_CNT   4
static newfunc status_provider_newfuncs[STATUS_PROVIDER_CNT] = {
	status_provider_mc5_new,
	status_provider_pidgin_new,
	status_provider_emesene_new,
	status_provider_telepathy_new
};
static StatusProvider * status_providers[STATUS_PROVIDER_CNT] = { 0 };

static const gchar * status_strings [STATUS_PROVIDER_STATUS_LAST] = {
  /* STATUS_PROVIDER_STATUS_ONLINE,    */ N_("Available"),
  /* STATUS_PROVIDER_STATUS_AWAY,      */ N_("Away"),
  /* STATUS_PROVIDER_STATUS_DND        */ N_("Busy"),
  /* STATUS_PROVIDER_STATUS_INVISIBLE  */ N_("Invisible"),
  /* STATUS_PROVIDER_STATUS_OFFLINE,   */ N_("Offline"),
  /* STATUS_PROVIDER_STATUS_DISCONNECTED*/ N_("Offline")
};

static const gchar * status_icons[STATUS_PROVIDER_STATUS_LAST] = {
  /* STATUS_PROVIDER_STATUS_ONLINE, */     "user-available-panel",
  /* STATUS_PROVIDER_STATUS_AWAY, */       "user-away-panel",
  /* STATUS_PROVIDER_STATUS_DND, */        "user-busy-panel",
  /* STATUS_PROVIDER_STATUS_INVISIBLE, */  "user-invisible-panel",
  /* STATUS_PROVIDER_STATUS_OFFLINE */     "user-offline-panel",
  /* STATUS_PROVIDER_STATUS_DISCONNECTED */"user-offline-panel"
};

static MeGwibberService * me_gwibber_service = NULL;

static DbusmenuMenuitem * root_menuitem = NULL;
static DbusmenuMenuitem * status_menuitems[STATUS_PROVIDER_STATUS_LAST] = {0};
static GMainLoop * mainloop = NULL;
static StatusServiceDbus * dbus_interface = NULL;
static StatusProviderStatus global_status = STATUS_PROVIDER_STATUS_DISCONNECTED;
static GFileMonitor *avatar_monitor = NULL;
static DbusmenuMenuitem *broadcast_field = NULL;
static DbusmenuMenuitem * useritem = NULL;

static void
status_update (void) {
	StatusProviderStatus oldglobal = global_status;
	global_status = STATUS_PROVIDER_STATUS_DISCONNECTED;
  gboolean indeterminate = FALSE;

	/* Ask everyone what they think the status should be, if
	   they're more connected, up the global level */
  StatusProviderStatus i;
	for (i = 0; i < STATUS_PROVIDER_CNT; i++) {
    if (status_providers[i] == NULL) continue;
		StatusProviderStatus localstatus = status_provider_get_status(status_providers[i]);
    g_debug ("provider[%d]: %s", i, status_strings[localstatus]);

    if (localstatus > STATUS_PROVIDER_STATUS_OFFLINE)
      /* offline and disconnected is similar for some providers
         so it's not meaningful to determine the 'indeterminate' status
      */
      continue;

		if (localstatus != global_status) {
      if (global_status < STATUS_PROVIDER_STATUS_OFFLINE)
        /* at least one provider was maintaining a status better than 'offline'
           and there's now another one with something different */
        indeterminate = TRUE;
			global_status = localstatus;
		}
	}

  /* Configure the icon on the panel */
  status_service_dbus_set_status(dbus_interface, indeterminate ? "user-indeterminate" : status_icons[global_status]);

  g_debug("Global status changed to: %s", indeterminate ? "indeterminate" : status_strings[global_status]);

  /* If we're now disconnected, make setting the statuses
     insensitive. */
  if (global_status == STATUS_PROVIDER_STATUS_DISCONNECTED) {
    StatusProviderStatus i;
    for (i = STATUS_PROVIDER_STATUS_ONLINE;
         i < STATUS_PROVIDER_STATUS_LAST; i++) {
      if (status_menuitems[i] == NULL) continue;
      dbusmenu_menuitem_property_set_bool(status_menuitems[i],
                                          DBUSMENU_MENUITEM_PROP_ENABLED,
                                          FALSE);
      dbusmenu_menuitem_property_set_int (status_menuitems[i],
                                          DBUSMENU_MENUITEM_PROP_TOGGLE_STATE,
                                          (i != STATUS_PROVIDER_STATUS_OFFLINE)
                                          ? DBUSMENU_MENUITEM_TOGGLE_STATE_UNCHECKED
                                          : DBUSMENU_MENUITEM_TOGGLE_STATE_CHECKED);
    }
    return;
  }

  /* Ensure items are sensititive if they were previoulsy disabled */
  if (oldglobal == STATUS_PROVIDER_STATUS_DISCONNECTED) {
    StatusProviderStatus i;
    for (i = STATUS_PROVIDER_STATUS_ONLINE;
         i < STATUS_PROVIDER_STATUS_LAST; i++) {
      if (status_menuitems[i] == NULL) continue;
      dbusmenu_menuitem_property_set_bool(status_menuitems[i], DBUSMENU_MENUITEM_PROP_ENABLED, TRUE);
    }
  }

  /* add a radio mark to the user status entry */
  for (i = STATUS_PROVIDER_STATUS_ONLINE;
       i < STATUS_PROVIDER_STATUS_LAST; i++) {
    if (status_menuitems[i] == NULL) continue;
    if (indeterminate
        || (i != global_status)) {
      dbusmenu_menuitem_property_set_int (status_menuitems[i],
                                          DBUSMENU_MENUITEM_PROP_TOGGLE_STATE,
                                          DBUSMENU_MENUITEM_TOGGLE_STATE_UNCHECKED);
    } else {
      dbusmenu_menuitem_property_set_int (status_menuitems[i],
                                          DBUSMENU_MENUITEM_PROP_TOGGLE_STATE,
                                          DBUSMENU_MENUITEM_TOGGLE_STATE_CHECKED);
    }
  }

	return;
}

static void
me_gwibber_status_update (MeGwibberService *instance, MeGwibberServiceStatus status, gpointer data) {
  g_return_if_fail (instance != NULL);

  g_debug ("gwibber service status changed to: %d", status);

  if (broadcast_field == NULL) {
    g_warning ("no broadcast field to update");
    return;
  }

  if (! me_gwibber_service_has_configured_accounts (instance)) {
    g_debug ("no configured accounts detected, so hiding the broadcast field");
    dbusmenu_menuitem_property_set_bool (broadcast_field, DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
    return;
  }

  dbusmenu_menuitem_property_set (broadcast_field, DBUSMENU_ENTRY_MENUITEM_PROP_HINT, me_gwibber_service_get_accounts_string (instance));

  if (! dbusmenu_menuitem_property_get_bool (broadcast_field, DBUSMENU_MENUITEM_PROP_VISIBLE)) {
    dbusmenu_menuitem_property_set_bool (broadcast_field, DBUSMENU_MENUITEM_PROP_VISIBLE, TRUE);
  }

  return;
}


static void
status_menu_click (DbusmenuMenuitem * mi, guint timestamp, gpointer data)
{
	StatusProviderStatus status = (StatusProviderStatus)GPOINTER_TO_INT(data);
	g_debug("Setting status: %d", status);
	int i;
	for (i = 0; i < STATUS_PROVIDER_CNT; i++) {
		status_provider_set_status(status_providers[i], status);
	}

	return;
}

static void
spawn_on_activate_cb (DbusmenuMenuitem *mi, guint timestamp, gpointer user_data)
{
	GError * error = NULL;

	if (!g_spawn_command_line_async(user_data, &error)) {
		g_warning("Unable to start %s: %s", (char *)user_data, error->message);
		g_error_free(error);
	}
}

static gboolean
program_is_installed (gchar *program)
{
	gchar *cmd = g_find_program_in_path(program);
	if (cmd != NULL) {
		g_free(cmd);
		return TRUE;
	}

	return FALSE;
}

static void
build_accounts_menuitems (gpointer data)
{
	DbusmenuMenuitem * root = DBUSMENU_MENUITEM(data);
	g_return_if_fail(root != NULL);

  DbusmenuMenuitem *im_accounts_mi = NULL;
  DbusmenuMenuitem *tw_accounts_mi = NULL;
  DbusmenuMenuitem *u1_accounts_mi = NULL;
  gboolean at_least_one = FALSE;

  if (program_is_installed ("empathy-accounts")) {
    im_accounts_mi = dbusmenu_menuitem_new();
    dbusmenu_menuitem_property_set(im_accounts_mi, DBUSMENU_MENUITEM_PROP_LABEL,
                                   _("Chat Accounts..."));
    g_signal_connect(G_OBJECT(im_accounts_mi), DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED,
                     G_CALLBACK(spawn_on_activate_cb), "empathy-accounts");
    at_least_one = TRUE;
  }

  if (program_is_installed ("gwibber-accounts")) {
    tw_accounts_mi = dbusmenu_menuitem_new();
    dbusmenu_menuitem_property_set(tw_accounts_mi, DBUSMENU_MENUITEM_PROP_LABEL,
                                   _("Broadcast Accounts..."));
    g_signal_connect(G_OBJECT(tw_accounts_mi), DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED,
                     G_CALLBACK(spawn_on_activate_cb), "gwibber-accounts");
    at_least_one = TRUE;
  }

  /* Removed ubuntuone-control-panel-gtk from MeMenu (LP: #711233)
  if (program_is_installed ("ubuntuone-control-panel-gtk")) {
    u1_accounts_mi = dbusmenu_menuitem_new();
    dbusmenu_menuitem_property_set(u1_accounts_mi, DBUSMENU_MENUITEM_PROP_LABEL,
                                   _("Ubuntu One..."));
    g_signal_connect(G_OBJECT(u1_accounts_mi), DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED,
                     G_CALLBACK(spawn_on_activate_cb), "ubuntuone-control-panel-gtk");
    at_least_one = TRUE;
  }
  */

  if (at_least_one) {
    DbusmenuMenuitem *separator = dbusmenu_menuitem_new();
    dbusmenu_menuitem_property_set(separator, DBUSMENU_MENUITEM_PROP_TYPE,
                                   DBUSMENU_CLIENT_TYPES_SEPARATOR);
    dbusmenu_menuitem_child_append(root, separator);

    if (im_accounts_mi)
      dbusmenu_menuitem_child_append(root, im_accounts_mi);
    if (tw_accounts_mi)
      dbusmenu_menuitem_child_append(root, tw_accounts_mi);
    if (u1_accounts_mi)
      dbusmenu_menuitem_child_append(root, u1_accounts_mi);
  }

}

static gboolean
build_providers (gpointer data)
{
	int i;
	for (i = 0; i < STATUS_PROVIDER_CNT; i++) {
		status_providers[i] = status_provider_newfuncs[i]();

		if (status_providers[i] != NULL) {
			g_signal_connect(G_OBJECT(status_providers[i]), STATUS_PROVIDER_SIGNAL_STATUS_CHANGED, G_CALLBACK(status_update), NULL);
		}
	}

	status_update();

  me_gwibber_service = me_gwibber_service_get ();
  if (me_gwibber_service != NULL) {
    g_signal_connect (G_OBJECT (me_gwibber_service), ME_GWIBBER_SERVICE_SIGNAL_STATUS_CHANGED, G_CALLBACK (me_gwibber_status_update), NULL);
  }

  return FALSE;
}

#define GCONF_NAMESPACE "/system/indicator/me"
#define GCONF_DISPLAY   "/system/indicator/me/display"

static void
display_mode_changed ()
{
  GConfClient *context = gconf_client_get_default ();
  g_return_if_fail (context != NULL);

  GConfValue *option = gconf_client_get (context, GCONF_DISPLAY, NULL);
  gint value = 1; /* username, by default */

  g_debug ("display_mode_changed");

  if (option != NULL &&
      option->type == GCONF_VALUE_INT)
    value = gconf_value_get_int (option);

  switch (value) {
  case 0: /* anonymous */
    status_service_dbus_set_username (dbus_interface, "");
    break;
  default:
  case 1:
    status_service_dbus_set_username (dbus_interface, g_get_user_name ());
    break;
  case 2:
    status_service_dbus_set_username (dbus_interface, g_get_real_name ());
    break;
  }

  dbusmenu_menuitem_property_set_bool (useritem,
                                       DBUSMENU_MENUITEM_PROP_VISIBLE,
                                       (value == 0) ? FALSE : TRUE);

  g_object_unref (context);

  return;
}


/* disabled for this release */
#if 0
static void
avatar_changed_cb (GFileMonitor * monitor, GFile * file, GFile * other_file, GFileMonitorEvent event_type, gpointer user_data)
{
	g_debug("avatar changed!");

  g_return_if_fail (DBUSMENU_IS_MENUITEM (user_data));

  DbusmenuMenuitem *mi = DBUSMENU_MENUITEM (user_data);

  /* whatever happened, we just update the icon property
     and let the indicator reload the image */
  gchar * path = g_file_get_path (file);
  dbusmenu_menuitem_property_set (mi, DBUSMENU_ABOUT_ME_MENUITEM_PROP_ICON, path);
}

static void
build_avatar_item (DbusmenuMenuitem * root)
{
  useritem = dbusmenu_menuitem_new();
  dbusmenu_menuitem_property_set(useritem, DBUSMENU_ABOUT_ME_MENUITEM_PROP_NAME, g_get_real_name ());
  dbusmenu_menuitem_property_set_bool(useritem, DBUSMENU_MENUITEM_PROP_ENABLED, FALSE);
  dbusmenu_menuitem_property_set_bool(useritem, DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_property_set(useritem, DBUSMENU_MENUITEM_PROP_TYPE, DBUSMENU_ABOUT_ME_MENUITEM_TYPE);
  dbusmenu_menuitem_child_append(root, useritem);

  /* avatar icon */
	gchar *filename = g_build_filename (g_get_home_dir (), ".face", NULL);
  dbusmenu_menuitem_property_set (useritem, DBUSMENU_ABOUT_ME_MENUITEM_PROP_ICON, filename);
  GFile *file = g_file_new_for_path (filename);
  avatar_monitor = g_file_monitor (file, G_FILE_MONITOR_NONE, NULL, NULL);
  if (avatar_monitor != NULL)
		g_signal_connect (G_OBJECT (avatar_monitor), "changed", G_CALLBACK (avatar_changed_cb), useritem);

  g_free (filename);

	return;
}
#endif

static gboolean
build_menu (gpointer data)
{
	DbusmenuMenuitem * root = DBUSMENU_MENUITEM(data);
	g_return_val_if_fail(DBUSMENU_IS_MENUITEM(root), FALSE);

  /* and receive display mode notifications to update it later */
  GConfClient *context = gconf_client_get_default ();
  if (context != NULL) {
    gconf_client_add_dir (context, GCONF_NAMESPACE,
                          GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);
    gconf_client_notify_add (context, GCONF_DISPLAY,
                             display_mode_changed, NULL, NULL, NULL);
    g_object_unref (context);
  }

  /* disabled for this release */
#if 0
  build_avatar_item(root);
#endif

  broadcast_field = DBUSMENU_MENUITEM (entry_menu_item_new());
  dbusmenu_menuitem_property_set (broadcast_field, DBUSMENU_ENTRY_MENUITEM_PROP_HINT, _("Post message..."));
  dbusmenu_menuitem_property_set_bool (broadcast_field, DBUSMENU_MENUITEM_PROP_ENABLED, TRUE);
  dbusmenu_menuitem_property_set_bool (broadcast_field, DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_child_append(root, broadcast_field);

	StatusProviderStatus i;
	for (i = STATUS_PROVIDER_STATUS_ONLINE; i < STATUS_PROVIDER_STATUS_LAST; i++) {
		if (i == STATUS_PROVIDER_STATUS_DISCONNECTED) {
			/* We don't want an item for the disconnected status.  Users
			   can't set that value through the menu :) */
			continue;
		}

		status_menuitems[i] = dbusmenu_menuitem_new();

		dbusmenu_menuitem_property_set(status_menuitems[i], DBUSMENU_MENUITEM_PROP_LABEL, _(status_strings[i]));
		dbusmenu_menuitem_property_set(status_menuitems[i], DBUSMENU_MENUITEM_PROP_ICON_NAME, status_icons[i]);
    dbusmenu_menuitem_property_set (status_menuitems[i],
                                    DBUSMENU_MENUITEM_PROP_TOGGLE_TYPE,
                                    DBUSMENU_MENUITEM_TOGGLE_RADIO);
    dbusmenu_menuitem_property_set_int (status_menuitems[i],
                                        DBUSMENU_MENUITEM_PROP_TOGGLE_STATE,
                                        DBUSMENU_MENUITEM_TOGGLE_STATE_UNCHECKED);
		if (global_status == STATUS_PROVIDER_STATUS_DISCONNECTED) {
			dbusmenu_menuitem_property_set_bool(status_menuitems[i], DBUSMENU_MENUITEM_PROP_ENABLED, FALSE);
		}
		g_signal_connect(G_OBJECT(status_menuitems[i]), DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED, G_CALLBACK(status_menu_click), GINT_TO_POINTER(i));

		dbusmenu_menuitem_child_append(root, status_menuitems[i]);

		g_debug("Built %s", status_strings[i]);
	}

	build_accounts_menuitems(root);

  /* add a standard "About Me..." menu item at the end of the menu */
	DbusmenuMenuitem *separator = dbusmenu_menuitem_new();
	dbusmenu_menuitem_property_set(separator, DBUSMENU_MENUITEM_PROP_TYPE,
                                 DBUSMENU_CLIENT_TYPES_SEPARATOR);
	dbusmenu_menuitem_child_append(root, separator);

  useritem = dbusmenu_menuitem_new();
  dbusmenu_menuitem_property_set(useritem, DBUSMENU_MENUITEM_PROP_LABEL, _("About Me..."));
  dbusmenu_menuitem_property_set_bool(useritem, DBUSMENU_MENUITEM_PROP_ENABLED, FALSE);
  dbusmenu_menuitem_property_set_bool(useritem, DBUSMENU_MENUITEM_PROP_VISIBLE, FALSE);
  dbusmenu_menuitem_child_append(root, useritem);

  gchar *gam = g_find_program_in_path("gnome-about-me");
  if (gam != NULL) {
    dbusmenu_menuitem_property_set_bool(useritem, DBUSMENU_MENUITEM_PROP_ENABLED, TRUE);
    g_signal_connect(G_OBJECT(useritem),
                     DBUSMENU_MENUITEM_SIGNAL_ITEM_ACTIVATED,
                     G_CALLBACK(spawn_on_activate_cb), "gnome-about-me");
    g_free(gam);
  }

  /* finally set the menu name and update items visibility according
     to a gconf key
  */
  display_mode_changed ();

	return FALSE;
}

void
service_shutdown (IndicatorService * service, gpointer user_data)
{
	g_debug("Service shutting down");

  if (avatar_monitor != NULL)
    g_object_unref (avatar_monitor);

	g_main_loop_quit(mainloop);
	return;
}

int
main (int argc, char ** argv)
{
  g_type_init();

	/* Setting up i18n and gettext.  Apparently, we need
	   all of these. */
	setlocale (LC_ALL, "");
	bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
	textdomain (GETTEXT_PACKAGE);

	IndicatorService * service = indicator_service_new_version(INDICATOR_ME_DBUS_NAME, INDICATOR_ME_DBUS_VERSION);
	g_signal_connect(G_OBJECT(service), INDICATOR_SERVICE_SIGNAL_SHUTDOWN, G_CALLBACK(service_shutdown), NULL);

	g_idle_add(build_providers, NULL);

  root_menuitem = dbusmenu_menuitem_new();
  DbusmenuServer * server = dbusmenu_server_new(INDICATOR_ME_DBUS_OBJECT);
  dbusmenu_server_set_root(server, root_menuitem);

	g_idle_add(build_menu, root_menuitem);

	dbus_interface = g_object_new(STATUS_SERVICE_DBUS_TYPE, NULL);

  mainloop = g_main_loop_new(NULL, FALSE);
  g_main_loop_run(mainloop);

  return 0;
}

