/*
 * Copyright (C) 2010 Neil Jagdish Patel
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Ken VanDine <ken@vandine.org>
 */

using Gwibber;

namespace GwibberGtk
{
  public class UserView : Gtk.Box
  {
    private Gtk.Adjustment     adjustment;
    private Gtk.VScrollbar     scrollbar;
    private TileBox            view_box;
    private int _position = 0;
    private int tiles_visible = 0;
    private uint refresh_id = 0;
    private bool _showing = false;
    public Gwibber.Streams streams;
    private List<Gee.HashMap> streams_list;


    public int position {
      get { return _position; }
      set {
        if (_position != value)
          {
            _position = value;
            _position = _position.clamp (0, (int)streams_list.length ());

            if (_position != adjustment.get_value ())
            {
              adjustment.set_value (_position);
              refresh ();
            }
          }
      }
    }

    public bool showing {
      get { return _showing; }
      set { 
        if (value != _showing)
        {
          _showing = value; 
        }
        do_refresh ();
      }
    }

    private SList<UserViewTile> tiles;

    private int last_width = 0;
    private int last_height = 0;

    public UserView ()
    {
      Object (homogeneous:false, spacing:0);
    }

    ~UserView ()
    {
    }

    construct
    {
      streams = new Gwibber.Streams ();

      adjustment = new Gtk.Adjustment (0, 0, 1, 1, 1, 1);
      adjustment.notify["value"].connect (()=> {
        double v = adjustment.get_value ();
        if ((double)position != v)
        {
          if ((double)position < v && v < (position + 1))
            position += 1;
          else if ((double)position > v && v > (position - 1))
            position -= 1;
          else
            position = (int)adjustment.get_value ();
        }
        refresh ();
      });

      view_box = new TileBox ();
      pack_start (view_box, true, true, 0);

      view_box.show ();

      scrollbar = new Gtk.VScrollbar (adjustment);
      pack_start (scrollbar, false, false, 0);   
      
      set_events(Gdk.EventMask.KEY_PRESS_MASK | Gdk.EventMask.SCROLL_MASK);
      scroll_event.connect ((source, event) => {
        Gdk.ScrollDirection direction = event.direction;
        if (event.direction == Gdk.ScrollDirection.UP)
        {
          if (position > 0)
          {
            position -= 1;
          }
        }
        if (event.direction == Gdk.ScrollDirection.DOWN)
        {
          if (position < (adjustment.upper - tiles_visible))
          {
            position += 1;
          }
        }
        return true;
      });

      key_press_event.connect ((source, event) => {
        var key = event.keyval;

        switch (event.keyval)
        {
          case 0xff55: // *_Page_Up
          case 0xff9a:
            position -= tiles_visible;
            break;
          case 0xff56: // *_Page_Down
          case 0xff9b:
            position += tiles_visible;
            break;
          case 0xff54: // *_Down
          case 0xff99:
            position += 1;
            break;
          case 0xff52: // *_Up
          case 0xff97:
            position -= 1;
            break;
          default:
            return false;
        }
        return true;
      });

      for (int i = 0; i < 10; i++)
      {
        var tile = new UserViewTile (streams);
        tile.show_transient.connect((id, name) => {
          show_transient (id, name);
          });
        tile.show_all ();
        view_box.pack_start (tile, false, false, 0);
        tiles.append (tile);
      }

      view_box.size_allocate.connect_after ((a) => {
        if (last_width != a.width || last_height != a.height)
        {
          last_width = a.width;
          last_height = a.height;
          view_box.last_height = last_height;
          refresh ();
        }
      });

    streams.created.connect((id) => {
      var stream_map = streams.lookup (id);
      var name = stream_map["name"];
      refresh ();
      show_transient (id, name);
    });

    streams.updated.connect((id) => {
      debug ("updated stream %s", id);
      var stream_map = streams.lookup (id);
      var name = stream_map["name"];
      refresh ();
      show_transient (id, name);
    });

    streams.deleted.connect((id) => {
      refresh ();
    });

    refresh ();
    }

    public override void get_preferred_height_for_width (int width,
                                                         out int mh,
                                                         out int nh)
    {
      mh = nh = last_height;
    }

    public override void get_preferred_height (out int mh, out int nh)
    {
      mh = nh = last_height;
    }


    [Signal (action=true)]
    public virtual signal void show_transient (string id, string name)
    {
    }

    /*
     * This is a wrapper method to ensure we don't get DOS'd by row-added signals
     */
    private void refresh ()
    {
      if (refresh_id == 0)
      {
        refresh_id = Idle.add (() => {
          do_refresh ();
          refresh_id = 0;
          return false;
        });
      }
    }

    /*
     * This refreshes the view with the latest model and position
     */
    private void do_refresh ()
    {
      int i = 0;

      fill_up_remaining_space ();
      streams_list = streams.list ();
      var n_rows = streams_list.length ();

      foreach (UserViewTile tile in tiles)
      {
        if ((streams_list.nth_data (position + i) != null) && (position + i < (int)n_rows))
        {
          tile.set_details (streams_list.nth_data(position + i));
          tiles_visible = i + 1;
          tile.set_no_show_all (false);
          tile.show_all ();
        }
        else
        {
            tile.set_no_show_all (true);
            tile.reset ();
        }
        i++;
      }

      Timeout.add (10, ()=>{
        queue_draw ();
        adjustment.set_upper ((double)n_rows);
        return false;
      });
    }

    private void fill_up_remaining_space ()
    {
      Gtk.Allocation alloc;
      int overall_height = 0;
      bool one_was_hidden = false;
      var n_rows = streams_list.length ();

      view_box.get_allocation (out alloc);

      foreach (UserViewTile tile in tiles)
      {
        Gtk.Allocation a;

        if (tile.get_visible ())
        {
            tile.get_allocation (out a);
            overall_height += a.height;
        }
        else
        {
          one_was_hidden = true;
        }
      }

      if (alloc.height > overall_height && !one_was_hidden && (int)tiles.length () < (int)n_rows)
      {
          var tile = new UserViewTile (streams);
          tile.show_transient.connect((id, name) => {
            show_transient (id, name);
            });
          tile.show ();
          view_box.pack_start (tile, false, false, 0);
          tiles.append (tile);
          do_refresh ();
      }
      Timeout.add (0,()=>{
        queue_draw ();
        return false;
      });
    }
  }
}
