/*
 * Copyright (C) 2011 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.

 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Ken VanDine <ken@vandine.org>
 */

using Gwibber;

public class UsersItem : TabBarItem
{
  private Gtk.Alignment _align;
  private Gtk.Box box;
  private Gtk.Box _view;
  private GwibberGtk.UserView _user_view;
  private GwibberGtk.StreamView _stream_view;
  public Gwibber.Streams streams { get; construct; }
  private ulong active_hander_id;
  private Gtk.Entry _entry;
  private Gtk.Box _profile_box;
  private Gtk.Box _vbox;
  private GwibberGtk.UserProfileTile _user = null;
  private uint user_profile_handler_id = 0;
  private TabWidgets.AccountSelectorBar _acctbar;

  public UsersItem (Gwibber.Streams streams)
  {
    Object (stream:"user", icon_name:"gwibber-avatar", tooltip:_("Users"), streams:streams);
  }

  construct
  {
    // View
    _view = new Gtk.Box (Gtk.Orientation.HORIZONTAL, 0);
    _view.set_homogeneous (true);
    _view.set_no_show_all (true);
    _view.show ();

    _user_view = new GwibberGtk.UserView ();
    _user_view.show_all ();
    
    _stream_view = new GwibberGtk.StreamView ();
    _stream_view.set_no_show_all (true);
    _stream_view.notify["prepared"].connect (() => {
      _stream_view.set_no_show_all (false);
      _stream_view.show_all ();
    });

    _stream_view.send.connect ((mid, account, sender, action) => {
        entry.text_view.mid = mid;
        if (action == "private" || stream == "private")
        {
          entry.text_view.action = "private";
          entry.private.show ();
        }
        else
          entry.text_view.action = "reply";
        entry.target_bar.selected = account;
        entry.showing = true;
        if (entry.text_view.action != "private")
          entry.text_view.buffer.text = sender + " ";
        entry.text_view.grab_focus ();
        });


    _view.pack_start(_user_view, true, true, 0);
    _view.pack_start(_stream_view, true, true, 0);


    // Toolbar
    _align = new Gtk.Alignment (0.5f, 0.5f, 1.0f, 1.0f);


    _vbox = new Gtk.Box (Gtk.Orientation.VERTICAL, 0);
    _profile_box = new Gtk.Box (Gtk.Orientation.HORIZONTAL, 0);
    _profile_box.set_no_show_all (true);
    box = new Gtk.Box (Gtk.Orientation.HORIZONTAL, 12);
    _vbox.pack_start (box, false, false, 0);
    _vbox.pack_start (_profile_box, false, false, 0);

    _align.add (_vbox);
    _vbox.show ();
    
    _entry = new Gtk.Entry ();
    _entry.set_icon_from_stock (Gtk.EntryIconPosition.SECONDARY, "gtk-clear");

    _entry.icon_press.connect ((pos, event) => {
      if (pos == Gtk.EntryIconPosition.SECONDARY)
      _entry.set_text ("");
    });

    _entry.key_press_event.connect ((t, e)=>
    {
      if (e.keyval == 65307) // Escape
      {
        _entry.set_text ("");
        return true;
      }
      return false;
    });

    Gdk.Color fg_color;
    Gdk.Color.parse ("black", out fg_color);

    _entry.modify_fg (Gtk.StateType.NORMAL, fg_color);

    _entry.activate.connect (() =>
    {
      string name = _entry.get_text ();
      if (name.has_prefix ("@"))
        name = name.replace("@", "");
      _entry.set_text ("");
      debug ("selected is %s", _acctbar.selected);
      streams.create (_acctbar.selected, name, "user_messages");
    });

    //_entry.set_no_show_all (true);
    _entry.show_all ();
    box.pack_start (_entry, false, false, 0);
    _entry.size_allocate.connect_after (()=> { queue_draw (); });

    streams.updated.connect((id) => {
      var stream_map = streams.lookup (id);
      var name = stream_map["name"];
      _user_view.hide ();
      _stream_view.hide ();
      var filtered_model = streams.transient_filter_model (streams.transient_model, id);
      _stream_view.model = filtered_model;
      _stream_view.stream = stream;
      raise (this);
      _stream_view.show_all ();
    });

    _user_view.show_transient.connect((id, name) => {
      _user_view.hide ();
      _stream_view.hide ();
      var filtered_model = streams.transient_filter_model (streams.transient_model, id);
      _stream_view.model = filtered_model;
      
      unowned Dee.ModelIter? iter = null;
      iter = filtered_model.get_first_iter ();


      if (_user != null)
      {
        _user.destroy ();
        _user = null;
      }

      _user = new GwibberGtk.UserProfileTile ();
      _profile_box.pack_start (_user, false, false, 0);

      int x = 0;
      user_profile_handler_id = Timeout.add (1, () => {
        if (x > 10)
        {
          user_profile_handler_id = 0;
          Idle.add (() => {
            _stream_view.hide ();
            _user_view.show ();
            queue_draw ();
	    return false;
          });
          return false;
        }
        var profile = streams.user_profile (id);
        string nick = (string)profile["nick"];
        if (_user != null)
          _user.hide ();
        if (nick == null)
        {
          x++;
          return true;
        }
        if (nick.length > 0) 
        {
          _user.set_details (id,
                      (string)profile["name"], 
                      (string)profile["nick"], 
                      (int)profile["followers"], 
                      (int)profile["friends"], 
                      (int)profile["statuses"], 
                      (bool)profile["following"], 
                      (bool)profile["protected"], 
                      (string)profile["url"], 
                      (string)profile["description"], 
                      (string)profile["account"], 
                      (string)profile["service"], 
                      (string)profile["image"]);
          _user.show ();
          _user.showing = true;
          _profile_box.show ();
          _stream_view.stream = stream;
          raise (this);
          _entry.hide ();
          _acctbar.hide ();
          user_profile_handler_id = 0;
          _user.notify["showing"].connect(() => {
            if (!_user.showing)
            {
              _profile_box.hide ();
              _stream_view.hide ();
              _stream_view.showing = false;
              _user_view.show_all ();
              _user_view.showing = true;
              _entry.show_all ();
              _acctbar.show_all ();
            }
            Idle.add (() => {
              queue_draw ();
	      return false;
            });
          });

          Idle.add (() => {
            _user_view.showing = false;
            _stream_view.show_all ();
            _stream_view.showing = true;
            _user.showing = true;
            _view.show ();
            queue_draw ();
	    return false;
          });
          return false;
        } else
        {
          x++;
          return true;
        }
      });

    });

    _acctbar = new TabWidgets.AccountSelectorBar ();
    box.pack_start (_acctbar, false, false, 0);

    active_hander_id = notify["active"].connect(() => {
      if (active)
      {
        _user_view.showing = true;
        if (active_hander_id > 0)
          disconnect (active_hander_id);
      }
    });

    _stream_view.notify["showing"].connect (() => {
      if (_stream_view.showing)
        _user_view.hide ();
    });

    _user_view.notify["showing"].connect (() => {
      if (_user_view.showing)
        _stream_view.hide ();
    });

    _align.show_all ();
  }

  [Signal (action=true)]
  public virtual signal void raise (TabBarItem item)
  {
  }

  public override Gtk.Widget? get_toolbar ()
  {
    return _align;
  }
  
  public override Gtk.Widget? get_view ()
  {
    return _view;
  }
}
