/*****************************************************************************
 *
 * grail - Gesture Recognition And Instantiation Library
 *
 * Copyright (C) 2010-2011 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 ****************************************************************************/

#ifndef GRAIL_H
#define GRAIL_H

#include <linux/input.h>
#include <grail-bits.h>
#include <grail-types.h>
#include <oif/frame.h>

#ifdef __cplusplus
extern "C" {
#endif

#define GRAIL_VERSION           0x00011000

#define DIM_GRAIL_TYPE		64
#define DIM_GRAIL_TYPE_BYTES	((DIM_GRAIL_TYPE + 7) >> 3)

#define DIM_GRAIL_PROP		32
#define DIM_GRAIL_PROP_BYTES	((DIM_GRAIL_PROP + 7) >> 3)

#define GRAIL_STATUS_BEGIN	0
#define GRAIL_STATUS_UPDATE	1
#define GRAIL_STATUS_END	2

#define GRAIL_EXPECT_DRAG_X	0x0001
#define GRAIL_EXPECT_DRAG_Y	0x0002
#define GRAIL_EXPECT_SCALE	0x0004
#define GRAIL_EXPECT_ANGLE	0x0008
#define GRAIL_EXPECT_MASK	0x000f

typedef float grail_prop_t;			/* gesture properties */
typedef oif_frame_time_t grail_time_t;	/* time in milliseconds */
typedef struct grail *grail_handle;		/* the grail instance handle */

/**
 * struct grail_get_version - get grail library version
 *
 * Report the version of the grail library, which can be different from
 * the value of GRAIL_VERSION in this header file.
 *
 * This function allows for fallback options from major interface
 * extensions within the same ABI version. For the normal cases of ABI
 * agnostic or backwards incompatible changes, this function is not
 * needed.
 */
unsigned int GRAIL_PUBLIC grail_get_version(void);

/**
 * struct grail_coord - coordinate in bounding box units
 * @x: the horizontal position (bbox units)
 * @y: the vertical position (bbox units)
 */
struct grail_coord {
	float x, y;
};

grail_handle GRAIL_PUBLIC grail_new_raw(oif_frame_handle fh,
					unsigned int num_frames,
					void *select,
					unsigned int version,
					unsigned int control_size,
					unsigned int frame_size,
					unsigned int slot_size);

/**
 * grail_new - allocate and initialize a new grail instance
 * @fh: oif frame handle to use
 * @num_frames: number of frames in cyclic buffer
 * @select: client selection callback
 *
 * Initialize the internal grail structures.
 *
 * Returns zero in case of failure.
 */
#define grail_new(fh, num_frames, select)		\
	grail_new_raw(fh, num_frames, select,		\
		      GRAIL_VERSION,			\
		      sizeof(struct grail_control),	\
		      sizeof(struct grail_frame),	\
		      sizeof(struct grail_element))

/**
 * grail_delete - destroy and delete grail instance
 * @ge: grail instance in use
 *
 * Deallocates all internal memory structures.
 */
void GRAIL_PUBLIC grail_delete(grail_handle ge);

/**
 * grail_get_control - get mutable control structure
 * @ge: the grail device in use
 *
 * Return the control struct of the grail instance.
 *
 * The control pointer is ABI agnostic, owned by the grail instance, and
 * has grail scope.
 */
struct grail_control GRAIL_PUBLIC *grail_get_control(grail_handle ge);

/**
 * grail_pump_frame - insert touch frames into grail
 * @ge: the grail device in use
 * @frame: the touch frame to insert
 *
 * Insert a new touch frame into the grail engine. If the frame induces a
 * new gesture frame, a pointer to the frame is returned.
 *
 * The grail frame pointer is ABI agnostic, owned by the grail instance, and
 * has grail scope.
 */
const struct grail_frame GRAIL_PUBLIC *
grail_pump_frame(grail_handle ge, const struct oif_frame *frame);

/**
 * struct grail_client_id - Gesture client information
 * @client: Client id
 * @root: Root window
 * @event: Window to route events to
 * @child: Window the event occured in
 *
 * This struct is treated opaquely, and only has meaning to the gesture
 * client. Details are subject to change.
 */
struct grail_client_id {
	int client;
	int root, event, child;
};

/**
 * struct grail_client_info - Gesture request information
 * @id: Gesture client id
 * @mask: Gestures the client is listening to
 */
struct grail_client_info {
	struct grail_client_id id;
	grail_mask_t mask[DIM_GRAIL_TYPE_BYTES];
};

/**
 * struct grail_control - control parameters of grail
 * @glue_ms: minimum time to hold activation (ms)
 * @bar_drag_x: horizontal distance to activate (surface width fraction)
 * @bar_drag_y: vertical distance to activate (surface height fraction)
 * @bar_scale: minimum scaling to activate (fraction)
 * @bar_angle: minimum angle to activate (radians)
 * @drop_x_ms: horizontal expect timeout (ms)
 * @drop_y_ms: vertical expect timeout (ms)
 * @drop_scale_ms: scaling expect timeout (ms)
 * @drop_angle_ms: rotation expect timeout (ms)
 *
 * The parameters are used to tune the behavior of the gesture recognition.
 *
 * The moveness is a number between zero and one denoting the
 * character of the current transform. Zero means pure rotate and
 * scale, one means pure drag.
 *
 * Later versions of this struct may grow in size, but will remain
 * binary compatible with older versions.
 */
struct grail_control {
	float glue_ms;
	float bar_drag_x;
	float bar_drag_y;
	float bar_scale;
	float bar_angle;
	float drop_x_ms;
	float drop_y_ms;
	float drop_scale_ms;
	float drop_angle_ms;
};

/**
 * struct grail_frame - frame of ongoing elementary transformations
 * @prev: pointer to the previous gesture frame
 * @touch: pointer to the touch frame triggering this gesture frame
 * @num_ongoing: number of elements in the ongoing array
 * @ongoing: array of ongoing transformation elements
 * @slots: array of all transformation slots
 *
 * A gesture frame consists of one or several touch frames glued
 * together into a stable transition, combined with information on
 * ongoing elementary gestural transformations. The array of ongoing
 * elements contains all elements with a nonzero expect mask.
 *
 * Later versions of this struct may grow in size, but will remain
 * binary compatible with older versions.
 */
struct grail_frame {
	const struct grail_frame *prev;
	const struct oif_frame *touch;
	unsigned int num_ongoing;
	struct grail_element **ongoing;
	struct grail_element **slots;
};

/**
 * struct grail_element - elementary gesture transformation
 * @prev: respective element of previous frame
 * @slot: the transformation slot occupied by this element
 * @id: unique identifier of the ongoing transformation
 * @num_touches: number of contacts of this element
 * @touches: array of contacts of this element
 * @start_time: start time of this element
 * @expect_mask: bitmask of expected gestures (grail main types)
 * @active_mask: bitmask of activated gestures (grail main types)
 * @center: gesture center position (surface units)
 * @velocity: current center velocity (surface units per second)
 * @radius: gesture radius from center (surface units)
 * @transform: the transformation matrix of the gesture
 * @rotation_center: current instant center of rotation (surface units)
 * @drag: accumulated transformation displacement (surface units)
 * @scale: accumulated scale (dimensionless)
 * @angle: accumulated rotation angle (radians)
 *
 * The grail element describes the ongoing gestural transformation of
 * a particular set of contacts. The expect mask describes which
 * gestural transformations may become active during the course of
 * events, and the active mask describes which have passed their
 * respective activation threshold. The set of expected gestures can
 * change over time, for instance by exclusion or timeout.
 *
 * Applications handling rotation, either by transformation matrix or
 * angle, should use the drag displacement. For other applications,
 * the center displacement may be used instead, as to not lose
 * movement accuracy.
 *
 * Later versions of this struct may grow in size, but will remain
 * binary compatible with older versions.
 */
struct grail_element {
	const struct grail_element *prev;
	int slot;
	int id;
	int num_touches;
	const struct oif_contact **touches;
	grail_time_t start_time;
	unsigned int expect_mask;
	unsigned int active_mask;
	struct grail_coord center;
	struct grail_coord velocity;
	float radius2;
	float transform[6];
	struct grail_coord rotation_center;
	struct grail_coord drag;
	float scale2;
	float angle;
};

/**
 * grail_element_transform - transform coordinates using element
 * @slot: the transformation element to use
 * @q: the grail coordinate to fill
 * @x: the grail coordinate to transform
 *
 * Performs the 3x3 transform *q = T *p, where T is the element
 * transform.
 */
static inline void grail_element_transform(const struct grail_element *slot,
					   struct grail_coord *q,
					   const struct grail_coord *p)
{
	const float *T = slot->transform;
	q->x = T[0] * (p->x - slot->center.x) +
	       T[1] * (p->y - slot->center.y) +
	       T[2] + slot->center.x;
	q->y = T[3] * (p->x - slot->center.x) +
	       T[4] * (p->y - slot->center.y) +
	       T[5] + slot->center.y;
}

/**
 * struct grail_event - Gesture event
 * @type: The gesture type
 * @id: Unique identifier foof the gesture instance
 * @status: Gesture status (begin, update, end)
 * @ntouch: Number of current touches
 * @nprop: Number of properties in the gesture
 * @pos: Focus point of the gesture (bbox coordinates)
 * @touch: Array of individual touch information
 * @client_id: The gesture client to route the gesture to
 * @time: Time of event (milliseconds)
 * @prop: Array of properties of the event
 *
 * Gesture events are passed to the client via the gesture() callback.
 */
struct grail_event {
	int type;
	int id;
	int status;
	int ntouch;
	int nprop;
	struct grail_coord pos;
	struct grail_client_id client_id;
	grail_time_t time;
	grail_prop_t prop[DIM_GRAIL_PROP];
};

/**
 * struct grail - Main grail device
 * @get_clients: Called at the onset of new gestures to retrieve the list
 * of listening clients.
 * @event: Callback for kernel events passing through grail.
 * @gesture: Main gesture callback.
 * @impl: Grail implementation details.
 * @gin: Gesture instatiation details.
 * @gru: Gesture recognition details.
 * @priv: Generic pointer to user-defined content.
 *
 * The grail device pulls events from the underlying device, detects
 * gestures, and passes them on to the client via the gesture()
 * callback. Events that are not gesture or for other reasons held back are
 * passed on via the event() callback. The user provides information about
 * windows and listening clients via the get_clients callback, which is
 * called during gesture instantiation.
 *
 */
struct grail {
	int (*get_clients)(struct grail *ge,
			   struct grail_client_info *client, int max_clients,
			   const struct grail_coord *coords, int num_coords,
			   const grail_mask_t *types, int type_bytes);
	void (*event)(struct grail *ge,
		      const struct input_event *ev);
	void (*gesture)(struct grail *ge,
			const struct grail_event *ev);
	struct grail_impl *impl;
	struct gesture_inserter *gin;
	struct gesture_recognizer *gru;
	void *priv;
};

/**
 * grail_open - open a grail device
 * @ge: the grail device to open
 * @fd: file descriptor of the kernel device
 *
 * Initialize the internal grail structures and configure it by reading the
 * protocol capabilities through the file descriptor.
 *
 * The callbacks, parameters and priv pointer should be set prior to this
 * call.
 *
 * Returns zero on success, negative error number otherwise.
 */
int GRAIL_PUBLIC grail_open(struct grail *ge, int fd);

/**
 * grail_idle - check state of kernel device
 * @ge: the grail device in use
 * @fd: file descriptor of the kernel device
 * @ms: number of milliseconds to wait for activity
 *
 * Returns true if the device is idle, i.e., there are no fetched
 * events in the pipe and there is nothing to fetch from the device.
 */
int GRAIL_PUBLIC grail_idle(struct grail *ge, int fd, int ms);

/**
 * grail_pull - pull and process available events from the kernel device
 * @ge: the grail device in use
 * @fd: file descriptor of the kernel device
 *
 * Pull all available events and process them. The grail callbacks are
 * invoked during this call.
 *
 * The underlying file descriptor must have O_NONBLOCK set, or this method
 * will not return until the file is closed.
 *
 * On success, returns the number of events read. Otherwise,
 * a standard negative error number is returned.
 */
int GRAIL_PUBLIC grail_pull(struct grail *ge, int fd);

/**
 * grail_close - close the grail device
 * @ge: the grail device to close
 * @fd: file descriptor of the kernel device
 *
 * Deallocates all memory associated with grail, and clears the grail
 * structure.
 */
void GRAIL_PUBLIC grail_close(struct grail *ge, int fd);

/**
 * grail_set_bbox - set the grail unit bounding box
 * @ge: the grail device in use
 * @min: the minimum (lower-left) corner of the bounding box
 * @max: the maximum (upper-right) corner of the bounding box
 *
 * Sets the box within which the device coordinates should be presented.
 */
void GRAIL_PUBLIC grail_set_bbox(struct grail *ge,
				 const struct grail_coord *min,
				 const struct grail_coord *max);

/**
 * grail_get_units - get device coordinate ranges
 * @ge: the grail device in use
 * @min: minimum x and y coordinates
 * @max: maximum x and y coordinates
 *
 * The grail event attributes pos, touch_major, touch_minor,
 * width_major, and width_minor are all given in device coordinate
 * units, unless specified otherwise using the grail_set_bbox()
 * function. This function reports the device coordinate ranges.
 *
 */
void GRAIL_PUBLIC grail_get_units(const struct grail *ge,
				  struct grail_coord *min, struct grail_coord *max);

/**
 * grail_get_contact_frame - get current contact frame
 * @ge: the grail device in use
 *
 * Return the contact frame current being processed. If called from
 * within a gesture callback, it is guaranteed to return the frame
 * corresponding to the gesture.
 *
 * The returned pointer can be NULL if no input has yet been extracted
 * through the grail instance.
 *
 * The frame pointer is ABI agnostic, owned by the grail instance, and
 * has grail scope.
 */
const struct oif_frame GRAIL_PUBLIC *
grail_get_contact_frame(const struct grail *ge);

#ifndef GRAIL_NO_LEGACY_API

struct grail_contact {
	int id;
	int tool_type;
	struct grail_coord pos;
	float touch_major;
	float touch_minor;
	float width_major;
	float width_minor;
	float angle;
	float pressure;
};

void GRAIL_PUBLIC grail_filter_abs_events(struct grail *ge, int usage);

int GRAIL_PUBLIC grail_get_contacts(const struct grail *ge,
				    struct grail_contact *touch, int max_touch);

#endif

#ifdef __cplusplus
}
#endif

#endif
