#!/usr/bin/env python

import difflib
import os
import subprocess
import unittest

srcdir = os.environ.get('srcdir', '.')

class ExpectedOutputTestCase(unittest.TestCase):

    def __init__(self, name, command, expected_output):
        super(ExpectedOutputTestCase, self).__init__()
        self.name = name
        self.command = command
        self.expected_output = expected_output

    def id(self):
        return self.name

    def __str__(self):
        return self.name

    def _ndiff_strings(self, a, b):
        """Return ndiff between two strings containing lines.

        A trailing newline is added if missing to make the strings
        print properly."""
        if b and b[-1] != '\n':
            b += '\n'
        if a and a[-1] != '\n':
            a += '\n'
        difflines = difflib.ndiff(a.splitlines(True),
                                  b.splitlines(True),
                                  linejunk=lambda x: False,
                                  charjunk=lambda x: False)
        return ''.join(difflines)

    def assertEqualDiff(self, a, b, message=None):
        """Assert two texts are equal, if not raise an exception.

        This is intended for use with multi-line strings where it can
        be hard to find the differences by eye.
        """
        # TODO: perhaps override assertEquals to call this for strings?
        if a == b:
            return
        if message is None:
            message = "texts not equal:\n"
        raise AssertionError(message +
                             self._ndiff_strings(a, b))

    def runTest(self):
        proc = subprocess.Popen(self.command,
                                stdin=subprocess.PIPE,
                                stdout=subprocess.PIPE)
        stdout, stderr = proc.communicate()
        self.assertEqual(proc.returncode, 0)
        self.assertEqualDiff(open(self.expected_output, 'rb').read(), stdout)


def suite_from_dir(command, directory, extension='.xml'):
    suite = unittest.TestSuite()
    for filename in os.listdir(os.path.join(srcdir, directory)):
        base, ext = os.path.splitext(filename)
        if ext != extension:
            continue
        test_name = '%s/%s' % (directory, base)
        input_file = os.path.join(srcdir, directory, filename)
        expected_output = os.path.join(srcdir, directory, base + '.out')
        suite.addTest(ExpectedOutputTestCase(
            test_name, [command, input_file], expected_output))
    return suite

def test_suite():
    suite = unittest.TestSuite()
    suite.addTest(suite_from_dir('./parse-obex-caps', 'capabilities'))
    suite.addTest(suite_from_dir('./parse-obex-folder-listing',
                                 'folder-listings'))
    return suite

if __name__ == '__main__':
    unittest.main(defaultTest='test_suite')

