/**
 * GMyth Library
 *
 * @file gmyth/gmyth_scheduler.c
 *
 * @brief <p> The scheduler encapsulates all functions for browsing, scheduling
 * and modifying the recorded content.
 *
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia.
 * @author Alexsandro Jose Virginio dos Santos <alexsandro.santos@indt.org.br>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <assert.h>

#include <glib.h>
#include <glib/gprintf.h>

#include "gmyth_scheduler.h"
#include "gmyth_util.h"
#include "gmyth_query.h"
#include "gmyth_socket.h"
#include "gmyth_debug.h"

static void     gmyth_scheduler_class_init(GMythSchedulerClass * klass);
static void     gmyth_scheduler_init(GMythScheduler * object);

static void     gmyth_scheduler_dispose(GObject * object);
static void     gmyth_scheduler_finalize(GObject * object);

static gboolean update_backend(GMythScheduler * scheduler, gint record_id);

G_DEFINE_TYPE(GMythScheduler, gmyth_scheduler, G_TYPE_OBJECT)
    static void     gmyth_scheduler_class_init(GMythSchedulerClass * klass)
{
    GObjectClass   *gobject_class;

    gobject_class = (GObjectClass *) klass;

    gobject_class->dispose = gmyth_scheduler_dispose;
    gobject_class->finalize = gmyth_scheduler_finalize;
}

static void
gmyth_scheduler_init(GMythScheduler * sched)
{
    sched->recordid = 0;
    sched->type = 0;
    sched->search = 0;
    sched->profile = g_string_new("");

    sched->dupin = 0;
    sched->dupmethod = 0;
    sched->autoexpire = 0;
    sched->autotranscode = 0;
    sched->transcoder = 0;

    sched->autocommflag = 0;
    sched->autouserjob1 = 0;
    sched->autouserjob2 = 0;
    sched->autouserjob3 = 0;
    sched->autouserjob4 = 0;

    sched->startoffset = 0;
    sched->endoffset = 0;
    sched->maxepisodes = 0;
    sched->maxnewest = 0;

    sched->recpriority = 0;
    sched->recgroup = g_string_new("");
    sched->playgroup = g_string_new("");

    sched->prefinput = 0;
    sched->inactive = 0;

    sched->search_type = g_string_new("");
    sched->search_what = g_string_new("");

    sched->msqlquery = gmyth_query_new();
}

static void
gmyth_scheduler_dispose(GObject * object)
{
    GMythScheduler *scheduler = GMYTH_SCHEDULER(object);

    if (scheduler->backend_info) {
        g_object_unref(scheduler->backend_info);
        scheduler->backend_info = NULL;
    }

    if (scheduler->msqlquery) {
        g_object_unref(scheduler->msqlquery);
        scheduler->msqlquery = NULL;
    }

    g_string_free(scheduler->profile, TRUE);
    g_string_free(scheduler->recgroup, TRUE);
    g_string_free(scheduler->playgroup, TRUE);
    g_string_free(scheduler->search_type, TRUE);
    g_string_free(scheduler->search_what, TRUE);

    G_OBJECT_CLASS(gmyth_scheduler_parent_class)->dispose(object);
}

static void
gmyth_scheduler_finalize(GObject * object)
{
    g_signal_handlers_destroy(object);

    G_OBJECT_CLASS(gmyth_scheduler_parent_class)->finalize(object);
}

/** Creates a new instance of GMythScheduler.
 * 
 * @return a new instance of GMythScheduler.
 */
GMythScheduler *
gmyth_scheduler_new()
{
    GMythScheduler *scheduler =
        GMYTH_SCHEDULER(g_object_new(GMYTH_SCHEDULER_TYPE, NULL));

    return scheduler;
}

gboolean
gmyth_scheduler_connect(GMythScheduler * scheduler,
                        GMythBackendInfo * backend_info)
{
    return gmyth_scheduler_connect_with_timeout(scheduler, backend_info,
                                                0);
}

/** Connects to the Mysql database in the backend. The backend address
 * is loaded from the GMythSettings instance.
 *
 * @param scheduler the GMythScheduler instance to be connected.
 * @return true if connection was success, false if failed.
 */
gboolean
gmyth_scheduler_connect_with_timeout(GMythScheduler * scheduler,
                                     GMythBackendInfo * backend_info,
                                     guint timeout)
{
    assert(scheduler);
    g_return_val_if_fail(backend_info != NULL, FALSE);

    if (scheduler->backend_info)
        g_object_unref(scheduler->backend_info);

    scheduler->backend_info = g_object_ref(backend_info);

    if (scheduler->msqlquery == NULL) {
        g_warning("[%s] GMythScheduler db initializing", __FUNCTION__);
        scheduler->msqlquery = gmyth_query_new();
    }

    if (!gmyth_query_connect_with_timeout(scheduler->msqlquery,
                                          scheduler->backend_info,
                                          timeout)) {
        g_warning("[%s] Error while connecting to db", __FUNCTION__);
        return FALSE;
    }

    return TRUE;
}

/** Disconnects from the Mysql database in the backend.
 *
 * @param scheduler the GMythScheduler instance to be disconnected
 * @return true if disconnection was success, false if failed.
 */
gboolean
gmyth_scheduler_disconnect(GMythScheduler * scheduler)
{
    assert(scheduler);

    if (scheduler->msqlquery != NULL) {
        gmyth_query_disconnect(scheduler->msqlquery);
    }

    return TRUE;
}

/** Retrieves from the backend Mysql database the list of recording schedules.
 * 
 * @param scheduler The GMythScheduler instance.
 * @param schedule_list the GList pointer to be filled with the loaded list of ScheduleInfo items.
 * @return The amount of schedules retrieved from database, or -1 if error.
 */
gint
gmyth_scheduler_get_schedule_list(GMythScheduler * scheduler,
                                  GList ** schedule_list)
{
    ScheduleInfo   *schedule;
    MYSQL_RES      *msql_res;
    GString        *query_str = g_string_new("");
    gchar          *date_time = NULL;

    assert(scheduler);

    g_string_printf(query_str,
                    "SELECT recordid,programid,chanid,starttime,startdate,"
                    "endtime,enddate,title,subtitle,description,category,type,parentid,seriesid FROM record;");

    if (scheduler->msqlquery == NULL) {
        g_warning("[%s] Scheduler db connection not initialized",
                  __FUNCTION__);
        return -1;
    }
    msql_res =
        gmyth_query_process_statement(scheduler->msqlquery,
                                      query_str->str);

    if (msql_res == NULL) {
        g_warning("DB retrieval of schedule list failed");
        return -1;
    } else {
        MYSQL_ROW       row;

        *schedule_list = NULL;

        while ((row = mysql_fetch_row(msql_res)) != NULL) {
            schedule = g_new0(ScheduleInfo, 1);
            gint type = 0;

            schedule->schedule_id =
                (guint) g_ascii_strtoull(row[0], NULL, 10);
            schedule->program_id = g_string_new (row[1]);
            schedule->channel_id = (gint) g_ascii_strtoull (row[2], NULL, 10);

            /*
             * generate a time_t from a time and a date db field 
             */
            date_time = g_strdup_printf("%sT%s", row[4], row[3]);
            schedule->start_time =
                gmyth_util_string_to_time_val(date_time);
            g_free(date_time);

            /*
             * generate a time_t from a time and a date db field 
             */
            date_time = g_strdup_printf("%sT%s", row[6], row[5]);
            schedule->end_time = gmyth_util_string_to_time_val(date_time);
            g_free(date_time);

            schedule->title = g_string_new(row[7]);
            schedule->subtitle = g_string_new(row[8]);
            schedule->description = g_string_new(row[9]);
            schedule->category = g_string_new(row[10]);
            type = g_ascii_strtoull (row[11], NULL, 10);
            if (type == 4) {
                schedule->type = GMYTH_SCHEDULE_ALL_OCCURRENCES;
            } else if (type == 1) {
                schedule->type = GMYTH_SCHEDULE_ONE_OCCURRENCE;
            } else if (type == 8) {
                schedule->type = GMYTH_SCHEDULE_EXCEPTION;
                schedule->parentid = (gint) g_ascii_strtoull (row[12], NULL, 10);
            }

            schedule->seriesid = g_string_new (row[13]);

            (*schedule_list) = g_list_append(*(schedule_list), schedule);
        }
    }

    mysql_free_result(msql_res);
    g_string_free(query_str, TRUE);

    return (*schedule_list == NULL) ? 0 : g_list_length(*schedule_list);
}

/** Retrieves from the backend Mysql database the list of recorded programs.
 * 
 * @param scheduler The GMythScheduler instance.
 * @param recorded_list the GList pointer to be filled with the loaded RecordInfo items.
 * @return The amount of recorded retrieved from database, or -1 if error.
 */
gint
gmyth_scheduler_get_recorded_list(GMythScheduler * scheduler,
                                  GList ** recorded_list)
{
    RecordedInfo   *record;
    MYSQL_RES      *msql_res;
    GString        *query_str = g_string_new("");

    assert(scheduler);

    g_string_printf(query_str,
                    "SELECT recordid,programid,chanid,starttime,progstart,"
                    "endtime,progend,title,subtitle,description,category,"
                    "filesize,basename,seriesid FROM recorded WHERE recgroup != 'LiveTV'");

    if (scheduler->msqlquery == NULL) {
        g_warning("[%s] Scheduler db connection not initialized",
                  __FUNCTION__);
        return -1;
    }

    msql_res =
        gmyth_query_process_statement(scheduler->msqlquery,
                                      query_str->str);

    if (msql_res == NULL) {
        g_warning("DB retrieval of recording list failed");
        return -1;
    } else {
        MYSQL_ROW       row;

        (*recorded_list) = NULL;

        while ((row = mysql_fetch_row(msql_res)) != NULL) {
            record = g_new0(RecordedInfo, 1);

            record->record_id = (guint) g_ascii_strtoull(row[0], NULL, 10);
            record->program_id = g_string_new (row[1]);
            record->channel_id = (gint) g_ascii_strtoull(row[2], NULL, 10);
            record->start_time = gmyth_util_string_to_time_val(row[3]);
            record->end_time = gmyth_util_string_to_time_val(row[5]);

            record->title = g_string_new(row[7]);
            record->subtitle = g_string_new(row[8]);
            record->description = g_string_new(row[9]);
            record->category = g_string_new(row[10]);
            record->filesize = g_ascii_strtoull(row[11], NULL, 10);
            record->basename = g_string_new(row[12]);
            record->seriesid = g_string_new(row[13]);

            (*recorded_list) = g_list_append((*recorded_list), record);
        }
    }

    mysql_free_result(msql_res);
    g_string_free(query_str, TRUE);

    return (*recorded_list == NULL) ? 0 : g_list_length(*recorded_list);
}

RecordedInfo*
gmyth_scheduler_get_recorded_info (GMythScheduler *scheduler,
                                   const gchar* basename)
{
    RecordedInfo   *record = NULL;
    MYSQL_RES      *msql_res;
    GString        *query_str = g_string_new("");

    assert(scheduler);

    g_string_printf(query_str,
                    "SELECT recordid,programid,chanid,starttime,progstart,"
                    "endtime,progend,title,subtitle,description,category,"
                    "filesize,basename,seriesid FROM recorded "
                    "WHERE recgroup != 'LiveTV' AND basename = '%s'", basename);

    if (scheduler->msqlquery == NULL) {
        g_warning("[%s] Scheduler db connection not initialized", 
                        __FUNCTION__);
        return NULL;
    }

    msql_res =
        gmyth_query_process_statement(scheduler->msqlquery,
                                      query_str->str);

    if (msql_res == NULL) {
        g_warning("DB retrieval of recording list failed");
        return NULL;
    } else {
        MYSQL_ROW       row;
        row = mysql_fetch_row(msql_res);
        if (row != NULL) {
            record = g_new0(RecordedInfo, 1);
            record->record_id = (guint) g_ascii_strtoull(row[0], NULL, 10);
            record->program_id = g_string_new (row[1]);
            record->channel_id = (gint) g_ascii_strtoull(row[2], NULL, 10);
            record->start_time = gmyth_util_string_to_time_val(row[3]);
            record->end_time = gmyth_util_string_to_time_val(row[5]);
            record->title = g_string_new(row[7]);
            record->subtitle = g_string_new(row[8]);
            record->description = g_string_new(row[9]);
            record->category = g_string_new(row[10]);
            record->filesize = g_ascii_strtoull(row[11], NULL, 10);
            record->basename = g_string_new(row[12]);
            record->seriesid = g_string_new(row[13]);
        }
    }

    mysql_free_result(msql_res);
    g_string_free(query_str, TRUE);

    return record;
}


static void
_set_value(GMythQuery * myth_query, char *field, gchar * value,
           gint rec_id)
{
    gchar          *query =
        g_strdup_printf
        ("UPDATE record SET recordid = %d, %s = \"%s\" WHERE recordid = %d;",
         rec_id, field, value, rec_id);

    gmyth_query_process_statement(myth_query, query);
    g_free(query);
}

static void
_set_int_value(GMythQuery * myth_query, char *field, gint value,
               gint rec_id)
{
    gchar *str_value = g_strdup_printf("%d", value);

    _set_value(myth_query, field, str_value, rec_id);
    g_free(str_value);
}

ScheduleInfo*
gmyth_scheduler_add_schedule_program (GMythScheduler * scheduler,
                                      GMythProgramInfo *program,
                                      GMythScheduleType type)
{
    ScheduleInfo *info;

    info = g_new0 (ScheduleInfo, 1);
    info->program_id = g_string_new (program->program_id->str);
    info->channel_id = program->channel_id;
    info->start_time = g_new0 (GTimeVal, 1);
    *info->start_time = *program->startts;
    info->end_time = g_new0 (GTimeVal, 1);
    *info->end_time = *program->endts;
    info->seriesid = g_string_new (program->seriesid->str);
    info->title = g_string_new (program->title->str);
    info->subtitle = g_string_new (program->subtitle->str);
    info->description = g_string_new (program->description->str);
    info->category = g_string_new (program->category->str);
    info->type = type;

    if (gmyth_scheduler_add_schedule_full (scheduler, info, type))
    {
        if (!program->recstartts)
            program->recstartts = g_new0 (GTimeVal, 1);
        *program->recstartts = *info->start_time;

        if (!program->recendts)
            program->recendts = g_new0 (GTimeVal, 1);
        *program->recendts = *info->end_time;

        program->recordid = info->schedule_id;
        return info;
    }

    gmyth_schedule_info_free (info);
    return NULL;
}


gboolean
gmyth_scheduler_add_schedule_full (GMythScheduler * scheduler,
                             ScheduleInfo * schedule_info, GMythScheduleType type)
{
    MYSQL_RES      *msql_res;
    gchar          *query_str = "INSERT record (recordid) VALUE (0);";
    gchar          *station = NULL;
    gulong          rec_id;

    g_return_val_if_fail (IS_GMYTH_SCHEDULER (scheduler), FALSE);

    if (scheduler->msqlquery == NULL) {
        g_warning("[%s] Scheduler db connection not initialized",
                  __FUNCTION__);
        return FALSE;
    }

    msql_res =
        gmyth_query_process_statement_with_increment(scheduler->msqlquery,
                                                     query_str, &rec_id);
    mysql_free_result(msql_res);

    // Retrieves the station info
    query_str =
        g_strdup_printf
        ("SELECT callsign FROM channel WHERE chanid = %d;",
         schedule_info->channel_id);
    msql_res =
        gmyth_query_process_statement(scheduler->msqlquery, query_str);
    if (msql_res == NULL) {
        g_warning("[%s] msql query returned NULL MYSQL_RES", __FUNCTION__);
        return FALSE;
    } else {
        MYSQL_ROW       row;

        if ((row = mysql_fetch_row(msql_res)) != NULL) {
            station = g_strdup(row[0]);
        }
    }
    mysql_free_result(msql_res);
    g_free(query_str);

    // _set_value (field, value, id);
    _set_int_value(scheduler->msqlquery, "chanid",
                   schedule_info->channel_id, rec_id);
    _set_value(scheduler->msqlquery, "station", station, rec_id);
    _set_value(scheduler->msqlquery, "title", schedule_info->title->str,
               rec_id);
    // / subtitle, description 
    _set_value(scheduler->msqlquery, "starttime",
               gmyth_util_time_to_string_only_time(schedule_info->
                                                   start_time), rec_id);
    _set_value(scheduler->msqlquery, "startdate",
               gmyth_util_time_to_string_only_date(schedule_info->
                                                   start_time), rec_id);
    _set_value(scheduler->msqlquery, "endtime",
               gmyth_util_time_to_string_only_time(schedule_info->
                                                   end_time), rec_id);
    _set_value(scheduler->msqlquery, "enddate",
               gmyth_util_time_to_string_only_date(schedule_info->
                                                   end_time), rec_id);
    // / category, series id, program id
    // _set_value (scheduler->msqlquery, "findday",
    // (gmyth_util_time_val_to_date( schedule_info->start_time
    // ))->tm_wday, rec_id);
    // _set_value (scheduler->msqlquery, "findtime",
    // gmyth_util_time_to_string_only_time( schedule_info->start_time),
    // rec_id);
    // _set_int_value (scheduler->msqlquery, "findid",
    // (gint)(schedule_info->start_time->tv_sec/60/60/24 + 719528),
    // rec_id);

    if (schedule_info->seriesid)
       _set_value(scheduler->msqlquery, "seriesid",
                  schedule_info->seriesid->str, rec_id);
 
    _set_value(scheduler->msqlquery, "parentid", "0", rec_id);
    _set_value(scheduler->msqlquery, "search", "0", rec_id);

    if (type == GMYTH_SCHEDULE_ALL_OCCURRENCES) {
        _set_int_value(scheduler->msqlquery, "type", 3, rec_id);
    } else if (type == GMYTH_SCHEDULE_ONE_OCCURRENCE) {
        _set_int_value(scheduler->msqlquery, "type", 1, rec_id);
    } else if (type == GMYTH_SCHEDULE_EXCEPTION) {
        _set_int_value(scheduler->msqlquery, "type", 8, rec_id);
        _set_int_value(scheduler->msqlquery, "parentid", schedule_info->parentid,
		       rec_id);
    }

    _set_value(scheduler->msqlquery, "recpriority", "0", rec_id);
    _set_value(scheduler->msqlquery, "startoffset", "0", rec_id);
    _set_value(scheduler->msqlquery, "endoffset", "0", rec_id);
    _set_value(scheduler->msqlquery, "dupmethod", "6", rec_id); // ?
    _set_value(scheduler->msqlquery, "dupin", "15", rec_id);    // ?

    _set_value(scheduler->msqlquery, "prefinput", "0", rec_id);
    _set_value(scheduler->msqlquery, "inactive", "0", rec_id);
    _set_value(scheduler->msqlquery, "profile", "Default", rec_id);
    _set_value(scheduler->msqlquery, "recgroup", "Default", rec_id);
    _set_value(scheduler->msqlquery, "storagegroup", "Default", rec_id);
    _set_value(scheduler->msqlquery, "playgroup", "Default", rec_id);
    _set_value(scheduler->msqlquery, "autoexpire", "1", rec_id);
    _set_value(scheduler->msqlquery, "maxepisodes", "0", rec_id);
    _set_value(scheduler->msqlquery, "maxnewest", "0", rec_id);
    _set_value(scheduler->msqlquery, "autocommflag", "1", rec_id);
    _set_value(scheduler->msqlquery, "autotranscode", "0", rec_id);
    _set_value(scheduler->msqlquery, "transcoder", "0", rec_id);

    _set_value(scheduler->msqlquery, "autouserjob1", "0", rec_id);
    _set_value(scheduler->msqlquery, "autouserjob2", "0", rec_id);
    _set_value(scheduler->msqlquery, "autouserjob3", "0", rec_id);
    _set_value(scheduler->msqlquery, "autouserjob4", "0", rec_id);

    schedule_info->schedule_id = rec_id;

    /* Notify the backend of changes */
    return update_backend(scheduler, rec_id);
}

/** Requests the Mysql database in the backend to add a new schedule.
 * 
 * @param scheduler the GMythScheduler instance.
 * @param schedule_info the ScheduleInfo with recording schedule information
 * to be added. record_id = -1 to add a new schedule, otherwise this
 * function will update the schedule in the db
 * @return gboolean returns FALSE if some error occurs, TRUE otherwise
 */
gboolean
gmyth_scheduler_add_schedule (GMythScheduler * scheduler,
                             ScheduleInfo * schedule_info)
{
    return gmyth_scheduler_add_schedule_full (scheduler, schedule_info, 
                                              GMYTH_SCHEDULE_ONE_OCCURRENCE);
}

/** Requests the Mysql database in the backend to remove an existing schedule.
 * 
 * @param scheduler the GMythScheduler instance.
 * @param schedule_id The schedule's record id to be removed
 * @return gboolean TRUE if success, FALSE if error
 */
gboolean
gmyth_scheduler_delete_schedule(GMythScheduler * scheduler, gint schedule_id)
{

    MYSQL_RES      *msql_res;
    GString        *query_str = NULL;

    g_return_val_if_fail (scheduler != NULL, FALSE);


    if (scheduler->msqlquery == NULL) {
        g_warning("[%s] Scheduler db connection not initialized",
                  __FUNCTION__);
        return FALSE;
    }

    query_str = g_string_new("");
    g_string_printf(query_str,
                    "DELETE FROM record WHERE recordid=%d", schedule_id);

    msql_res =
        gmyth_query_process_statement(scheduler->msqlquery,
                                      query_str->str);


    mysql_free_result(msql_res);
    g_string_free(query_str, TRUE);

    // Notify the backend of the changes
    return update_backend(scheduler, schedule_id);
}

/*
 * Add an exception program to be removed from the schedule list, when programs
 * where scheduled with the GMYTH_SCHEDULE_ALL_OCCURRENCES option.
 * @param scheduler the GMythScheduler instance.
 * @param schedule_id the schedule id of the all occurrence schedule to be changed
 * @param exception_info the ScheduleInfo to be removed from all schedule occurrences
 * @return TRUE if success, FALSE if any error happens
 */
gboolean
gmyth_scheduler_add_exception (GMythScheduler *scheduler, gint schedule_id,
			       ScheduleInfo *exception_info)
{
    gboolean res;

    g_return_val_if_fail (scheduler != NULL, FALSE);
    g_return_val_if_fail (exception_info != NULL, FALSE);

    exception_info->parentid = schedule_id;
    res = gmyth_scheduler_add_schedule_full (scheduler, exception_info, GMYTH_SCHEDULE_EXCEPTION);

    return res;
}

/** Requests the Mysql database in the backend to remove an existing recorded item.
 * 
 * @param scheduler the GMythScheduler instance.
 * @param record_id The recorded item id to be removed
 * @return gboolean TRUE if success, FALSE if error
 */
gboolean
gmyth_scheduler_delete_recorded(GMythScheduler * scheduler, gint record_id)
{

    MYSQL_RES      *msql_res;

    GString        *query_str = g_string_new("");

    assert(scheduler);

    if (scheduler->msqlquery == NULL) {
        g_warning("[%s] Scheduler db connection not initialized",
                  __FUNCTION__);
        return FALSE;
    }
    // ========================================
    g_string_printf(query_str,
                    "DELETE FROM recorded WHERE recordid=%d", record_id);

    // FIXME: Mythtv implementation runs also: DELETE FROM oldfind WHERE
    // recordid = x

    msql_res =
        gmyth_query_process_statement(scheduler->msqlquery,
                                      query_str->str);

    mysql_free_result(msql_res);
    g_string_free(query_str, TRUE);

    // Notify the backend of the changes
    return update_backend(scheduler, record_id);
}


gboolean gmyth_scheduler_was_recorded_before(GMythScheduler* scheduler, gint channel_id,
                                             time_t start_time)
{
    MYSQL_RES      *msql_res;
    GString        *query_str = g_string_new("");

    assert(scheduler);
    g_string_printf(query_str, "SELECT callsign FROM channel "
                    "WHERE chanid = \"%d\"", channel_id);

    msql_res = gmyth_query_process_statement(scheduler->msqlquery, query_str->str);

    if (msql_res) {
        MYSQL_ROW msql_row = mysql_fetch_row(msql_res);
        if (msql_row) {
            GString* callsign = g_string_new(msql_row[0]);
            GString* startts = gmyth_util_time_to_string(start_time);
            g_string_printf(query_str, "SELECT * FROM oldrecorded "
                            "WHERE station = \"%s\" AND starttime = \"%s\"",
                            callsign->str, startts->str);
            msql_res = gmyth_query_process_statement(scheduler->msqlquery, query_str->str);
            g_string_free(callsign, TRUE);
            g_string_free(startts, TRUE);
            g_string_free(query_str, TRUE);
            if (mysql_fetch_row(msql_res)) return TRUE;
        }
    }
    return FALSE;
}


gboolean gmyth_scheduler_reactivate_schedule(GMythScheduler* scheduler, gint channel_id,
                                             time_t start_time)

{
    MYSQL_RES      *msql_res;
    GString        *query_str = g_string_new("");

    assert(scheduler);
    g_string_printf(query_str, "SELECT callsign FROM channel "
                    "WHERE chanid = \"%d\"", channel_id);

    msql_res = gmyth_query_process_statement(scheduler->msqlquery, query_str->str);
    if (msql_res) {
        MYSQL_ROW msql_row = mysql_fetch_row(msql_res);
        if (msql_row) {
            GString* callsign = g_string_new(msql_row[0]);
            GString* startts = gmyth_util_time_to_string(start_time);
            g_string_printf(query_str, "UPDATE oldrecorded SET reactivate = 1 "
                            "WHERE station = \"%s\" AND starttime = \"%s\"",
                            callsign->str, startts->str);
            gmyth_query_process_statement(scheduler->msqlquery, query_str->str);
            g_string_free(callsign, TRUE);
            g_string_free(startts, TRUE);
            g_string_free(query_str, TRUE);
            return TRUE;
        }

    }

    return FALSE;
}


/*
 * This should only be used in special situations. We do not know the time that
 * the recording was set. We just know that it is an "ongoing" record and then
 * we have to use this to get it's info. It's always the oldest one -> first on list
 *
 */
GMythProgramInfo*
gmyth_scheduler_get_recorded_on_time(GMythScheduler* scheduler,
                                     guint channel_id)
{
    MYSQL_RES      *msql_res;
    GMythProgramInfo *proginfo = NULL;
    GString        *query_str = g_string_new("");

    assert(scheduler);

    g_string_printf(query_str,
                    "SELECT recorded.chanid,starttime,endtime,title,"
                    "subtitle,description,channel.channum,"
                    "channel.callsign,channel.name,channel.commfree,"
                    "channel.outputfilters,seriesid,programid,filesize,"
                    "lastmodified,stars,previouslyshown,originalairdate,"
                    "hostname,recordid,transcoder,playgroup,"
                    "recorded.recpriority,progstart,progend,basename,recgroup,"
                    "category,findid,duplicate "
                    "FROM recorded " "LEFT JOIN channel "
                    "ON recorded.chanid = channel.chanid "
                    "WHERE recorded.chanid = %d "
                    "ORDER BY starttime DESC", channel_id);

    msql_res =
        gmyth_query_process_statement(scheduler->msqlquery, query_str->str);

    if (msql_res) {
        MYSQL_ROW msql_row = mysql_fetch_row(msql_res);

        if (msql_row) {
            proginfo = gmyth_program_info_new();

            proginfo->channel_id = (gint) g_ascii_strtoull (msql_row[0], NULL, 10);
            proginfo->recstartts = gmyth_util_string_to_time_val(msql_row[1]);
            proginfo->recendts = gmyth_util_string_to_time_val(msql_row[2]);

            proginfo->title = g_string_new(msql_row[3]);
            proginfo->subtitle = g_string_new(msql_row[4]);
            proginfo->description = g_string_new(msql_row[5]);

            proginfo->chanstr = g_string_new(msql_row[6]);
            proginfo->chansign = g_string_new(msql_row[7]);
            proginfo->channame = g_string_new(msql_row[8]);
            proginfo->chancommfree = (gint) g_ascii_strtoull(msql_row[9], NULL, 10);
            proginfo->chanOutputFilters = g_string_new(msql_row[10]);
            proginfo->seriesid = g_string_new(msql_row[11]);
            proginfo->program_id = g_string_new(msql_row[12]);
            proginfo->filesize = g_ascii_strtoull(msql_row[13], NULL, 10);

            proginfo->lastmodified = gmyth_util_string_to_time_val(msql_row[14]);
            proginfo->stars = g_ascii_strtod(msql_row[15], NULL);
            proginfo->repeat = (gint)g_ascii_strtoull(msql_row[16], NULL, 10);

            if (msql_row[17] == NULL) {
                proginfo->originalAirDate = 0;
                proginfo->hasAirDate = FALSE;
            } else {
                proginfo->originalAirDate = gmyth_util_string_to_time_val(msql_row[17]);
                proginfo->hasAirDate = TRUE;
            }

            proginfo->hostname = g_string_new(msql_row[18]);
            proginfo->recordid = (gint) g_ascii_strtoull(msql_row[19], NULL, 10);
            proginfo->transcoder = (gint) g_ascii_strtoull(msql_row[20], NULL, 10);

            proginfo->playgroup = g_string_new(msql_row[21]);
            proginfo->recpriority = (gint) g_ascii_strtoull(msql_row[22], NULL, 10);

            proginfo->startts = gmyth_util_string_to_time_val(msql_row[23]);
            proginfo->endts = gmyth_util_string_to_time_val(msql_row[24]);
            proginfo->pathname = g_string_new(g_strdup(msql_row[25]));
            proginfo->recgroup = g_string_new(msql_row[26]);
            proginfo->category = g_string_new(msql_row[27]);
            proginfo->findid = (gint) g_ascii_strtoull(msql_row[28], NULL, 10);

            proginfo->recpriority2 = 0;

            g_string_printf(query_str,
                            "SELECT dupmethod,dupin,parentid,type "
                            "FROM record WHERE recordid = \"%d\"", proginfo->recordid);

            msql_res =
                gmyth_query_process_statement(scheduler->msqlquery,
                                              query_str->str);

            if (msql_res) {
                MYSQL_ROW msql_row = mysql_fetch_row(msql_res);

                if (msql_row) {
                    proginfo->dupmethod = (gint) g_ascii_strtoull(msql_row[0], NULL, 10);
                    proginfo->dupin = (gint) g_ascii_strtoull(msql_row[1], NULL, 10);
                    proginfo->parentid = (gint) g_ascii_strtoull(msql_row[2], NULL, 10);
                    proginfo->rectype = 0;
                }
            }


            g_string_printf(query_str,
                            "SELECT sourceid,cardid,cardinputid,shareable "
                            "FROM cardinput");

            msql_res =
                gmyth_query_process_statement(scheduler->msqlquery,
                                              query_str->str);

            if (msql_res) {
                MYSQL_ROW msql_row = mysql_fetch_row(msql_res);

                if (msql_row) {
                    proginfo->sourceid = 0;
                    proginfo->cardid = 0;
                    proginfo->inputid = 0;
                    if (msql_row[3] != NULL && g_ascii_strcasecmp("Y", msql_row[3]) == 0)
                        proginfo->shareable = 1;
                    else
                        proginfo->shareable = 0;
                }
            }



        }
    }

    g_string_free(query_str, TRUE);
    return proginfo;
}

/** Retrieves an existing recorded item information from database. The information
 * is used to fill the returned GMythProgramInfo.
 * 
 * @param scheduler The GMythScheduler instance.
 * @param channel The channel associated to the record
 * @param starttime The record start time
 * @return A GMythProgramInfo struct with the requested record item
 * information, or NULL if error.
 */
GMythProgramInfo *
gmyth_scheduler_get_recorded(GMythScheduler * scheduler,
                             GString * channel, GTimeVal * starttime)
{
    MYSQL_RES      *msql_res;
    GMythProgramInfo *proginfo = NULL;
    GString        *query_str = g_string_new("");
    gchar          *time_str =
        gmyth_util_time_to_string_from_time_val(starttime);

    assert(scheduler);

    gmyth_debug("[%s] channel: %s", __FUNCTION__, channel->str);

    if (scheduler->msqlquery == NULL) {
        g_warning("[%s] Scheduler db connection not initialized",
                  __FUNCTION__);
        return NULL;
    }

    g_string_printf(query_str,
                    "SELECT recorded.chanid,starttime,endtime,title, "
                    "subtitle,description,channel.channum, "
                    "channel.callsign,channel.name,channel.commfree, "
                    "channel.outputfilters,seriesid,programid,filesize, "
                    "lastmodified,stars,previouslyshown,originalairdate, "
                    "hostname,recordid,transcoder,playgroup, "
                    "recorded.recpriority,progstart,progend,basename,recgroup "
                    "FROM recorded " "LEFT JOIN channel "
                    "ON recorded.chanid = channel.chanid "
                    "WHERE recorded.chanid = \"%s\" "
                    "AND starttime = \"%s\" ;", channel->str, time_str);

    msql_res =
        gmyth_query_process_statement(scheduler->msqlquery,
                                      query_str->str);

    if (msql_res /* && query.size() > 0 */ ) {
        MYSQL_ROW       msql_row = mysql_fetch_row(msql_res);

        if (msql_row) {
            proginfo = gmyth_program_info_new();

            proginfo->channel_id = (gint) g_ascii_strtoull (msql_row[0], NULL, 10);
            proginfo->startts =
                gmyth_util_string_to_time_val(msql_row[23]);
            proginfo->endts = gmyth_util_string_to_time_val(msql_row[24]);
            proginfo->recstartts =
                gmyth_util_string_to_time_val(msql_row[1]);
            proginfo->recendts =
                gmyth_util_string_to_time_val(msql_row[2]);
            proginfo->title = g_string_new(msql_row[3]);
            proginfo->subtitle = g_string_new(msql_row[4]);
            proginfo->description = g_string_new(msql_row[5]);

            proginfo->chanstr = g_string_new(msql_row[6]);
            proginfo->chansign = g_string_new(msql_row[7]);
            proginfo->channame = g_string_new(msql_row[0]);
            proginfo->chancommfree =
                (gint) g_ascii_strtoull(msql_row[9], NULL, 10);
            proginfo->chanOutputFilters = g_string_new(msql_row[10]);
            proginfo->seriesid = g_string_new(msql_row[11]);
            proginfo->program_id = g_string_new(msql_row[12]);
            proginfo->filesize = g_ascii_strtoull(msql_row[13], NULL, 10);

            proginfo->lastmodified =
                gmyth_util_string_to_time_val(msql_row[14]);
            proginfo->stars = g_ascii_strtod(msql_row[15], NULL);
            proginfo->repeat =
                (gint) g_ascii_strtoull(msql_row[16], NULL, 10);

            if (msql_row[17] == NULL) {
                proginfo->originalAirDate = 0;
                proginfo->hasAirDate = FALSE;
            } else {
                proginfo->originalAirDate =
                    gmyth_util_string_to_time_val(msql_row[17]);
                proginfo->hasAirDate = TRUE;
            }

            proginfo->hostname = g_string_new(msql_row[18]);
            proginfo->recordid =
                (gint) g_ascii_strtoull(msql_row[19], NULL, 10);
            proginfo->transcoder =
                (gint) g_ascii_strtoull(msql_row[20], NULL, 10);
            // proginfo->spread = -1;
            // proginfo->programflags = proginfo->getProgramFlags();

            proginfo->recgroup = g_string_new(msql_row[26]);
            proginfo->playgroup = g_string_new(msql_row[21]);
            proginfo->recpriority =
                (gint) g_ascii_strtoull(msql_row[22], NULL, 10);

            proginfo->pathname = g_string_new(g_strdup(msql_row[25]));

            gmyth_debug("One program info loaded from mysql database\n");
        }
    }

    mysql_free_result(msql_res);
    g_string_free(query_str, TRUE);
    g_free(time_str);

    return proginfo;
}

gboolean
gmyth_scheduler_stop_recording (GMythScheduler * scheduler,
                                gint channel_id)
{
    GMythProgramInfo *program;
    GMythSocket    *socket;
    gboolean        res = FALSE;
    GMythStringList *slist;

    socket = gmyth_backend_info_get_connected_socket (scheduler->backend_info);
    program = gmyth_scheduler_get_recorded_on_time (scheduler, channel_id);

    if (program) {
        slist = gmyth_string_list_new();
        gmyth_string_list_append_char_array(slist, "STOP_RECORDING");

        gmyth_string_list_append_string(slist, program->title);        /* 0 */
        gmyth_string_list_append_string(slist, program->subtitle);     /* 1 */
        gmyth_string_list_append_string(slist, program->description);  /* 2 */
        gmyth_string_list_append_string(slist, program->category);     /* 3 */
        gmyth_string_list_append_int(slist, program->channel_id);   /* 4 */
        gmyth_string_list_append_string(slist, program->chanstr);      /* 5 */
        gmyth_string_list_append_string(slist, program->chansign);     /* 6 */
        gmyth_string_list_append_string(slist, program->channame);     /* 7 */
        gmyth_string_list_append_string(slist, program->pathname);     /* 8 */
        gmyth_string_list_append_int64(slist, program->filesize);      /* 9 */

        if (program->startts)
            gmyth_string_list_append_int(slist, program->startts->tv_sec); /* 10 */
        else
            gmyth_string_list_append_int(slist, 0);

        if (program->endts)
            gmyth_string_list_append_int(slist, program->endts->tv_sec);   /* 11 */
        else
            gmyth_string_list_append_int(slist, 0);

        gmyth_string_list_append_int(slist, program->duplicate);   /* 12 */
        gmyth_string_list_append_int(slist, program->shareable);   /* 13 */
        gmyth_string_list_append_int(slist, program->findid);      /* 14 */
        gmyth_string_list_append_string(slist, program->hostname); /* 15 */
        gmyth_string_list_append_int(slist, program->sourceid);    /* 16 */
        gmyth_string_list_append_int(slist, program->cardid);      /* 17 */
        gmyth_string_list_append_int(slist, program->inputid);     /* 18 */
        gmyth_string_list_append_int(slist, program->recpriority); /* 19 */

        // recstatus == recording
        gmyth_string_list_append_int(slist, -3);                   /* 20 */

        gmyth_string_list_append_int(slist, program->recordid);    /* 21 */
        gmyth_string_list_append_int(slist, program->rectype);     /* 22 */
        gmyth_string_list_append_int(slist, program->dupin);       /* 23 */
        gmyth_string_list_append_int(slist, program->dupmethod);   /* 24 */


        //fixme
        program->recstartts->tv_sec -= (60*60);

        gmyth_string_list_append_int(slist, 
                                     program->recstartts != NULL ?
                                     program->recstartts->tv_sec : 0);   /* 26 */

        gmyth_string_list_append_int(slist, 
                                     program->recendts != NULL ?
                                     program->recendts->tv_sec : 0);    /* 27 */

        gmyth_string_list_append_int(slist, program->repeat);       /* 28 */
        gmyth_string_list_append_int(slist, program->programflags); /* 29 */

        gmyth_string_list_append_char_array(slist, 
                                            program->recgroup != NULL ?
                                            program->recgroup->str : "Default");  /* 30 */

        gmyth_string_list_append_int(slist, program->chancommfree);         /* 31 */
        gmyth_string_list_append_string(slist, program->chanOutputFilters); /* 32 */
        gmyth_string_list_append_string(slist, program->seriesid);          /* 33 */
        gmyth_string_list_append_string(slist, program->program_id);     /* 34 */

        gmyth_string_list_append_int(slist,
                                     program->lastmodified != NULL ?
                                     program->lastmodified->tv_sec : 0);   /* 35 */

        gmyth_string_list_append_float(slist, program->stars); /* 36 */

        gmyth_string_list_append_int(slist,
                                     program->originalAirDate != NULL ?
                                     program->originalAirDate->tv_sec : 0); /* 37 */

        gmyth_string_list_append_int(slist, program->hasAirDate);  /* 38 */

        gmyth_string_list_append_char_array(slist,
                                            program->playgroup != NULL ?
                                            program->playgroup->str : "Default");  /* 39 */

        gmyth_string_list_append_int(slist, program->recpriority2); /* 40 */
        gmyth_string_list_append_int(slist, program->recpriority2); /* 40 */

        gmyth_socket_sendreceive_stringlist(socket, slist);
        res = (gmyth_string_list_get_int(slist, 0) == 1);

        g_object_unref (program);
        g_object_unref (slist);
    }

    g_object_unref (socket);
    return res;
}


/** Notifies the backend of an update in the db.
 * 
 * @param record_id the id of the modified recording.
 */
// fixme: put void and discovery record_id inside
static gboolean
update_backend (GMythScheduler * scheduler, 
                gint record_id)  
{
    GMythSocket    *socket;
    GMythStringList *strlist = gmyth_string_list_new();
    GString        *datastr = g_string_new("RESCHEDULE_RECORDINGS ");
    gboolean        ret = FALSE;

    g_string_append_printf(datastr, "%d", record_id);
    gmyth_string_list_append_string(strlist, datastr);

    socket = gmyth_backend_info_get_connected_socket (scheduler->backend_info);
    if (socket != NULL) { 
        ret = (gmyth_socket_sendreceive_stringlist(socket, strlist) > 0);
        g_object_unref (socket);
    } else {
        g_warning("[%s] Connection to backend failed!", __FUNCTION__);
    }

    g_string_free(datastr, TRUE);
    g_object_unref(strlist);
    return ret;
}

void
gmyth_scheduler_recorded_info_get_preview(RecordedInfo * info,
                                          GByteArray * data)
{
}

void
gmyth_recorded_info_free(RecordedInfo * info)
{
    g_return_if_fail (info != NULL);

    if (info->program_id)
        g_string_free (info->program_id, TRUE);

    if (info->title != NULL)
        g_string_free(info->title, TRUE);

    if (info->subtitle != NULL)
        g_string_free(info->subtitle, TRUE);

    if (info->description != NULL)
        g_string_free(info->description, TRUE);

    if (info->category != NULL)
        g_string_free(info->category, TRUE);

    if (info->basename != NULL)
        g_string_free(info->basename, TRUE);

    if (info != NULL)
        g_free(info->start_time);

    if (info != NULL)
        g_free(info->end_time);

    g_free(info);
}

static void
free_recorded_info_item(gpointer data, gpointer user_data)
{
    RecordedInfo   *info = (RecordedInfo *) data;

    gmyth_recorded_info_free(info);
}

void
gmyth_recorded_info_list_free(GList * list)
{
    g_return_if_fail(list != NULL);

    g_list_foreach(list, free_recorded_info_item, NULL);
    g_list_free(list);
}

void
gmyth_schedule_info_free(ScheduleInfo * info)
{

    g_return_if_fail(info != NULL);

    if (info->program_id)
        g_string_free (info->program_id, TRUE);

    if (info->title != NULL)
        g_string_free(info->title, TRUE);

    if (info->subtitle != NULL)
        g_string_free(info->subtitle, TRUE);

    if (info->description != NULL)
        g_string_free(info->description, TRUE);

    if (info->category != NULL)
        g_string_free(info->category, TRUE);

    if (info != NULL)
        g_free(info->start_time);

    if (info != NULL)
        g_free(info->end_time);

    g_free(info);
}

static void
free_schedule_info_item(gpointer data, gpointer user_data)
{
    ScheduleInfo   *info = (ScheduleInfo *) data;

    gmyth_schedule_info_free(info);
}

void
gmyth_schedule_info_list_free(GList * list)
{
    g_return_if_fail(list != NULL);

    g_list_foreach(list, free_schedule_info_item, NULL);
    g_list_free(list);
}
